(**************************************************************************
  *********                     tutil.ml                          *********
  **************************************************************************)

open Generic;;
open Term;;
open Names;;
open Std;;
open Pp;;
open Vectops;;
open Util;;

let map_oper f_op t =
 let rec f =
  function
     | DOP0 op -> DOP0 (f_op op)
     | DOP1 (op, t1) -> DOP1 (f_op op, f t1)
     | DOP2 (op, t1, t2) -> DOP2 (f_op op, f t1, f t2)
     | DOPN (op, v) -> DOPN (f_op op, Array.map f v)
     | DOPL (op, l) -> DOPL (f_op op, List.map f l)
     | DLAM (na, t) -> DLAM (na, f t)
     | DLAMV (na, v) -> DLAMV (na, Array.map f v)
     | VAR id -> VAR id
     | Rel i -> Rel i in
 f t;;

(***************************************************************************)
let apply_to_oper f_op t =
 match t with
 | DOP0 op ->
  let op' = f_op op in
  if op == op' then t
   else DOP0 op'
 | DOP1 (op, t1) ->
  let op' = f_op op in
  if op == op' then t
   else DOP1 (op', t1)
 | DOP2 (op, t1, t2) ->
  let op' = f_op op in
  if op == op' then t
   else DOP2 (op', t1, t2)
 | DOPN (op, v) ->
  let op' = f_op op in
  if op == op' then t
   else DOPN (op', v)
 | DOPL (op, l) ->
  let op' = f_op op in
  if op == op' then t
   else DOPL (op', l)
 | _ -> t;;

(***************************************************************************)
let rec apply_to_sons f t =
 match t with
 | DOP1 (op, t1) ->
  let t1' = f t1 in
  if t1 == t1' then t
   else DOP1 (op, t1')
 | DOP2 (op, t1, t2) ->
  let t1' = f t1
  and t2' = f t2 in
  if t1 == t1' & t2 == t2' then t
   else DOP2 (op, t1', t2')
 | DOPN (op, v) ->
  let v' = Array.map f v in
  if for_all_vect2 ( (==)) v v' then t
   else DOPN (op, v')
 | DOPL (op, l) ->
  let l' = List.map f l in
  if for_all2 ( (==)) l l' then t
   else DOPL (op, l')
 | DLAM (na, t1) ->
  let t1' = f t1 in
  if t1 == t1' then t
   else DLAM (na, t1')
 | DLAMV (na, v) ->
  let v' = Array.map f v in
  if for_all_vect2 ( (==)) v v' then t
   else DLAMV (na, v')
 | _ -> t;;

let rec apply_top_down f c = apply_to_sons (apply_top_down f) (f c);;

let rec apply_down_top f c = f (apply_to_sons (apply_down_top f) c);;

(***************************************************************************)
let do_to_sons f =
 function
    | DOP1 (_, c1) -> f c1; ()
    | DOP2 (_, c1, c2) ->
     f c1;
     f c2;
     ()
    | DOPN (_, cv) -> Array.iter f cv
    | DOPL (_, cl) -> List.iter f cl
    | DLAM (_, c') -> f c'; ()
    | DLAMV (_, cv) -> Array.iter f cv
    | _ -> ();;

let rec do_top_down f c =
 f c; do_to_sons (do_top_down f) c;;

let rec do_down_top f c =
 do_to_sons (do_down_top f) c;
 f c;
 ();;

let def_id = id_of_string "_";;

let rec f_R_to_V idl =
 function
    | Rel i -> begin
      try VAR (List.nth idl (i - 1))
      with
      | Failure "nth" -> Rel i
    end
    | VAR id -> VAR id
    | DOP0 op -> DOP0 op
    | DOP1 (op, c1) -> DOP1 (op, f_R_to_V idl c1)
    | DOP2 (op, c1, c2) -> DOP2 (op, f_R_to_V idl c1, f_R_to_V idl c2)
    | DOPL (op, l) -> DOPL (op, List.map (f_R_to_V idl) l)
    | DOPN (op, v) -> DOPN (op, Array.map (f_R_to_V idl) v)
    | DLAM ((Name id), c) ->
     let id = next_ident_away id idl in
     DLAM (Name id, f_R_to_V (id::idl) c)
    | DLAM (Anonymous, c) -> DLAM (Anonymous, f_R_to_V (def_id::idl) c)
    | DLAMV ((Name id), v) ->
     let id = next_ident_away id idl in
     DLAMV (Name id, Array.map (f_R_to_V (id::idl)) v)
    | DLAMV (Anonymous, v) ->
     DLAMV (Anonymous, Array.map (f_R_to_V (def_id::idl)) v);;

let rec f_V_to_R idl =
 function
    | Rel i -> Rel i
    | VAR id -> begin
      try Rel (index id idl)
      with
      | Failure "index" -> VAR id
    end
    | DOP0 op -> DOP0 op
    | DOP1 (op, c1) -> DOP1 (op, f_V_to_R idl c1)
    | DOP2 (op, c1, c2) -> DOP2 (op, f_V_to_R idl c1, f_V_to_R idl c2)
    | DOPL (op, l) -> DOPL (op, List.map (f_V_to_R idl) l)
    | DOPN (op, v) -> DOPN (op, Array.map (f_V_to_R idl) v)
    | DLAM ((Name id), c) -> DLAM (Name id, f_V_to_R (id::idl) c)
    | DLAM (Anonymous, c) -> DLAM (Anonymous, f_V_to_R (def_id::idl) c)
    | DLAMV ((Name id), v) -> DLAMV (Name id, Array.map (f_V_to_R (id::idl)) v)
    | DLAMV (Anonymous, v) ->
     DLAMV (Anonymous, Array.map (f_V_to_R (def_id::idl)) v);;

let from_Rel_to_VAR c = f_R_to_V [] c;;

let from_VAR_to_Rel c = f_V_to_R [] c;;

(***************************************************************************)
let rec apply_head =
 function
    | DOP2 (Cast, c, _) -> apply_head c
    | DOPN (AppL, v) -> apply_head v.(0)
    | c -> c;;

let rec count_prod =
 function
    | DOP2 (Prod, _, (DLAM (_, c'))) -> 1 + count_prod c'
    | DOP2 (Cast, c', _) -> count_prod c'
    | _ -> 0;;

let rec strip_outer_app_cast =
 function
    | DOP2 (Cast, c, _) -> strip_outer_app_cast c
    | DOPN (AppL, v) ->
     (match Array.length v with
     | 0 ->
      errorlabstrm "tutil__strip_outer_app_cast"
       [< 'sTR "malformed AppL with no sons" >]
     | 1 -> strip_outer_app_cast v.(0)
     | l ->
      let head = strip_outer_app_cast v.(0)
      and args = Array.sub v 1 (l - 1) in
      (match head with
       | DOPN (AppL, v) -> DOPN (AppL, Array.append v args)
       | _ -> DOPN (AppL, cons_vect head args)))
    | c -> c;;

(***************************************************************************)
let select_list_of_vect f v =
 let l = ref [] in
 let f x = if f x then l:=x::!l in
 for i = Array.length v - 1 downto 0 do f v.(i) done; !l;;

let rec select_list_fisrt f =
 function
    | x :: l -> if f x then x
                 else select_list_fisrt f l
    | [] -> raise Not_found;;

let select_vect_first f v =
 let n = Array.length v in
 let rec f_rec i = if i < n then (if f v.(i) then i
  else f_rec (i + 1))
  else raise Not_found in
 v.(f_rec 0);;

let flat_map_list_of_vect f v =
 let n = Array.length v in
 let rec f_rec i = if i < n then f v.(i) @ f_rec (i + 1)
  else [] in
 f_rec 0;;

