(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                machops.ml                                *)
(****************************************************************************)

open Std;;
open Names;;
open Vectops;;
open Impuniv;;
open Generic;;
open Evd;;
open Term;;
open Reduction;;
open Environ;;
open Typing;;
open Termenv;;
open Himsg;;
open Pp;;
open Extraction;;

(* Functions for interpreting a constant, initially in  termenv.ml 
   using eq_constr instead of conv - Christine *)

let search_var env str = VAR(fst(lookup_glob str env));;


(* this function verifies that HL1 is a sub-sequence of HL2, by structural
 * equality on members.  Members must be found in the same order - that
 * is the meaning of "sequence" here.
 *)
let hyp_seq_ordered_inclusion (idl1,tyl1) (idl2,tyl2) =
  let mt = Evd.mt_evd() in
  let rec aux = function
      ([], [], _, _) -> true
    | (_, _, [], []) -> false
    | ((id1::idl1), (ty1::tyl1), idl2, tyl2) ->
        let rec search = function
            ([], []) -> false
          | ((id2::idl2), (ty2::tyl2)) ->
              if id1 = id2
              then (conv mt (body_of_type ty1) (body_of_type ty2))
		& aux (idl1,tyl1,idl2,tyl2)
              else search (idl2,tyl2)
            | (_, _) -> invalid_arg "hyp_seq_ordered_inclusion"
        in search (idl2,tyl2)

    | (_, _, _, _) -> invalid_arg "hyp_seq_ordered_inclusion"
  in aux (idl1,tyl1,idl2,tyl2) 
;;

(* ici, oper est Const, MutConstruct ou MutInd *)
let construct_reference id env (oper,hyps) =
  if hyp_seq_ordered_inclusion hyps (get_globals env) then
    DOPN(oper,Array.of_list(List.map (fun id -> VAR id) (ids_of_sign hyps)))
  else errorlabstrm
         "construct_reference" 
         [<'sTR ("the constant  "^(string_of_id id)^
      	         " referred to variables which are not in the context")>]
;;

let global_reference env sp id =
    try construct_reference (basename sp) env (global_operator sp id)
    with Not_found -> 
            errorlabstrm 
               "global_reference"
               [<'sTR ("Cannot find reference "^(string_of_path sp))>]
;;

(* comme global_reference, mais retourne en plus les args implicites *)

let global_reference1 env sp id =
    try let x,l = (global_operator1 sp id) in
        (construct_reference (basename sp) env x),l
    with Not_found -> 
            errorlabstrm 
               "global_reference"
               [<'sTR ("Cannot find reference "^(string_of_path sp))>]
;;

(* search_reference env str gives the constr of name str
 * in the environment env *)
let search_reference env str =
  let sp = (* try *) Nametab.sp_of_id CCI str
           (* with Not_found -> error "search_reference" *)
  in global_reference env sp str
;;

(* comme search_reference, mais retourne en plus les args implicites *)

let search_reference1 env str =
  let sp = (* try *) Nametab.sp_of_id CCI str
           (* with Not_found -> error "search_reference" *)
  in global_reference1 env sp str
;;

let search_freference env str =
  let sp = (* try *) Nametab.sp_of_id FW str
           (* with Not_found -> error "search_freference" *)
  in global_reference env sp str
;;

(* comme search_freference, mais retourne en plus les args implicites *)

let search_freference1 env str =
  let sp = (* try *) Nametab.sp_of_id FW str
           (* with Not_found -> error "search_reference" *)
  in global_reference1 env sp str
;;

let search env id =
    try search_var env id
    with Not_found -> search_reference env id
;;

let global env id =
    try search env id
    with Not_found -> errorlabstrm "Machops.global"
      	       	       	       	[< 'sTR((string_of_id id) ^ " not declared") >]
;;

let lookup_exist sigma env sp =
  try let evd = Evd.map sigma sp
  in if hyp_seq_ordered_inclusion evd.hyps (get_globals env)
     then DOPN(Const sp,
	     Array.of_list(List.map (fun id -> VAR id) (ids_of_sign evd.hyps)))
     else errorlabstrm "Machops.lookup_exist"
       [< 'sTR"Hum - cannot instanciate the constant";'sPC;
          'sTR(string_of_path sp);'sPC;
          'sTR"because there weren't enough assumptions">]
with Not_found -> errorlabstrm "Machops.lookup_exist"
                    [< 'sTR (string_of_path sp); 'sPC; 'sTR "not declared">]
;;

let type_of_existential b sigma env k = 
  let (sp,args) = destConst k in
  try let evd = Evd.map sigma sp in
      if for_all2eq_vect 
          (fun id var -> var = VAR id)
          (Array.of_list (ids_of_sign evd.hyps)) args
          & hyp_seq_ordered_inclusion evd.hyps (get_globals env)
      then evd.concl
      else if b
          then invalid_arg "An existential was found with a bad argument list"
          else const_type sigma k
  with Not_found -> error ((string_of_path sp) ^ " not declared")
     | Failure _ -> invalid_arg "the current environment and the constant's env of construction are incompatible"
;;


let cast_fully = ref false;;
let j_val j =
  if !cast_fully then mkCast j._VAL (mkCast j._TYPE j._KIND)
  else j._VAL;;

let j_val_cast j = mkCast j._VAL j._TYPE;;
let j_val_only j = j._VAL;;


let assumption_of_judgement sigma env j =
  let typ = whd_betadeltaiota sigma j._TYPE in
    match typ with
        DOP0(Sort s) -> {body = j._VAL; typ=s}
      | _ -> error_assumption CCI env j._VAL;;

let type_judgement sigma env j =
  let typ = whd_betadeltaiota sigma j._TYPE in
    match typ with
        DOP0(Sort s) -> {body = j._VAL; typ = s}
      | _ -> error_not_type CCI env j._VAL;;

let fcn_proposition cts =
    {_VAL=DOP0(Sort(Prop cts));
     _TYPE=DOP0(Sort type_0);
     _KIND = DOP0(Sort type_1)};;

let fcn_type_with_universe u =
   let v = super u in
   let sv = super v in 
     {_VAL=DOP0(Sort(Type u));
      _TYPE=DOP0(Sort(Type v));
      _KIND = DOP0(Sort(Type sv))};;

(* F| Execution of a construction *)

let gen_rel sigma k env name var j =
  let jtyp = whd_betadeltaiota sigma j._TYPE in
  let jkind = whd_betadeltaiota sigma j._KIND in
  let j = {_VAL = j._VAL; _TYPE = jtyp; _KIND = jkind} in
    if isprop jkind then error "Proof objects can only be abstracted" 
    else
      match jtyp with
        DOP0(Sort s) ->
          let res_type = DOP0 (Sort (sort_of_product var.typ s)) in
            {_VAL = mkProd name (mkCast var.body (DOP0 (Sort var.typ))) (j_val_cast j);
             _TYPE = res_type;
             _KIND = type_of_sort res_type}
      | _ -> error_generalization k env (name,var) j
;;

let abs_rel sigma name var j =
  let rngtyp = whd_betadeltaiota sigma j._KIND in
  let cvar = incast_type var in
    {_VAL = mkLambda name cvar (j_val j);
     _TYPE = mkProd name cvar j._TYPE;
     _KIND = mkSort (sort_of_product var.typ (destSort rngtyp))};;

let abs_var id var (c,typ) =
  let name = Name id in
  let cvar = incast_type var in
  let c'   = mkLambda name cvar (subst_var id c) in
  let typ' =
    {body = mkProd name cvar (subst_var id typ.body);
     typ  = sort_of_product var.typ typ.typ} in
  (c',typ');;

let abs_var_type id var typ =
  {body = mkProd (Name id) (incast_type var) (subst_var id typ.body);
   typ  = sort_of_product var.typ typ.typ};;


(* We cannot assume, as in the normal machine operation, that the 
   types of variables are closed terms, and thus we must lift them *)

(* Avant: le _TYPE restait cast *)
let relative sigma n env = 
  try
    let (_,typ) = lookup_rel n env
    in {_VAL  = Rel(n);
        _TYPE = lift n typ.body;
        _KIND = DOP0(Sort typ.typ)}
  with Not_found -> error_unbound_rel CCI env n;;


let apply_rel_list sigma env nocheck argjl funj =
  let rec apply_rec typ = function
      [] -> {_VAL  = applist(j_val_only funj,List.map j_val_only argjl);
             _TYPE = typ;
             _KIND = funj._KIND}
    | hj::restjl ->
        match whd_betadeltaiota sigma typ with
            DOP2(Prod,c1,DLAM(_,c2)) ->
              if nocheck or conv_leq sigma hj._TYPE c1 then
                apply_rec (subst1 hj._VAL c2) restjl
              else error_cant_apply "Type Error" CCI env funj argjl
          | _ ->
              error_cant_apply "Non-functional construction" CCI env funj argjl
  in apply_rec funj._TYPE argjl;;

let cast_rel sigma env cj tj =
    if conv_leq sigma cj._TYPE tj._VAL then
        {_VAL=j_val_only cj;
         _TYPE=tj._VAL;
         _KIND = whd_betadeltaiota sigma tj._TYPE}
    else error_actual_type CCI env cj tj
;;

let type_of_constant sigma env k =
  let (sp,cl) = destConst k in
    match global_reference env sp (basename sp) with
	(DOPN(_,cl') as k') ->
	  if for_all2eq_vect (conv_x sigma) cl cl' then
            const_type sigma k
	  else error ("the current environment and the constant's env of construction are incompatible")
      | _ -> assert false
;;

(* type_of_const existait aussi dans trad.ml avec un comportement
   legerement different de type_of_existential (pourquoi ?). Maintenant,
   c'est le booleen qui fait la difference *)

let type_of_const_aux b sigma env k = 
  let (sp,_) = destConst k in
    if is_existential_id (basename sp)
    then type_of_existential b sigma env k
    else type_of_constant sigma env k
;;

let type_of_const sigma env t = type_of_const_aux true sigma env t;;
let type_of_const2 sigma env t = type_of_const_aux false sigma env t;;

let inf_of_const sigma (env,fenv) k = 
  let (sp, args) = destConst k in
  let gl = Const sp in
(*  (DOPN(Const sp as gl,args) as k) *)
    try (let (_,cb) = Environ.const_of_path sp in  
	   if informative_reference k then
             let fsp = fwsp_of sp in
             let infK = global_reference fenv fsp (id_of_global gl) in
             let (typ,kind) = destCast(const_type sigma infK) in
               Inf{_VAL= infK;
		   _TYPE = typ;
		   _KIND = kind}
           else Logic)
    with Not_found -> error ((string_of_path sp) ^ " not declared")
      | Failure _ -> invalid_arg "cannot convert a constant to an fconstant"
;;

let type_of_mconstr sigma env m = 
  let (x,y,i,cl) = destMutConstruct m in
  let gl = MutConstruct((x,y),i) in
  let i_0 = DOPN(MutInd (x,y),cl) in
    match global_reference env x (id_of_global gl) with
      | DOPN(_,cl') as i' ->
	  if for_all2eq_vect (conv_x sigma) cl cl' then
            type_mconstruct sigma i i_0
	  else error ("the current environment and the constant's env of construction are incompatible")
      | _ -> assert false
;;

let type_of_mind sigma env i = 
  let (sp, tyi, cl) = destMutInd i in
  let gl = MutInd (sp, tyi) in
    match global_reference env sp (id_of_global gl) with
	DOPN(_,cl') as i' ->
	  if for_all2eq_vect (conv_x sigma) cl cl' then
            outcast_type (* sigma *)(mind_arity i)
(* HH 6/99
	  else error ("the current environment and the constant's env of construction are incompatible")*)
	  else anomaly ("the current environment and the constant's env of construction are incompatible")
      | _ -> assert false
;;

let type_of_case sigma env pj cj lfj =
  let lft = Array.map (fun j -> j._TYPE) lfj in
  let (mind,bty,rslty) =
    type_case_branches env sigma cj._TYPE pj._TYPE pj._VAL cj._VAL in
  let kind = sort_of_arity sigma pj._TYPE in
    check_branches_message env sigma (cj._VAL,cj._TYPE) (bty,lft);
    {_VAL  = mkMutCaseA (ci_of_mind mind) (j_val pj) (j_val cj)
              (Array.map j_val lfj);
     _TYPE  = rslty;
     _KIND = kind}
;;

(*
let verify_wf_env env =
    if not (List.for_all isCast (vals_of_sign (get_globals env)))
      or not (List.for_all (fun (_,t) -> isCast t) (get_rels env))
    then invalid_arg "the constructive engine was fed a malformed environment"
;;
*)

let verify_wf_env env = ()
;;

let cast_of_judgement j = 
  let jv = j._VAL in
  match jv with 
    | DOP2(Cast,_,_) -> jv 
    | _              -> mkCast jv j._TYPE;;

(* $Id: machops.ml,v 1.17 1999/10/07 13:23:59 herbelin Exp $ *)
