(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                refiner.ml                                *)
(****************************************************************************)

open Std;;
open More_util;;
open Printer;;
open Initial;;
open Names;;
open Vectops;;
open Generic;;
open Evd;;
open Term;;
open Reduction;;
open Himsg;;
open Termenv;;
open Mach;;
open Proof_trees;;
open Pp;;
open Logic;;

type 'a sigma = {it : 'a ; sigma : global_constraints};;

type validation = (proof_tree list -> proof_tree);;
type tactic = goal sigma -> (goal list sigma * validation);;
type transformation_tactic = proof_tree -> (goal list * validation);;
type validation_list = proof_tree list -> proof_tree list;;

type tactic_list = (goal list sigma) -> 
                    (goal list sigma) * validation_list;;

(* hypotheses : 'a goalty -> constr signature *)
let hypotheses gl = gl.hyps;;
(* conclusion : 'a goalty -> constr *)
let conclusion gl = gl.concl;;

(* sig_it : 'a sigma -> 'a*)
let sig_it x = x.it;;
(* sig_sig : 'a sigma -> global_constraints *)
let sig_sig x = x.sigma;;

(*let Project gls =
    abs_rc(ts_mk {focus = gls.it.lc;
                  sign = gls.it.hyps;
                  decls = ts_it (repr_gc gls.sigma)});;
*)


(* Projects takes a goal sigma
 { it=(gls:Proof.goal) ; sigma = (gc:Proof.global_constraints) }
and gives readable_constraints, where
focus = local constraints of gls
sign = signature of gls
decls = existential var declarations of gc
*)
let project_with_focus gls = rc_of_gc (gls.sigma) (gls.it);;

let pf_status pf = pf.status;;

let is_complete pf = (COMPLETE_PROOF = (pf_status pf));;

let on_open_proofs f pf = if is_complete pf then pf else f pf;;

let rec and_status = function
    [] -> COMPLETE_PROOF
  | COMPLETE_PROOF::l -> and_status l
  | _ -> INCOMPLETE_PROOF
;;


(* Normalizing evars in a proof tree. Called when the sigma of
   the proof tree changes. Does normalise all the goals it finds: maybe
   a bit heavy, but we must keep the invariant on trees (the goals of the
   frontier of the subproof are equal to the goals of the ref field),
   otherwise frontier will fail.
   Rq: does not normalize the tactics, so Show Proof may still yield
   a term with solved evars in it.
 *)
let norm_goal sigma gl =
  let red_fun = nf_ise1 sigma in
  let ncl = red_fun gl.Evd.concl in
  { Evd.concl = ncl;
    Evd.hyps = map_sign_typ (type_app red_fun) gl.Evd.hyps;
    Evd.body = gl.Evd.body;
    Evd.info = gl.Evd.info }
;;
let rec norm_evar_pf sigma p =
  let new_sprf =
    match p.subproof with
    | None -> None
    | Some spf -> Some (norm_evar_pf sigma spf) in
  let new_ref =
    match p.ref with
    | None -> None
    | Some(r,pfl) -> Some(r, List.map (norm_evar_pf sigma) pfl) in
  { status = p.status;
    subproof = new_sprf;
    goal = norm_goal sigma p.goal;
    ref = new_ref };;


(* mapshape [ l1 ; ... ; lk ] [ v1 ; ... ; vk ] [ p_1 ; .... ; p_(l1+...+lk) ]
gives
[ (v1 [p_1 ... p_l1]) ; (v2 [ p_(l1+1) ... p_(l1+l2) ]) ; ... ;
(vk [ p_(l1+...+l(k-1)+1) ... p_(l1+...lk) ])
]
*)
let rec mapshape nl (fl:((proof_tree list) -> proof_tree) list) (l:proof_tree list) =
    match nl with
    [] -> []
  | h::t ->
    let m,l = chop_list h l
    in (List.hd fl m)::(mapshape t (List.tl fl) l)
;;

(* given a proof p, frontier p gives (l,v) where l is the list of goals
to be solved to complete the proof, and v is the corresponding validation *)
let rec frontier p =
  match p.ref with
      None -> ([p.goal],
	       (fun lp' -> 
		  let p' = List.hd lp' in
                    if p'.goal = p.goal then p'
                    else errorlabstrm "Refiner.frontier"
                      [< 'sTR"frontier was handed back a ill-formed proof." >]
	       ))
    | Some(r,pfl) ->
    	let gll,vl = List.split(List.map frontier pfl) in
    	  (List.flatten gll,
           (fun retpfl ->
              let pfl' = mapshape (List.map List.length gll) vl retpfl
              in {status = and_status (List.map pf_status pfl');
                  subproof=p.subproof;
                  goal = p.goal;
                  ref = Some(r,pfl')}
	   ))
;;

(* list_pf p is the lists of goals to be solved in order to complete the
proof p *)
let list_pf p = fst(frontier p);;

let rec nb_unsolved_goals pf = 
if is_complete pf then 0 else 
   if is_leaf_proof pf then 1 else 
     let lpf = children_of_proof pf in 
       List.fold_left 
	 (fun n pf1 -> n + nb_unsolved_goals pf1)
	 0 lpf
;;

(* leaf g is the canonical incomplete proof of a goal g *)
let leaf g = {status=INCOMPLETE_PROOF;
              subproof=None;
              goal=g;
              ref=None};;



(* Tactic table *)
let tac_tab = (Mhm.create 17);;

let lookup_tactic s =
    try  (Mhm.map tac_tab s)
    with Not_found -> errorlabstrm "Refiner.lookup_tactic"
                      [< 'sTR"The tactic " ; 'sTR s ; 'sTR" is not installed" >]
;;

let pr_tactic (s,l) =
      gentacpr (Ast.ope (s,(List.map ast_of_cvt_arg l)))
;;

(* refiner r is a tactic applying the rule r *)
let refiner r =
  match r with
    PRIM pr ->
      let prim_fun = prim_refiner pr in
        (fun goal_sigma ->
           let sgl = 
             (try prim_fun (ts_it goal_sigma.sigma) goal_sigma.it
              with UserError _ as e -> raise e
                | e -> (mSGNL(Errors.explain_user_exn e);
                        Errors.reraise_user_exn e))

           in ({it=sgl; sigma = goal_sigma.sigma},
               (fun pfl ->
                  if for_all2eq (fun g pf -> g = pf.goal) sgl pfl then
                    {status = and_status (List.map pf_status pfl);
                     goal = goal_sigma.it;
                     subproof = None;
                     ref = Some(r,pfl)}
                  else errorlabstrm "Refiner.refiner"
                    [< 'sTR"Bad subproof in validation.">])))


  | TACTIC(s,targs) ->
      let tacfun = lookup_tactic s targs in
        (fun goal_sigma ->
           let (sgl_sigma,v) = tacfun goal_sigma in
           let hidden_proof = v (List.map leaf sgl_sigma.it)
           in (sgl_sigma,
               fun spfl ->
                 if for_all2eq (fun g pf -> g = pf.goal) sgl_sigma.it spfl then
                   {status=and_status (List.map pf_status spfl);
                    goal=goal_sigma.it;
                    subproof=Some hidden_proof;
                    ref=Some(r,spfl)}
                 else errorlabstrm "Refiner.refiner"
                   [< 'sTR"Bad subproof in validation.">]))

   | ((CONTEXT ctxt) as r) ->
       (fun goal_sigma ->
          let gl = goal_sigma.it in
          let sg = mkGOAL ctxt gl.hyps gl.concl
          in ({it=[sg];sigma=goal_sigma.sigma},
              (fun pfl -> 
		 let pf = List.hd pfl in
                   if pf.goal = sg then
                     {status=pf.status;
                      goal=gl;
                      subproof=None;
                      ref=Some(r,[pf])}
                   else errorlabstrm "Refiner.refiner"
                     [< 'sTR "Bad subproof in validation.">]
	      )))

   (* [LOCAL_CONSTRAINTS lc] makes the local constraints be [lc] *)

   | ((LOCAL_CONSTRAINTS lc) as r) ->
       (fun goal_sigma ->
          let gl = goal_sigma.it  in
          let ctxt = outSOME gl.info in 
          let sg = mkGOAL ctxt gl.hyps gl.concl
          in ({it=[sg];sigma=goal_sigma.sigma},
              (fun pfl -> 
		 let pf = List.hd pfl in
                   if pf.goal = sg then
                     {status=pf.status;
                      goal=gl;
                      subproof=None;
                      ref=Some(r,[pf])}
                   else errorlabstrm "Refiner.refiner"
                     [<'sTR"Bad subproof in validation.">]
	     )))
;;

(* rc_of_pfsigma : proof sigma -> readable_constraints *)
let rc_of_pfsigma sigma = rc_of_gc sigma.sigma sigma.it.goal;;
(* rc_of_glsigma : proof sigma -> readable_constraints *)
let rc_of_glsigma sigma = rc_of_gc sigma.sigma sigma.it;;




(* extract_open_proof : constr signature -> proof
-> constr * (int * constr) list
takes a constr signature corresponding to global definitions
and a (not necessarly complete) proof
and gives a pair (pfterm,obl)
where pfterm is the constr corresponding to the proof
and obl is an int*constr list [ (m1,c1) ; ... ; (mn,cn)]
where the mi are metavariables numbers, and ci are their types.
Their proof should be completed in order to complete the initial proof
*)

let extract_open_proof sign pf =

let open_obligations = ref [] in

let rec proof_extractor vl = function

    {ref=Some(PRIM _,_)} as pf -> prim_extractor proof_extractor vl pf

  | {ref=Some(TACTIC _,spfl); subproof=Some hidden_proof} ->
      let sgl,v = frontier hidden_proof in
      let flat_proof = v spfl in
     	proof_extractor vl flat_proof

  | {ref=Some(CONTEXT ctxt,[pf])} -> (proof_extractor vl) pf

  | {ref=Some(LOCAL_CONSTRAINTS lc,[pf])} -> (proof_extractor vl) pf

  | {ref=None;goal=goal} ->
      let rel_env = get_rels vl in
      let n_rels = List.length rel_env in
      let visible_rels =
        map_succeed
          (fun id ->
             match lookup_id id vl with
		 GLOBNAME _ -> failwith "caught"
               | RELNAME(n,_) -> (n,id))
          (ids_of_sign goal.hyps) in
      let sorted_rels = 
	Sort.list (fun (n1,_) (n2,_) -> n1>n2) visible_rels in

      let abs_concl =
        List.fold_right
          (fun (_,id) concl -> 
	     mkNamedProd id (incast_type (snd(lookup_sign id goal.hyps))) concl)
          sorted_rels
          goal.concl in

      let mv = newMETA() in
      	open_obligations := (mv,abs_concl):: !open_obligations;
        applist(DOP0(Meta mv),List.map (fun (n,_) -> Rel n) sorted_rels) 


  | _ -> anomaly "Bug : a case has been forgotten in proof_extractor"

in
  
let pfterm = proof_extractor (gLOB sign) pf in
  
  (pfterm, List.rev !open_obligations)
;;


(* extracts a constr from a proof, and raises an error if the proof is
incomplete *)
let extract_proof sign pf =
    match extract_open_proof sign pf with
    t,[] -> t
  | _ ->
    errorlabstrm "extract_proof"
    [< 'sTR "Attempt to save an incomplete proof" >]
;;

let context ctxt = refiner (CONTEXT ctxt);;
let vernac_tactic texp = refiner (TACTIC texp);;

(* Adding new tactics to the table. *)
let add_tactic s t =
if Mhm.in_dom tac_tab s
  then
    errorlabstrm ("Refiner.add_tactic: "^s)
   [<'sTR "Cannot redeclare a tactic.">]
  else
    (Mhm.add tac_tab (s,t))
;;

let overwriting_add_tactic s t =
    if Mhm.in_dom tac_tab s then
        (Mhm.rmv tac_tab s;
         warning ("Overwriting definition of tactic "^s));
    Mhm.add tac_tab (s,t)
;;

(* hide_tactic s tac pr registers a tactic s under the name s *)
let hide_tactic s tac =
    (add_tactic s tac;
     (fun args -> vernac_tactic(s,args)))
;;

(* overwriting_register_tactic s tac pr registers a tactic s under the name
s even if a tactic of the same name is already registered *)
let overwrite_hidden_tactic s tac =
    (overwriting_add_tactic s tac;
     (fun args -> vernac_tactic(s,args)))
;;




(*********************)
(*   Tacticals       *)
(*********************)

(* unTAC : tactic -> goal sigma -> proof sigma *)
let unTAC tac g =
let (gl_sigma,v) = tac g
in {it=v(List.map leaf gl_sigma.it); sigma=gl_sigma.sigma};;

let unpackage glsig = (ref (glsig.sigma)),glsig.it;;
let repackage r v = {it=v;sigma = !r};;
let apply_sig_tac r tac g =
    let glsigma,v = tac (repackage r g)
    in (r := glsigma.sigma;
        (glsigma.it,v))
;;

let idtac_valid = function
    [pf] -> pf
  | _ -> anomaly "Refiner.idtac_valid"
;;

(* goal_goal_list : goal sigma -> goal list sigma *)
let goal_goal_list gls = {it=[gls.it];sigma=gls.sigma};;

(* the identity tactic *)
let tclIDTAC gls = (goal_goal_list gls, idtac_valid);;

(* General failure tactic *)
let tclFAIL_s s gls = errorlabstrm "Refiner.tclFAIL_s" [< 'sTR s>];;
let (tclFAIL:tactic) = tclFAIL_s "Failtac always fails.";;


let start_tac gls =
  let (sigr,g) = unpackage gls in
  (sigr,[g],idtac_valid)
;;

let finish_tac (sigr,gl,p) = (repackage sigr gl, p);;

let thens_tac tac2l taci (sigr,gs,p) =
  let (gl,gi) =
    try chop_list (List.length tac2l) gs
    with Failure _ -> errorlabstrm "Refiner.combine_tactics"
        [<'sTR "Wrong number of tactics.">] in
  let tac2gl = List.combine gl tac2l @ map_i (fun i g -> (g, taci i)) 0 gi in
  let gll,pl =
    List.split(List.map (fun (g,tac2) -> apply_sig_tac sigr tac2 g) tac2gl) in
  (sigr, List.flatten gll, comp p (mapshape (List.map List.length gll) pl))
;;

let then_tac tac = thens_tac [] (fun _ -> tac);;


let non_existent_goal n =
  errorlabstrm ("No such goal: "^(string_of_int n))
    [< 'sTR"Trying to apply a tactic to a non existent goal" >]
;;

(* Apply tac on the i-th goal (if i>0). If i<0, then start counting from
   the last goal (i=-1). *)
let theni_tac i tac ((_,gl,_) as subgoals) =
  let nsg = List.length gl in
  let k = if i < 0 then nsg + i else (i-1) in
  if nsg < 1 then errorlabstrm "theni_tac" [< 'sTR"No more subgoals.">]
  else if k >= 0 & k < nsg then
    thens_tac [] (fun i -> if i = k then tac else tclIDTAC) subgoals
  else non_existent_goal (k+1) 
;;


(* tclTHENSi tac1 [t1 ; ... ; tn] taci gls applies the tactic tac1 to gls and
   applies t1,..., tn to the n first resulting subgoals, and (taci i) to the
   (i+n+1)-th goal. Raises an error if the number of resulting subgoals
   is less than n *)
let tclTHENSi tac1 tac2l taci gls =
  finish_tac (thens_tac tac2l taci (then_tac tac1 (start_tac gls)))
;;

(* tclTHEN tac1 tac2 gls applies the tactic tac1 to gls and applies tac2 to
   every resulting subgoals *)
let tclTHEN tac1 tac2 = tclTHENSi tac1 [] (fun _ -> tac2);;

(* tclTHEN_i tac1 tac2 n gls applies the tactic tac1 to gls and applies
   tac2 (i+n-1) to the i_th resulting subgoal *)
let tclTHEN_i tac1 tac2 n = tclTHENSi tac1 [] (fun i -> tac2 (i+n));;

(* tclTHENS tac1 [t1 ; ... ; tn] gls applies the tactic tac1 to gls and applies
   t1,..., tn to the n resulting subgoals. Raises an error if the number of
   resulting subgoals is not n *)
let tclTHENS tac1 tac2l =
  tclTHENSi tac1 tac2l (fun _ -> tclFAIL_s "Wrong number of tactics.");;

(* Same as tclTHENS but completes with Idtac if the number resulting subgoals
  is strictly less than n *)
let tclTHENSI tac1 tac2l = tclTHENSi tac1 tac2l (fun _ -> tclIDTAC);;


(* tclTHENL tac1 tac2 gls applies the tactic tac1 to gls and tac2
   to the last resulting subgoal *)
let tclTHENL (tac1:tactic) (tac2:tactic) (gls:goal sigma) =
  finish_tac (theni_tac (-1) tac2 (then_tac tac1 (start_tac gls)))
;;

(* tclTHENLIST [t1;..;tn] applies t1;..tn. More convenient than tclTHEN
   when n is large. *)
let rec tclTHENLIST = function
    [] -> tclIDTAC
  | t1::tacl -> tclTHEN t1 (tclTHENLIST tacl)
;;



(* various progress criterions *)
let same_goal subgoal ptree = 
  (hypotheses subgoal) = (hypotheses ptree.it) &
  eq_constr (conclusion subgoal) (conclusion ptree.it) &
  (subgoal.info = ptree.it.info)
;;

let weak_progress gls ptree =
  (List.length gls.it <> 1) or 
  (not (same_goal (List.hd gls.it) ptree))
;;

let progress gls ptree =
  (weak_progress gls ptree) or
  (not (ts_eq ptree.sigma gls.sigma))
;;

(* PROGRESS tac ptree applies tac to the goal ptree and fails if tac leaves
the goal unchanged *)
let tclPROGRESS tac ptree =
  let rslt = tac ptree in
  if progress (fst rslt) ptree then rslt
  else errorlabstrm "Refiner.PROGRESS" [< 'sTR"Failed to progress.">]

(* weak_PROGRESS tac ptree applies tac to the goal ptree and fails 
   if tac leaves the goal unchanged, possibly modifying sigma *)
let tclWEAK_PROGRESS tac ptree =
  let rslt = tac ptree in
  if weak_progress (fst rslt) ptree then rslt
  else errorlabstrm "Refiner.tclWEAK_PROGRESS" [< 'sTR"Failed to progress.">]
;;

(* Same as tclWEAK_PROGRESS but fails also if tactics generates several goals,
   one of them being identical to the original goal *)
let tclNOTSAMEGOAL (tac:tactic) (ptree : goal sigma) =
  let rslt = tac ptree in
  let gls = (fst rslt).it in
  if List.exists (fun gl -> same_goal gl ptree) gls 
  then errorlabstrm "Refiner.tclNOTSAMEGOAL" 
      [< 'sTR"Tactic generated a subgoal identical to the original goal.">]
  else rslt
;;


(* ORELSE f1 f2 tries to apply f1 and if it fails, applies f2 *)
let tclORELSE0 f1 f2 g =
    try f1 g
    with UserError _ | Stdpp.Exc_located(_,UserError _) -> f2 g ;;

(* tclPROGRESS must be inside try ... with *)
let tclORELSE (f1:tactic) (f2:tactic) (g:goal sigma) =
    try tclPROGRESS f1 g
    with UserError _ | Stdpp.Exc_located(_,UserError _) -> f2 g ;;

(* TRY f tries to apply f, and if it fails, leave the goal unchanged *)
let tclTRY (f:tactic) = (tclORELSE f tclIDTAC);;

let tclTHENTRY (f:tactic) (g:tactic) = (tclTHEN f (tclTRY g));;

(*Try the first tactic that does not fail in a list of tactics*)
let rec tclFIRST = function
    []      -> tclFAIL_s "No applicable tactic."
 |  t::rest -> tclORELSE0 t (tclFIRST rest)
;;

let tclCOMPLETE tac = tclTHEN tac (tclFAIL_s "Proof is not complete.");;

(* Try the first thats solves the current goal *)
let tclSOLVE tacl = tclFIRST (List.map tclCOMPLETE tacl)


(* Beware: call by need of Caml, g is needed *)
let rec tclREPEAT tac gls =
  (tclORELSE (tclTHEN tac (tclREPEAT tac)) tclIDTAC) gls;;

let tclAT_LEAST_ONCE t = (tclTHEN t (tclREPEAT t));;

(* Iteration tactical *)
let tclDO n tac =
  let rec dorec n =
    if n = 0 then tclIDTAC
    else tclTHEN tac (fun gls -> dorec (n-1) gls) in
  if n < 0 then errorlabstrm "Refiner.tclDO"
      [<'sTR"Wrong argument : Do needs a positive integer.">]
  else dorec n
;;


(* A tactical Info for printing the underlying script *)

let pr_rule = function
    (PRIM r) -> pr_prim_rule r
  | (TACTIC texp) -> hOV 0 (pr_tactic texp)
  | CONTEXT ctxt -> pr_ctxt ctxt
  | LOCAL_CONSTRAINTS lc -> [< 'sTR"Local constraint change" >]

;;

let thin_sign osign (x,y) =
    let com_nsign = List.combine x y
    in List.split(map_succeed (fun (id,ty) ->
                              if (not (mem_sign osign id))
                                  or (id,ty) <> (lookup_sign id osign) then
                                  (id,ty)
                              else failwith "caught") com_nsign)
;;

let rec print_proof sigma osign pf =
let {hyps=sign;concl=cl;info=info;Evd.body=body} = pf.goal in
let sign' = thin_sign osign sign
in match pf.ref with

    None -> hOV 0 [< pr_seq {hyps=sign';concl=cl;info=info;Evd.body=body} >]

  | Some(r,spfl) ->
    hOV 0 [< hOV 0 (pr_seq {hyps=sign';concl=cl;info=info;Evd.body=body});
            'sPC ; 'sTR" BY ";
            hOV 0 [< pr_rule r >]; 'fNL ;
            'sTR"  ";
	    hOV 0 (prlist_with_sep pr_fnl (print_proof sigma sign) spfl)
              >]
;;

let pr_change gl = [< 'sTR"Change " ; prterm gl.concl ; 'sTR"." ; 'fNL>];;

let rec print_script nochange sigma osign pf =
  let {hyps=sign;concl=cl} = pf.goal in
  let sign' = thin_sign osign sign in
    match pf.ref with
        None ->
           if nochange then [< 'sTR"<Your Tactic Text here>" >]
           else pr_change pf.goal

      | Some(r,spfl) ->
          [< (if nochange then [< >] else (pr_change pf.goal));
             pr_rule r ; 'sTR"." ; 'fNL ;
             prlist_with_sep pr_fnl
               (print_script nochange sigma sign) spfl >]
;;


let rec print_treescript sigma osign pf =
  let {hyps=sign;concl=cl} = pf.goal in
  let sign' = thin_sign osign sign in
    match pf.ref with
        None -> [< >]

      | Some(r,spfl) ->
          [< pr_rule r ; 'sTR"." ; 'fNL ;
             let prsub =
               prlist_with_sep pr_fnl (print_treescript sigma sign) spfl
             in
               if List.length spfl > 1 then [< 'sTR"  "; hOV 0 prsub >]
               else prsub >]
;;

let rec print_info_script sigma osign pf =
  let {hyps=sign;concl=cl} = pf.goal in
  let sign' = thin_sign osign sign in
    match pf.ref with

        None -> [< >]

      | Some(r,spfl) ->
          [<  pr_rule r ; 
              match spfl with
                  [pf1] ->
                    if pf1.ref = None then [<'sTR "."; 'fNL >]
                    else [< 'sTR";" ; 'bRK(1,3) ;
                            print_info_script sigma sign pf1 >]
                | _ -> [< 'sTR"." ; 'fNL ;
                          prlist_with_sep pr_fnl
                            (print_info_script sigma sign) spfl >]
          >]
;;

let format_print_info_script sigma osign pf = 
    hOV 0 (print_info_script sigma osign pf);;

let print_subscript sigma sign pf = 
    if is_tactic_proof pf 
    then format_print_info_script sigma sign (subproof_of_proof pf)
    else format_print_info_script sigma sign pf;;

let tclINFO (tac:tactic) gls = 
    let (sgl,v) as res = tac gls in 
    (try let pf = v (List.map leaf (sig_it sgl))
    in  mSGNL(hOV 0 [< 'sTR" == "; 
                    print_subscript 
                     ((comp ts_it sig_sig) gls) (sig_it gls).hyps pf >])
    with UserError _ -> 
    mSGNL(hOV 0 [< 'sTR "Info failed to apply validation" >]))
    ; res
;;

(* Functions working on goal list for correct backtracking in Prolog *)

let tclFIRSTLIST = tclFIRST;;
let tclIDTAC_list gls = (gls, fun x -> x);;

(* first_goal : goal list sigma -> goal sigma *)

let first_goal gls = let gl = gls.it and sig_0 = gls.sigma in 
    if gl = [] then error "first_goal" else {it=List.hd gl; sigma = sig_0};;

(* tactic -> tactic_list : Apply a tactic to the first goal in the list *)

let apply_tac_list tac glls = 
      let (sigr,lg) = unpackage glls in
      match lg with (g1::rest) ->
      let (gl,p) = apply_sig_tac sigr tac g1 in
      let n = List.length gl in 
          repackage sigr (gl@rest), 
          (function pfl -> let (pfg,pfrest)=chop_list n pfl in (p pfg)::pfrest)
                | _ -> error "apply_tac_list";;

let then_tactic_list tacl1 tacl2 glls = 
  let (glls1,pl1) = tacl1 glls in
  let (glls2,pl2) = tacl2 glls1 in
  (glls2, comp pl1 pl2);;

(* Transform a tactic_list into a tactic *)
let tactic_list_tactic tac gls = 
    let (glres,vl) = tac (goal_goal_list gls) in
    (glres, comp idtac_valid vl)
;;

(* Applies tacl (may change several goals at the same time) to a proof tree.
   We detect if the evar environment changed, in which case we normalize
   the evars in every subgoal of the proof tree. *)
let solve_subgoal tacl pf_sigma =
  let (sigr,pf) = unpackage pf_sigma in
  let gl,p = frontier pf in
  let (sigr,gll,pl) = tacl (sigr,gl,p) in
  let pfs = repackage sigr (pl (List.map leaf gll)) in
  if (ts_it pfs.sigma) == (ts_it pf_sigma.sigma) then pfs
  else { it = norm_evar_pf (ts_it pfs.sigma) pfs.it; sigma=pfs.sigma}
;;

(* The type of proof-trees state and a few utilities 
A proof-tree state is built from a proof-tree, a set of global
   constraints, and a stack which allows to navigate inside the
   proof-tree remembering how to rebuild the global proof-tree
   possibly after modification of one of the focused children proof-tree.
   The number in the stack corresponds to 
   either the selected subtree and the validation is a function from a
   proof-tree list consisting only of one proof-tree to the global
   proof-tree 
   or -1 when the move is done behind a registered tactic in which
   case the validation corresponds to a constant function giving back 
   the original proof-tree.
*)

type pftreestate = {tpf      : proof_tree ;
                    tpfsigma : global_constraints;
                    tstack   : (int * validation) list}
;;
let proof_of_pftreestate pts    = pts.tpf;;
let is_top_pftreestate pts    = pts.tstack = [] ;;
let cursor_of_pftreestate pts   = List.map fst pts.tstack;;
let evc_of_pftreestate pts      = pts.tpfsigma;;
let top_goal_of_pftreestate pts = {it    = goal_of_proof pts.tpf; 
                                   sigma = pts.tpfsigma};;
let nth_goal_of_pftreestate n pts =
  let goals = fst (frontier pts.tpf) in
    try {it = List.nth goals (n-1); sigma = pts.tpfsigma }
    with Failure "nth" -> non_existent_goal n
;;

let descend n p =
  match p.ref with
      None        -> error "It is a leaf."
    | Some(r,pfl) ->
    	if List.length pfl >= n then
	  (match chop_list (n-1) pfl with 
	       left,(wanted::right) ->
		 (wanted,
		  (fun pfl' ->
                     if (List.length pfl' = 1) 
		       & (List.hd pfl').goal = wanted.goal 
		     then
                       let pf'       = List.hd pfl' in
                       let spfl      = left@(pf'::right) in
                       let newstatus = and_status (List.map pf_status spfl)
                       in {status   = newstatus;
			   goal     = p.goal;
			   subproof = p.subproof;
			   ref      = Some(r,spfl)}
			    
                     else error "descend: validation"))
	     | _ -> assert false)
    	else error "Too few subproofs"
;;

let traverse n pts = match n with 
    (* go to the parent *)
    0 ->
    (match  pts.tstack with
     [] -> error "traverse: no ancestors"
   | (_,v)::tl ->
     {tpf = v [pts.tpf];
      tstack = tl;
      tpfsigma = pts.tpfsigma})

    (* go to the hidden tactic-proof, if any, otherwise fail *)
  | -1 ->
    (match pts.tpf.subproof with
     None -> error "traverse: not a tactic-node"
   | Some spf ->
     let v = (fun pfl -> pts.tpf)
     in {tpf = spf;
         tstack = (-1,v)::pts.tstack;
         tpfsigma = pts.tpfsigma})

    (* when n>0, go to the nth child *)
  | n ->
    let (npf,v) = descend n pts.tpf
    in {tpf = npf;
        tpfsigma = pts.tpfsigma;
        tstack = (n,v):: pts.tstack}
;;

let change_constraints_pftreestate newgc pts = 
     {tpf = pts.tpf; tpfsigma = newgc;  tstack = pts.tstack};;


(* solve the nth subgoal with tactic tac *)
let solve_nth_pftreestate n tac pts =
    let pf    = pts.tpf in
    let rslts =
      solve_subgoal (theni_tac n tac) {it = pts.tpf;sigma = pts.tpfsigma}
    in {tpf      = rslts.it;
        tpfsigma = rslts.sigma;
        tstack   = pts.tstack}
;;

let solve_pftreestate = solve_nth_pftreestate 1
;;

 (* This function implements a poor man's undo at the current goal.
    This is a gross approximation as it does not attempt to clean correctly
    the global constraints given in tpfsigma. *)
let weak_undo_pftreestate pts =
   let pf = leaf pts.tpf.goal in
     {tpf = pf;
     tpfsigma = pts.tpfsigma;
     tstack = pts.tstack};;

(* Gives a new proof (a leaf) of a goal gl *)
let mk_pftreestate g =
       {tpf      = leaf g;
        tstack   = [];
        tpfsigma = ts_mk(Evd.mt_evd())}
;;

(* Extracts a constr from a proof-tree state ; raises an error if the proof is
   not complete or the state does not correspond to the head of the proof-tree *)

let extract_pftreestate pts =
    if   pts.tstack = [] 
    then strong (whd_betadeltatiota (ts_it pts.tpfsigma))
           (extract_proof pts.tpf.goal.hyps pts.tpf)
    else errorlabstrm "extract_pftreestate"
        [< 'sTR"Cannot extract from a proof-tree in which we have descended;" ;
           'sPC ; 'sTR"Please ascend to the root" >]
;;

(* Focus on the first leaf proof in a proof-tree state *)

let rec first_unproven pts =
let pf = (proof_of_pftreestate pts)
in if is_complete_proof pf then
    errorlabstrm "first_unproven"
    [< 'sTR"No unproven subgoals" >]
   else if is_leaf_proof pf then
       pts
   else
       try let childnum =
       try_find_i 
       (fun n pf -> if not(is_complete_proof pf) then n else failwith "caught")
       1
       (children_of_proof pf)
           in first_unproven (traverse childnum pts)
       with e -> errorlabstrm "first_unproven" (Errors.explain_sys_exn e)
;;

(* Focus on the last leaf proof in a proof-tree state *)

let rec last_unproven pts =
let pf = (proof_of_pftreestate pts)
in if is_complete_proof pf then
    errorlabstrm "last_unproven"
    [< 'sTR"No unproven subgoals" >]
   else if is_leaf_proof pf then
       pts
   else let children = (children_of_proof pf) in
           let nchilds = List.length children in
       try 
           let childnum =
               try_find_i 
               (fun n pf ->
                    if not(is_complete_proof pf) then n else failwith "caught")
               1
               (List.rev children)
           in last_unproven (traverse (nchilds-childnum+1) pts)
       with e -> errorlabstrm "last_unproven" (Errors.explain_sys_exn e)
;;

let rec nth_unproven n pts =
let pf = proof_of_pftreestate pts
in if is_complete_proof pf then
    errorlabstrm "nth_unproven"
    [< 'sTR"No unproven subgoals" >]
   else if is_leaf_proof pf then
        if n = 1 then pts else  errorlabstrm "nth_unproven"
    [< 'sTR"Not enough unproven subgoals" >]
   else let children = children_of_proof pf in
        let rec process i k = function
	    [] -> errorlabstrm "nth_unproven"
		[< 'sTR"Not enough unproven subgoals" >]
	  | pf1::rest -> let k1 = nb_unsolved_goals pf1 in 
	      if k1<k then process (i+1) (k-k1) rest
	      else nth_unproven k (traverse i pts)
	in process 1 n children
;;

let rec node_prev_unproven loc pts =
let pf = proof_of_pftreestate pts
in match cursor_of_pftreestate pts with
    [] -> last_unproven pts
  | n::l ->
    if is_complete_proof pf or loc = 1 then
        node_prev_unproven n (traverse 0 pts)
   else if is_complete_proof (List.nth (children_of_proof pf) (loc - 2)) then
       node_prev_unproven (loc - 1) pts
   else first_unproven (traverse (loc - 1) pts)
;;


let rec node_next_unproven loc pts =
let pf = proof_of_pftreestate pts
in match cursor_of_pftreestate pts with
    [] -> first_unproven pts
  | n::l ->
    if is_complete_proof pf or loc = (List.length (children_of_proof pf)) then
        node_next_unproven n (traverse 0 pts)
   else if is_complete_proof (List.nth (children_of_proof pf) loc) then
       node_next_unproven (loc + 1) pts
   else last_unproven(traverse (loc + 1) pts)
;;

let next_unproven pts =
let pf = proof_of_pftreestate pts
in if is_leaf_proof pf then
    match cursor_of_pftreestate pts with
    [] -> error "next_unproven"
  | n::_ -> node_next_unproven n (traverse 0 pts)
   else node_next_unproven (List.length (children_of_proof pf)) pts
;;

let prev_unproven pts =
let pf = proof_of_pftreestate pts
in if is_leaf_proof pf then
    match cursor_of_pftreestate pts with
    [] -> error "prev_unproven"
  | n::_ -> node_prev_unproven n (traverse 0 pts)
   else node_prev_unproven 1 pts
;;

let rec top_of_tree pts = 
  if is_top_pftreestate pts then pts
  else top_of_tree(traverse 0 pts)
;;

(* $Id: refiner.ml,v 1.39 1999/11/12 11:11:24 herbelin Exp $ *)
