(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                             evar_refiner.ml                              *)
(****************************************************************************)

open Std;;
open Initial;;
open Names;;
open Vectops;;
open Generic;;
open Term;;
open Reduction;;
open Himsg;;
open Termenv;;
open Mach;;
open Proof_trees;;
open Pp;;
open Logic;;
open Refiner;;
open Evd;;


let rc_of_pfsigma sigma = rc_of_gc sigma.sigma sigma.it.goal;;
let rc_of_glsigma sigma = rc_of_gc sigma.sigma sigma.it;;

type walking_constraints = readable_constraints idstamped;;
type 'a result_w_tactic = walking_constraints -> walking_constraints * 'a;;
type w_tactic = walking_constraints -> walking_constraints;;


let local_Constraints lc gs = refiner (LOCAL_CONSTRAINTS lc) gs;;

let on_wc f wc = ids_mod f wc;;

let startWalk gls =
    let evc = project_with_focus gls in
    let wc = (ids_mk evc)
    in (wc,
        (fun wc' gls' ->
             if ids_eq wc wc' & gls.it = gls'.it then
                 if  Spset.equal (get_lc gls.it) (get_focus (ids_it wc')) then
                     tclIDTAC {it=gls'.it; sigma = get_gc (ids_it wc')}
                 else
                 (local_Constraints (get_focus (ids_it wc'))
                  {it=gls'.it; sigma = get_gc (ids_it wc')})
             else error "Walking"))
;;

let walking_THEN wt rt gls =
    let (wc,kONT) = startWalk gls in
    let (wc',rslt) = wt wc
    in tclTHEN (kONT wc') (rt rslt) gls
;;

let walking wt = walking_THEN (fun wc -> (wt wc,())) (fun () -> tclIDTAC);;


let extract_decl sp evc =
    let evdmap = (ts_it evc).decls in
    let evd = Evd.map evdmap sp 
    in (ts_mk{sign = evd.hyps;
              focus = (get_lc evd);
              decls = Evd.rmv evdmap sp})
;;

let restore_decl sp evd evc =
    let newctxt = {lc     = (ts_it evc).focus;
                   mimick = (get_mimick evd);
                   pgm    = (get_pgm evd)} in
    let newgoal = {hyps=evd.hyps; concl = evd.concl;
                   info=Some newctxt;body = evd.body}
    in (rc_add evc (sp,newgoal))
;;

(* [w_Focusing sp wt wc]
 *
 * Focuses the walking context WC onto the declaration SP, given that
 * this declaration is UNDEFINED.  Then, it runs the walking_tactic,
 * WT, on this new context.  When the result is returned, we recover
 * the resulting focus (access list) and restore it to SP's declaration.
 *
 * It is an error to cause SP to change state while we are focused on it.
 *)
let w_Focusing_THEN sp (wt:'a result_w_tactic) (wt':'a -> w_tactic)
                       (wc:walking_constraints) =
    let focus = (ts_it (ids_it wc)).focus
    and sign  = (ts_it (ids_it wc)).sign
    and evd   = Evd.map ((ts_it(ids_it wc)).decls) sp in
    let (wc':walking_constraints) = (ids_mod (extract_decl sp) wc) in
    let (wc'',rslt) = wt wc'
    in if ids_eq wc wc'' then 
        if ts_eq  (ids_it wc') (ids_it wc'') then wt' rslt wc
        else let wc''' = (ids_mod (restore_decl sp evd) wc'')
             in wt' rslt
                 (ids_mod
                   (ts_mod(fun evc ->
                                {sign  = sign;
                                 focus = focus;
                                 decls = evc.decls}))
                    wc''')
       else error "w_saving_focus"
;;

let w_add_sign (id,t) (wc:walking_constraints) =
  ids_mk((ts_mod
         (fun evr ->
              {focus = evr.focus;
               sign = add_sign (id,t) evr.sign;
               decls = evr.decls})
          (ids_it wc)))
;;

let ctxt_type_of evc c = type_of (ts_it evc).decls (ts_it evc).sign c;;

let w_IDTAC wc = wc;;
let w_Focusing sp wt = 
 w_Focusing_THEN sp (fun wc -> (wt wc,())) (fun _ -> w_IDTAC);;

let w_Focus sp wc = (ids_mod (extract_decl sp) wc);;

let w_Underlying wc = (ts_it (ids_it wc)).decls;;
let w_type_of wc c  = ctxt_type_of (ids_it wc) c;;
let w_hyps    wc    = get_hyps (ids_it wc);;
let w_ORELSE wt1 wt2 wc = try wt1 wc with UserError _ -> wt2 wc;;

let w_Declare sp c (wc:walking_constraints) =
     (match c with (DOP2(Cast,_,_)) -> ()
      | _ -> error "Cannot declare an un-casted evar");
     let _       = w_type_of wc c in
     let access  = get_focus (ids_it wc)
     and sign    = get_hyps  (ids_it wc)in
     let newdecl = mkGOAL (mt_ctxt access) sign c
     in ((ids_mod (fun evc -> (rc_add evc (sp,newdecl))) wc): walking_constraints)
;;

let w_Declare_At sp sp' c = w_Focusing sp (w_Declare sp' c);;

let evars_of sigma c =
  List.fold_right Spset.add 
    (uniquize(process_opers_of_term
             (function (Const sp) -> Evd.is_evar (ts_it sigma).decls sp
                     | _ -> false)
             (function (Const sp) -> sp
	     	| _ -> assert false) [] c))
  Spset.empty
;;


let w_Define sp c wc =
let spdecl = Evd.map (w_Underlying wc) sp in
let cty = try ctxt_type_of (ids_it (w_Focus sp wc))
                       (DOP2(Cast,c,spdecl.concl))
          with Not_found -> error "Instantiation contains unlegal variables"
in match spdecl.body with
    EVAR_EMPTY ->
    let access = evars_of (ids_it wc) c in
    let spdecl' = {hyps=spdecl.hyps;
                   concl=spdecl.concl;
                   info=Some (mt_ctxt access);
                   body = EVAR_DEFINED c}
    in (ids_mod (fun evc -> (Proof_trees.remap evc (sp,spdecl'))) wc)

  | _ -> error "define_evar"
;;

(* $Id: evar_refiner.ml,v 1.14 1999/06/29 07:47:33 loiseleu Exp $ *)
