(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA                        ENS-CNRS                *)
(*              Rocquencourt                        Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               Jul 10th 1997                              *)
(*                                                                          *)
(****************************************************************************)
(*                               searchisos.ml                              *)
(****************************************************************************)

open Pp;;
open Std;;
open Vectops;;
open More_util;;
open Constrtypes;;
open Environ;;
open Generic;;
open Libobject;;
open Library;;
open Names;;
open Reduction;;
open Summary;;
open System;;
open Term;;
open Printer;;
open Pretty;;

let search_in_lib = ref false;;

(* Searchisos does not use existential variables,
 * but we need an evar_map anyway.
 *)
let empty_evd = Evd.mt_evd();;

(******** Updating of the inductive types table ********)
(* The list of the inductive types with one constructor *)
let tbl_ind_one=Hashtbl.create 53;;

(* Substitutes a rel by a term *)
let rec subs_rel rel term=function
   DOP1(a,b) -> DOP1(a,subs_rel rel term b)
  |DOP2(a,b,c) -> DOP2(a,subs_rel rel term b,subs_rel rel term c)
  |DOPN(a,b) -> DOPN(a,Array.map (subs_rel rel term) b)
  |DOPL(a,b) -> DOPL(a,List.map (subs_rel rel term) b)
  |DLAM(a,b) -> DLAM(a,subs_rel (rel+1) term b)
  |DLAMV(a,b) -> DLAMV(a,Array.map (subs_rel (rel+1) term) b)
  |(Rel a) as b -> if a=rel then term else b
  |_ as a -> a;;

(* Modify the unpractical types of constructors to correct types *)
let modify_constructors name pk mindlc=
  let (lna,lC)=decomp_all_DLAMV_name mindlc
  in
    let lst=ref (Array.to_list lC)
    and lgra=List.length lna
    in
      for cmr=0 to (lgra-1) do
        lst:=List.map (subs_rel (lgra-cmr) (DOPN(MutInd(path_of_string
             (name^"."^(string_of_kind pk)),cmr),Array.of_list []))) (!lst)
      done;
      !lst;;

(* Puts off all the Cast *)
let rec put_off_casts=function
   DOP1(a,b) -> DOP1(a,put_off_casts b)
  |DOP2(Cast,a,b) -> put_off_casts a
  |DOP2(a,b,c) -> DOP2(a,put_off_casts b,put_off_casts c)
  |DOPN(a,b) -> DOPN(a,Array.map put_off_casts b)
  |DOPL(a,b) -> DOPL(a,List.map put_off_casts b)
  |DLAM(a,b) -> DLAM(a,put_off_casts b)
  |DLAMV(a,b) -> DLAMV(a,Array.map put_off_casts b)
  |a -> a;;

(* Transforms a string to a list of char *)
let list_of_string str=
  let lst=ref []
  in
    for cmr=0 to ((String.length str)-1) do
      lst:=(!lst)@[String.get str cmr]
    done;
    !lst;;

(* Transforms a list of char in a string *)
let string_of_list lst=
  let str=String.create (List.length lst)
  in
    let rec fill cmr=function
       [] -> str
      |a::b ->
        String.set str cmr a;
        fill (cmr+1) b
    in
      fill 0 lst;;

(* Extract the name of the section_path *)
let extract sp=
  let lsts=list_of_string(string_of_path sp)
  and lstc=ref []
  in
    let rec inspect cmr=function
       [] -> failwith "extract"
      |a::b ->
        if cmr<4 then
          inspect (cmr+1) b
        else
          if not(a='#') then
            (lstc:=a::(!lstc);
             inspect cmr b)
    in
      inspect 0 (List.rev lsts);
      string_of_list(!lstc);;

(* Tests if the ind_one is really inductive *)
let rec occ name=function
   DOP1(_,a) -> occ name a
  |DOP2(_,a,b) -> (occ name a) or (occ name b)
  |DOPN(a,b) ->
    (match a with
        MutInd(c,_) ->
          (extract c)=name or (List.exists (occ name) (Array.to_list b))
       |MutConstruct((c,_),_) ->
          (extract c)=name or (List.exists (occ name) (Array.to_list b))
       |_ -> List.exists (occ name) (Array.to_list b))
  |DOPL(_,a) -> List.exists (occ name) a
  |DLAM(_,a) -> occ name a
  |DLAMV(_,a) -> List.exists (occ name) (Array.to_list a)
  |_ -> false
and is_rec name=function
   DOP2(Prod,a,DLAM(b,c)) ->
     (occ name a) or (is_rec name c)
  |_ -> false;;     

(* For mutind_one and mutconstruct_one *)
exception Mutind_one;;
exception Mutconstruct_one;;

(* Tests if muti is a MutInd with one constructor and gives the new type
   corresponding *)
let mutind_one muti=
  let name=
    match muti with
       MutInd(a,_) -> extract a
      |_ -> failwith "mutind_one"
  in
    try (fst(Hashtbl.find tbl_ind_one name))
    with
       Not_found -> raise Mutind_one;;

(* Tests if mutc is the only MutConstruct of a MutInd and gives the new term
   corresponding *)
let mutconstruct_one mutc=
  let name=
    match mutc with
       MutConstruct((a,_),_) -> extract a
      |_ -> failwith "mutconstruct_one"
  in
    try (snd(Hashtbl.find tbl_ind_one name))
    with
       Not_found -> raise Mutconstruct_one;;

(* Rewrites all inductives types with one constructor to a generic pair Sigma
   with the constructor Csigma *)
let rec ind_rew=function
   DOP1(a,b) -> DOP1(a,ind_rew b)
  |DOP2(a,b,c) -> DOP2(a,ind_rew b,ind_rew c)
  |DOPN(a,b) ->
    (match a with
        MutInd(truc,_) ->
          (try
             (mutind_one a)
           with Mutind_one ->
              DOPN(a,Array.map ind_rew b))
       |MutConstruct((_,_),_) ->
         (try
            (mutconstruct_one a)
          with Mutconstruct_one ->
             DOPN(a,Array.map ind_rew b))
       |_ -> DOPN(a,Array.map ind_rew b))
  |DOPL(a,b) -> DOPL(a,List.map ind_rew b)
  |DLAM(a,b) -> DLAM(a,ind_rew b)
  |DLAMV(a,b) -> DLAMV(a,Array.map ind_rew b)
  |a -> a;;

(* Use ind_rew till the normal form *)
let total_ind_rew typ=
  let mem=ref typ
  and cal=ref typ
  in
    cal:=ind_rew(!mem);
    while not((!mem)=(!cal)) do
      mem:=(!cal);
      cal:=ind_rew(!mem)
    done;
    !mem;;

(* Transforms prod to (array,dest) *)
let decomp_prod_array term=
  let nbr=nb_prod term
  in
    let tab=Array.create nbr (Anonymous,DOP0(Implicit))
    and cmr=ref 0
    in
      let rec inspect=function
         DOP2(Prod,a,DLAM(b,c)) ->
           Array.set tab (!cmr) (b,a);
           incr cmr;
           inspect c
        |_ as a -> (tab,a)
      in
        inspect term;;

(* Tool for upd_lst_one in object MUTUALINDUCTIVE *)
let upd_mutual name mib=
 let {mINDKIND=pk ; mINDPACKETS=mipv ; mINDNPARAMS=nbparex}=mib
 in
   let chg_mutind nbpar typ_cons=
     let nbtpar=nb_prod typ_cons
     in
       let rec fill cmr=function
          DOP2(Prod,a,DLAM(b,c)) as d ->
            if cmr<nbpar then
              DOP2(Lambda,a,DLAM(b,fill (cmr+1) c))
            else
              if cmr<nbtpar-1 then
                DOP2(XTRA("Sigma",[]),a,DLAM(b,fill (cmr+1) c))
              else
                a
         |a->
           DOPN(MutInd(path_of_string "#Datatypes#unit.cci",0),Array.of_list
           [])
       in
         fill 0 typ_cons
   and chg_mutconstruct nbpar typ_cons=
     let nbtpar=nb_prod typ_cons
     in
       let body nbr=
         let rec construct cmr=
           if cmr>1 then
             DOP2(XTRA("Csigma",[]),Rel cmr,construct (cmr-1))
           else
             Rel cmr
         in
           construct nbr
       in
         let rec fill cmr=function
            DOP2(Prod,a,DLAM(b,c)) ->
              if cmr<nbpar-1 then
                DOP2(Lambda,a,DLAM(b,fill (cmr+1) c))
              else if cmr=nbpar-1 then
                if (nbtpar-nbpar)=0 then
                  DOP2(Lambda,a,DLAM(b,DOPN(MutConstruct((path_of_string
                  "#Datatypes#unit.cci",0),1),Array.of_list [])))
                else
                  DOP2(Lambda,a,DLAM(b,fill (cmr+1) c))
              else
                if cmr<nbtpar-1 then
                  DOP2(Lambda,a,DLAM(b,fill (cmr+1) c))
                else
                  DOP2(Lambda,a,DLAM(b,body (nbtpar-nbpar)))
           |a ->
             DOPN(MutConstruct((path_of_string "#Datatypes#unit.cci",0),1),
             Array.of_list [])
         in
           fill 0 typ_cons
   in
     let explore_mip mip=
       let {mINDCONSNAMES=mcs ; mINDTYPENAME=typ ; mINDLC=mlc ;
            mINDARITY=ari}=mip
       in
         let nbpar=nb_prod ari.body
         and (par_tab,_)=decomp_prod_array ari.body
         in
           if ((Array.length mcs)=1)&((nbpar-nbparex)=0) then
             (let tpc=put_off_casts(List.hd (modify_constructors
                      (name^(string_of_id typ)) pk mlc))
              and name=string_of_id typ
              in
                if not(is_rec name tpc) then
                  Hashtbl.add tbl_ind_one name (total_ind_rew (chg_mutind
                  nbparex tpc),chg_mutconstruct nbparex tpc))
     in
       if Array.length mipv = 1 then
         explore_mip mipv.(0)
       else
         Array.iter explore_mip mipv;;

(* Tool for upd_lst_one in Lib.leaf *)
let upd_leaf_entry name (spopt,lobj) =
  let tag = object_tag lobj in
  match (spopt,tag) with
    (_,"MUTUALINDUCTIVE") ->
    let (cmap,_) = outMutualInductive lobj in
    if List.mem_assoc CCI cmap then
      upd_mutual name (assoc_kind CCI cmap)
  |_ -> ()
;;

(* Tools for upd_lst_one in Lib.node and Lib.library_segment *)
let rec upd_library_entry depth name=function
   (sp,Lib.LEAF lobj) -> upd_leaf_entry name (sp,lobj)
  |(_,Lib.ClosedDir(a,_,_,ctxt)) ->
    if depth=0 then
      upd_context 1 (name^a^"#") ctxt
  |_ -> ()
and upd_context depth name= 
 let rec prec = function
    [] -> ()
   |h::rest ->
     upd_library_entry depth name h;
     prec rest
 in prec
;;

(* Update tbl_ind_one *)
let upd_tbl_ind_one ()=
  Hashtbl.clear tbl_ind_one;
  upd_context 0 "#" (List.rev(Lib.contents_after None));;

(******** Types' reduction ********)
(* Decrements all rel>nbr *)
let rec decr_rel nbr=function
   DOP1(a,b) -> DOP1(a,decr_rel nbr b)
  |DOP2(a,b,c) -> DOP2(a,decr_rel nbr b,decr_rel nbr c)
  |DOPN(a,b) -> DOPN(a,Array.map (decr_rel nbr) b)
  |DOPL(a,b) -> DOPL(a,List.map (decr_rel nbr) b)
  |DLAM(a,b) -> DLAM(a,decr_rel (nbr+1) b)
  |DLAMV(a,b) -> DLAMV(a,Array.map (decr_rel (nbr+1)) b)
  |(Rel a) as b ->
    if a>nbr then
      Rel(a-1)
    else
      b
  |a -> a;;

(* Elims the unit in the list *)
let elim_unit lst=
  let subs_tt lst=
    let rec inspect rel=function
       [] -> []
      |a::b ->
        (fst a,(subs_rel rel (DOPN(MutConstruct((path_of_string
        "#Datatypes#unit.cci",0),1),Array.of_list [])) (snd a)))::(inspect
        (rel+1) b)
    in
      inspect 1 lst
  in
    let rec iter=function
       [] -> []
      |a::b ->
        if (snd a)=(DOPN(MutInd(path_of_string "#Datatypes#unit.cci",0),
        Array.of_list [])) then
          if (fst a)=Anonymous then
            iter b
          else
            iter (subs_tt b)
        else
          a::(iter b)
    in
      iter lst;;

(* Transforms the normal form to list of components *)
let rec list_of_sigma=function
   DOP2(XTRA("Sigma",[]),a,DLAM(b,c)) ->
     [(b,a)]@(list_of_sigma c)
  |a -> [(Anonymous,a)];;

(* Apply associativity on a sigma term *)
let sigma_test term=
  let rec apply_assoc (name,body)=function
     DOP2(XTRA("Sigma",[]),a,DLAM(b,c)) ->
       DOP2(XTRA("Sigma",[]),a,DLAM(name,apply_assoc (b,(subs_rel 1
       (DOP2(XTRA("Csigma",[]),Rel 2,Rel 1)) body)) c))
    |a -> DOP2(XTRA("Sigma",[]),a,DLAM(name,snd(sigma_test_rec body)))
  and sigma_test_rec=function
     DOP2(XTRA("Sigma",[]),a,DLAM(b,c)) ->
       let resa=sigma_test_rec a
       in
         if (fst resa) then
           (true,apply_assoc (b,c) (snd resa))
         else
           (true,DOP2(XTRA("Sigma",[]),a,DLAM(b,(snd (sigma_test_rec c)))))
    |a -> (false,a)
  and elim_unit=function
     DOP2(XTRA("Sigma",[]),a,DLAM(b,c)) ->
       if c=(DOPN(MutInd(path_of_string "#Datatypes#unit.cci",0),Array.of_list
       [])) then
         a
       else
         if a=(DOPN(MutInd(path_of_string "#Datatypes#unit.cci",0),
         Array.of_list [])) then
           elim_unit (decr_rel 1 (subs_rel 1 (DOPN(MutConstruct((path_of_string
           "#Datatypes#unit.cci",0),1),Array.of_list [])) c))
         else
           DOP2(XTRA("Sigma",[]),a,DLAM(b,elim_unit c))
    |a -> a
  and final_test=function
     DOP2(XTRA("Sigma",[]),_,_) -> true
    |_ -> false
  in
    let resterm=sigma_test_rec term
    in
      if (fst resterm) then
        (let newterm=elim_unit (snd resterm)
         in
           (final_test newterm,newterm))
      else
        resterm;;

(* Transforms (list,dest) to prod *)
let make_prod (lst,dest)=
  let rec inspect=function
     [] -> dest
    |a::b -> DOP2(Prod,snd a,DLAM(fst a,inspect b))
  in
    inspect lst;;

(* Applies curry *)
let rec apply_curry (name,body) lstres=function
   [] -> failwith "apply_curry"
  |a::b ->
    if b=[] then
      (lstres@[(name,snd a)],body)
    else
      apply_curry (fst a,subs_rel 1 (DOP2(XTRA("Csigma",[]),Rel 2,Rel 1)) body)
      (lstres@[(name,snd a)]) b;;

(* Makes a list of rel n...1 *)
let make_rel nbr=
  let lst=ref []
  in
    for cmr=nbr downto 1 do
      lst:=(!lst)@[(Rel cmr)]
    done;
    !lst;;

(* Increments a list of subs *)
let incr_lst lstsubs=
  let rec incr_lst_rel=function
     [] -> []
    |a::b ->
      (match a with
          (Rel c) -> (Rel (c+1))::(incr_lst_rel b)
         |_ -> failwith "incr_lst")
  and incr_tot=function
     [] -> []
    |a::b ->
      ((fst a)+1,incr_lst_rel (snd a))::(incr_tot b)
  in
    incr_tot lstsubs;;

(* Makes the total substitution for distrib *)
let rec subs_tot lstsubs nbr=function
   DOP1(a,b) -> DOP1(a,subs_tot lstsubs nbr b)
  |DOP2(a,b,c) -> DOP2(a,subs_tot lstsubs nbr b,subs_tot lstsubs nbr c)
  |DOPN(a,b) ->
    if (a=AppL) then
      (match b.(0) with
          Rel c ->
            (try (DOPN(a,Array.of_list((List.assoc c lstsubs)@(List.map
                  (subs_tot lstsubs nbr) (List.tl (Array.to_list b))))))
             with
               Not_found -> DOPN(a,Array.map (subs_tot lstsubs nbr) b))
         |_ -> DOPN(a,Array.map (subs_tot lstsubs nbr) b))
    else
      DOPN(a,Array.map (subs_tot lstsubs nbr) b)
  |DOPL(a,b) -> DOPL(a,List.map (subs_tot lstsubs nbr) b)
  |DLAM(a,b) -> DLAM(a,subs_tot (incr_lst lstsubs) (nbr+1) b)
  |DLAMV(a,b) -> DLAMV(a,Array.map (subs_tot (incr_lst lstsubs) (nbr+1)) b)
  |(Rel a) as b ->
    (try (DOPN(AppL,Array.of_list(List.assoc a lstsubs)))
     with
        Not_found ->
          if a>nbr then
            Rel(a-nbr)
          else
            b)
  |a -> a;;

(* Puts name to anonymous prod *)
let put_name lstc=
  let rec modify cmr=function
     [] -> []
    |a::b ->
      if (fst a)=Anonymous then
        (Name (id_of_string ("#"^(string_of_int cmr))),snd a)::(modify (cmr+1)
        b)
      else
        a::(modify cmr b)
  in
    modify 0 lstc;;

(* Increments the first rel of lstsubs *)
let rec incr_fst_rel=function
   [] -> []
  |a::b -> ((fst a)+1,snd a)::(incr_fst_rel b);;

(* Applies distrib *)
let apply_distrib lstc termsig=
  let lgr=List.length lstc
  and newlstc=put_name lstc
  in
    let rec apply_distrib_rec lstsubs cmr=function
       DOP2(XTRA("Sigma",[]),a,DLAM(b,c)) ->
         DOP2(XTRA("Sigma",[]),make_prod (newlstc,subs_tot lstsubs cmr a),
         DLAM(b,apply_distrib_rec ((incr_fst_rel lstsubs)@[(1,make_rel
         (lgr+1))]) (cmr+1) c))
      |a -> make_prod (newlstc,subs_tot lstsubs cmr a)
    in
      apply_distrib_rec [] 0 termsig;;

(* Applies distrib *)
(*let apply_distrib lstc sgm=
  let rec one_prod (name,typ)=function
     DOP2(XTRA("Sigma",[]),a,DLAM(b,c)) ->
      DOP2(XTRA("Sigma",[]),DOP2(Prod,typ,DLAM(name,a)),DLAM(b,one_prod (name,
      typ) (subs_distr 1 (DOPN(AppL,Array.of_list [Rel 2;Rel 1])) c)))
    |a -> DOP2(Prod,typ,DLAM(name,a))
  in
    let rec iter term=function
       [] -> term
      |a::b ->
        iter (one_prod a term) b
    in
      iter sgm (List.rev lstc);;*)

(* Transforms a list of sigma to a sigma term *)
let rec sigma_of_list=function
   [] -> failwith "sigma_of_list"
  |a::b ->
    if b=[] then
      (snd a)
    else
      DOP2(XTRA("Sigma",[]),(snd a),DLAM(fst a,sigma_of_list b));;

(* New reduction for dependences, more performant I hope *)
let rec cur_dep lst=function
   DOP2(Prod,a,DLAM(b,c)) ->
     let (res,tms)=sigma_test a
     in
       if res then
         (let (lstres,bd)=apply_curry (b,c) [] (list_of_sigma tms)
          in
            cur_dep (lst@lstres) bd)
       else
         cur_dep (lst@[(b,a)]) c
  |a -> (lst,a)
and nf_dep term=
  let (lst,tc)=cur_dep [] term
  in
    if tc=(DOPN(MutInd(path_of_string "#Datatypes#unit.cci",0),Array.of_list
    [])) then
      DOPN(MutInd(path_of_string "#Datatypes#unit.cci",0),Array.of_list
      [])
    else if lst=[] then
      snd(sigma_test tc)
    else
      let lstt=elim_unit (lst@[(Anonymous,tc)])
      in
        let lstc=List.rev(List.tl (List.rev lstt))
        and lstb=(snd (List.hd (List.rev lstt)))
        in
          let (res,tms)=sigma_test lstb
          in
            if res then
              apply_distrib lstc tms
            else
              make_prod (lstc,tms);;

(******** Swap ********)
(* For permut *)
exception Permut_finished;;
exception Permut_impossible;;
exception Intern;;

(* Generates all permutations of 0..n-1 *)
let permut n=
  let cour=ref(Array.create n 0)
  and indc=ref 0
  and lstt=ref []
  and lstc=ref []
  in
    let init()=
      for cmr=0 to (n-1) do
        Array.set (!cour) cmr cmr
      done
    and swap()=
      let aux=Array.copy (!cour)
      in
        Array.set aux 0 (!cour).(!indc);
        Array.set aux (!indc) (!cour).(0);
        aux
    in
      let rec serve()=
        let tab=swap()
        in
          if (List.mem tab (!lstt)) then
            (incr indc;
             (if (!indc)=n then
                (cour:=
                   (try
                      (List.hd (!lstc))
                    with
                       Failure "hd" -> raise Permut_finished);
                 (try
                    (lstc:=(List.tl (!lstc)))
                  with
                     Failure "tl" -> lstc:=[]);
                 indc:=0));
             serve())
          else
            (lstt:=(!lstt)@[tab];
             lstc:=(!lstc)@[tab];
             incr indc;
             (if (!indc)=n then
                (cour:=
                   (try
                      (List.hd (!lstc))
                    with
                       Failure "hd" -> raise Permut_finished);
                 (try
                    (lstc:=(List.tl (!lstc)))
                  with
                     Failure "tl" -> lstc:=[]);
                 indc:=0));
             tab)
      in
        init();
        serve;;

(* Transforms array(permutation) to list of couples(always permutation) *)
let permut_array_list tab=
  let lst=ref []
  in
    for cmr=0 to ((Array.length tab)-1) do
      lst:=(!lst)@[(cmr,tab.(cmr))]
    done;
    !lst;;

(* Transforms (array,dest) to prod *)
let recompose (tab,dest)=
  let rec inspect=function
     [] -> dest
    |a::b -> DOP2(Prod,snd a,DLAM(fst a,inspect b))
  in
    inspect (Array.to_list tab);;

(* Returns the list of (Name, Negative Rel) *)
let ident_rel_list bgn term=
  let (tab,_)=decomp_prod_array term
  and lst=ref []
  and cmr=ref bgn
  in
    for aux=0 to ((Array.length tab)-1) do
      if not((fst (tab.(aux)))=Anonymous) then
        (lst:=(!lst)@[(fst (tab.(aux)),!cmr)];
         decr cmr)
    done;
    !lst;;

(* Puts the negative rel *)
let put_neg_rel lst_rel term=
  let rec apply lst_rel=function
     DOP2(Prod,a,DLAM(b,c)) ->
        if b=Anonymous then
          DOP2(Prod,a,DLAM(b,apply lst_rel c))
        else
          DOP2(Prod,a,DLAM(b,apply lst_rel (subs_rel 1 (Rel (List.assoc b
          lst_rel)) c)))
    |a -> a
  in
    apply lst_rel term;;

(* Puts off one negative rel *)
let rec subs_rel2 relneg relpos=function
   DOP1(a,b) -> DOP1(a,subs_rel2 relneg relpos b)
  |DOP2(a,b,c) ->
    DOP2(a,subs_rel2 relneg relpos b,subs_rel2 relneg relpos c)
  |DOPN(a,b) -> DOPN(a,Array.map (subs_rel2 relneg relpos ) b)
  |DOPL(a,b) -> DOPL(a,List.map (subs_rel2 relneg relpos) b)
  |DLAM(a,b) -> DLAM(a,subs_rel2 relneg (relpos+1) b)
  |DLAMV(a,b) -> DLAMV(a,Array.map (subs_rel2 relneg relpos) b)
  |(Rel a) as b -> if a=relneg then Rel(relpos) else b
  |_ as a -> a;;

(* Puts off the negative rel *)
let put_off_neg_rel lst term=
  let (tab,dest)=decomp_prod_array term
  in
    let taba=ref tab
    and desta=ref dest
    and lgr=Array.length tab
    in
      for cmr=0 to (lgr-1) do
        (if not((fst ((!taba).(cmr)))=Anonymous) then
           let (tabx,destx)=
             decomp_prod_array(subs_rel2 (List.assoc (fst ((!taba).(cmr)))
             lst) 1 (recompose (Array.sub (!taba) (cmr+1) (lgr-cmr-1),!desta)))
           in
             taba:=
               Array.append (Array.sub (!taba) 0 (cmr+1)) tabx;
             desta:=destx)
      done;
      recompose (!taba,!desta);;

(* Tells if relneg occurs in the term *)
let rec rel_neg_occur relneg=function
   DOP1(_,a) -> rel_neg_occur relneg a
  |DOP2(_,a,b) ->
    (rel_neg_occur relneg a) or (rel_neg_occur relneg b)
  |DOPN(_,a) -> List.exists (rel_neg_occur relneg) (Array.to_list a)
  |DOPL(_,a) -> List.exists (rel_neg_occur relneg) a
  |DLAM(_,a) -> rel_neg_occur relneg a
  |DLAMV(_,a) -> List.exists (rel_neg_occur relneg) (Array.to_list a)
  |(Rel a) -> a=relneg
  |_ -> false;;

(* Permuts a term *)
let permut_term lst termneg permcoup=
  let (tabneg,destneg)=decomp_prod_array termneg
  and tabcoup=Array.of_list permcoup
  and occur_in_array relneg tab=
    try
      (for cmr=0 to ((Array.length tab)-1) do
         if rel_neg_occur relneg (snd (tab.(cmr))) then
           raise Intern
       done);
      false
    with
       Intern -> true
  in
    let tabperm=ref (Array.copy tabneg)
    and lgr=Array.length tabneg
    in
      for cmr=0 to (lgr-1) do
        Array.set (!tabperm) cmr (tabneg.(snd (tabcoup.(cmr))))
      done;
      for cmr=0 to (lgr-1) do
        (if not((fst ((!tabperm).(cmr)))=Anonymous) then
          if (occur_in_array (List.assoc (fst ((!tabperm).(cmr))) lst)
             (Array.sub (!tabperm) 0 cmr)) then
            raise Permut_impossible)
      done;
      put_off_neg_rel lst (recompose (!tabperm,destneg));;

(* Equality between 2 terms modulo alpha-conversion and universes *)
let rec eq_constr_rec2 m n = if m=n then true else
  match (strip_head_cast m,strip_head_cast n) with
     (DOP2(Cast,c1,_),c2) 	       -> eq_constr_rec2 c1 c2
   | (c1,DOP2(Cast,c2,_))              -> eq_constr_rec2 c1 c2
   | (Rel p1,Rel p2)                   -> p1=p2
   | (DOPN(oper1,cl1),DOPN(oper2,cl2)) ->
      	       	 oper1=oper2 & for_all2eq_vect eq_constr_rec2 cl1 cl2
   | (DOP0 oper1,DOP0 oper2)           -> 
     (match (oper1,oper2) with
         (Sort(Type a),Sort(Type b)) -> true
        |(_,_) -> oper1=oper2)
   | (DOP1(i,c1),DOP1(j,c2))           -> (i=j) & eq_constr_rec2 c1 c2
   | (DOP2(i,c1,c1'),DOP2(j,c2,c2'))   ->
      	       	 (i=j) & eq_constr_rec2 c1 c2 & eq_constr_rec2 c1' c2'
   | (DLAM(_,c1),DLAM(_,c2)) 	       -> eq_constr_rec2 c1 c2
   | (DLAMV(_,cl1),DLAMV(_,cl2))       -> 
      	       	 for_all2eq_vect eq_constr_rec2 cl1 cl2
   | _ 				       -> false
;;

(* Equality between 2 constrs modulo alpha-conversion and universes *)
let eq_constr2 ((m:constr),(n:constr)) = eq_constr_rec2 m n;;

(* Small equality modulo universes *)
let eq_univ (n,m)=
  if n=m then
    true
  else
    match (n,m) with
       (DOP0 oper1,DOP0 oper2) ->
         (match (oper1,oper2) with
             (Sort(Type a),Sort(Type b)) -> true
            |(_,_) -> false)
      |(_,_) -> false;;

(* idem as List.assoc but modulo eq_univ *)
let rec assoc_univ elem=function
   [] -> raise Not_found
  |a::b ->
    if eq_univ(elem,fst a) then
      (snd a)
    else
      (assoc_univ elem b);;

(* Test if the type is dependant or not *)
let rec is_dependant=function
   DOP0 _ -> false
  |DOP1(_,a) -> is_dependant a
  |DOP2(_,a,b) -> (is_dependant a) or (is_dependant b)
  |DOPN(_,a) -> List.exists is_dependant (Array.to_list a)
  |DOPL(_,a) -> List.exists is_dependant a
  |DLAM(_,a) -> is_dependant a
  |DLAMV(_,a) -> List.exists is_dependant (Array.to_list a)
  |VAR _ -> false
  |Rel _ -> true;;

(* Give, for a prod type, the list of not-dependant types with their number of
   occurences *)
let list_not_dep typ=
  let (tab,_)=decomp_prod_array typ
  and lst=ref []
  and elem=ref (DOP0(Implicit))
  in
    let rec update_lst elem=function
       [] -> failwith "list_not_dep"
      |a::b ->
        if eq_univ((fst a),elem) then
          (fst a,(snd a)+1)::b
        else
          a::(update_lst elem b)
    in
      for cmr=0 to ((Array.length tab)-1) do
        elem:=snd(tab.(cmr));
        if not(is_dependant(!elem)) then
          if List.exists (function a -> eq_univ((!elem),fst a)) (!lst) then
            lst:=update_lst (!elem) (!lst)
          else
            lst:=(!lst)@[(!elem,1)]
      done;
      !lst;;

(* Optimization before compare modulo swap *)
let opt_not_dep typ0 typ1=
  let lst0=ref (list_not_dep typ0)
  and lst1=ref (list_not_dep typ1)
  in
    let rec put_off elem=function
       [] -> failwith "opt_not_dep"
      |a::b ->
        if eq_univ((fst a),elem) then
          b
        else
          a::(put_off elem b)
    and comp()=
      if (!lst0=[]) then
        true
      else
        (let elem=List.hd (!lst0)
         in
           try
             (if (assoc_univ (fst elem) (!lst1))=(snd elem) then
                (lst0:=List.tl (!lst0);
                 lst1:=put_off (fst elem) (!lst1);
                 comp())
              else
                false)
           with
              Not_found -> false)
    in
      if (List.length (!lst0))=(List.length (!lst1)) then
        comp()
      else
        false;;

(* Another optimization which compares the goals *)
let goals_comp typ0 typ1=
  let (_,dest0)=decomp_prod_array typ0
  and (_,dest1)=decomp_prod_array typ1
  and coh_lst=ref []
  in
    let rec is_coherent (rel0,rel1)=
      try
        (rel1=(List.assoc rel0 (!coh_lst)))
      with
        Not_found ->
          coh_lst:=(!coh_lst)@[(rel0,rel1)];
          true
    and comp_rec n m =
      if m=n then
        true
      else
        match (strip_head_cast m,strip_head_cast n) with
           (DOP2(Cast,c1,_),c2) 	     -> comp_rec c1 c2
         | (c1,DOP2(Cast,c2,_))              -> comp_rec c1 c2
         | (Rel p1,Rel p2)                   ->
             (is_coherent (p1,p2))
         | (DOPN(oper1,cl1),DOPN(oper2,cl2)) ->
      	       	 oper1=oper2 & for_all2eq_vect comp_rec cl1 cl2
         | (DOP0 oper1,DOP0 oper2)           -> 
           (match (oper1,oper2) with
               (Sort(Type a),Sort(Type b)) -> true
              |(_,_) -> oper1=oper2)
         | (DOP1(i,c1),DOP1(j,c2))           -> (i=j) & comp_rec c1 c2
         | (DOP2(i,c1,c1'),DOP2(j,c2,c2'))   ->
      	       	 (i=j) & comp_rec c1 c2 & comp_rec c1' c2'
         | (DLAM(_,c1),DLAM(_,c2)) 	     -> comp_rec c1 c2
         | (DLAMV(_,cl1),DLAMV(_,cl2))       -> 
      	       	 for_all2eq_vect comp_rec cl1 cl2
         | _ 				     -> false
    in
      let comp_rec_constr ((m:constr),(n:constr))=comp_rec m n
      in
        comp_rec_constr (dest0,dest1);;

(* Compare 2 types modulo swap *)
let swap_iso cmr equal (a,b)=
  let nba=nb_prod a
  and nbb=nb_prod b
  and aut=ref false
  and lst=ident_rel_list cmr a
  and permc=ref [||]
  in
    let aneg=put_neg_rel lst a
    in
      let perm=permut nba
      in
        if (nba=nbb) then
          if (nba=0) or (nba=1) then
            (equal(a,b),[||])
          else if (opt_not_dep a b)&(goals_comp a b) then
            (try
               (while not(!aut) do
                  try
                    (permc:=perm();
                     aut:=equal((permut_term lst aneg (permut_array_list
                          (!permc))),b))
                  with
                     Permut_impossible -> ()
                done;
                (!aut,!permc))
             with
                Permut_finished -> (false,[||]))
          else
            (false,[||])
        else
          (false,[||]);;

(******** Normal forms' comparison ********)
let nf_bse=ref (DOP0(Implicit));;

(* Gives the number of bounded variables with an array of occurences for a
   term *)
let bnd_var term=
  let lst=ref []
  in
    let rec update_lst rel_neg=function
       [] -> [(rel_neg,1)]
      |a::b ->
        if (fst a)=rel_neg then
          ((fst a),(snd a)+1)::b
        else
          a::(update_lst rel_neg b)
    and fill=function
       DOP0 _ -> ()
      |DOP1(_,a) -> fill a
      |DOP2(_,a,b) ->
        fill a;
        fill b
      |DOPN(_,a) -> List.iter fill (Array.to_list a)
      |DOPL(_,a) -> List.iter fill a
      |DLAM(_,a) -> fill a
      |DLAMV(_,a) -> List.iter fill (Array.to_list a)
      |VAR _ -> ()
      |Rel a -> lst:=update_lst a (!lst)
    and sort tab=
      let modf=ref true
      and cmr=ref 0
      and lgr=Array.length tab
      and aux=ref 0
      in
        while (!modf) do
          modf:=false;
          while (!cmr)<(lgr-1) do
            if tab.(!cmr)>tab.((!cmr)+1) then
              (aux:=tab.(!cmr);
               Array.set tab (!cmr) (tab.((!cmr)+1));
               Array.set tab ((!cmr)+1) (!aux);
               modf:=true);
            incr cmr
          done;
          cmr:=0
        done             
    and fill_tab tab cmr=function
       [] -> ()
      |a::b ->
        Array.set tab cmr (snd a);
        fill_tab tab (cmr+1) b
    in
      fill term;
      (let tab=Array.create (List.length (!lst)) 0
       in
         fill_tab tab 0 (!lst);
         sort tab;
         (List.length (!lst),tab));;

(* Order on the arrays of occurences *)
let strict_gt_occ (tab0,tab1)=
  let ans=ref false
  in
    for cmr=0 to ((Array.length tab0)-1) do
      if tab0.(cmr)>tab1.(cmr) then
        ans:=true
    done;
    !ans;;

(* Returns a list of (Name,Negative Rel) *)
let rel_neg_lst lst_sigma=
  let lst=ref []
  and cmr=ref (-1)
  and tab_sigma=Array.of_list lst_sigma
  in
    for aux=0 to ((Array.length tab_sigma)-1) do
      if not((fst (tab_sigma.(aux)))=Anonymous) then
        (lst:=(!lst)@[(fst (tab_sigma.(aux)),!cmr)];
         decr cmr)
    done;
    (!lst,!cmr);;

(* Puts the negative Rels *)
let chg_neg_rel lst_rel lst_sigma=
  let rec one_rel rel_neg rel=function
     [] -> []
    |a::b ->
      if (fst a)=Anonymous then
        [(fst a,subs_rel rel (Rel rel_neg) (snd a))]@(one_rel rel_neg rel b)
      else
        [(fst a,subs_rel rel (Rel rel_neg) (snd a))]@(one_rel rel_neg (rel+1)
        b)
  and apply lst_rel=function
     [] -> []
    |a::b ->
      if (fst a)=Anonymous then
        a::(apply lst_rel b)
      else
        a::(apply lst_rel (one_rel (List.assoc (fst a) lst_rel) 1 b))
  in
    apply lst_rel lst_sigma;;

(* Forms the list of classes from list_sigma *)
let lst_classes neg_lst_sigma=
  let lst_cls=ref []
  in
    let rec update_lst_cls (cls,cpl)=function
       [] -> [(cls,[cpl])]
      |a::b ->
        if (fst(fst a))=(fst cls) then
          if (snd(fst a))=(snd cls) then
            (fst a,(snd a)@[cpl])::b
          else if strict_gt_occ (snd(fst a),snd cls) then
            (cls,[cpl])::a::b
          else
            a::(update_lst_cls (cls,cpl) b)
        else if (fst(fst a))>(fst cls) then
          (cls,[cpl])::a::b
        else
          a::(update_lst_cls (cls,cpl) b)
    and apply=function
       [] -> ()
      |a::b -> lst_cls:=update_lst_cls (bnd_var (snd a),a) (!lst_cls)
    in
      apply neg_lst_sigma;
      !lst_cls;;

(* Equality which constructs a list of coherence for Negative Rel *)
let equal_self (typ0,typ1)=
  let coh_lst=ref []
  in
    let rec is_coherent (rel0,rel1)=
      try
        (rel1=(List.assoc rel0 (!coh_lst)))
      with
        Not_found ->
          coh_lst:=(!coh_lst)@[(rel0,rel1)];
          true
    and comp_rec n m =
      if m=n then
        true
      else
        match (strip_head_cast m,strip_head_cast n) with
           (DOP2(Cast,c1,_),c2) 	     -> comp_rec c1 c2
         | (c1,DOP2(Cast,c2,_))              -> comp_rec c1 c2
         | (Rel p1,Rel p2)                   ->
           if p1>0 then
             p1=p2
           else
             (is_coherent (p1,p2))
         | (DOPN(oper1,cl1),DOPN(oper2,cl2)) ->
      	       	 oper1=oper2 & for_all2eq_vect comp_rec cl1 cl2
         | (DOP0 oper1,DOP0 oper2)           -> 
           (match (oper1,oper2) with
               (Sort(Type a),Sort(Type b)) -> true
              |(_,_) -> oper1=oper2)
         | (DOP1(i,c1),DOP1(j,c2))           -> (i=j) & comp_rec c1 c2
         | (DOP2(i,c1,c1'),DOP2(j,c2,c2'))   ->
      	       	 (i=j) & comp_rec c1 c2 & comp_rec c1' c2'
         | (DLAM(_,c1),DLAM(_,c2)) 	     -> comp_rec c1 c2
         | (DLAMV(_,cl1),DLAMV(_,cl2))       -> 
      	       	 for_all2eq_vect comp_rec cl1 cl2
         | _ 				     -> false
    in
      let comp_rec_constr ((m:constr),(n:constr))=comp_rec m n
      in
        comp_rec_constr (typ0,typ1);;

(* Optimization based on the idea of classes *)
(*let opt_cls (nlst0,cmr0) nlst1=
  let lst_cls0=lst_classes nlst0
  and lst_cls1=lst_classes nlst1
  in
    let rec comp_lists=function
       ((a::b),(c::d)) ->
         if (fst a)=(fst c) then
           if (List.length (snd a))=(List.length (snd c)) then
             comp_lists (b,d)
           else
             false
         else
           false
      |([],[]) -> true
      |(_,_) -> failwith "opt_cls"
    and comp_classes cls0 cls1=
      let rec one_list elem=function
         [] -> []
        |a::b ->
          if swap_iso cmr0 equal_self (snd a,snd elem) then
            b
          else
            a::(one_list elem b)
      and two_two lst0 lst1=
        let lgr0=List.length lst0
        and aux1=ref lst1
        and mem=ref lst1
        in
          try
            (for cmr0=0 to (lgr0-1) do
               aux1:=one_list (List.nth lst0 cmr0) !aux1;
               if (!mem)=(!aux1) then
                 failwith "fail_opt_cls"
               else
                 mem:=(!aux1)
             done;
             true)
          with
             Failure "fail_opt_cls" -> false
      in
        two_two (snd cls0) (snd cls1)
    in
      if (List.length lst_cls0)=(List.length lst_cls1) then
        if comp_lists (lst_cls0,lst_cls1) then
          List.for_all2 comp_classes lst_cls0 lst_cls1
        else
          false
      else
        false;;*)

(* Permuts a term of components *)
let permut_sigma_term lst_neg lst_sigma_neg permcoup=
  let tab_term=Array.of_list lst_sigma_neg
  and tabcoup=Array.of_list permcoup
  and occur_in_array relneg tab=
    try
      (for cmr=0 to ((Array.length tab)-1) do
         if rel_neg_occur relneg (snd (tab.(cmr))) then
           raise Intern
       done);
      false
    with
       Intern -> true
  in
    let tabperm=Array.copy tab_term
    and lgr=Array.length tab_term
    in
      for cmr=0 to (lgr-1) do
        Array.set tabperm cmr (tab_term.(snd (tabcoup.(cmr))))
      done;
      for cmr=0 to (lgr-1) do
        (if not((fst (tabperm.(cmr)))=Anonymous) then
          if (occur_in_array (List.assoc (fst (tabperm.(cmr))) lst_neg)
             (Array.sub tabperm 0 cmr)) then
            raise Permut_impossible)
      done;
      Array.to_list tabperm;;

(* Idem as List.assoc but with ((a,b),_) *)
let rec assoc2 a=function
   [] -> raise Not_found
  |((b,c),_)::d ->
    if a=b then
      c
    else
      assoc2 a d;;

(* Equality modulo a fixed list of coherence *)
let equal_comp coh_lst (typ0,typ1)=
  let rec is_coherent (rel0,rel1)=
    try
      (rel1=(assoc2 rel0 coh_lst))
    with
       Not_found ->
         failwith "equal_comp"
  and comp_rec n m =
      if m=n then
        true
      else
        match (strip_head_cast m,strip_head_cast n) with
           (DOP2(Cast,c1,_),c2) 	     -> comp_rec c1 c2
         | (c1,DOP2(Cast,c2,_))              -> comp_rec c1 c2
         | (Rel p1,Rel p2)                   ->
           if p1>0 then
             p1=p2
           else
             (is_coherent (p1,p2))
         | (DOPN(oper1,cl1),DOPN(oper2,cl2)) ->
      	       	 oper1=oper2 & for_all2eq_vect comp_rec cl1 cl2
         | (DOP0 oper1,DOP0 oper2)           -> 
           (match (oper1,oper2) with
               (Sort(Type a),Sort(Type b)) -> true
              |(_,_) -> oper1=oper2)
         | (DOP1(i,c1),DOP1(j,c2))           -> (i=j) & comp_rec c1 c2
         | (DOP2(i,c1,c1'),DOP2(j,c2,c2'))   ->
      	       	 (i=j) & comp_rec c1 c2 & comp_rec c1' c2'
         | (DLAM(_,c1),DLAM(_,c2)) 	     -> comp_rec c1 c2
         | (DLAMV(_,cl1),DLAMV(_,cl2))       -> 
      	       	 for_all2eq_vect comp_rec cl1 cl2
         | _ 				     -> false
    in
      let comp_rec_constr ((m:constr),(n:constr))=comp_rec m n
      in
        comp_rec_constr (typ0,typ1);;

(* Idem as List.assoc but with ((a,_),b) *)
let rec assoc3 a=function
   [] -> raise Not_found
  |((b,_),c)::d ->
    if a=b then
      c
    else
      assoc3 a d;;

(* Applies a permutation perm to tab *)
let apply_perm perm tab=
  let tabperm=Array.copy tab
  and lgr=Array.length tab
  in
    for cmr=0 to (lgr-1) do
      tabperm.(cmr) <- (tab.(perm.(cmr)))
    done;
    tabperm;;

(* Applies the coherence list to a term *)
let rec apply_coh_lst coh_lst=function
   DOP1(a,b) -> DOP1(a,apply_coh_lst coh_lst b)
  |DOP2(a,b,c) -> DOP2(a,apply_coh_lst coh_lst b,apply_coh_lst coh_lst c)
  |DOPN(AppL,a) ->
    (match (a.(0)) with
        Rel b ->
          (try
             (let lgr_par=(Array.length a)-1
              and perm=assoc3 b coh_lst
              in
                if (lgr_par=(Array.length perm)) then
                  DOPN(AppL,Array.map (apply_coh_lst coh_lst) (Array.append
                  [|Rel b|] (apply_perm perm (Array.sub a 1 lgr_par))))
                else
                  DOPN(AppL,Array.map (apply_coh_lst coh_lst) a))
           with
              Not_found -> DOPN(AppL,Array.map (apply_coh_lst coh_lst) a))
       |_ -> DOPN(AppL,Array.map (apply_coh_lst coh_lst) a))
  |DOPN(a,b) -> DOPN(a,Array.map (apply_coh_lst coh_lst) b)
  |DOPL(a,b) -> DOPL(a,List.map (apply_coh_lst coh_lst) b)
  |DLAM(a,b) -> DLAM(a,apply_coh_lst coh_lst b)
  |DLAMV(a,b) -> DLAMV(a,Array.map (apply_coh_lst coh_lst) b)
  |a -> a;;

(* Compares sigma expressions *)
let sigma_iso (nlst0,cmr0) nlst1 ns_lst0 ns_lst1=
  let coh_lst=ref []
  and mem=ref []
  in
    let term_to_term cpl0 cpl1=
      if (fst cpl0)=Anonymous then
        if (fst cpl1)=Anonymous then
          fst (swap_iso cmr0 (equal_comp (!coh_lst)) (apply_coh_lst (!coh_lst)
          (snd cpl0),(snd cpl1)))
        else
          false
      else
        if (fst cpl1)=Anonymous then
          false
        else
          (let res=swap_iso cmr0 (equal_comp (!mem)) (apply_coh_lst (!coh_lst)
                   (snd cpl0),(snd cpl1))
           in
             coh_lst:=(!coh_lst)@[((List.assoc (fst cpl0) nlst0,List.assoc (fst
                      cpl1) nlst1),snd res)];
             fst res)
    in
      (*if (opt_cls (ns_lst0,cmr0) ns_lst1) then*)
      if (List.length ns_lst0)=(List.length ns_lst1) then
        (let perm=permut (List.length ns_lst0)
         and ans=ref false
         in
           (try
              (while not(!ans) do
                 try
                   (ans:=List.for_all2 term_to_term (permut_sigma_term nlst0
                         ns_lst0 (permut_array_list (perm()))) ns_lst1)
                 with
                    Permut_impossible -> ()
               done)
            with
               Permut_finished -> ());
           !ans)
      else
        false;;

(******** Decision procedure ********)
(* Renames if a variable occurs more than one time *)
let rename term=
  let rec rename_rec lstv=function
     DOP1(a,b) ->
       let resb=rename_rec lstv b
       in
         (fst resb,DOP1(a,snd resb))
    |DOP2(a,b,c) ->
      let resb=rename_rec lstv b
      in
        let resc=rename_rec (fst resb) c
        in
          (fst resc,DOP2(a,snd resb,snd resc))
    |DOPN(a,b) ->
      let lgr=Array.length b
      in
        if lgr=0 then
          (lstv,DOPN(a,b))
        else
          let aux=ref lstv
          and tab=Array.create lgr ([],DOP0(Implicit))
          in
            for cmr=0 to (lgr-1) do
              tab.(cmr) <- (rename_rec (!aux) (b.(cmr)));
              aux:=(fst (tab.(cmr)))
            done;
            (!aux,DOPN(a,Array.map snd tab))
    |DOPL(a,b) -> 
      let c=Array.of_list b
      and lgr=List.length b
      in
        if lgr=0 then
          (lstv,DOPL(a,b))
        else
          let aux=ref lstv
          and tab=Array.create lgr ([],DOP0(Implicit))
          in
            for cmr=0 to (lgr-1) do
              tab.(cmr) <- (rename_rec (!aux) (c.(cmr)));
              aux:=(fst (tab.(cmr)))
            done;
            (!aux,DOPL(a,Array.to_list(Array.map snd tab)))
    |DLAM(a,b) ->
      (match a with
          Anonymous ->
            let resb=rename_rec lstv b
            in
              (fst resb,DLAM(a,snd resb))
         |Name c ->
           let aux=ref (string_of_id c)
           in
             while (List.mem (!aux) lstv) do
               aux:=("#"^(!aux))
             done;
             (let resb=rename_rec (lstv@[!aux]) b
              in
                (fst resb,DLAM(Name (id_of_string (!aux)),snd resb))))
    |DLAMV(a,b) ->
      let lgr=Array.length b
      in
        if lgr=0 then
          (lstv,DLAMV(a,b))
        else
          let aux=ref lstv
          and tab=Array.create lgr ([],DOP0(Implicit))
          in
            for cmr=0 to (lgr-1) do
              tab.(cmr) <- (rename_rec (!aux) (b.(cmr)));
              aux:=(fst (tab.(cmr)))
            done;
            (!aux,DLAMV(a,Array.map snd tab))
    |a -> (lstv,a)
  in
    snd (rename_rec [] term);;

(* Tests if user_type and nfb are isomorphic *)
let iso_dep nfb=
  let lsta=list_of_sigma(!nf_bse)
  and lstb=list_of_sigma(nfb)
  in
    let (nlsta,cmra)=rel_neg_lst lsta
    and (nlstb,_)=rel_neg_lst lstb
    in
      let ns_lsta=chg_neg_rel nlsta lsta
      and ns_lstb=chg_neg_rel nlstb lstb
      in
        sigma_iso (nlsta,cmra) nlstb ns_lsta ns_lstb;;

(******** Types' display ********)
(* The table of the normal forms *)
let rew_tab=Hashtbl.create 51;;

(* To know the search time *)
let seetime=ref false;;


(* Prints a variable and its type *)
let print_var2 name typ =
    [< 'sTR "*** [" ; 'sTR name ; 'sTR " : ";
     prterm typ; 
     'sTR "]" >];;

(* For print_mutual2 *)
let rec stream_map=function
   [] -> [<>]
  |a::b -> [< a ; stream_map b >];;

(* Applies iso_dep to mutual inductive types *)
let print_mutual2 name mib=
 let {mINDKIND=pk; mINDPACKETS=mipv; mINDNPARAMS=nparams} = mib in 
 let (lpars,_) = decomp_n_prod empty_evd nparams mipv.(0).mINDARITY.body in
 let lparsname = List.map fst lpars in
 let lparsprint = assumptions_for_print lparsname in
 let inspect_constructors mip=
   let lst=modify_constructors (name^(string_of_id mipv.(0).mINDTYPENAME)) pk
           mip.mINDLC
   in
     let lidC=
       List.map2 (fun id c -> (id,c)) (Array.to_list mip.mINDCONSNAMES) lst
     in
       let rec inspect_rec=function
          [] -> [<>]
         |a::b ->
           let nf=
             try
               (Hashtbl.find rew_tab (name^"<"^
                (string_of_id mip.mINDTYPENAME)^">#"^(string_of_id (fst a))))
             with
                Not_found ->
                   let nfcal=rename(nf_dep(Reduction.nf_betaiota (ind_rew
                     (Reduction.nf_betaiota (snd a)))))
                   in
                     Hashtbl.add rew_tab (name^"<"^
                     (string_of_id mip.mINDTYPENAME)^">#"^(string_of_id
                     (fst a))) nfcal;
                     nfcal
           in
             if (iso_dep nf) then
               [< 'sTR (name^"<"^
                  (string_of_id mip.mINDTYPENAME)^">#"^(string_of_id (fst a))^
                  ":") ; prterm (snd a) ; 'fNL ; inspect_rec b >]
             else
               inspect_rec b
       in
         inspect_rec lidC
 in
 let print_oneind name namedis mip = 
     let arity0=mip.mINDARITY.body
     and (_,arity1)=decomp_n_prod empty_evd nparams mip.mINDARITY.body in
       let nf=
         try
           (Hashtbl.find rew_tab (name^(string_of_id mip.mINDTYPENAME)))
         with
            Not_found ->
              let nfcal=rename(nf_dep(Reduction.nf_betaiota (ind_rew
                (Reduction.nf_betaiota arity0))))
              in
                Hashtbl.add rew_tab (name^(string_of_id mip.mINDTYPENAME))
                nfcal;
                nfcal
       in
       (if (iso_dep nf)  then
         (hOV 0 [< 'sTR namedis; print_id mip.mINDTYPENAME ;
                   if nparams = 0 then [<>]
                   else [< 'sTR" ["; print_env CCI (List.rev lpars); 
                   'sTR "]">]; 'bRK(1,5);  'sTR " : ";
                   term0 lparsprint arity1; 'fNL ;
                   inspect_constructors mip >] )
        else
          [< inspect_constructors mip >]) in
  let mip = mipv.(0) in
(* Case one [co]inductive *)
 if Array.length mipv = 1 then 
   (let arity0=mip.mINDARITY.body
    and (_,arity1)=decomp_n_prod empty_evd nparams mip.mINDARITY.body in
      let nf=
         try
           (Hashtbl.find rew_tab (name^(string_of_id mip.mINDTYPENAME)))
         with
            Not_found ->
              let nfcal=rename(nf_dep(Reduction.nf_betaiota (ind_rew
                (Reduction.nf_betaiota arity0))))
              in
                Hashtbl.add rew_tab (name^(string_of_id mip.mINDTYPENAME))
                nfcal;
                nfcal
      in
        (if (iso_dep nf) then
      (let sfinite = if mip.mINDFINITE then "Inductive " else "CoInductive " in
       (hOV 0 [< 'sTR (name^sfinite) ; print_id mip.mINDTYPENAME ;
       if nparams = 0 then [<>] 
                      else [< 'sTR" ["; print_env CCI (List.rev lpars); 
                 'sTR "]">]; 'bRK(1,5); 'sTR ": "; 
                 term0 lparsprint arity1 ; 'fNL ; inspect_constructors
                 mip >]))
     else
       [<inspect_constructors mip>]))
(* Mutual [co]inductive definitions *)
 else 
  let (mipli,miplc) = List.fold_left 
      (fun (li,lc) mi -> if mi.mINDFINITE then (li@[mi],lc) else (li,lc@[mi]))
      ([],[]) (Array.to_list mipv) in 
        let strind =
          if mipli = [] then [<>] 
          else  [< stream_map (List.map (print_oneind name
                   (name^"Mutual Inductive ")) mipli) >]
      and strcoind = 
        if miplc = [] then [<>] 
        else  [< stream_map (List.map (print_oneind name
                   (name^"Mutual CoInductive ")) miplc) >] 
  in
 (hOV 0 [< if mip.mINDFINITE then [< strind; strcoind >] else [<strcoind;
           strind>] >])
;;

(* Applies iso_dep to objects that are variable, constants or mutual inductive
   types *)
let print_leaf_entry2 name2 (spopt,lobj)=
  let tag = object_tag lobj in
  match (spopt,tag) with
    (_,"VARIABLE") ->
    let (name,(typ,_),_,l,_,_) = outVariable lobj
    in
      let nf=
        try
          (Hashtbl.find rew_tab (name2^(string_of_id name)))
        with
           Not_found ->
             let nfcal=rename(nf_dep(Reduction.nf_betaiota (ind_rew
               (Reduction.nf_betaiota typ.body))))
             in
               Hashtbl.add rew_tab (name2^(string_of_id name)) nfcal;
               nfcal
      in
        if (iso_dep nf) then
          [< 'sTR name2 ; print_var2 (string_of_id name) typ.body; 
            print_impl_args (list_of_implicits l) ; 'fNL>]
        else
          [<>]
  | (sp,"CONSTANT") ->
    let (cmap,_,_) = outConstant lobj in
    if List.mem_assoc CCI cmap then
      let {cONSTBODY=val_0;cONSTTYPE=typ;cONSTIMPARGS=l} = List.assoc CCI cmap
      in
        let nf=
          try
            (Hashtbl.find rew_tab (name2^(string_of_id (basename sp))))
          with
             Not_found ->
               let nfcal=rename(nf_dep(Reduction.nf_betaiota (ind_rew
                 (Reduction.nf_betaiota typ.body))))
               in
                 Hashtbl.add rew_tab (name2^(string_of_id (basename sp)))
                 nfcal;
                 nfcal
        in
          if (iso_dep nf) then 
            hOV 0 [< (match val_0 with 
              None -> [< 'sTR (name2^"--* [") ;
                         'sTR (string_of_id (basename sp)) ;  'sTR " : ";
                         'cUT ; prtype typ ; 'sTR"]" >]
            | _    -> [< 'sTR(name2^(string_of_id (basename sp))) ;
                         'sTR ":" ; 'cUT ;  prtype typ  >]);
                         print_impl_args (list_of_implicits l) ; 'fNL >]
          else
            [<>]
    else
      hOV 0 [< 'sTR(name2^"Fw constant ") ; 
               'sTR (print_basename (fwsp_of sp)) ; 'fNL>]

  | (sp,"MUTUALINDUCTIVE") ->
    let (cmap,_) = outMutualInductive lobj in
    if List.mem_assoc CCI cmap then
      [< print_mutual2 name2 (List.assoc CCI cmap) >]
    else
      hOV 0 [< 'sTR(name2^"Fw inductive definition ") ; 
               'sTR (print_basename (fwsp_of sp)) ; 'fNL >]
  |_ -> [<>]
;;

(* Searchs in context *)
let rec print_library_entry2 depth name=function
   (sp,Lib.LEAF lobj) ->
     (*print_endline (string_of_path sp);
     flush stdout;*)
     [< print_leaf_entry2 name (sp,lobj) >]
  |(_,Lib.ClosedDir(a,_,_,ctxt)) ->
    if depth=0 then
      if not(a="Cpo") then
        [< print_context2 1 (name^a^"#") ctxt >]
      else
        [<>]
    else
      [<>]
  |_ -> [<>]
and print_context2 depth name= 
 let rec prec = function
    [] -> [<>]
   |h::rest -> [< print_library_entry2 depth name h ; prec rest >]
 in prec
;;

(* Search2 for coqtop *)
let type_search_in_env typ_bse=
  let begin_time = System.timestamp () in
  upd_tbl_ind_one();
  nf_bse:=rename(nf_dep(Reduction.nf_betaiota (ind_rew (Reduction.nf_betaiota
    typ_bse))));
  mSG(print_context2 0 "#" (List.rev(Lib.contents_after None)));
  if (!seetime) then
    (let end_time = System.timestamp ()
     in mSGNL [< 'sTR"Finished transaction in " ;
                 System.fmt_time_difference begin_time end_time >]
     )
;;

(* Search2 for coq_search2_top *)
let type_search_in_lib typ_bse=
  let begin_time = System.timestamp () in
  nf_bse:=rename(nf_dep(Reduction.nf_betaiota (ind_rew (Reduction.nf_betaiota
    typ_bse))));
  mSG(print_context2 0 "#" (List.rev(Lib.contents_after None)));
  if (!seetime) then
     (let end_time = System.timestamp ()
     in mSGNL [< 'sTR"Finished transaction in " ;
                 System.fmt_time_difference begin_time end_time >]
     )
;;

let type_search typ_bse =
  if !search_in_lib then type_search_in_lib typ_bse
  else type_search_in_env typ_bse
;;

(******** Light Require ********)
(* Filter which keeps only constants, variables and mutual inductive types *)
let filter_s node=
  let rec filter_node node=
    (match node with
        Lib.ClosedDir(st,od,cd,ls) -> Lib.ClosedDir(st,od,cd,filter_ls ls)
       |_ -> failwith "filter")
  and filter_ls=function
     [] -> []
    |a::b ->
      (match a with
          (sp,Lib.LEAF obj) ->
            let tag=object_tag obj
            in
              if (tag="CONSTANT") or (tag="GRAMMAR") or (tag="MUTUALINDUCTIVE")
                 or (tag="PPSYNTAX") or (tag="TOKEN") or (tag="VARIABLE") then
                (sp,Lib.LEAF (extract_object_specification obj))::(filter_ls b)
              else
                filter_ls b
         |(_,Lib.Import(_,_,_)) -> a::(filter_ls b)
         |_ -> filter_ls b)
  in
    filter_node node;;

(* Light internings *)
let rec intern_module2 spec_only s ofile =
    let filename = match ofile with Some f -> f | None -> s in
    let (sp,d) = disk_intern_module spec_only filename in
    (match (filter_s d) with
      Lib.ClosedDir(_,odc,_,_) as d ->
        let impbefore = needed_modules d in
        List.iter
	  (fun sp ->
            if not(module_is_read sp) then
              intern_module2 spec_only (string_of_id(basename sp)) None)
          impbefore;
        Lib.add_module (sp,d)
    | _ -> failwith "Searchisos.intern_module_gen2")
;;

let require_module2 spec name ofile locally =
  let filename = match ofile with Some f -> f | None -> name in
  if not (module_is_known name)
  then intern_module2 (spec_on_path spec filename) name ofile;
  import_export_module name locally
;;

(******** Load the rew_tab ********)
(* Rewrite the type of the mutual inductive type and the types of its
   constructors *)
let load_mutual name mib=

 let {mINDKIND=pk; mINDPACKETS=mipv; mINDNPARAMS=nparams} = mib in 
 let (lpars,_) = decomp_n_prod empty_evd nparams mipv.(0).mINDARITY.body in
 let lparsname = List.map fst lpars in
 let lparsprint = assumptions_for_print lparsname in
 let inspect_constructors mip=
   let lst=modify_constructors (name^(string_of_id mipv.(0).mINDTYPENAME)) pk
           mip.mINDLC
   in
     let lidC=
       List.map2 (fun id c -> (id,c)) (Array.to_list mip.mINDCONSNAMES) lst
     in
       let rec inspect_rec=function
          [] -> ()
         |a::b ->
           Hashtbl.add rew_tab (name^"<"^(string_of_id mip.mINDTYPENAME)^">#"^
           (string_of_id(fst a))) (rename(nf_dep(Reduction.nf_betaiota (ind_rew
             (Reduction.nf_betaiota (snd a))))));
           inspect_rec b
       in
         inspect_rec lidC
 in
 let print_oneind name mip = 
     let arity0=mip.mINDARITY.body
     and (_,arity1)=decomp_n_prod empty_evd nparams mip.mINDARITY.body in
       Hashtbl.add rew_tab (name^(string_of_id mip.mINDTYPENAME)) (rename
       (nf_dep(Reduction.nf_betaiota (ind_rew (Reduction.nf_betaiota
       arity0)))));
       inspect_constructors mip in
  let mip = mipv.(0) in
(* Case one [co]inductive *)
 if Array.length mipv = 1 then 
   (let arity0=mip.mINDARITY.body
    and (_,arity1)=decomp_n_prod empty_evd nparams mip.mINDARITY.body in
      Hashtbl.add rew_tab (name^(string_of_id mip.mINDTYPENAME)) (rename(nf_dep
      (Reduction.nf_betaiota (ind_rew (Reduction.nf_betaiota arity0)))));
      inspect_constructors mip)
(* Mutual [co]inductive definitions *)
 else 
  let (mipli,miplc) = List.fold_left 
      (fun (li,lc) mi -> if mi.mINDFINITE then (li@[mi],lc) else (li,lc@[mi]))
      ([],[]) (Array.to_list mipv) in 
    if mip.mINDFINITE then
      List.iter (print_oneind name) (mipli@miplc)
    else
      List.iter (print_oneind name) (miplc@mipli)
;;

(* Rewrite the types of the variables, constants and mutual inductive types *)
let load_leaf_entry name2 (spopt,lobj)=
  let tag = object_tag lobj in
  match (spopt,tag) with
    (_,"VARIABLE") ->
    let (name,(typ,_),_,l,_,_) = outVariable lobj
    in
      Hashtbl.add rew_tab (name2^(string_of_id name)) (rename(nf_dep
      (Reduction.nf_betaiota (ind_rew (Reduction.nf_betaiota typ.body)))))
  | (sp,"CONSTANT") ->
    let (cmap,_,_) = outConstant lobj in
    if List.mem_assoc CCI cmap then
      let {cONSTBODY=val_0;cONSTTYPE=typ;cONSTIMPARGS=l} = List.assoc CCI cmap
      in
        Hashtbl.add rew_tab (name2^(string_of_id (basename sp))) (rename(nf_dep
        (Reduction.nf_betaiota (ind_rew (Reduction.nf_betaiota typ.body)))))
  | (sp,"MUTUALINDUCTIVE") ->
    let (cmap,_) = outMutualInductive lobj in
    if List.mem_assoc CCI cmap then
      load_mutual name2 (List.assoc CCI cmap)
  |_ -> ()
;;

(* $Id: searchisos.ml,v 1.15 1999/10/29 23:19:09 barras Exp $ *)
