(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA                        ENS-CNRS                *)
(*              Rocquencourt                        Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               Jul 10th 1997                              *)
(*                                                                          *)
(****************************************************************************)
(*                               cmd_line.ml                                *)
(****************************************************************************)

open Std;;
open Pp;;
open Coqtoplevel;;
open Cmdobj;;
open Initial;;
open States;;
open System;;

let print_header () =
  Printf.printf "Welcome to Coq V%s (%s)\n" Coq_config.version Coq_config.date;
  flush stdout;;

let batch_mode = ref false
let set_batch_mode () = batch_mode := true

let remove_top_ml () = Mltop.remove ()

let inputstate = ref "tactics.coq"
let set_inputstate s = inputstate:= s
let inputstate () =if !inputstate <> "" then intern_restore_state !inputstate

let outputstate = ref ""
let set_outputstate s = outputstate:=s
let outputstate () = if !outputstate <> "" then extern_state !outputstate

let load_vernacular_list = ref ([]:string list)
let add_load_vernacular s =
  load_vernacular_list := (make_suffix s ".v")::(!load_vernacular_list)
let load_vernacular () =
  List.iter
    (fun s -> Vernac.load_vernac false s)
    (List.rev !load_vernacular_list)

let load_vernacular_obj = ref ([]:string list)
let add_vernac_obj s = load_vernacular_obj := s::(!load_vernacular_obj)
let load_vernac_obj () = 
  List.iter
    (fun s -> Library.read_module (Filename.basename s) (Some s))
    (List.rev !load_vernacular_obj)

let require_list = ref ([]:string list)
let add_require s = require_list := s::(!require_list)
let require () =
  List.iter
    (fun s -> Library.require_module None (Filename.basename s) (Some s) false)
    (List.rev !require_list)


(*
 * Parsing of the command line.
 *
 * We no longer use Arg.parse, in order to use Usage.print_usage to be coherent
 * with launch/coqtop and launch/coqc.
 *)

let usage () =
  if !batch_mode then
    Usage.print_usage_coqc ()
  else
    Usage.print_usage_coqtop () ;
  flush stderr ;
  exit 1


let parse_args () =
  let rec parse = function
      [] -> ()

    | ("-I"|"-include") :: d :: rem -> push_include d ; parse rem
    | ("-I"|"-include") :: []       -> usage ()

    | ("-R") :: d :: rem -> rec_include d ; parse rem
    | ("-R") :: []       -> usage ()

    | "-q" :: rem -> no_load_rc () ; parse rem

    | "-batch" :: rem -> set_batch_mode () ; parse rem
	     
    | "-outputstate" :: s :: rem -> set_outputstate s ; parse rem
    | "-outputstate" :: []       -> usage ()

    | "-nois" :: rem -> set_inputstate "" ; parse rem
	     
    | ("-inputstate"|"-is") :: s :: rem -> set_inputstate s ; parse rem
    | ("-inputstate"|"-is") :: []       -> usage ()

    | "-load-ml-object" :: f :: rem -> Mltop.dir_ml_load f ; parse rem
    | "-load-ml-object" :: []       -> usage ()

    | "-load-ml-source" :: f :: rem -> Mltop.dir_ml_use f ; parse rem
    | "-load-ml-source" :: []       -> usage ()

    | "-load-vernac-source" :: f :: rem -> add_load_vernacular f ; parse rem
    | "-load-vernac-source" :: []       -> usage ()

    | "-load-vernac-object" :: f :: rem -> add_vernac_obj f ; parse rem
    | "-load-vernac-object" :: []       -> usage ()

    | "-require" :: f :: rem -> add_require f ; parse rem
    | "-require" :: []       -> usage ()

    | "-unsafe" :: f :: rem -> add_unsafe f ; parse rem
    | "-unsafe" :: []       -> usage ()

    | "-debug" :: rem -> set_debug () ; parse rem

    | "-emacs" :: rem -> Printer.print_emacs := true ; parse rem

    | "-init-file" :: f :: rem -> set_rcfile f ; parse rem
    | "-init-file" :: []       -> usage ()

    | "-user" :: u :: rem -> set_rcuser u ; parse rem
    | "-user" :: []       -> usage ()

    | "-notactics" :: rem -> remove_top_ml () ; parse rem

    | "-just-parsing" :: rem -> Vernac.just_parsing := true ; parse rem

    | "-tac-debug" :: rem -> Logic.set_debug true ; parse rem

    | s :: _ -> prerr_endline ("Don't know what to do with " ^ s) ; usage ()

  in
    try
      parse (List.tl (Array.to_list Sys.argv))
    with 
      	UserError(_,s) as e -> begin
          try
	    Stream.empty s; exit 1
          with
	      Stream.Failure ->
		mSGNL (Errors.explain_sys_exn e); exit 1
	end
      | e -> begin mSGNL (Errors.explain_sys_exn e); exit 1 end
;;

(* To prevent from doing the initialization twice *)
let initialized = ref false;;

(* Ctrl-C is fatal during the initialisation *)
let start () =
  if not !initialized then
    begin
      initialized := true;
      Sys.catch_break false;
      try
        parse_args ();
        print_header ();
        init_load_path ();
        inputstate ();
 	load_vernac_obj ();
        require ();
        load_rcfile();
        load_vernacular ();
        outputstate ()
      with e ->
        flush_all();
        if not !batch_mode then message "Error during initialization :";
        mSGNL (Coqtoplevel.print_toplevel_error e);
        exit 1
    end;
  if !batch_mode then (flush_all(); Profile.print_profile ();exit 0);
  Coqtoplevel.go();;

(* $Id: cmd_line.ml,v 1.29 1999/11/30 19:25:43 mohring Exp $ *)
