/*	File:	  unparse.c
 *      Author:	  Henning Makholm <makholm@diku.dk>
 *      	  Sebastian Skalberg <skalberg@diku.dk>
 *      Content:  C-Mix speclib: Manage names and print residual program
 *
 *      Copyright  1999. The TOPPS group at DIKU, U of Copenhagen.
 *      Redistribution and modification are allowed under certain
 *      terms; see the file COPYING.cmix for details.
 */
  
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#define cmixSPECLIB_SOURCE
#include "speclib.h"
#include "code.h"
#include <assert.h>

/***************************************************************************/
/* Adding taboos to a name table                                           */
/***************************************************************************/

static void
addTaboo(struct cmixNameRec names[],unsigned const namesSorted[],
         unsigned toohigh, const char *taboo)
{
  char mycopy[31] ;
  char *pc ;
  unsigned thisseq ;
  unsigned lower ;
  if ( strlen(taboo)>=31 ||
       taboo[0] == '\0' ||
       isdigit((unsigned char)taboo[0]) )
    return ; /* generated names have maximal length of 30 */
  strcpy(mycopy,taboo);
  
  /* Strip of any trailing sequence of digits in the taboo */
  for ( pc=mycopy ; pc[0] ; pc++ )
    ;
  while ( isdigit((unsigned char)pc[-1]) )
    pc-- ;
  if ( strlen(pc) > 4 && (pc != mycopy+1 || mycopy[0] != 'v') )
    return; /* generated names have at most 4 digits at the end */
  
  if ( pc[0] ) {
    thisseq = cmix_atoi(pc);
    pc[0] = '\0' ;
  } else
    thisseq = 0 ;
  
  /* NOW do a binary search in the name table */
  lower = 0 ;
  while(lower < toohigh) {
    unsigned mid = (lower+toohigh)/2 ;
    int cmpres = strcmp(names[namesSorted[mid]].base,mycopy);
    if ( cmpres == 0 ) {
      unsigned index = namesSorted[mid] ;
      if ( names[index].nextseq_global <= thisseq )
        names[index].nextseq_global = thisseq+1 ;
      return ;
    }
    if ( cmpres < 0 )
      lower = mid+1 ;
    else
      toohigh = mid ;
  }
  /* the name part wasn't found: all is well */
}   

/***************************************************************************/
/* Traversing expressions                                                  */
/***************************************************************************/

typedef void (*Callback)(struct cmixExprName *,struct cmixNameRec[]) ;

static void traverseExpr(union cmixExpr *,Callback,struct cmixNameRec[]);
static void
traverseInner(struct cmixExprInner *e,Callback cb,struct cmixNameRec names[])
{
  unsigned i ;
  for ( i=0; e->child[i] ; i++ )
    traverseExpr(e->child[i],cb,names);
}
static void
traverseExpr(union cmixExpr *e,Callback cb,struct cmixNameRec names[])
{
  if ( e == NULL )
    return ;
  switch(e->common.tag) {
  case Inner:
    traverseInner(&e->inner,cb,names);
    break ;
  case NameRequest:
  case Name:
    cb(&e->name,names);
    break ;
  default:
    break ;
  }
}

/***************************************************************************/
/* Computing needed declarations                                           */
/***************************************************************************/

static void mentionName(struct cmixExprName *,struct cmixNameRec []);

static void
mentionDecl(struct cmixDecl *d)
{
  if ( d->status == Done )
    return ;
  d->status = Done ;
  traverseInner(&d->decl,mentionName,NULL);
  for ( d = d->members ; d ; d = d->next )
    if ( d->status == Sure )
      mentionDecl(d) ;
}

static void
mentionName(struct cmixExprName *n,struct cmixNameRec dummy[])
{
  struct cmixDecl *d ;
  for ( d = n->decl ; d ; d = d->sibling )
    mentionDecl(d);
}

static void
mentionFun(struct cmixFun *f)
{
  union cmixStmt *s ;
  int stmtReachable ;
  
  stmtReachable = 1 ;
  for ( s = f->stmts ; s ; s=s->common.next )
    switch(s->common.tag) {
    case Label:
      if ( stmtReachable )
        s->label.refcount++ ;
      stmtReachable = 1 ;
      break ;
    case Plain:
      traverseInner(&s->plain.expr,mentionName,NULL);
      break ;
    case Goto:
      s->jump.target->refcount++ ;
      stmtReachable = 0 ;
      break ;
    case If:
      s->cond.then_target->refcount++;
      s->cond.else_target->refcount++;
      traverseInner(s->cond.cond,mentionName,NULL);
      stmtReachable = 0 ;
      break ;
    case Return:
      traverseInner(&s->plain.expr,mentionName,NULL);
      stmtReachable = 0 ;
      break ;
    case Abort:
      traverseInner(&s->plain.expr,mentionName,NULL);
      stmtReachable = 0 ;
      break ;
    }
}

/***************************************************************************/
/* Name management functions                                               */
/***************************************************************************/

static void
assignGlobalName(struct cmixExprName *e,struct cmixNameRec names[])
{
  if ( e->tag != NameRequest )
    return ;
  e->tag = Name ;
 tryagain:
  if ( e->seq < names[e->index].nextseq_global )
    e->seq = names[e->index].nextseq_global ;
  if ( e->seq >= 10000 && e->index ) {
    /* panic! too many of these names defined - revert to
     * index 0 whose base name is "v".
     */
    e->index = 0 ;
    e->seq = 0 ;
    goto tryagain ;
  }
  names[e->index].nextseq_global = e->seq + 1 ;
}

static void
prepareLocalName(struct cmixExprName *e,struct cmixNameRec names[])
{
  names[e->index].nextseq_local = names[e->index].nextseq_global ;
}

static void
assignLocalName(struct cmixExprName *e,struct cmixNameRec names[])
{
  if ( e->tag != NameRequest )
    return ;
  e->tag = Name ;
 tryagain:
  if ( e->seq < names[e->index].nextseq_local )
    e->seq = names[e->index].nextseq_local ;
  if ( e->seq >= 10000 && e->index ) {
    /* panic! too many of these names defined - revert to
     * index 0 whose base name is "v".
     */
    e->index = 0 ;
    e->seq = 0 ;
    goto tryagain ;
  }
  names[e->index].nextseq_local = e->seq + 1 ;
}

/***************************************************************************/
/* Printing a cmixExpr                                                     */
/***************************************************************************/

static int
ExprOK(union cmixExpr *exp)
{
  if ( exp == NULL )
    return 0 ;
  if (exp->common.tag != Inner)
    return 1 ;
  return cmixInnerOK(&exp->inner);
}

int
cmixInnerOK(struct cmixExprInner *exp)
{
  char const *pc ;
  unsigned i = 0 ;
  unsigned nolevel = 0 ;
  unsigned yeslevel = 0 ;
  int inPling = 0 ;
  int printing = 1 ;
  for ( pc = exp->string ; *pc ; pc++ )
    switch(*pc) {
    case '?':
      if ( printing && !ExprOK(exp->child[i]) )
        return 0 ;
      i++ ;
      break ;
    case ':':
      if ( nolevel != 0 || exp->child[i] == NULL )
        nolevel++, printing = 0 ;
      else
        yeslevel++ ; /* printing == 1 */
      break ;
    case '\'':
      if ( inPling )
        printing = nolevel == 0 ;
      else {
        if ( nolevel )
          nolevel-- , printing = nolevel == 0 ;
        else if ( yeslevel )
          yeslevel-- , printing = 0 ;
      }
      inPling = !inPling ;
      break ;
    default:
      break ;
    }
  return 1 ;
}

static void printExpr(union cmixExpr *,struct cmixNameRec names[],FILE *f);

static void
printInner(struct cmixExprInner *e,struct cmixNameRec names[],FILE *f)
{
  char const *pc ;
  unsigned i = 0 ;
  unsigned nolevel = 0 ;
  unsigned yeslevel = 0 ;
  int inPling = 0 ;
  int printing = 1 ;
  for ( pc = e->string ; *pc ; pc++ )
    switch(*pc) {
    case '#':
      if ( printing ) switch(*++pc) {
      case '=':
        putc('#',f);
        break ;
      case '!':
        putc('?',f);
        break ;
      case ';':
        putc(':',f);
        break ;
      case ',':
        putc('\'',f);
        break ;
      default:
        cmixFatal("strange char in expr format\n");
      }
      break ;
    case '?':
      if ( printing ) printExpr(e->child[i],names,f);
      i++ ;
      break ;
    case ':':
      if ( inPling )
        cmixFatal("Pling imbalance in skip string\n");
      if ( nolevel != 0 || e->child[i] == NULL )
        nolevel++, printing = 0 ;
      else
        yeslevel++ ; /* printing == 1 */
      break ;
    case '\'':
      if ( inPling )
        printing = nolevel == 0 ;
      else {
        if ( nolevel )
          nolevel-- , printing = nolevel == 0 ;
        else if ( yeslevel )
          yeslevel-- , printing = 0 ;
        else
          cmixFatal("Too many plings in skip string\n");
      }
      inPling = !inPling ;
      break ;
    default:
      if ( printing ) putc(*pc,f);
      break ;
    }
  if ( nolevel || yeslevel || inPling )
    cmixFatal("Too many hashes in skip string\n");
  return ;
}

static void
printExpr(union cmixExpr *e,struct cmixNameRec names[],FILE *f)
{
  if ( e == NULL ) {
    fprintf(f,"{0}") ;
    return ;
  }
  switch(e->common.tag) {
  case Inner:
    printInner(&e->inner,names,f);
    break ;
  case Name:
    if ( e->name.seq == 0 )
      fprintf(f,"%s",names[e->name.index].base) ;
    else
      fprintf(f,"%s%u",names[e->name.index].base,e->name.seq) ;
    break ;
  case LiftChar:
    switch( e->lift_int.data ) {
    case '\\': fprintf(f,"'\\\\'"); break ;
    case '\'': fprintf(f,"'\\''"); break ;
    case '\a': fprintf(f,"'\\a'"); break ;
    case '\b': fprintf(f,"'\\b'"); break ;
    case '\f': fprintf(f,"'\\f'"); break ;
    case '\n': fprintf(f,"'\\n'"); break ;
    case '\r': fprintf(f,"'\\r'"); break ;
    case '\t': fprintf(f,"'\\t'"); break ;
    case '\v': fprintf(f,"'\\v'"); break ;
    default: {
      unsigned char data = e->lift_int.data ;
      if ( isprint(data) )
        fprintf(f,"'%c'",data);
      else if ( data < 512 ) /* i.e representable as 3 octal digits */
        fprintf(f,"'\\%o'",data);
      else
        fprintf(f,"'\\x%X'",data);
      break ; }
    }
    break ;
  case LiftInt:
    fprintf(f,"%lu",e->lift_int.data);
    break ;
  case LiftFloat:
  case LiftDouble:
  case LiftLongDouble:
    e->lift_float.write(f,e->common.tag,e->lift_float.data);
    break ;
  case NameRequest:
    cmixFatal("Some name didn't get managed\n");
  default:
    cmixFatal("bad expression magic in unparse()\n");
  }
}

/***************************************************************************/
/* Printing declarations and function bodies                               */
/***************************************************************************/

static void
printDecl(struct cmixDecl *d,struct cmixNameRec names[], FILE *of)
{
  printInner(&d->decl,names,of);
  if ( d->members ) {
    int gotany = 0 ;
    struct cmixDecl *m ;
    fprintf(of," {");
    for(m=d->members; m; m=m->next)
      if ( m->status == Done ) {
        gotany = 1 ;
        printDecl(m,names,of);
      }
    if ( !gotany )
      fprintf(of," int:1; }");
    else
      fprintf(of,"\n}");
  }
}

static void
printIndent(int indent,FILE *f)
{
  while(indent--) {
    putc(' ',f);
    putc(' ',f);
  }
}

static void
printGoto(int indent,struct cmixStmtLabel *target,FILE *f)
{
  printIndent(indent,f);
  fprintf(f,"goto L%u;\n",target->number);
}

static void
printStmtSeq(union cmixStmt *s,int indent,struct cmixNameRec names[], FILE *f)
{
  while(1) {
    if ( s == NULL ) {
      fprintf(f,"/* FALLING OFF THE END OF THE FUNCTION! */\n");
      return ;
    }
    switch(s->common.tag) {
    case Label:
      if ( s->label.refcount > 1 ) {
        printGoto(indent,&s->label,f);
        return ;
      }
      /* else print nothing */
      break ;
    case Plain:
      printIndent(indent,f);
      printInner(&s->plain.expr,names,f);
      fprintf(f,";\n");
      break ;
    case Goto:
      if ( s->jump.target->refcount > 1 ) {
        printGoto(indent,s->jump.target,f);
        return ;
      } else {
        s = s->jump.target->next ;
        continue ;
      }
    case If:
      printIndent(indent,f);
      fprintf(f,"if (");
      printInner(s->cond.cond,names,f);
      putc(')',f);
      if ( s->cond.then_target->refcount > 1 ) {
        putc('\n',f);
        printGoto(indent+1,s->cond.then_target,f);
        printIndent(indent,f);
      } else {
        fprintf(f," {\n");
        printStmtSeq(s->cond.then_target->next,indent+1,names,f);
        printIndent(indent,f);
        fprintf(f,"} ");
      }
      fprintf(f,"else");
      if ( s->cond.else_target->refcount > 1 ) {
        putc('\n',f);
        printGoto(indent+1,s->cond.else_target,f);
      } else {
        fprintf(f," {\n");
        printStmtSeq(s->cond.else_target->next,indent+1,names,f);
        printIndent(indent,f);
        fprintf(f,"}\n");
      }
      return ;
    case Return:
      printIndent(indent,f);
      fprintf(f,"return ");
      printInner(&s->plain.expr,names,f);
      fprintf(f,";\n");
      return ;
    case Abort:
      printIndent(indent,f);
      fprintf(f,"abort();\n");
      printIndent(indent,f);
      fprintf(f,"/* residual deref of spectime NULL pointer:\n");
      printIndent(indent,f);
      fprintf(f," * ");
      printInner(&s->plain.expr,names,f);
      fprintf(f," */\n");
      return ;
    }
    s = s->common.next ;
  }
}

static void
printStmts(union cmixStmt *s,struct cmixNameRec names[],FILE *f)
{
  union cmixStmt *s2 ;
  unsigned labelcounter = 1;
  /* first, assign numbers to the important labels */
  for ( s2 = s ; s2 ; s2 = s2->common.next )
    if ( s2->common.tag == Label && s2->label.refcount > 1 )
      s2->label.number = labelcounter++ ;
  /* then we can print the body */
  printStmtSeq(s,1,names,f);
  for ( ; s ; s = s->common.next )
    if ( s->common.tag == Label && s->label.refcount > 1 ) {
      fprintf(f,"L%u:\n",s->label.number);
      printStmtSeq(s->label.next,1,names,f);
    }
}

static void
printPlain( Interval *interval, int indent,
            struct cmixNameRec names[], FILE *f)
{
  union cmixStmt *stmt;
  
  stmt = (union cmixStmt *)interval->member.basic.label;
  do {
    stmt = stmt->common.next;
    if ( stmt->common.tag == Plain ) {
      printIndent(indent,f);
      printInner(&stmt->plain.expr,names,f);
      fprintf(f,";\n");
    }
  } while ( stmt != interval->member.basic.control );
}

static unsigned
prepareInterval( Interval *interval, Loop *loop, Interval *ifFollow,
                 unsigned nextFree )
{
  struct basic *info;
  Interval *latch = NULL;
  Loop *old_loop = NULL;
  
  if ( interval == NULL )
    return nextFree;
  
  if ( loop != NULL )
    latch = loop->latch;
  
  assert( interval->type == basic );
  
  info = &interval->member.basic;
  
  if ( interval == latch || interval == ifFollow )
    return nextFree;
  
  if ( interval->number != -1 ||
       ( interval->back_refs != NULL &&
         interval->back_refs->next != NULL &&
         loop != info->loop &&
         ( info->loop == NULL || interval != info->loop->header ) ) ) {
    if ( loop != NULL ) {
      /* we're in a loop */
      if ( ( interval == loop->latch && loop->type == dowile ) ||
           ( interval == loop->header &&
             loop->type != dowile ) )
        return nextFree;
      else if ( interval == loop->follow )
        return nextFree;
    }
    if ( interval->number == 0 || interval->number == -1 )
      interval->number = nextFree++;
    return nextFree;
  }
  
  interval->number = 0;
  if ( info->loop != NULL && info->loop->header == interval ) {
    switch ( info->loop->type ) {
    case endless:
      assert( interval != info->loop->latch );
      break;
      
    case wile:
      assert( info->control->common.tag == If );
      if ( info->loop->latch->number == -1 )
        info->loop->latch->number = 0;
      nextFree = prepareInterval(interval->forw_refs->next->interval,
                                 info->loop,ifFollow,nextFree);
      return prepareInterval(info->loop->follow,loop,ifFollow, nextFree);
      
    case dowile:
      if ( interval == info->loop->latch ) {
        if ( info->loop->latch->number == -1 )
          info->loop->latch->number = 0;
        return prepareInterval(info->loop->follow,loop,ifFollow,nextFree);
      }
      break;
      
    default:
      assert( 0 == 1 );
    }
    old_loop = loop;
    loop = info->loop;
    latch = loop->latch;
   
  }
  
  if ( interval->forw_refs == NULL )
    return nextFree;
  
  switch ( info->control->common.tag ) {
  case If:
    assert(interval->forw_refs->next->interval != info->ifFollow);
    nextFree = prepareInterval(interval->forw_refs->next->interval,loop,
                               info->ifFollow,nextFree);
    if ( interval->forw_refs->interval != latch &&
         interval->forw_refs->interval != info->ifFollow )
      nextFree = prepareInterval(interval->forw_refs->interval,
                                 loop,info->ifFollow,nextFree);
    nextFree = prepareInterval(info->ifFollow,loop,ifFollow,nextFree);
    break;
    
  case Goto:
    nextFree = prepareInterval(interval->forw_refs->interval,loop,
                               ifFollow,nextFree);
    break;
    
  default:
    fprintf(stderr,"Tag = %d\n", info->control->common.tag);
    assert( 0 == 1 );
  }
  
  if ( info->loop != NULL && info->loop->header == interval ) {
    switch ( info->loop->type ) {
    case endless:
      if ( info->loop->latch->number == -1 )
        info->loop->latch->number = 0;
      break;

    case dowile:
      if ( info->loop->latch->number == -1 )
        info->loop->latch->number = 0;
      break;

    default:
      assert( 0 == 1 );
    }

    nextFree = prepareInterval(info->loop->follow,old_loop,ifFollow,nextFree);
  }
  
  return nextFree;
}

static void
printInterval( Interval *interval, int indent, struct cmixNameRec names[],
               Loop *loop, Interval *ifFollow, FILE *f )
{
  union cmixStmt *stmt;
  struct basic *info;
  Interval *latch = NULL;
  Loop *old_loop = NULL;
  
  if ( interval == NULL )
    return;
  
  if ( loop != NULL )
    latch = loop->latch;
  
  assert( interval->type == basic );
  
  info = &interval->member.basic;
  
  if ( interval == latch || interval == ifFollow ) {
    return;
  }

  
  if ( info->printed ||
       ( interval->back_refs != NULL &&
         interval->back_refs->next != NULL &&
         loop != info->loop &&
         ( info->loop == NULL || interval != info->loop->header ) ) ) {
    printIndent(indent,f);
    if ( loop != NULL ) {
      /* we're in a loop */
      if ( ( interval == loop->latch && loop->type == dowile ) ||
           ( interval == loop->header &&
             loop->type != dowile ) ) {
        fprintf(f,"continue;\n");
        return;
      } else if ( interval == loop->follow ) {
        fprintf(f,"break;\n" );
        return;
      }
    }

    fprintf(f,"goto L%u;\n",interval->number);
    
    assert( interval->number != -1 );
    assert( interval->number != 0 );
    
    return;
  }
  
  info->printed = 1;
  
  stmt = (union cmixStmt *)info->label;
  if ( interval->number != 0 && interval->number != -1 )
    fprintf(f,"L%u:\n",interval->number);
  
  if ( info->loop != NULL && info->loop->header == interval ) {
    switch ( info->loop->type ) {
    case endless:
      assert( interval != info->loop->latch );
      printIndent(indent,f);
      fprintf(f,"for ( ; ; ) {\n");
      break;

    case wile:
      assert( info->control->common.tag == If );
      printIndent(indent,f);
      fprintf(f,"while (");
      printInner(info->control->cond.cond,names,f);
      fprintf(f,") {\n");
      printInterval(interval->forw_refs->next->interval,indent+1,names,
                    info->loop,ifFollow,f);
      if ( info->loop->latch->number != 0 &&
           info->loop->latch->number != -1 )
        fprintf(f,"L%u:\n",info->loop->latch->number);
      info->loop->latch->member.basic.printed = 1;
      printPlain(info->loop->latch,indent+1,names,f);
      printIndent(indent,f);
      fprintf(f,"}\n");
      printInterval(info->loop->follow,indent,names,loop,ifFollow,f);
      return;
      
    case dowile:
      printIndent(indent,f);
      fprintf(f,"do {\n");
      if ( interval == info->loop->latch ) {
        if ( info->loop->latch->number != 0 &&
             info->loop->latch->number != -1 )
          fprintf(f,"L%u:\n",info->loop->latch->number);
        info->loop->latch->member.basic.printed = 1;
        printPlain(info->loop->latch,indent+1,names,f);
        printIndent(indent,f);
        fprintf(f,"} while (");
        printInner(info->loop->latch->member.basic.control->cond.cond,
                   names,f);
        fprintf(f,");\n");
        printInterval(info->loop->follow,indent,names,loop,ifFollow,f);
        return;
      }
      break;
      
    default:
      assert( 0 == 1 );
    }
    
    old_loop = loop;
    loop = info->loop;
    latch = loop->latch;
    indent++;
  }
  
  printPlain(interval,indent,names,f);
  
  if ( interval->forw_refs == NULL ) {
    switch ( info->control->common.tag ) {
    case Return:
      printIndent(indent,f);
      fprintf(f,"return ");
      printInner(&info->control->plain.expr,names,f);
      fprintf(f,";\n");
      break;
    default:
      assert( 0 == 1 );
    }
    return;
  }
  
  /* Not the header of a loop, maybe a structured conditional? */
  
  switch ( info->control->common.tag ) {
  case If:
    assert(interval->forw_refs->next->interval != info->ifFollow);
    printIndent(indent,f);
    fprintf(f,"if (");
    printInner(info->control->cond.cond,names,f);
    fprintf(f,") {\n");
    printInterval(interval->forw_refs->next->interval,indent+1,names,loop,
                  info->ifFollow,f);
    if ( interval->forw_refs->interval != latch &&
         interval->forw_refs->interval != info->ifFollow ) {
      printIndent(indent,f);
      fprintf(f,"} else {\n");
      printInterval(interval->forw_refs->interval,indent+1,names,loop,
                    info->ifFollow,f);
      printIndent(indent,f);
      fprintf(f,"}\n");
    } else {
      printIndent(indent,f);
      fprintf(f,"}\n");
    }
    
    printInterval(info->ifFollow,indent,names,loop,ifFollow,f);
    break;
    
  case Goto:
    printInterval(interval->forw_refs->interval,indent,names,loop,
                  ifFollow,f);
    break;
    
  default:
    assert( 0 == 1 );
  }
  
  if ( info->loop != NULL && info->loop->header == interval ) {
    indent--;
    switch ( info->loop->type ) {
    case endless:
      if ( info->loop->latch->number != 0 &&
           info->loop->latch->number != -1 )
        fprintf(f,"L%u:\n",info->loop->latch->number);
      info->loop->latch->member.basic.printed = 1;
      printPlain(info->loop->latch,indent+1,names,f);
      printIndent(indent,f);
      fprintf(f,"}\n");
      break;

    case dowile:
      if ( info->loop->latch->number != 0 &&
           info->loop->latch->number != -1 )
        fprintf(f,"L%u:\n",info->loop->latch->number);
      info->loop->latch->member.basic.printed = 1;
      printPlain(info->loop->latch,indent+1,names,f);
      printIndent(indent,f);
      fprintf(f,"} while (");
      printInner(info->loop->latch->member.basic.control->cond.cond,
                 names,f);
      fprintf(f,");\n");
      break;

    default:
      assert( 0 == 1 );
    }

    printInterval(info->loop->follow,indent,names,old_loop,ifFollow,f);
  }
}


/*****************************************************************************/

int cmixRestruct = 1;

void
cmixUnparse(struct cmixNameRec names[], unsigned const namesSorted[],
            unsigned namesCount, FILE *outfile)
{
  int cls ;
  struct cmixFun *f ;
  struct cmixDecl *d, *m ;
  
  /* step 1: add the latter-day taboos from the generator functions.
   */
  for ( f = cmixFuns ; f ; f=f->next ) {
    if ( f->resname->common.tag == Inner )
      addTaboo(names,namesSorted,namesCount,
               f->resname->inner.string);
  }
  
  /* step 2: walk through all of the code, marking those declarations
   * that are actually mentioned, and counting predecessors for each
   * label.
   */
  for ( f = cmixFuns ; f ; f=f->next )
    mentionFun(f);
  for( cls = 0 ; cls < cmixDECLCLASSES ; cls++ )
    for(d = cmixDecls[cls] ; d ; d=d->next )
      if ( d->status == Sure )
        mentionDecl(d);
  
  /* step 3: assign names to global variables and functions
   */
  for( cls = 0 ; cls < cmixDECLCLASSES ; cls++ )
    for(d = cmixDecls[cls] ; d ; d=d->next )
      if ( d->status == Done )
        traverseInner(&d->decl,assignGlobalName,names);
  for(f = cmixFuns ; f ; f=f->next )
    if ( f->resname->common.tag == NameRequest )
      assignGlobalName(&f->resname->name,names);
  
  /* step 4a: assign names within each function
   */
  for(f = cmixFuns ; f ; f=f->next ) {
    traverseInner(&f->heading,prepareLocalName,names);
    for(d = f->locals ; d ; d=d->next )
      traverseInner(&d->decl,prepareLocalName,names);
    names[0].nextseq_local = names[0].nextseq_global ;
    
    traverseInner(&f->heading,assignLocalName,names);
    for(d = f->locals ; d ; d=d->next )
      if ( d->status == Done )
        traverseInner(&d->decl,assignLocalName,names);
  }
  /* step 4b: .. and within each struct
   */
  for( cls = 0 ; cls < cmixDECLCLASSES ; cls++ )
    for(d = cmixDecls[cls]; d ; d=d->next ) {
      for( m=d->members; m ; m=m->next )
        traverseInner(&m->decl,prepareLocalName,names);
      for( m=d->members; m ; m=m->next )
        if ( m->status == Done )
          traverseInner(&m->decl,assignLocalName,names);
    }
  
  /* step 5: print the program
   */
  for( cls = 0 ; cls < cmixDECLCLASSES ; cls++ ) {
    for(d = cmixDecls[cls] ; d ; d=d->next ) {
      if ( d->status == Done ) {
        printDecl(d,names,outfile);
        fprintf(outfile,";\n");
      }
    }
    putc('\n',outfile);
    if ( cls == cmixGlobal ) {
      for(f = cmixFuns ; f ; f=f->next )
        if ( f->shared ) {
          printInner(&f->heading,names,outfile);
          fprintf(outfile,";\n");
        }
      for(f = cmixFuns ; f ; f=f->next ) {
        putc('\n',outfile);
        printInner(&f->heading,names,outfile);
        fprintf(outfile,"\n{\n");
        for(d = f->locals ; d ; d=d->next )
          if (d->status == Done) {
            fprintf(outfile,"  ");
            printDecl(d,names,outfile);
            fprintf(outfile,";\n");
          }
        if ( cmixRestruct ) {
          Restruct *res;
          int printSep = 0;
          int idx;
          int nextFree = 1;
          res = cmixRestructStmts(f->stmts);
          do {
            for ( idx = 1; idx <= res->count; idx++ ) {
              if ( res->order[idx]->number == -1 ) {
                nextFree = prepareInterval(res->order[idx],NULL,NULL,nextFree);
                continue;
              }
            }
            break;
          } while ( 1 );
          
          do {
            for ( idx = 1; idx <= res->count; idx++ ) {
              if ( ! res->order[idx]->member.basic.printed ) {
                if ( printSep ) {
                  fprintf(outfile,"\n/******************/\n");
                  fprintf(outfile,  "/** END OF BLOCK **/\n");
                  fprintf(outfile,  "/******************/\n\n");
                }
                printSep = 1;
                
                printInterval(res->order[idx],1,names,NULL,NULL,outfile);
                continue;
              }
            }
            break;
          } while ( 1 );
        } else {
          printStmts(f->stmts,names,outfile);
        }
        fprintf(outfile,"}\n");
      }
      putc('\n',outfile);
    }
  }
}
