/* Authors:  Peter Holst Andersen (txix@diku.dk)
 *           Jens Peter Secher (jpsecher@diku.dk)
 * Content:  C-Mix system: Namespace and scope utilities for the parser.
 *
 * Copyright  1998. The TOPPS group at DIKU, U of Copenhagen.
 * Redistribution and modification are allowed under certain
 * terms; see the file COPYING.cmix for details.
 */

#include <cmixconf.h>
#include "options.h"
#include "liststack.h"
#include "parser.h"
#include "symboltable.h"
#include <strstream.h>

#define debug debugstream_gram

//////// NAMESPACES ////////
// ANSI C sec. 6.1.2.3 :

//////////// Namespace 1
SymbolTable<Indirection> goto_labels;
// Goto labels (we only need one table, since labels are in scope of the entire
// function they appear in) 

/////////// Namespace 2
Scope<ObjectDecl> names;
// Names of Function definitions, Variables, Enum constants, Function
// declarations and Typedefs.

ListStack<VarDecl> objects;
// List of real objects.

static Plist<VarDecl>* enumconsts = new Plist<VarDecl>();
// List of enumconsts in the enum being declared.

static Plist<FunDef>* functions = new Plist<FunDef>();

/////////// Namespace 3
Scope<UserDecl> usernames;
// Names of structs/unions/enums.

ListStack<UserDecl> userdecls;
// Stack of user declarations; needed because UDs pop up in the middle of
// ordinary declarations.

/////////// Namespace 4
static Scope<MemberDecl> membernames;
// Members of the struct or union type currently being declared. Several levels
// are needed because of embedded declarations.

static ListStack<MemberDecl> members;
// List of members.

//////// namespaces end ////////


static int old_function = 0;
// A value of means 1 when we are parsing the declaration list at the head of
// an old function declaration. Set in function_defitinion and checked in the
// function add_declaration.

static int reenter_scope_at_next_block = 0;
// Set in rule function_definition and tested/reset in function enter_scope().

static int last_decl_can_have_init = 0;
// Controlled by add_declaration() and used by add_initializer().

static bool
equal_constants(Expr *e1, Expr *e2)
{
  if ( e1->valueKnown && e2->valueKnown )
    return e1->knownValue == e2->knownValue ;
  Diagnostic di(WARNING,e1->pos);
  di << "assuming that `" ;
  e1->show(di);
  di << '\'' ;
  di.addline(e2->pos) << "and `" ;
  e2->show(di);
  di << "' will evaluate to the same constant" ;
  return true ;
}

// Convert both types to their common "composite type" (ISO-C 6.1.2.6).
// During this unifications, usertypes may be collapsed.
// return true if the unification succeded, false if it failed.
static bool
unify_types(Type* t1, Type* t2)
{
  assert(t1 && t2);
  // The types have to have the same qualifiers.
  if (t1->cv.cst != t2->cv.cst || t1->cv.vol != t2->cv.vol)
    return false ;
  // Compare the types.
  if (t1->realtype != t2->realtype)
    return false ;
  switch (t1->realtype) {
  case EllipsisT:
    return true ;
  case BaseT: {
    BaseType* b1 = (BaseType*) t1;
    BaseType* b2 = (BaseType*) t2;
    return b1->tag == b2->tag ; }
  case PtrT: {
    PtrType* p1 = (PtrType*) t1;
    PtrType* p2 = (PtrType*) t2;
    return unify_types(p1->next,p2->next); }
  case ArrayT: {
    ArrayType* p1 = (ArrayType*) t1;
    ArrayType* p2 = (ArrayType*) t2;
    if (p1->size==NULL)
      p1->size = p2->size ;
    else if (p2->size==NULL)
      p2->size = p1->size ;
    else {
      if ( !equal_constants(p1->size,p2->size) )
        return false ;
    }
    return unify_types(p1->next,p2->next); }
  case FunT: {
    FunType* f1 = (FunType*) t1;
    FunType* f2 = (FunType*) t2;
    // If one of the parameter lists is unspecified, use the other one.
    if ( f2->unspecified() ) {
      f2->params->clear() ;
      f2->params->splice(f2->params->begin(),*f1->params);
    } else if ( f1->unspecified() ) {
      f1->params->clear() ;
      f1->params->splice(f1->params->begin(),*f2->params);
    } else {
      if ( f1->params->size() != f2->params->size() )
        return false ;
      double_foreach(p1, p2, *f1->params, *f2->params, Plist<Type>) {
        if ( !unify_types(*p1,*p2) )
          return false ;
      }
    }
    return unify_types(f1->ret,f2->ret); }
  case StructT:
  case UnionT:
  case EnumT: {
    UserDecl* d1 = ((UserType*) t1)->def;
    while ( d1->refer_to != NULL ) d1 = d1->refer_to ;
    UserDecl* d2 = ((UserType*) t2)->def;
    while ( d2->refer_to != NULL ) d2 = d2->refer_to ;

    if ( d1 == d2 )
      return true ;
    assert(d1->tag == d2->tag );

    if ( d1->tag != Enum && ((StructDecl*)d1)->member_list->empty() ) {
      // We do not complete one struct with the member types of
      // another, lest we import member names from a different
      // file and later meet different names in this file.
      // We do, however, make sure that as soon as we meet some
      // definition *with* member types, that is made the top
      // of the definition group, so that all subsequent
      // unifications get a chance at matching the members.
      UserDecl *dt = d1 ; d1 = d2 ; d2 = dt ;
    }
    // Remember to redirect one of the types BEFORE unifying subtypes!
    d2->refer_to = d1 ;
    
    if ( d1->tag != Enum ) {
      StructDecl *s1 = (StructDecl*) d1 ;
      StructDecl *s2 = (StructDecl*) d2 ;
      if( !s2->member_list->empty() ) {
        assert( !s1->member_list->empty() );
	if ( s1->member_list->size() != s2->member_list->size() )
          return false ;
        // XXX: The order of the members of unions should be irrelevant,
        // but we require them to be identical, anyway.
        double_foreach(m1,m2,*s1->member_list,*s2->member_list,
                       Plist<MemberDecl>) {
          if ( m1->bitfield ) {
            if ( m2->bitfield ) {
              if ( !equal_constants(m1->bitfield,m2->bitfield) )
                return false ;
            } else
              return false ;
          } else {
            if ( m2->bitfield )
              return false ;
            else
              ;
          }
          if ( !unify_types(m1->var->type,m2->var->type) )
            return false ;
        }
      }
    } else {
      // unifying two enum definitions
      EnumDecl *e1 = (EnumDecl*) d1 ;
      EnumDecl *e2 = (EnumDecl*) d2 ;
      if ( e1->member_list->size() != e2->member_list->size() )
        return false ;
      double_foreach(m1,m2,*e1->member_list,*e2->member_list,
                     Plist<VarDecl>) {
        m2->refer_to = *m1 ;
        if ( m1->init == NULL )
          m1->init = m2->init ;
        else if ( m2->init == NULL )
          m2->init = m1->init ;
        else {
          assert(m1->init->isSimple());
          assert(m2->init->isSimple());
          if ( !equal_constants(m1->init->expr(),m2->init->expr()) )
            return false ;
        }
      }
    }
    return true ; }
  case AbstractT: {
    AbsType *a1 = (AbsType*) t1 ;
    AbsType *a2 = (AbsType*) t2 ;
    return ( a1->name == a2->name || strcmp(a1->name,a2->name) == 0 ); }
  }
  Diagnostic(INTERNAL,ccchere)
      << "fell through switch in unify_types()" ;
  return false ;
}

void
cccParser :: lone_sue(Parse_UserType* d)
{
  // a lone "enum foobar;" is syntactically allowed, just as
  // "unsigned long;" is, but neither has any meaning.
  if (d->type != Enum)
    introduce_struct_decl(d->type,d->id);
}

static UserDecl *
make_or_reuse_struct(UserDecl *old,UserTag type, char const *name)
{
  assert(type != Enum);
  if ( old != NULL ) {
    if ( old->tag == type )
      return old ;
    else {
      Diagnostic d(ERROR,ccchere) ;
      d << name << " is a " << usertype2str(old->tag) << ", not a "
        << usertype2str(type) ;
      d.addline(old->pos) << "(declaration of " << usertype2str(old->tag)
                          << ' ' << old->name << ')' ;
    }
  }
  StructDecl* d = new StructDecl(name,ccchere,type,NULL);
  userdecls.push_back(d);
  usernames.insert(name, d);
  return d ;
}

// This is meant to introduce struct/union names. Updates of members are done
// later on by update_struct_decl.
void
cccParser::introduce_struct_decl(UserTag type, char const* name)
{
  // Check for _local_ name clashes.
  make_or_reuse_struct(usernames.lookup_local(name),type,name);
}

void
cccParser::find_struct_decl(UserTag type, char const* name)
{
  // If a structure of this name does exists, use that one (if it matches).
  make_or_reuse_struct(usernames.lookup(name),type,name);
}

// This can only be called after introduce_struct().
void
cccParser :: update_struct_decl(char const* name)
{
  // A previous declaration has to be present in local scope.
  UserDecl* oldDecl = usernames.lookup_local(name);
  assert(oldDecl!=NULL);
  StructDecl* old = (StructDecl*) oldDecl;
  if ( !old->member_list->empty() ) {
    Diagnostic d(ERROR,ccchere);
    d << "redeclaration of struct or union " << name ;
    d.addline(old->pos) << "previous declaration here";
  } else {
    Plist<MemberDecl>* memberlst = members.get_list();
    delete old->member_list ;
    old->member_list=memberlst;
    old->assign_defseqnum() ;
    if (debug^4) debug << "[updating: " << name << "]";
  }
}

char const *
cccParser::fresh_anonymous_name()
{
  static unsigned long counter = 0 ;
  char buffer[11] ;
  sprintf(buffer,"$%lu",++counter);
  return stringDup(buffer);
}

void
cccParser::anonymous_struct_decl(UserTag type, char const* name)
{
  if (debug^4) debug << "[anonymous: " << name << "]";
  StructDecl* d = new StructDecl(name,ccchere,type,members.get_list());
  d->assign_defseqnum();
  userdecls.push_back(d);
  usernames.insert(name, d);
}

void
cccParser::add_enum_decl(char const* name)
{
  // Check for name clashes.
  UserDecl* oldDecl = usernames.lookup_local(name);
  if (oldDecl) {
    Diagnostic di(ERROR,ccchere);
    di << usertype2str(oldDecl->tag) << ' ' << name << " redeclared as enum" ;
    di.addline(oldDecl->pos) << "(previous declaration here)";
  }
  // Insert the enum.
  EnumDecl* d = new EnumDecl(name,ccchere,enumconsts);
  userdecls.push_back(d);
  usernames.insert(name,d);
  // "Clear" the global list of enum constants.
  enumconsts = new Plist<VarDecl>();
  if (debug^4) debug << "[add_enum_decl: " << name << "]";
}

void
cccParser :: find_enum_decl(char const* name)
{
  UserDecl* oldDecl = usernames.lookup(name);
  if (!oldDecl)
    Diagnostic(ERROR,ccchere) << "enum " << name << " undeclared" ;
}


///////////////////////////////////////////////


BaseType*
cccParser::make_basetype (Parse_Type* pt)
{
  BaseTypeTag tag = pt->determine_type();
  BaseType *result = new BaseType(tag);
  result->cv = pt->qualifiers;
  return result ;
}

Type*
cccParser::make_typedeftype(Parse_TypedefType* tt)
{
  Type* t = tt->type->type;
  // If there are no qualifiers, just return the unfolded type.
  if ( tt->qualifiers.subsetof(constvol()) )
    return t;
  // Put the qualifiers at the end of an array chain.
  t = t->copy();
  Type* handle = t;
  while (t->realtype == ArrayT) t = ((ArrayType*)t)->next;
  if ( t->realtype == FunT ) {
    Diagnostic(WARNING,tt->type->pos)
      << "ignoring qualifiers on funcion type (through typedef)" ;
    return handle ;
  }
  // Check for multiple qualifiers.
  if ( tt->qualifiers.cst && t->cv.cst )
    Diagnostic(WARNING,tt->type->pos)
      << "multiple const qualifiers when using this typedef";
  if ( tt->qualifiers.vol && t->cv.vol )
    Diagnostic(WARNING,tt->type->pos)
      << "multiple volatile qualifiers when using this typedef";
  t->cv += tt->qualifiers ;
  return handle;
}

Parse_Decl*
cccParser::make_decl (Parse_Type* pt)
{
  return new Parse_Decl(make_basetype(pt),pt->storage);
}

Parse_Decl*
cccParser::make_decl (Parse_UserType* ut)
{
  return new Parse_Decl(make_usertype(ut),ut->storage);
}

Parse_Decl*
cccParser::make_decl (Parse_TypedefType* tt)
{
  return new Parse_Decl(make_typedeftype(tt),tt->storage);
}


UserType*
cccParser::make_usertype (const Parse_UserType* const usert)
{
  UserDecl* def = usernames.lookup(usert->id);
  UserType* result ;
  assert(def != NULL);
  switch (usert->type) {
  case Struct: result = new StructType(def); break;
  case Union:  result = new UnionType(def); break;
  case Enum:   result = new EnumType(def); break;
  default: 
    Diagnostic(INTERNAL,RIGHT_HERE) << "cccParser::make_usertype";
    return NULL;
  }
  result->cv = usert->qualifiers ;
  return result ;
}

void
cccParser::enter_scope (void)
{
  if (reenter_scope_at_next_block) {
    if (debug^5) debug << "[reenter scope]";
    reenter_scope_at_next_block = 0;
    names.reenter_previous_scope();
    objects.reenter();
    usernames.reenter_previous_scope();
    userdecls.reenter();
  }
  else {
    if (debug^5) debug << "[enter scope]";
    names.enter_scope();
    objects.enter();
    usernames.enter_scope();
    userdecls.enter();
  }
}

void
cccParser::leave_scope (void)
{
  names.leave_scope();
  objects.leave();

  if (debug^5) debug << "[leave scope]";
  
  usernames.leave_scope();
  
  // Ugly hack: in case there are userdecls left here, we export
  // them to the enclosing scope, so they won't be forgot in the
  // c2core phase.
  Plist<UserDecl> *olduserdecls = userdecls.get_list() ;
  userdecls.leave();
  foreach(i,*olduserdecls,Plist<UserDecl>)
      userdecls.push_back(*i);
  delete olduserdecls ;
}


void
cccParser::members_enter_scope (void)
{
  membernames.enter_scope();
  members.enter();
}

void
cccParser::members_leave_scope (void)
{
  membernames.leave_scope();
  members.leave();
}


Parse_TypedefType*
cccParser::make_typedef (const char* name)
{
  VarDecl* tdef = (VarDecl*) names.lookup(name);
  assert(tdef != NULL);
  return new Parse_TypedefType(tdef);
}

MemberDecl*
cccParser::add_member (Parse_Decl *d, Expr* bits)
{
  // Check that the name was not seen before.
  if (membernames.lookup_local(d->name)) {
    Diagnostic(ERROR,ccchere) << "redeclaration of member " << d->name ;
  }
  VarDecl* var = new VarDecl(d->name,d->pos,d->type,NoLinkage,VarMu);
  MemberDecl* decl = new MemberDecl(var,bits);
  // Make the name known.
  membernames.insert(d->name, decl) ;
  // Put the actual definition in the back of the list of members. This list
  // is collected in update_struct_decl() and anonymous_struct_decl(). 
  members.push_back(decl); 
  if (debug^4) {
    debug << "add_member: ";
    decl->show(debug);
    debug << "\n";
  }
  return decl;
}

// Construct an enum constant and put it in current scope.
VarDecl*
cccParser::add_enumconst (char* n, Position pos, Expr *value)
{
  assert(n != NULL);
  // try to construct a default value if one is not given
  if ( value == NULL ) {
    if ( enumconsts->empty() )
      value = new ConstExpr("0",Integer,pos);
    else if ( enumconsts->back()->init &&
              enumconsts->back()->init->expr()->valueKnown ) {
      // Only generate an enumeration value if we could evaluate
      // the previous one. We do not trust our own constant folding
      // abilities absolutely, so we like to keep the original
      // expression in the new value specification.
      // However, if that would result in an unevaluatable expression
      // we would gain little and subject the poor user to numerous
      // "assuming this and that evaluates to the same" warnings
      // when unifying such enum definitions.
      value = enumconsts->back()->init->expr()->copy_plusone() ;
    }
  }
  // An enum constant is always an int.
  VarDecl* decl = new VarDecl(n,pos,BaseType::get(Int),NoLinkage,VarEnum);
  if ( value )
    decl->set_initializer(new Init(value));
  if (debug^5) {
    debug << "add_enumconst: ";
    decl->show(debug);
    debug << "\n";
  }
  // Check that the name was not seen before.
  ObjectDecl* old = names.lookup_local(n);
  if (old) {
    // Typedefs are allowed to be overwritten (which sucks).
    // Abstract types are usually typedef in Real Life
    if (old->linkage != Typedef &&
        old->linkage != AbstractType)
      Diagnostic(ERROR,ccchere) << n << " redeclared as enum constant";
  }
  names.insert(n, decl) ;
  enumconsts->push_back(decl);
  return decl;
}

FunDef*
cccParser::make_final_FunDef (Parse_Decl* board,
                              Plist<VarDecl>* ds,
                              Plist<UserDecl>* us)
{
  if ( !board->type->isFun ) {
    Diagnostic(ERROR,ccchere) << board->name
                              << " is not a function so it cannot "
                              << " have a body" ;
    return NULL ;
  }
  Linkage link = External;
  switch (board->storage) {
  case P_NoStorageClass:
  case P_Extern:          link=External; break;
  case P_Static:          link=Internal; break;
  case P_Typedef:         
  case P_Register:
  case P_Auto:
    Diagnostic(ERROR,board->pos) << board->name <<
      "() has a bad storage class specifier" ;
  }
  return new FunDef(board->name,board->pos,board->type,link,ds,us);
}

FunDef*
cccParser :: common_add_Fun (FunDef* f)
{
  // check for prototypes
  ObjectDecl *proto = names.lookup(f->name) ;
  if ( proto != NULL ) {
    // reuse a prototype.
    if ( f->linkage == External )
      f->linkage = proto->linkage ;
    if ( f->linkage != proto->linkage ) {
      Diagnostic d(ERROR,f->pos);
      d << f->name << " redeclared with different linkage" ;
      d.addline(proto->pos) << "(previous declaration here)" ;
    }
    // Check that the declarations match.
    if (!unify_types(f->type,proto->type))
      type_mismatch(f->name,proto->type,proto->pos,f->type,f->pos);
    assert(proto->tag==Fundef);
    FunDef* x = (FunDef*) proto;
    if (x->stmts != NULL) {
      Diagnostic d(ERROR,f->pos);
      d << "redefinition of function " << x->name;
      d.addline(x->pos) << "(previous definition here)" ;
    }
    // Use the existing object but transfer body and position.
    if (debug^5) {
      debug << "add_fundef: using existing function "
            << x->name << "\n";
    }
    x->pos = f->pos;
    x->decls = f->decls;
    x->userdecls = f->userdecls;
    x->stmts = f->stmts;
    x->effects = f->effects ;
    x->calltime = f->calltime ;
    delete f ;
    // if this prototype is not the lead element in the union, promote it
    if( x->refer_to != NULL ) {
      x->refer_to = NULL ;
      external_linker(x,true);
    }
    reenter_scope_at_next_block = 1;
    return x;
  } else {
    // There is no prototype in scope
    // Make the name and definion known.    
    names.insert(f->name,f);
    functions->push_back(f);
    if ( f->linkage == External )
      external_linker(f,true);
    // Signal that forthcomming declarations in the body of the
    // function belongs to the same name scope.
    reenter_scope_at_next_block = 1;
    if (debug^4) {
      debug << "add_FunDef: ";
      f->show(debug);
      debug << "\n";
    }
    return f;
  }
}

FunDef*
cccParser :: add_FunDef (Parse_Decl* board)
{
  // Create a function definition with all the current parameters
  // from the object scope that has just been left.
  reenter_scope_at_next_block = 1;
  enter_scope();
  FunDef* f = make_final_FunDef(board,objects.get_list(),userdecls.get_list());
  leave_scope();
  return common_add_Fun(f);
}

void
cccParser::old_style_decl_list (void)
{
  old_function=1;
  reenter_scope_at_next_block = 1;
  enter_scope();
}

FunDef*
cccParser::old_style_add_FunDef (Parse_Decl* board)
{
  old_function=0;
  // Create a function definition with all the current parameters
  // from the object scope that has just been left.
  reenter_scope_at_next_block = 1;
  enter_scope();
  // Collect the types for the arguments. 
  Plist<VarDecl>* parameters = objects.get_list();
  Plist<Type>* types = new Plist<Type>();
  Plist<VarDecl>::iterator first = parameters->begin();
  while(first) {
    // If no type is specified it gets to be integer.
    if ((*first)->type == NULL)
      (*first)->type = BaseType::get(Int);
    types->push_back((*first)->type);
    first++;
  }
  board->type = new FunType(types,board->type);
  FunDef* f = make_final_FunDef(board,parameters,userdecls.get_list());
  leave_scope();
  //     if (debug^4) {
  //         debug << "old_fun: ";
  //         f->show(debug);
  //         debug << "\n";
  //     }
  return common_add_Fun(f);
}

void
cccParser::add_Label (LabelStmt* stmt)
{
  Indirection* i = goto_labels.lookup(stmt->label);
  if (i==NULL) {
    // The label has not been mentioned before. Insert it into
    // the symboltable, pointing to the right statement.
    goto_labels.insert(stmt->label,new Indirection(stmt->pos,stmt));
  }
  else {
    // The label has been seen before. Check whether it was a
    // real declaration or just a forward reference.
    if (i->stmt == NULL) {
      // Backpatch with the real statement.
      i->stmt = stmt;
    }
    else
      Diagnostic(ERROR,ccchere) << "redeclaration of label "
                                << stmt->label ;
  }
}

Indirection*
cccParser::add_goto (char* label, Position p)
{
  Indirection* i = goto_labels.lookup(label);
  if (i==NULL) {
    // The label has not been mentioned before. Insert it into
    // the symboltable, pointing to nothing.
    Indirection* ind = new Indirection(p, NULL);
    goto_labels.insert(label,ind);
    return ind;
  }
  else {
    // The label has been seen before.
    return i;
  }
}

void cccParser :: update_arg_old_fun_decl(Parse_Decl* d)
{
  // Find the dummy declaration for the parameter.
  ObjectDecl* oldDecl = names.lookup_local(d->name);
  if (!oldDecl) {
    Diagnostic di(ERROR,ccchere);
    di << "old style function declaration:" ;
    di.addline() << "no such parameter " << d->name ;
  }
  else {
    assert(oldDecl->tag == Vardecl);
    VarDecl* old = (VarDecl*) oldDecl;
    if (old->type) {
      Diagnostic di(ERROR,ccchere);
      di << "old style function declaration:" ;
      di.addline() << "redeclaration of " << d->name ;
      if (debug^2) {
        di.addline() << "old type was ";
        old->type->show(di);
        di << "]";
      }
      return ;
    }
    // Update the placeholder.
    old->pos = d->pos;
    old->type = d->type;
    if (debug^4) {
      debug << "update parameter: ";
      old->show(debug);
      debug << "\n";
    }
    // The varmode is already set: old-style parameters start out
    // their life as ordinary 'noclass' declarations with type NULL!
  }
}

void
cccParser::add_noclass (Parse_Decl* d)
{
  // this is only used for *local* noclass definitions of *objects*.
  VarDecl* decl = new VarDecl(d->name, d->pos,d->type,NoLinkage,VarIntAuto);
  if (debug^4) {
    debug << "add_noclass: ";
    decl->show(debug);
    debug << "\n";
  }
  names.insert(d->name, decl);
  objects.push_back(decl);
}

void
cccParser::external_linker(ObjectDecl *d,bool is_defn)
{
  static SymbolTable<ObjectDecl> externals;

  assert(d->linkage==External);
  ObjectDecl *prev = externals.lookup(d->name) ;
  if ( prev == NULL ) {
    externals.insert(d->name,d);
    return ;
  }
  assert(prev->refer_to == NULL);
  assert(d->refer_to == NULL);
  if ( is_defn ) {
    if ( prev->is_definition() ) {
      Diagnostic di(ERROR,d->pos) ;
      di << "multiple definitions of " << d->name ;
      di.addline(prev->pos) << "(previous definition here)" ;
      return ;
    } else {
      // this is a definition; prev is not. Swap them
      ObjectDecl *temp = prev ; prev = d ; d = temp ;
      externals.remove(prev->name);
      externals.insert(prev->name,prev);
    }
  }
  if ( !unify_types(prev->type,d->type) )
    type_mismatch(d->name,prev->type,prev->pos,d->type,d->pos) ;
  d->refer_to = prev ;
}

void
cccParser::add_extern(Parse_Decl* d, bool tentative_defn)
{
  ObjectDecl* ex = names.lookup_global(d->name);
  if (ex != NULL) {
    // If there is a visible file scope definition we use it and
    // inherit its linkage.
    // Check that the declarations match.
    if ( !unify_types(ex->type,d->type) ) {
      type_mismatch(d->name,ex->type,ex->pos,d->type,d->pos);
      return ;
    }
    // if this is a tentative definition (which only occurs
    // for redeclarations with file scope) and the declaration
    // we already know is not a definition, we make this
    // definition the main one.
    if ( tentative_defn ) {
      assert(ex->tag==Vardecl) ;
      VarDecl *v = (VarDecl*) ex ;
      if ( v->varmode == VarExtDefault ) {
        v->varmode = VarIntAuto ;
        v->pos = ccchere ;
      }
    }
    if (debug^5) {
      debug << "add_extern: using existing object for " << d->name << endl ;
    }
    // Insert a placeholder (pointing to the global).
    names.insert(d->name,ex);
  } else {
    // There is no visible global definition for this. Make a new
    // one; with external linkage.
    ObjectDecl* decl;
    // Function prototypes should eventually be a real function,
    // so they are inserted as a function.
    if (d->type->isFun) {
      FunDef* fun = new FunDef(d->name, d->pos,d->type,External,NULL,NULL);
      functions->push_back(fun);
      decl = fun;
    } else {
      VariableMode thevarmode = tentative_defn ? VarIntAuto : VarExtDefault ;
      VarDecl* var
        = new VarDecl(d->name, d->pos,d->type,External,thevarmode);
      objects.push_back_global(var);
      decl = var;
    }
    if (debug^4) {
      debug << "add_extern: ";
      decl->show(debug);
      debug << "\n";
    }
    names.insert(d->name, decl);
    external_linker(decl,tentative_defn);
  }
}

void
cccParser::add_static(Parse_Decl* d)
{
  ObjectDecl* decl;
  if (d->type->isFun) {
    FunDef* fun = new FunDef(d->name,d->pos,d->type,Internal,NULL,NULL);
    functions->push_back(fun);
    decl = fun;
  } else {
    VarDecl* var = new VarDecl(d->name,d->pos,d->type,Internal,VarIntAuto);
    var->static_local = true ;
    objects.push_back(var);
    decl = var;
  }
  if (debug^4) {
    debug << "add_static: ";
    decl->show(debug);
    debug << "\n";
  }
  names.insert(d->name, decl);
}

// Construct a final declaration, check the integrity of it,
// and put it in current scope.
void cccParser::add_declaration (Parse_Decl *d)
{
  assert(d->name != NULL);

  last_decl_can_have_init = 1;
  if (old_function) {
    last_decl_can_have_init = 0;
    update_arg_old_fun_decl(d);
    return ;
  }

  // Find a previous declaration if such exists.
  ObjectDecl* oldDecl = names.lookup_local(d->name);
  // If we're at anything but file scope, it is not allowed to
  // redeclare a name.
  if ( names.get_level() > 0 && oldDecl ) {
    Diagnostic di(ERROR,ccchere) ;
    di << "redeclaration of " << d->name ;
    di.addline(oldDecl->pos) << "(previous declaration here)"  ;
    return ;
  }

  if (d->type != NULL && d->type->isFun) {
    last_decl_can_have_init = 0;
    // Function prototypes in non-global scope can only have extern or no
    // storage specifier.
    if ( names.get_level() > 0 && d->storage != P_Extern ) {
      Diagnostic(ERROR,ccchere) << "prototype can only have external linkage";
      d->storage = P_Extern ;
    }
  }
  // If the declaration is extern or static we put it in the
  // global list of declarations. 
  switch (d->storage) {
  case P_Extern:
    add_extern(d,false);
    break;
  case P_Static:
    add_static(d);
    break;
  case P_Typedef:
    last_decl_can_have_init = 0;
    if (oldDecl)
      Diagnostic(ERROR,ccchere) << "redeclaration of " << d->name;
    else {
      VarDecl *decl = new VarDecl(d->name, d->pos,d->type,Typedef,VarMu);
      if (debug^4) {
        debug << "add_typedef: ";
        decl->show(debug);
        debug << "\n";
      }
      names.insert(d->name, decl);
      objects.push_back(decl);
    }
    break;
  case P_Register: // We dont care about register/auto.
  case P_Auto:
    if (names.get_level() == 0)
      Diagnostic(ERROR,ccchere) << "register or auto specified for "
                                   "top-level declaration " << d->name ;
    add_noclass(d);
    break;
  case P_NoStorageClass:
    // external declarations with no explicit storage class are
    // considered external. When an object is declared, the
    // declaration counts as a (tentative) definition; see
    // the Standard, 6.7.2
    if ( names.get_level()==0 ) {
      // NULL types should occur only in old-style parameter lists.
      assert(d->type != NULL);
      add_extern(d,!d->type->isFun);
    } else
      add_noclass(d);
    break;
  default:
    Diagnostic(INTERNAL,d->pos) << "unknown storage class";
  }
}

void
cccParser :: add_initializer (Init* init)
{
  if ( init == NULL )
    return ;
  ObjectDecl* lastOne = names.last_added();
  if (!last_decl_can_have_init) {
    Diagnostic(ERROR,ccchere) << "initializer not allowed here";
    return ;
  }
  assert(lastOne->tag == Vardecl);
  lastOne->set_initializer(init);
  // now the declaration is a definition. If it has external linkage
  // we need to make it the main copy of the definition.
  // We can see that by the refer_to pointer which is NULL unless
  // the external linker has been around.
  if ( lastOne->refer_to != NULL ) {
    // the external_linker will do the job for us if we temporarily
    // untie this declaration from the tree.
    lastOne->refer_to = NULL ;
    external_linker(lastOne,true);
  }
}

void
cccParser::type_mismatch(char const *n,
                         Type *t1,Position p1,
                         Type *t2, Position p2)
{
  Diagnostic d(ERROR,Position());
  d << "conflicting types given for " << n << ':' ;
  t1->show(d.addline(p1));
  t2->show(d.addline(p2));
}

VarExpr*
cccParser :: add_var_ref (char* name, Position pos)
{
  ObjectDecl* d = names.lookup(name);
  if (d==NULL) {
    Diagnostic(ERROR,ccchere) << "unknown variable " << name ;
    // XXX this should be an implicit function declaration ...
    d = new VarDecl(name,pos,BaseType::get(Int),NoLinkage,VarMu);
  }
  return new VarExpr(d,pos);
}

void
cccParser :: clean_up_between_files()
{
  names.restart();
  usernames.restart();
  membernames.restart();
}

CProgram*
cccParser :: get_program()
{
  CProgram* pgm = new CProgram;
  pgm->functions = functions;
  pgm->definitions = objects.get_list();
  pgm->usertypes = userdecls.get_list();
  return pgm;
}
