/* File: int.c
 * Time-stamp: <99/03/18 13:15:39 panic>
 * Author: Arne John Glenstrup <panic@diku.dk>
 * Description: Interpreter example: syntax constructors
 */

#include <string.h>
#ifdef __CMIX__
#  include "stdio.h"
#  include "stdlib.h"
#else
#  include <stdio.h>
#  include <stdlib.h>
#endif

#include "cint.h"

Exp* CONST(Value constValue) {
  Exp* e = (Exp*)malloc(1 * sizeof(Exp));
  e->tag = Const;
  e->constValue = constValue;
  return e;
}

Exp* VAR(const char* varName) {
  Exp* e = (Exp*)malloc(1 * sizeof(Exp));
  char* tmp = (char*)malloc(strlen(varName) + 1);
  strcpy(tmp, varName);
  e->tag = Var;
  e->varName = tmp;
  return e;
}

Exp* UNOP(OpTag operat, Exp* exp1) {
  Exp* e = (Exp*)malloc(1 * sizeof(Exp));
  e->tag = UnOp;
  e->operat = operat;
  e->exp1 = exp1;
  return e;
}

Exp* BINOP(OpTag operat, Exp* exp1, Exp* exp2) {
  Exp* e = (Exp*)malloc(1 * sizeof(Exp));
  e->tag = BinOp;
  e->operat = operat;
  e->exp1 = exp1;
  e->exp2 = exp2;
  return e;
}

void DELEXP(Exp* e) {
  if (e) {
    switch (e->tag) {
    case Const  : break;
    case Var    : free((char*)(e->varName)); break;
    case UnOp   : DELEXP(e->exp1); break;
    case BinOp  : DELEXP(e->exp1); DELEXP(e->exp2); break;
    }
    free(e);
  }
}

Decl* DECL(const char* name, Decl* next) {
  Decl* d = (Decl*)malloc(1 * sizeof(Decl));
  char* tmp = (char*)malloc(strlen(name + 1));
  strcpy(tmp, name);
  d->name = tmp; d->next = next;
  return d;
}  

void DELDECL(Decl* d) {
  if (d) {
    DELDECL(d->next);
    free((char*)(d->name));
    free(d);
  }
}

Stmt* EXPRESSION(Exp* exp, Stmt* next) {
  Stmt* s = (Stmt*)malloc(1 * sizeof(Stmt));
  s->tag = Expression;
  s->exp = exp;
  s->next = next;
  return s;
}

Stmt* IFELSE(Exp* exp, Stmt* ifStmt, Stmt* elseStmt, Stmt* next) {
  Stmt* s = (Stmt*)malloc(1 * sizeof(Stmt));
  s->tag = IfElse;
  s->exp = exp;
  s->ifStmt = ifStmt;
  s->elseStmt = elseStmt;
  s->next = next;
  return s;
}

Stmt* WHILE(Exp* exp, Stmt* whileStmts, Stmt* next) {
  Stmt* s = (Stmt*)malloc(1 * sizeof(Stmt));
  s->tag = While;
  s->exp = exp;
  s->whileStmts = whileStmts;
  s->next = next;
  return s;
}

Stmt* RETURN(Exp* exp, Stmt* next) {
  Stmt* s = (Stmt*)malloc(1 * sizeof(Stmt));
  s->tag = Return;
  s->exp = exp;
  s->next = next;
  return s;
}

Stmt* COMPOUND(Decl* decls, Stmt* stmts, Stmt* next) {
  Stmt* s = (Stmt*)malloc(1 * sizeof(Stmt));
  s->tag = Compound;
  s->decls = decls;
  s->stmts = stmts;
  s->next = next;
  return s;
}

void DELSTMT(Stmt* s) {
  if (s) {
    switch (s->tag) {
    case Expression : DELEXP(s->exp); break;
    case IfElse     :
      DELEXP(s->exp); DELSTMT(s->ifStmt); DELSTMT(s->elseStmt); break;
    case While      : DELEXP(s->exp); DELSTMT(s->whileStmts); break;
    case Return     : DELEXP(s->exp); break;
    case Compound   : DELDECL(s->decls); DELSTMT(s->stmts); break;
    }
    DELSTMT(s->next);
    free(s);
  }
}

Formal* FORMAL(const char* name, Formal* next) {
  Formal* p = (Formal*)malloc(1 * sizeof(Formal));
  char* tmp = (char*)malloc(strlen(name) + 1);
  strcpy(tmp, name);
  p->name = tmp;
  p->next = next;
  return p;
}

void DELFORMAL(Formal* p) {
  if (p) {
    free((char*)(p->name));
    DELFORMAL(p->next);
    free(p);
  }
}

Func* FUNC(const char* name, Formal* params, Stmt* stmts, Func* next) {
  Func* f = (Func*)malloc(1 * sizeof(Func));
  char* tmp = (char*)malloc(strlen(name) + 1);
  strcpy(tmp, name);
  f->name = tmp;
  f->params = params; f->stmts = stmts; f->next = next;
  return f;
}

void DELFUNC(Func* f) {
  if (f) {
    free(f->name);
    DELFORMAL(f->params);
    DELSTMT(f->stmts);
    DELFUNC(f->next);
  }
}

Prog* PROG(Func* funcs) {
  Prog* p = (Prog*)malloc(1 * sizeof(Prog));
  p->funcs = funcs;
  return p;
}

void DELPROG(Prog* p) {
  if (p) {
    DELFUNC(p->funcs);
    free(p);
  }
}

boolean val2boolean(Value v) { return v; }
int  val2int (Value v) { return v; }
Value int2val(int i)   { return i; }

Value uminus  (Value v)            { return -v; }
Value negate  (Value v)            { return !v; }
Value plus    (Value v1, Value v2) { return v1 +  v2; }
Value minus   (Value v1, Value v2) { return v1 -  v2; }
Value multiply(Value v1, Value v2) { return v1 *  v2; }
Value divide  (Value v1, Value v2) { return v1 /  v2; }
Value eq      (Value v1, Value v2) { return v1 == v2; }
Value ne      (Value v1, Value v2) { return v1 != v2; }
Value gt      (Value v1, Value v2) { return v1 >  v2; }
Value lt      (Value v1, Value v2) { return v1 <  v2; }
Value ge      (Value v1, Value v2) { return v1 >= v2; }
Value le      (Value v1, Value v2) { return v1 <= v2; }

