/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel2.h"

void ATL_drefgemv
(
   const enum ATLAS_TRANS     TRANS,
   const int                  M,
   const int                  N,
   const double               ALPHA,
   const double               * A,
   const int                  LDA,
   const double               * X,
   const int                  INCX,
   const double               BETA,
   double                     * Y,
   const int                  INCY
)
{
/*
 * Purpose
 * =======
 *
 * ATL_drefgemv performs one of the matrix-vector operations
 *
 *    y := alpha * op( A ) * x + beta * y,
 *
 * where op( X ) is one of
 *
 *    op( X ) = X   or   op( X ) = X'.
 *
 * where  alpha and beta are scalars, x and y are vectors and op( A ) is
 * an m by n matrix.
 *
 * Arguments
 * =========
 *
 * TRANS   (input)                       const enum ATLAS_TRANS
 *         On entry,  TRANS  specifies the  operation to be performed as
 *         follows:
 *
 *            TRANS = AtlasNoTrans    y := alpha*A *x + beta*y,
 *
 *            TRANS = AtlasConj       y := alpha*A *x + beta*y,
 *
 *            TRANS = AtlasTrans      y := alpha*A'*x + beta*y,
 *
 *            TRANS = AtlasConjTrans  y := alpha*A'*x + beta*y.
 *
 *         Unchanged on exit.
 *
 * M       (input)                       const int
 *         On entry,  M  specifies  the number of rows of  the matrix  A
 *         when TRANS = AtlasNoTrans or TRANS = AtlasConj,  and the num-
 *         ber of columns of the matrix  A otherwise. M must be at least
 *         zero. Unchanged on exit.
 *
 * N       (input)                       const int
 *         On entry, N  specifies  the number of columns of the matrix A
 *         when TRANS = AtlasNoTrans or TRANS = AtlasConj,  and the num-
 *         ber of rows of the matrix A otherwise. N must be at least ze-
 *         ro. Unchanged on exit.
 *
 * ALPHA   (input)                       const double
 *         On entry, ALPHA specifies the scalar alpha.   When  ALPHA  is
 *         supplied as zero then  A and X  need not be set on input. Un-
 *         changed on exit.
 *
 * A       (input)                       const double *
 *         On entry,  A  points  to an array of size equal to or greater
 *         than   LDA * ka * sizeof(   double  ), where  ka  is  n  when
 *         TRANS = AtlasNotrans or TRANS = AtlasConj, and  m  otherwise.
 *         Before entry, when TRANS = AtlasNotrans or TRANS = AtlasConj,
 *         the leading m by n part of the array  A  must contain the ma-
 *         trix coefficients,  and otherwise the leading n by m  part of
 *         the array A  must contain the matrix coefficients.  Unchanged
 *         on exit.
 *
 * LDA     (input)                       const int
 *         On entry, LDA  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDA  must be  at least
 *         MAX( 1, m )  when  TRANS = AtlasNotrans or TRANS = AtlasConj,
 *         and MAX( 1, n ) otherwise. Unchanged on exit.
 *
 * X       (input)                       const double *
 *         On entry,  X  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCX ) ) * sizeof(   double  ),
 *         that contains the vector x. Unchanged on exit.
 *
 * INCX    (input)                       const int
 *         On entry, INCX specifies the increment for the elements of X.
 *         INCX must not be zero. Unchanged on exit.
 *
 * BETA    (input)                       const double
 *         On entry,  BETA  specifies the scalar  beta.   When  BETA  is
 *         supplied as zero then Y  need not be set on input.  Unchanged
 *         on exit.
 *
 * Y       (input/output)                double *
 *         On entry,  Y  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( m - 1 ) * abs( INCY ) ) * sizeof(   double  ),
 *         that contains the vector y.  Before entry with BETA non-zero,
 *         the incremented array  Y  must contain the vector y. On exit,
 *         Y is overwritten by the updated vector y.
 *
 * INCY    (input)                       const int
 *         On entry, INCY specifies the increment for the elements of Y.
 *         INCY must not be zero. Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   int                        i, iaij, ix, iy, j, jaj, jx, jy;
   register double            t0;
/* ..
 * .. Executable Statements ..
 *
 */
   if( ( M == 0 ) || ( N == 0 ) ||
       ( ( ALPHA == ATL_dZERO ) && ( BETA == ATL_dONE  ) ) ) return;

   if(      BETA == ATL_dZERO )
   { for( i = 0, iy = 0; i < M; i++, iy += INCY ) Y[iy]  = ATL_dZERO; }
   else if( BETA != ATL_dONE )
   { for( i = 0, iy = 0; i < M; i++, iy += INCY ) Y[iy] *= BETA;      }

   if( ALPHA == ATL_dZERO ) return;

   if( ( TRANS == AtlasNoTrans ) || ( TRANS == AtlasConj ) )
   {
      for( j = 0,      jaj  = 0,   jx  = 0;
           j < N; j++, jaj += LDA, jx += INCX )
      {
         t0 = ALPHA * X[jx];
         for( i = 0,      iaij  = jaj, iy  = 0;
              i < M; i++, iaij += 1,   iy += INCY )
         {
            Y[iy] += A[iaij] * t0;
         }
      }
   }
   else
   {
      for( j = 0,      jaj  = 0,   jy  = 0;
           j < M; j++, jaj += LDA, jy += INCY )
      {
         t0 = ATL_dZERO;
         for( i = 0,      iaij  = jaj, ix  = 0;
              i < N; i++, iaij += 1,   ix += INCX )
         {
            t0 += A[iaij] * X[ix];
         }
         Y[jy] += ALPHA * t0;
      }
   }
/*
 * End of ATL_drefgemv
 */
}
