/*
 * =====================================================================
 * Include files
 * =====================================================================
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "atlas_misc.h"
#include "atlas_tst.h"
/*
 * =====================================================================
 * #define macro constants
 * =====================================================================
 */
#define    MEGA                     1000000.0
#if defined( SREAL ) || defined( SCPLX )
#define    THRESH                        50.0f
#else
#define    THRESH                        50.0
#endif

#define    ATLAS_DEBUG
/*
 * =====================================================================
 * # macro functions
 * =====================================================================
 *
 * The following and mutually exclusive  macros  allow to select various
 * BLAS implementations to test the ATLAS implementation against:
 *
 *    USE_F77_BLAS     : Fortran 77 BLAS interface,
 *    USE_L2_REFERENCE : C ATLAS reference implementation,
 *
 * If none of these macros is defined at compile time, the  ATLAS imple-
 * mentation is to be tested against itself,  after all this is the only
 * version we are sure to have available.
 */
#define USE_F77_BLAS

#if   defined( USE_F77_BLAS ) /* Trusted BLAS version to test against */
#define  TP2      Mjoin( PATL,   f77 )
#elif defined( USE_L2_REFERENCE )
#include "atlas_reflevel2.h"
#define  TP2      Mjoin( PATL,   ref )
#else /* defined( USE_L2_ATLAS ) */  /* use ATLAS itself !! (default) */
#include "atlas_level2.h"
#define  TP2      PATL
#endif

#define trusted_gbmv(   TA,      M, N, KL, KU, al, A, lA, X, iX, be, Y, iY) \
Mjoin(TP2,gbmv) (       TA,      M, N, KL, KU, al, A, lA, X, iX, be, Y, iY)
#define trusted_gemv(   TA,      M, N,         al, A, lA, X, iX, be, Y, iY) \
Mjoin(TP2,gemv) (       TA,      M, N,         al, A, lA, X, iX, be, Y, iY)

#ifdef TREAL
#define trusted_sbmv(UP,         N,    K,      al, A, lA, X, iX, be, Y, iY) \
Mjoin(TP2,sbmv) (    UP,         N,    K,      al, A, lA, X, iX, be, Y, iY)
#define trusted_spmv(UP,         N,            al, A,     X, iX, be, Y, iY) \
Mjoin(TP2,spmv) (    UP,         N,            al, A,     X, iX, be, Y, iY)
#define trusted_symv(UP,         N,            al, A, lA, X, iX, be, Y, iY) \
Mjoin(TP2,symv) (    UP,         N,            al, A, lA, X, iX, be, Y, iY)
#else
#define trusted_sbmv(UP,         N,    K,      al, A, lA, X, iX, be, Y, iY) \
Mjoin(TP2,hbmv) (    UP,         N,    K,      al, A, lA, X, iX, be, Y, iY)
#define trusted_spmv(UP,         N,            al, A,     X, iX, be, Y, iY) \
Mjoin(TP2,hpmv) (    UP,         N,            al, A,     X, iX, be, Y, iY)
#define trusted_symv(UP,         N,            al, A, lA, X, iX, be, Y, iY) \
Mjoin(TP2,hemv) (    UP,         N,            al, A, lA, X, iX, be, Y, iY)
#endif

#define trusted_tbmv(UP, TA, DI, N,    K,          A, lA, X, iX) \
Mjoin(TP2,tbmv) (    UP, TA, DI, N,    K,          A, lA, X, iX)
#define trusted_tpmv(UP, TA, DI, N,                A,     X, iX) \
Mjoin(TP2,tpmv) (    UP, TA, DI, N,                A,     X, iX)
#define trusted_trmv(UP, TA, DI, N,                A, lA, X, iX) \
Mjoin(TP2,trmv) (    UP, TA, DI, N,                A, lA, X, iX)

#define trusted_tbsv(UP, TA, DI, N,    K,          A, lA, X, iX) \
Mjoin(TP2,tbsv) (    UP, TA, DI, N,    K,          A, lA, X, iX)
#define trusted_tpsv(UP, TA, DI, N,                A,     X, iX) \
Mjoin(TP2,tpsv) (    UP, TA, DI, N,                A,     X, iX)
#define trusted_trsv(UP, TA, DI, N,                A, lA, X, iX) \
Mjoin(TP2,trsv) (    UP, TA, DI, N,                A, lA, X, iX)

#ifdef TREAL
#define trusted_geru(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(TP2,ger)  (                M, N,         al, X, iX, Y, iY,     A, lA)
#define trusted_gerc(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(TP2,ger)  (                M, N,         al, X, iX, Y, iY,     A, lA)
#else
#define trusted_geru(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(TP2,geru) (                M, N,         al, X, iX, Y, iY,     A, lA)
#define trusted_gerc(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(TP2,gerc) (                M, N,         al, X, iX, Y, iY,     A, lA)
#endif

#ifdef TREAL
#define  trusted_spr(UP,         N,            al, X, iX,            A    ) \
Mjoin(TP2,spr)  (    UP,         N,            al, X, iX,            A    )
#define  trusted_syr(UP,         N,            al, X, iX,            A, lA) \
Mjoin(TP2,syr)  (    UP,         N,            al, X, iX,            A, lA)
#else
#define  trusted_spr(UP,         N,            al, X, iX,            A    ) \
Mjoin(TP2,hpr)  (    UP,         N,            al, X, iX,            A    )
#define  trusted_syr(UP,         N,            al, X, iX,            A, lA) \
Mjoin(TP2,her)  (    UP,         N,            al, X, iX,            A, lA)
#endif

#ifdef TREAL
#define trusted_spr2(UP,         N,            al, X, iX, Y, iY,     A    ) \
Mjoin(TP2,spr2) (    UP,         N,            al, X, iX, Y, iY,     A    )
#define trusted_syr2(UP,         N,            al, X, iX, Y, iY,     A, lA) \
Mjoin(TP2,syr2) (    UP,         N,            al, X, iX, Y, iY,     A, lA)
#else
#define trusted_spr2(UP,         N,            al, X, iX, Y, iY,     A    ) \
Mjoin(TP2,hpr2) (    UP,         N,            al, X, iX, Y, iY,     A    )
#define trusted_syr2(UP,         N,            al, X, iX, Y, iY,     A, lA) \
Mjoin(TP2,her2) (    UP,         N,            al, X, iX, Y, iY,     A, lA)
#endif
/*
 * ATLAS version of the BLAS to test.
 */
#include "atlas_level2.h"

#define    test_gbmv(   TA,      M, N, KL, KU, al, A, lA, X, iX, be, Y, iY) \
Mjoin(PATL,gbmv)    (   TA,      M, N, KL, KU, al, A, lA, X, iX, be, Y, iY)
#define    test_gemv(   TA,      M, N,         al, A, lA, X, iX, be, Y, iY) \
Mjoin(PATL,gemv)    (   TA,      M, N,         al, A, lA, X, iX, be, Y, iY)

#ifdef TREAL
#define    test_sbmv(UP,         N,    K,      al, A, lA, X, iX, be, Y, iY) \
Mjoin(PATL,sbmv)    (UP,         N,    K,      al, A, lA, X, iX, be, Y, iY)
#define    test_spmv(UP,         N,            al, A,     X, iX, be, Y, iY) \
Mjoin(PATL,spmv)    (UP,         N,            al, A,     X, iX, be, Y, iY)
#define    test_symv(UP,         N,            al, A, lA, X, iX, be, Y, iY) \
Mjoin(PATL,symv)    (UP,         N,            al, A, lA, X, iX, be, Y, iY)
#else
#define    test_sbmv(UP,         N,    K,      al, A, lA, X, iX, be, Y, iY) \
Mjoin(PATL,hbmv)    (UP,         N,    K,      al, A, lA, X, iX, be, Y, iY)
#define    test_spmv(UP,         N,            al, A,     X, iX, be, Y, iY) \
Mjoin(PATL,hpmv)    (UP,         N,            al, A,     X, iX, be, Y, iY)
#define    test_symv(UP,         N,            al, A, lA, X, iX, be, Y, iY) \
Mjoin(PATL,hemv)    (UP,         N,            al, A, lA, X, iX, be, Y, iY)
#endif

#define    test_tbmv(UP, TA, DI, N,    K,          A, lA, X, iX) \
Mjoin(PATL,tbmv)    (UP, TA, DI, N,    K,          A, lA, X, iX)
#define    test_tpmv(UP, TA, DI, N,                A,     X, iX) \
Mjoin(PATL,tpmv)    (UP, TA, DI, N,                A,     X, iX)
#define    test_trmv(UP, TA, DI, N,                A, lA, X, iX) \
Mjoin(PATL,trmv)    (UP, TA, DI, N,                A, lA, X, iX)

#define    test_tbsv(UP, TA, DI, N,    K,          A, lA, X, iX) \
Mjoin(PATL,tbsv)    (UP, TA, DI, N,    K,          A, lA, X, iX)
#define    test_tpsv(UP, TA, DI, N,                A,     X, iX) \
Mjoin(PATL,tpsv)    (UP, TA, DI, N,                A,     X, iX)
#define    test_trsv(UP, TA, DI, N,                A, lA, X, iX) \
Mjoin(PATL,trsv)    (UP, TA, DI, N,                A, lA, X, iX)

#ifdef TREAL
#define    test_geru(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(PATL,ger)     (            M, N,         al, X, iX, Y, iY,     A, lA)
#define    test_gerc(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(PATL,ger)     (            M, N,         al, X, iX, Y, iY,     A, lA)
#else
#define    test_geru(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(PATL,geru)    (            M, N,         al, X, iX, Y, iY,     A, lA)
#define    test_gerc(            M, N,         al, X, iX, Y, iY,     A, lA) \
Mjoin(PATL,gerc)    (            M, N,         al, X, iX, Y, iY,     A, lA)
#endif

#ifdef TREAL
#define     test_spr(UP,         N,            al, X, iX,            A    ) \
Mjoin(PATL,spr)     (UP,         N,            al, X, iX,            A    )
#define     test_syr(UP,         N,            al, X, iX,            A, lA) \
Mjoin(PATL,syr)     (UP,         N,            al, X, iX,            A, lA)
#else
#define     test_spr(UP,         N,            al, X, iX,            A    ) \
Mjoin(PATL,hpr)     (UP,         N,            al, X, iX,            A    )
#define     test_syr(UP,         N,            al, X, iX,            A, lA) \
Mjoin(PATL,her)     (UP,         N,            al, X, iX,            A, lA)
#endif

#ifdef TREAL
#define    test_spr2(UP,         N,            al, X, iX, Y, iY,     A    ) \
Mjoin(PATL,spr2)    (UP,         N,            al, X, iX, Y, iY,     A    )
#define    test_syr2(UP,         N,            al, X, iX, Y, iY,     A, lA) \
Mjoin(PATL,syr2)    (UP,         N,            al, X, iX, Y, iY,     A, lA)
#else
#define    test_spr2(UP,         N,            al, X, iX, Y, iY,     A    ) \
Mjoin(PATL,hpr2)    (UP,         N,            al, X, iX, Y, iY,     A    )
#define    test_syr2(UP,         N,            al, X, iX, Y, iY,     A, lA) \
Mjoin(PATL,her2)    (UP,         N,            al, X, iX, Y, iY,     A, lA)
#endif
/*
 * =====================================================================
 * typedef definitions
 * =====================================================================
 */
enum LVL2_ROUT /* 17 + 1 = 18 */
{
   GEMV=0, GBMV, SBMV, SPMV, SYMV, TBMV, TPMV, TRMV,
   TBSV,   TPSV, TRSV,
   GERU,   GERC, SYR,  SPR,  SYR2, SPR2,
   ALLROUTS
};
/*
 * =====================================================================
 * Static variables
 * =====================================================================
 */
static TYPE eps;
/*
 * =====================================================================
 * Prototypes for the testing routines
 * =====================================================================
 */
double     opbl2
( const enum LVL2_ROUT,           const int,      const int,
  const int,      const int );
void       tbddom
( const enum ATLAS_UPLO,          const int,      const int,
  TYPE *,         const int );
void       tpddom
( const enum ATLAS_UPLO,          const int,      TYPE * );
void       trddom
( const enum ATLAS_UPLO,          const int,      TYPE *,
  const int );

TYPE       gmvtst
( const enum LVL2_ROUT,           const int,      const enum ATLAS_TRANS,
  const int,      const int,      const int,      const int,
  const SCALAR,   const int,      const int,      const SCALAR,
  const int,      double *,       double *,       double *,
  double * );
TYPE       smvtst
( const enum LVL2_ROUT,           const int,      const enum ATLAS_UPLO,
  const int,      const int,      const SCALAR,   const int,
  const int,      const SCALAR,   const int,      double *,
  double *,       double *,       double * );
TYPE       tmvtst
( const enum LVL2_ROUT,           const int,      const enum ATLAS_UPLO,
  const enum ATLAS_TRANS,         const enum ATLAS_DIAG,
  const int,      const int,      const int,      const int,
  double *,       double *,       double *,       double * );
TYPE       tsvtst
( const enum LVL2_ROUT,           const int,      const enum ATLAS_UPLO,
  const enum ATLAS_TRANS,         const enum ATLAS_DIAG,
  const int,      const int,      const int,      const int,
  double *,       double *,       double *,       double * );
TYPE       gr1tst
( const enum LVL2_ROUT,           const int,      const int,
  const int,      const SCALAR,   const int,      const int,
  const int,      double *,       double *,       double *,
  double * );
TYPE       sr1tst
( const enum LVL2_ROUT,           const int,      const enum ATLAS_UPLO,
  const int,      const TYPE,     const int,      const int,
  double *,       double *,       double *,       double * );
TYPE       sr2tst
( const enum LVL2_ROUT,           const int,      const enum ATLAS_UPLO,
  const int,      const SCALAR,   const int,      const int,
  const int,      double *,       double *,       double *,
  double * );

int        gmvcase
( const enum LVL2_ROUT,           const int,      const int,
  const enum ATLAS_TRANS,         const int,      const int,
  const int,      const int,      const SCALAR,   const int,
  const int,      const SCALAR,   const int,      double *,
  double *,       double *,       double * );
int        smvcase
( const enum LVL2_ROUT,           const int,      const int,
  const enum ATLAS_UPLO,          const int,      const int,
  const SCALAR,   const int,      const int,      const SCALAR,
  const int,      double *,       double *,       double *,
  double * );
int        tmvcase
( const enum LVL2_ROUT,           const int,      const int,
  const enum ATLAS_UPLO,          const enum ATLAS_TRANS,
  const enum ATLAS_DIAG,          const int,      const int,
  const int,      const int,      double *,       double *,
  double *,       double * );
int        tsvcase
( const enum LVL2_ROUT,           const int,      const int,
  const enum ATLAS_UPLO,          const enum ATLAS_TRANS,
  const enum ATLAS_DIAG,          const int,      const int,
  const int,      const int,      double *,       double *,
  double *,       double * );
int        gr1case
( const enum LVL2_ROUT,           const int,      const int,
  const int,      const int,      const SCALAR,   const int,
  const int,      const int,      double *,       double *,
  double *,       double * );
int        sr1case
( const enum LVL2_ROUT,           const int,      const int,
  const enum ATLAS_UPLO,          const int,      const TYPE,
  const int,      const int,      double *,       double *,
  double *,       double * );
int        sr2case
( const enum LVL2_ROUT,           const int,      const int,
  const enum ATLAS_UPLO,          const int,      const SCALAR,
  const int,      const int,      const int,      double *,
  double *,       double *,       double * );

void       RungbmCase
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_TRANS *,             int,
  int,            int,            int,            int,
  int,            int,            int,            int,
  int,            int,            int,            int,
  TYPE *,         int,            TYPE *,         int,
  int *,          int,            int *,          int *,
  int * );
void       RungemCase
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_TRANS *,             int,
  int,            int,            int,            int,
  int,            int,            TYPE *,         int,
  TYPE *,         int,            int *,          int,
  int *,          int *,          int * );
void       RunsbCase
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_UPLO *,              int,
  int,            int,            int,            int,
  int,            int,            int,            int,
  int,            TYPE *,         int,            TYPE *,
  int,            int *,          int,            int *,
  int *,          int * );
void       RunspCase
( enum LVL2_ROUT, int,            int,            int,
  enum ATLAS_UPLO *,              int,            int,
  int,            int,            TYPE *,         int,
  TYPE *,         int,            int *,          int,
  int *,          int *,          int * );
void       RunsyCase
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_UPLO *,              int,
  int,            int,            int,            TYPE *,
  int,            TYPE *,         int,            int *,
  int,            int *,          int *,          int * );
void       RuntbCase
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_UPLO *,              int,
  enum ATLAS_TRANS *,             int,            enum ATLAS_DIAG *,
  int,            int,            int,            int,
  int,            int,            int,            int,
  int,            int,            int *,          int *,
  int * );
void       RuntpCase
( enum LVL2_ROUT, int,            int,            int,
  enum ATLAS_UPLO *,              int,            enum ATLAS_TRANS *,
  int,            enum ATLAS_DIAG *,              int,
  int,            int,            int,            int *,
  int *,          int * );
void       RuntrCase
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_UPLO *,              int,
  enum ATLAS_TRANS *,             int,            enum ATLAS_DIAG *,
  int,            int,            int,            int,
  int *,          int *,          int * );
void       Rungr1Case
( enum LVL2_ROUT, int,            int,            int,
  int,            int,            int,            int,
  int,            int,            int,            TYPE *,
  int,            int *,          int,            int *,
  int *,          int * );
void       Runsp1Case
( enum LVL2_ROUT, int,            int,            int,
  enum ATLAS_UPLO *,              int,            int,
  int,            int,            TYPE *,         int,
  int *,          int *,          int * );
void       Runsr1Case
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_UPLO *,              int,
  int,            int,            int,            TYPE *,
  int,            int *,          int *,            int * );
void       Runsp2Case
( enum LVL2_ROUT, int,            int,            int,
  enum ATLAS_UPLO *,              int,            int,
  int,            int,            TYPE *,            int,
  int *,          int,            int *,            int *,
  int * );
void       Runsr2Case
( enum LVL2_ROUT, int,            int,            int,
  int,            enum ATLAS_UPLO *,              int,
  int,            int,            int,            TYPE *,
  int,            int *,          int,            int *,
  int *,          int * );
void       RunCases
( int,            int,            int,            int,
  enum ATLAS_UPLO *,              int,            enum ATLAS_TRANS *,
  int,            enum ATLAS_DIAG *,              int,
  int,            int,            int,            int,
  int,            int,            int,            int,
  int,            int,            int,            int,
  TYPE *,         int,            TYPE *,         int,
  int *,          int,            int *,          int,
  enum LVL2_ROUT * );

void       PrintUsage
( char * );

void       GetFlags
( int,            char **,        int *,          enum LVL2_ROUT **,
  int *,          int *,          int *,          int *,
  enum ATLAS_UPLO **,             int *,          enum ATLAS_TRANS **,
  int *,          enum ATLAS_DIAG **,             int *,
  int *,          int *,          int *,          int *,
  int *,          int *,          int *,          int *,
  int *,          int *,          int *,          int *,
  TYPE **,        int *,          TYPE **,        int *,
  int **,         int *,          int ** );

int        main
( int,            char ** );
/*
 * =====================================================================
 * and finally, here is the code ...
 * =====================================================================
 */
double opbl2
(
   const enum LVL2_ROUT       ROUT,
   const int                  M,
   const int                  N,
   const int                  KKL,
   const int                  KKU
)
{
   double                     i0, i1, j, adds = 0.0, em, en, ek, kl, ku,
                              muls = 0.0;

   if( ( M <= 0 ) || ( N <= 0 ) ) return( 0.0 );

   kl = (double)( Mmax( 0, Mmin( M - 1, KKL ) ) );
   ku = (double)( Mmax( 0, Mmin( N - 1, KKU ) ) );
   em = (double)(M);
   en = (double)(N);
   ek = kl;

   if(      ROUT == GEMV )
   {
      muls = em * ( en + 1.0 );
      adds = em * en;

   }
   else if( ROUT == GBMV )
   {
      for( j = 0; j < N; j++ )
      {
         i1 = (double)(j) + kl; i0 = em - 1.0; i1 = Mmin( i1, i0 );
         i0 = (double)(j) - ku; i0 = Mmax( i0, 0.0 );
         i1 -= i0 - 1;
         muls += Mmax( i1, 0.0 );
      }
      adds = muls;
/*
 *    muls = em * ( en + 1.0 ) - ( em - 1.0 - kl ) * ( em - kl ) / 2.0 -
 *                               ( en - 1.0 - ku ) * ( en - ku ) / 2.0;
 *    adds = em * ( en + 1.0 ) - ( em - 1.0 - kl ) * ( em - kl ) / 2.0 -
 *                               ( en - 1.0 - ku ) * ( en - ku ) / 2.0;
 */
   }
   else if( ( ROUT == SPMV ) || ( ROUT == SYMV ) )
   {                       /* ( ( ROUT == HPMV ) || ( ROUT == HEMV ) ) */
      muls = em * ( em + 1.0 );
      adds = em * em;
   }
   else if( ROUT == SBMV )
   {                                               /* ( ROUT == HBMV ) */
      muls = em * ( em + 1.0 ) - ( em - 1.0 - ek ) * ( em - ek );
      adds = em * em           - ( em - 1.0 - ek ) * ( em - ek );
   }
   else if( ( ROUT == TPMV ) || ( ROUT == TRMV ) )
   {
      muls = em * ( em + 1.0 ) / 2.0;
      adds = ( em - 1.0 ) * em / 2.0;
   }
   else if( ROUT == TBMV )
   {
      muls = em * ( em + 1.0 ) / 2.0 - ( em - ek - 1.0 ) * ( em - ek ) / 2.0;
      adds = ( em - 1.0 ) * em / 2.0 - ( em - ek - 1.0 ) * ( em - ek ) / 2.0;
   }
   else if( ( ROUT == TPSV ) || ( ROUT == TRSV ) )
   {
      muls = em * ( em + 1.0 ) / 2.0;
      adds = ( em - 1.0 ) * em / 2.0;
   }
   else if( ROUT == TBSV )
   {
      muls = em * ( em + 1.0 ) / 2.0 - ( em - ek - 1.0 ) * ( em - ek ) / 2.0;
      adds = ( em - 1.0 ) * em / 2.0 - ( em - ek - 1.0 ) * ( em - ek ) / 2.0;
   }
   else if( ( ROUT == GERC ) || ( ROUT == GERU ) ) /* ( ROUT == GER  ) */
   {
      muls = em * en + Mmin( em, en );
      adds = em * en;
   }
   else if( ( ROUT == SPR ) || ( ROUT == SYR ) )
   {                         /* ( ( ROUT == HPR ) || ( ROUT == HER ) ) */
      muls = em * ( em + 1.0 ) / 2.0 + em;
      adds = em * ( em + 1.0 ) / 2.0;
   }
   else if( ( ROUT == SPR2 ) || ( ROUT == SYR2 ) )
   {                       /* ( ( ROUT == HPR2 ) || ( ROUT == HER2 ) ) */
      muls = em * ( em + 1.0 ) + 2.0 * em;
      adds = em * ( em + 1.0 );
   }
#ifdef TREAL
   return(       muls +       adds );
#else
   return( 6.0 * muls + 2.0 * adds );
#endif
}

void tbddom
(
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const int                  K,
   TYPE                       * A,
   const int                  LDA
)
{
/*
 * Scale strictly lower (resp. upper) part of triangular band matrix by 1 / N
 * to make it diagonally dominant.
 */
   int                        i, i0, i1, iaij, j, jaj, l, lda2 = ( LDA SHIFT );
   TYPE                       alpha;

   if( N <= 0 ) return;

   alpha = ATL_rone / (TYPE)(N);

   if( UPLO == AtlasUpper )
   {
      for( j = 0, jaj  = 0; j < N; j++, jaj += lda2 )
      {
         l       = K - j;
         i0      = ( j - K > 0 ? j - K : 0 );

         for( i = i0, iaij = ((l+i0) SHIFT)+jaj; i < j; i++, iaij += (1 SHIFT) )
         {
            A[iaij  ] *= alpha;
#ifdef TCPLX
            A[iaij+1] *= alpha;
#endif
         }
         if( A[iaij  ] >= ATL_rzero ) A[iaij  ] += ATL_rone;
         else                         A[iaij  ] -= ATL_rone;
#ifdef TCPLX
         if( A[iaij+1] >= ATL_rzero ) A[iaij+1] += ATL_rone;
         else                         A[iaij+1] -= ATL_rone;
#endif
      }
   }
   else
   {
      for( j = N-1, jaj = (N-1)*lda2; j >= 0; j--, jaj -= lda2 )
      {
         if( A[jaj  ] >= ATL_rzero ) A[jaj  ] += ATL_rone;
         else                        A[jaj  ] -= ATL_rone;
#ifdef TCPLX
         if( A[jaj+1] >= ATL_rzero ) A[jaj+1] += ATL_rone;
         else                        A[jaj+1] -= ATL_rone;
#endif
         i1   = ( N - 1 > j + K ? j + K : N - 1 );
         for( i  = j+1, iaij = (1 SHIFT)+jaj; i <= i1; i++, iaij += (1 SHIFT) )
         {
            A[iaij  ] *= alpha;
#ifdef TCPLX
            A[iaij+1] *= alpha;
#endif
         }
      }
   }
}

void tpddom
(
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   TYPE                       * A
)
{
/*
 * Scale strictly lower (resp. upper) part of triangular packed matrix by 1 / N
 * to make it diagonally dominant.
 */
   int                        i, iaij, j;
   TYPE                       alpha;

   if( N <= 0 ) return;

   alpha = ATL_rone / (TYPE)(N);

   if( UPLO == AtlasUpper )
   {
      for( j = 0, iaij= 0; j < N; j++ )
      {
         for( i = 0; i < j; i++, iaij += (1 SHIFT) )
         {
            A[iaij  ] *= alpha;
#ifdef TCPLX
            A[iaij+1] *= alpha;
#endif
         }
         if( A[iaij  ] >= ATL_rzero ) A[iaij  ] += ATL_rone;
         else                         A[iaij  ] -= ATL_rone;
#ifdef TCPLX
         if( A[iaij+1] >= ATL_rzero ) A[iaij+1] += ATL_rone;
         else                         A[iaij+1] -= ATL_rone;
#endif
         iaij += (1 SHIFT);
      }
   }
   else
   {
      for( j = N-1, iaij = ((((N-1)*(N+2)) >> 1) SHIFT); j >= 0; j-- )
      {
         if( A[iaij  ] >= ATL_rzero ) A[iaij  ] += ATL_rone;
         else                         A[iaij  ] -= ATL_rone;
#ifdef TCPLX
         if( A[iaij+1] >= ATL_rzero ) A[iaij+1] += ATL_rone;
         else                         A[iaij+1] -= ATL_rone;
#endif
         iaij += (1 SHIFT);
         for( i = j+1; i < N; i++, iaij += (1 SHIFT) )
         {
            A[iaij  ] *= alpha;
#ifdef TCPLX
            A[iaij+1] *= alpha;
#endif
         }
         iaij -= ( ( N - j ) << (1 SHIFT) ) + (1 SHIFT);
      }
   }
}

void trddom
(
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   TYPE                       * A,
   const int                  LDA
)
{
/*
 * Scale strictly lower (resp. upper) part of triangular matrix by 1 / N
 * to make it diagonally dominant.
 */
   int                        i, iaij, j, jaj, lda2 = ( LDA SHIFT ),
                              ldap12 = (( LDA + 1 ) SHIFT);
   TYPE                       alpha;

   if( N <= 0 ) return;

   alpha = ATL_rone / (TYPE)(N);

   if( UPLO == AtlasUpper )
   {
      for( j = 0, jaj = 0; j < N; j++, jaj += lda2 )
      {
         for( i = 0, iaij = jaj; i < j; i++, iaij += (1 SHIFT) )
         {
            A[iaij  ] *= alpha;
#ifdef TCPLX
            A[iaij+1] *= alpha;
#endif
         }
         if( A[iaij  ] >= ATL_rzero ) A[iaij  ] += ATL_rone;
         else                         A[iaij  ] -= ATL_rone;
#ifdef TCPLX
         if( A[iaij+1] >= ATL_rzero ) A[iaij+1] += ATL_rone;
         else                         A[iaij+1] -= ATL_rone;
#endif
      }
   }
   else
   {
      for( j = N-1, jaj = (N-1)*ldap12; j >= 0; j--, jaj -= ldap12 )
      {
         if( A[jaj  ] >= ATL_rzero ) A[jaj  ] += ATL_rone;
         else                        A[jaj  ] -= ATL_rone;
#ifdef TCPLX
         if( A[jaj+1] >= ATL_rzero ) A[jaj+1] += ATL_rone;
         else                        A[jaj+1] -= ATL_rone;
#endif
         for( i = j+1, iaij = jaj+(1 SHIFT); i < N; i++, iaij += (1 SHIFT) )
         {
            A[iaij  ] *= alpha;
#ifdef TCPLX
            A[iaij+1] *= alpha;
#endif
         }
      }
   }
}

TYPE gmvtst
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const enum ATLAS_TRANS     TRANS,
   const int                  M,
   const int                  N,
   const int                  KL,
   const int                  KU,
   const SCALAR               ALPHA,
   const int                  LDA,
   const int                  INCX,
   const SCALAR               BETA,
   const int                  INCY,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        i, mA, nA;
   int                        Aseed, Xseed, Yseed;
   const int                  aincX = Mabs( INCX ), aincY = Mabs( INCY ),
                              L2size = L2SIZE / sizeof( double );
   TYPE                       normA, normD, normY, normX, resid;
   double                     ops, t0, ttest, ttrust;
   TYPE                       * A  = NULL, * X = NULL, * Y = NULL, * Y0,
                              * YD = NULL, * x, * y;
   double                     * L2 = NULL;

   *TTRUST0 = *TTEST0 = *MFTEST0 = *MFTRUST0 = 0.0;
   if( ( M == 0 ) || ( N == 0 ) ) { return( ATL_rzero ); }

   if( TRANS == AtlasNoTrans ) { mA = M; nA = N; }
   else                        { mA = N; nA = M; }

   ops = opbl2( ROUT, mA, nA, KL, KU );
/*
 * Allocate L2 cache space, A, X, Y and Y0
 */
   L2 = (double *)malloc( L2size * sizeof( double )        );
   A  = (TYPE   *)malloc( ATL_MulBySize( LDA ) * nA        );
   X  = (TYPE   *)malloc( ATL_MulBySize( N   ) * aincX     );
   Y  = (TYPE   *)malloc( ATL_MulBySize( M   ) * aincY * 2 );

   if( ( L2 == NULL ) || ( A == NULL ) || ( X == NULL ) || ( Y == NULL ) )
   {
      if( L2 ) free( L2 );
      if( A  ) free( A  );
      if( X  ) free( X  );
      if( Y  ) free( Y  );
      return( ATL_rnone );
   }

   Y0 = Y + M * ( aincY SHIFT );
/*
 * Generate random operands
 */
   Aseed = mA * nA + LDA;
   Xseed = N  * aincX * 27 + 213;
   Yseed = M  * aincY;

   if( ROUT == GBMV ) Mjoin( PATL, gegen )( KL+KU+1, nA, A, LDA, Aseed );
   else               Mjoin( PATL, gegen )( mA,      nA, A, LDA, Aseed );
   Mjoin( PATL, gegen )( 1,  N, X,  aincX, Xseed );
   Mjoin( PATL, gegen )( 1,  M, Y,  aincY, Yseed );
   Mjoin( PATL, gegen )( 1,  M, Y0, aincY, Yseed );
/*
 * Compute the norm of Y for later use in testing
 */
   if( TEST )
   {
      normY = Mjoin( PATL, infnrm )( M, Y, aincY );

#ifdef TREAL
      if( Mabs( BETA ) > ATL_rone ) normY *= Mabs( BETA  );
#else
      if( Mabs( *BETA ) + Mabs( BETA[1] ) > ATL_rone )
         normY *= ( Mabs( *BETA ) + Mabs( BETA[1] ) );
#endif
      if( normY == ATL_rzero ) normY = ATL_rone;
   }
/*
 * Start cold cache timing operations for the trusted routine
 */
   x = X;  if( INCX < 0 ) x = X  + ( ( ( 1 - N ) * INCX ) SHIFT );
   y = Y0; if( INCY < 0 ) y = Y0 + ( ( ( 1 - M ) * INCY ) SHIFT );

   if( ROUT == GBMV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_gbmv( TRANS, M, N, KL, KU, ALPHA, A, LDA, x, INCX, BETA, y,
                    INCY );
      ttrust = time00() - t0;
   }
   else
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_gemv( TRANS, M, N, ALPHA, A, LDA, x, INCX, BETA, y, INCY );
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 )
   { *TTRUST0 = ttrust; *MFTRUST0 = ops / ( ttrust * MEGA ); }
/*
 * Start cold cache timing operations for the tested routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - N ) * INCX ) SHIFT );
   y = Y; if( INCY < 0 ) y = Y + ( ( ( 1 - M ) * INCY ) SHIFT );

   if( ROUT == GBMV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_gbmv( TRANS, M, N, KL, KU, ALPHA, A, LDA, x, INCX, BETA, y, INCY );
      ttest = time00() - t0;
   }
   else
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_gemv( TRANS, M, N, ALPHA, A, LDA, x, INCX, BETA, y, INCY );
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 )
   { *TTEST0  = ttest;  *MFTEST0  = ops / ( ttest  * MEGA ); }
/*
 * if timing only, I am done ... so leave.
 */
   free( L2 );

   if( !( TEST ) ) { free( A ); free( X ); free( Y ); return( ATL_rzero ); }
/*
 * else perform error check
 */
   if( ROUT == GBMV )
      normA = Mjoin( PATL, gbnrm1 )( mA, nA, KL, KU, A, LDA );
   else
      normA = Mjoin( PATL, genrm1 )( mA, nA,         A, LDA );
   if( normA == ATL_rzero ) normA = ATL_rone;

   free( A  );

   normX = Mjoin( PATL, infnrm )( N, X, aincX );
#ifdef TREAL
   if( Mabs( ALPHA ) > ATL_rone ) normX *= Mabs( ALPHA );
#else
   if( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) > ATL_rone )
      normX *= ( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) );
#endif
   if( normX == ATL_rzero ) normX = ATL_rone;

   free( X );
/*
 * Ensure the difference of the output operands is relatively tiny enough
 */
   YD = (TYPE *)malloc( ATL_MulBySize( M ) );
   if( YD == NULL ) { free( Y ); return( ATL_rnone ); }

   Mjoin( PATL, vdiff )( M, Y, aincY, Y0, aincY, YD, 1 );

   normD = Mjoin( PATL, infnrm )( M, YD, 1 );
   resid = normD / ( normA * normX * normY * eps * Mmax( M, N ) );

   if( ( resid > THRESH ) || ( resid != resid ) )
   {
      (void) fprintf( stderr,
      "ERROR:  resid=%f, normD=%f, normA=%f, normX=%f, normY=%f, eps=%e\n",
      resid, normD, normA, normX, normY, eps );
#ifdef ATLAS_DEBUG
      Mjoin( PATL, geprint )( "Y_trusted", 1, M, Y0, aincY );
      Mjoin( PATL, geprint )( "Y_test",    1, M, Y,  aincY );
#endif
   }

   free( Y  );
   free( YD );

   return( resid );
}

TYPE smvtst
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const int                  K,
   const SCALAR               ALPHA,
   const int                  LDA,
   const int                  INCX,
   const SCALAR               BETA,
   const int                  INCY,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        i, lA;
   int                        Aseed, Xseed, Yseed;
   const int                  aincX = Mabs( INCX ), aincY = Mabs( INCY ),
                              L2size = L2SIZE / sizeof( double );
   TYPE                       normA, normD, normY, normX, resid;
   double                     ops, t0, ttest, ttrust;
   TYPE                       * A  = NULL, * X = NULL, * Y = NULL, * Y0,
                              * YD = NULL, * x, * y;
   double                     * L2 = NULL;

   *TTRUST0 = *TTEST0 = *MFTEST0 = *MFTRUST0 = 0.0;
   if( N == 0 ) { return( ATL_rzero ); }

   if( ROUT == SPMV ) lA = ( N * ( N + 1 ) ) >> 1;   /* ( ROUT == HPMV ) */
   else               lA = LDA * N;

   ops = opbl2( ROUT, N, N, K, K );
/*
 * Allocate L2 cache space, A, X, Y and Y0
 */
   L2 = (double *)malloc( L2size * sizeof( double )        );
   A  = (TYPE   *)malloc( ATL_MulBySize( lA )              );
   X  = (TYPE   *)malloc( ATL_MulBySize( N  ) * aincX      );
   Y  = (TYPE   *)malloc( ATL_MulBySize( N  ) * aincY * 2  );

   if( ( L2 == NULL ) || ( A == NULL ) || ( X == NULL ) || ( Y == NULL ) )
   {
      if( L2 ) free( L2 );
      if( A  ) free( A  );
      if( X  ) free( X  );
      if( Y  ) free( Y  );
      return( ATL_rnone );
   }

   Y0 = Y + N * ( aincY SHIFT );
/*
 * Generate random operands
 */
   Aseed = N * N + LDA;
   Xseed = N * aincX * 27 + 213;
   Yseed = N * aincY;

   if(      ROUT == SBMV )       /* ( ROUT == HBMV ) */
      Mjoin( PATL, gegen )( K+1, N, A, LDA, Aseed );
   else if( ROUT == SPMV )       /* ( ROUT == HPMV ) */
      Mjoin( PATL, gegen )( lA,  1, A, lA,  Aseed );
   else /* if( ROUT == SYMV ) */ /* ( ROUT == HEMV ) */
      Mjoin( PATL, gegen )( N,   N, A, LDA, Aseed );

   Mjoin( PATL, gegen )( 1, N, X,  aincX, Xseed );
   Mjoin( PATL, gegen )( 1, N, Y,  aincY, Yseed );
   Mjoin( PATL, gegen )( 1, N, Y0, aincY, Yseed );
/*
 * Compute the norm of Y for later use in testing
 */
   if( TEST )
   {
      normY = Mjoin( PATL, infnrm )( N, Y, aincY );

#ifdef TREAL
      if( Mabs( BETA ) > ATL_rone ) normY *= Mabs( BETA  );
#else
      if( Mabs( *BETA ) + Mabs( BETA[1] ) > ATL_rone )
         normY *= ( Mabs( *BETA ) + Mabs( BETA[1] ) );
#endif
      if( normY == ATL_rzero ) normY = ATL_rone;
   }
/*
 * Start cold cache timing operations for the trusted routine
 */
   x = X;  if( INCX < 0 ) x = X  + ( ( ( 1 - N ) * INCX ) SHIFT );
   y = Y0; if( INCY < 0 ) y = Y0 + ( ( ( 1 - N ) * INCY ) SHIFT );

   if(      ROUT == SBMV ) /* ( ROUT == HBMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_sbmv( UPLO, N, K, ALPHA, A, LDA, x, INCX, BETA, y, INCY );
      ttrust = time00() - t0;
   }
   else if( ROUT == SPMV ) /* ( ROUT == HPMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_spmv( UPLO, N,    ALPHA, A,      x, INCX, BETA, y, INCY );
      ttrust = time00() - t0;
   }
   else /* if( ROUT == SYMV ) */ /* ( ROUT == HEMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_symv( UPLO, N,    ALPHA, A, LDA, x, INCX, BETA, y, INCY );
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 )
   { *TTRUST0 = ttrust; *MFTRUST0 = ops / ( ttrust * MEGA ); }
/*
 * Start cold cache timing operations for the tested routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - N ) * INCX ) SHIFT );
   y = Y; if( INCY < 0 ) y = Y + ( ( ( 1 - N ) * INCY ) SHIFT );

   if(      ROUT == SBMV ) /* ( ROUT == HBMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_sbmv( UPLO, N, K, ALPHA, A, LDA, x, INCX, BETA, y, INCY );
      ttest = time00() - t0;
   }
   else if( ROUT == SPMV ) /* ( ROUT == HPMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_spmv( UPLO, N,    ALPHA, A,      x, INCX, BETA, y, INCY );
      ttest = time00() - t0;
   }
   else /* if( ROUT == SYMV ) */ /* ( ROUT == HEMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_symv( UPLO, N,    ALPHA, A, LDA, x, INCX, BETA, y, INCY );
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 )
   { *TTEST0  = ttest;  *MFTEST0  = ops / ( ttest  * MEGA ); }
/*
 * if timing only, I am done ... so leave.
 */
   free( L2 );

   if( !( TEST ) ) { free( A ); free( X ); free( Y ); return( ATL_rzero ); }
/*
 * else perform error check
 */
#ifdef TREAL
   if(      ROUT == SBMV ) normA = Mjoin( PATL, sbnrm )( UPLO, N, K, A, LDA );
   else if( ROUT == SPMV ) normA = Mjoin( PATL, spnrm )( UPLO, N,    A      );
   else                    normA = Mjoin( PATL, synrm )( UPLO, N,    A, LDA );
#else
   if(      ROUT == SBMV ) normA = Mjoin( PATL, hbnrm )( UPLO, N, K, A, LDA );
   else if( ROUT == SPMV ) normA = Mjoin( PATL, hpnrm )( UPLO, N,    A      );
   else                    normA = Mjoin( PATL, henrm )( UPLO, N,    A, LDA );
#endif
   if( normA == ATL_rzero ) normA = ATL_rone;

   free( A  );

   normX = Mjoin( PATL, infnrm )( N, X, aincX );
#ifdef TREAL
   if( Mabs( ALPHA ) >  ATL_rone ) normX *= Mabs( ALPHA );
#else
   if( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) > ATL_rone )
      normX *= ( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) );
#endif
   if( normX == ATL_rzero ) normX = ATL_rone;

   free( X  );
/*
 * Ensure the difference of the output operands is relatively tiny enough
 */
   YD = (TYPE *)malloc( ATL_MulBySize( N ) );
   if( YD == NULL ) { free( Y ); return( ATL_rnone ); }

   Mjoin( PATL, vdiff )( N, Y, aincY, Y0, aincY, YD, 1 );

   normD = Mjoin( PATL, infnrm )( N, YD, 1 );
   resid = normD / ( normA * normX * normY * eps * (TYPE)(N) );

   if( ( resid > THRESH ) || ( resid != resid ) )
   {
      (void) fprintf( stderr,
      "ERROR:  resid=%f, normD=%f, normA=%f, normX=%f, normY=%f, eps=%e\n",
      resid, normD, normA, normX, normY, eps );
#ifdef ATLAS_DEBUG
      Mjoin( PATL, geprint )( "Y_trusted", 1, N, Y0, aincY );
      Mjoin( PATL, geprint )( "Y_test",    1, N, Y,  aincY );
#endif
   }

   free( Y  );
   free( YD );

   return( resid );
}

TYPE tmvtst
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const enum ATLAS_UPLO      UPLO,
   const enum ATLAS_TRANS     TRANS,
   const enum ATLAS_DIAG      DIAG,
   const int                  N,
   const int                  K,
   const int                  LDA,
   const int                  INCX,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        i, lA;
   int                        Aseed, Xseed;
   const int                  aincX = Mabs( INCX ),
                              L2size = L2SIZE / sizeof( double );
   TYPE                       normA, normD, normX, resid;
   double                     ops, t0, ttest, ttrust;
   TYPE                       * A = NULL, * X = NULL, * X0, * XD = NULL, * x;
   double                     * L2;

   *TTRUST0 = *TTEST0 = *MFTEST0 = *MFTRUST0 = 0.0;
   if( N == 0 ) { return( ATL_rzero ); }

   if( ROUT == TPMV ) lA = ( N * ( N + 1 ) ) >> 1;
   else               lA = LDA * N;

   ops = opbl2( ROUT, N, N, K, K );
/*
 * Allocate L2 cache space, A, X and X0
 */
   L2 = (double *)malloc( L2size * sizeof( double )       );
   A  = (TYPE   *)malloc( ATL_MulBySize( lA )             );
   X  = (TYPE   *)malloc( ATL_MulBySize( N  ) * aincX * 2 );

   if( ( L2 == NULL ) || ( A == NULL ) || ( X == NULL ) )
   {
      if( L2 ) free( L2 );
      if( A  ) free( A  );
      if( X  ) free( X  );
      return( ATL_rnone );
   }
   X0 = X + N * ( aincX SHIFT );
/*
 * Generate random operands
 */
   Aseed = N * N + LDA;
   Xseed = N * aincX * 27 + 213;

   if(      ROUT == TBMV ) Mjoin( PATL, gegen )( K+1, N, A,  LDA, Aseed );
   else if( ROUT == TPMV ) Mjoin( PATL, gegen )( lA,  1, A,  lA,  Aseed );
   else                    Mjoin( PATL, gegen )( N,   N, A,  LDA, Aseed );
   Mjoin( PATL, gegen )( 1, N, X,  aincX, Xseed );
   Mjoin( PATL, gegen )( 1, N, X0, aincX, Xseed );
/*
 * Compute the norm of X for later use in testing
 */
   if( TEST )
   {
      normX = Mjoin( PATL, infnrm )( N, X, aincX );
      if( normX == ATL_rzero ) normX = ATL_rone;
   }
/*
 * Start cold cache timing operations for the trusted routine
 */
   x = X0; if( INCX < 0 ) x = X0 + ( ( ( 1 - N ) * INCX ) SHIFT );

   if(      ROUT == TBMV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_tbmv( UPLO, TRANS, DIAG, N, K, A, LDA, x, INCX );
      ttrust = time00() - t0;
   }
   else if( ROUT == TPMV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_tpmv( UPLO, TRANS, DIAG, N,    A,      x, INCX );
      ttrust = time00() - t0;
   }
   else /* if( ROUT == TRMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_trmv( UPLO, TRANS, DIAG, N,    A, LDA, x, INCX );
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 )
   { *TTRUST0 = ttrust; *MFTRUST0 = ops / ( ttrust * MEGA ); }
/*
 * Start cold cache timing operations for the tested routine
 */
   x = X;  if( INCX < 0 ) x = X  + ( ( ( 1 - N ) * INCX ) SHIFT );

   if(      ROUT == TBMV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_tbmv( UPLO, TRANS, DIAG, N, K, A, LDA, x, INCX );
      ttest = time00() - t0;
   }
   else if( ROUT == TPMV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_tpmv( UPLO, TRANS, DIAG, N,    A,      x, INCX );
      ttest = time00() - t0;
   }
   else /* if( ROUT == TRMV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_trmv( UPLO, TRANS, DIAG, N,    A, LDA, x, INCX );
      ttest = time00() - t0;
   }
   if( ttest > 0.0 )
   { *TTEST0  = ttest;  *MFTEST0  = ops / ( ttest  * MEGA ); }
/*
 * if timing only, I am done ... so leave.
 */
   free( L2 );

   if( !( TEST ) ) { free( A ); free( X ); return( ATL_rzero ); }
/*
 * else perform error check
 */
   if(      ROUT == TBMV )
      normA = Mjoin( PATL, tbnrm1 )( UPLO, DIAG, N, K, A, LDA );
   else if( ROUT == TPMV )
      normA = Mjoin( PATL, tpnrm1 )( UPLO, DIAG, N,    A      );
   else /* if( ROUT == TRMV ) */
      normA = Mjoin( PATL, trnrm1 )( UPLO, DIAG, N,    A, LDA );
   if( normA == ATL_rzero ) normA = ATL_rone;

   free( A  );
/*
 * Ensure the difference of the output operands is relatively tiny enough
 */
   XD = (TYPE * )malloc( ATL_MulBySize( N ) );
   if( XD == NULL ) { free( X ); return( -1 ); }

   Mjoin( PATL, vdiff )( N, X, aincX, X0, aincX, XD, 1 );
   normD = Mjoin( PATL, infnrm )( N, XD, 1 );
   resid = normD / ( normA * normX * eps * N );

   if( ( resid > THRESH ) || ( resid != resid ) )
   {
      (void) fprintf(stderr,
      "ERROR:   resid=%f, normD=%f, normA=%f, normX=%f, eps=%e\n",
      resid, normD, normA, normX, eps );
#ifdef ATLAS_DEBUG
      Mjoin( PATL, geprint )( "X_trusted", 1, N, X0, aincX );
      Mjoin( PATL, geprint )( "X_test",    1, N, X,  aincX );
#endif
   }

   free( X  );
   free( XD );

   return( resid );
}

TYPE tsvtst
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const enum ATLAS_UPLO      UPLO,
   const enum ATLAS_TRANS     TRANS,
   const enum ATLAS_DIAG      DIAG,
   const int                  N,
   const int                  K,
   const int                  LDA,
   const int                  INCX,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        i, lA;
   int                        Aseed, Xseed;
   const int                  aincX = Mabs( INCX ),
                              L2size = L2SIZE / sizeof( double );
   TYPE                       normA, normD, normX, resid;
   double                     ops, t0, ttest, ttrust;
   TYPE                       * A = NULL, * X = NULL, * X0, * XD = NULL, * x;
   double                     * L2;

   *TTRUST0 = *TTEST0 = *MFTEST0 = *MFTRUST0 = 0.0;
   if( N == 0 ) { return( ATL_rzero ); }

   if( ROUT == TPSV ) lA = ( N * ( N + 1 ) ) >> 1;
   else               lA = LDA * N;

   ops = opbl2( ROUT, N, N, K, K );
/*
 * Allocate L2 cache space, A, X and X0
 */
   L2 = (double *)malloc( L2size * sizeof( double )       );
   A  = (TYPE   *)malloc( ATL_MulBySize( lA )             );
   X  = (TYPE   *)malloc( ATL_MulBySize( N  ) * aincX * 2 );

   if( ( L2 == NULL ) || ( A == NULL ) || ( X == NULL ) )
   {
      if( L2 ) free( L2 );
      if( A  ) free( A  );
      if( X  ) free( X  );
      return( ATL_rnone );
   }
   X0 = X + N * ( aincX SHIFT );
/*
 * Generate random operands
 */
   Aseed = N * N + LDA;
   Xseed = N * aincX * 27 + 213;

   if(      ROUT == TBSV )
   {
      Mjoin( PATL, gegen )( K+1, N, A, LDA, Aseed );
      tbddom( UPLO, N, K, A, LDA );
   }
   else if( ROUT == TPSV )
   {
      Mjoin( PATL, gegen )( lA,  1, A, lA,  Aseed );
      tpddom( UPLO, N,    A      );
   }
   else
   {
      Mjoin( PATL, gegen )( N,   N, A, LDA, Aseed );
      trddom( UPLO, N,    A, LDA );
   }
   Mjoin( PATL, gegen )( 1, N, X,  aincX, Xseed );
   Mjoin( PATL, gegen )( 1, N, X0, aincX, Xseed );
/*
 * Compute the norm of X for later use in testing
 */
   if( TEST )
   {
      normX = Mjoin( PATL, infnrm )( N, X, aincX );
      if( normX == ATL_rzero ) normX = ATL_rone;
   }
/*
 * Start cold cache timing operations for the trusted routine
 */
   x = X0; if( INCX < 0 ) x = X0 + ( ( ( 1 - N ) * INCX ) SHIFT );

   if(      ROUT == TBSV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_tbsv( UPLO, TRANS, DIAG, N, K, A, LDA, x, INCX );
      ttrust = time00() - t0;
   }
   else if( ROUT == TPSV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_tpsv( UPLO, TRANS, DIAG, N,    A,      x, INCX );
      ttrust = time00() - t0;
   }
   else /* if( ROUT == TRSV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_trsv( UPLO, TRANS, DIAG, N,    A, LDA, x, INCX );
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 )
   { *TTRUST0 = ttrust; *MFTRUST0 = ops / ( ttrust * MEGA ); }
/*
 * Start cold cache timing operations for the tested routine
 */
   x = X;  if( INCX < 0 ) x = X  + ( ( ( 1 - N ) * INCX ) SHIFT );

   if(      ROUT == TBSV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_tbsv( UPLO, TRANS, DIAG, N, K, A, LDA, x, INCX );
      ttest = time00() - t0;
   }
   else if( ROUT == TPSV )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_tpsv( UPLO, TRANS, DIAG, N,    A,      x, INCX );
      ttest = time00() - t0;
   }
   else /* if( ROUT == TRSV ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_trsv( UPLO, TRANS, DIAG, N,    A, LDA, x, INCX );
      ttest = time00() - t0;
   }
   if( ttest > 0.0 )
   { *TTEST0  = ttest;  *MFTEST0  = ops / ( ttest  * MEGA ); }
/*
 * if timing only, I am done ... so leave.
 */
   free( L2 );

   if( !( TEST ) ) { free( A ); free( X ); return( ATL_rzero ); }
/*
 * else perform error check
 */
   if(      ROUT == TBSV )
      normA = Mjoin( PATL, tbnrm1 )( UPLO, DIAG, N, K, A, LDA );
   else if( ROUT == TPSV )
      normA = Mjoin( PATL, tpnrm1 )( UPLO, DIAG, N,    A      );
   else /* if( ROUT == TRSV ) */
      normA = Mjoin( PATL, trnrm1 )( UPLO, DIAG, N,    A, LDA );
   if( normA == ATL_rzero ) normA = ATL_rone;

   free( A  );
/*
 * Ensure the difference of the output operands is relatively tiny enough
 */
   XD = (TYPE * )malloc( ATL_MulBySize( N ) );
   if( XD == NULL ) { free( X ); return( -1 ); }

   Mjoin( PATL, vdiff )( N, X, aincX, X0, aincX, XD, 1 );
   normD = Mjoin( PATL, infnrm )( N, XD, 1 );
   resid = normD / ( normA * normX * eps * N );

   if( ( resid > THRESH ) || ( resid != resid ) )
   {
      (void) fprintf(stderr,
      "ERROR:   resid=%f, normD=%f, normA=%f, normX=%f, eps=%e\n",
      resid, normD, normA, normX, eps );
#ifdef ATLAS_DEBUG
      Mjoin( PATL, geprint )( "X_trusted", 1, N, X0, aincX );
      Mjoin( PATL, geprint )( "X_test",    1, N, X,  aincX );
#endif
   }

   free( X  );
   free( XD );

   return( resid );
}

TYPE gr1tst
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  M,
   const int                  N,
   const SCALAR               ALPHA,
   const int                  INCX,
   const int                  INCY,
   const int                  LDA,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        i;
   int                        Aseed, Xseed, Yseed;
   const int                  aincX = Mabs( INCX ), aincY = Mabs( INCY ),
                              L2size = L2SIZE / sizeof( double );
   TYPE                       normA, normD, normY, normX, resid;
   double                     ops, t0, ttest, ttrust;
   TYPE                       * A = NULL, * A0, * X = NULL, * Y = NULL, * x,
                              * y;
   double                     * L2 = NULL;

   *TTRUST0 = *TTEST0 = *MFTEST0 = *MFTRUST0 = 0.0;
   if( ( M == 0 ) || ( N == 0 ) ) { return( ATL_rzero ); }

   ops = opbl2( ROUT, M, N, 0, 0 );
/*
 * Allocate L2 cache space, A, A0, X and Y
 */
   L2 = (double *)malloc( L2size * sizeof( double )    );
   A  = (TYPE   *)malloc( ATL_MulBySize( LDA ) * N * 2 );
   X  = (TYPE   *)malloc( ATL_MulBySize( M   ) * aincX );
   Y  = (TYPE   *)malloc( ATL_MulBySize( N   ) * aincY );

   if( ( L2 == NULL ) || ( A == NULL ) || ( X == NULL ) || ( Y == NULL ) )
   {
      if( L2 ) free( L2 );
      if( A  ) free( A  );
      if( X  ) free( X  );
      if( Y  ) free( Y  );
      return( ATL_rnone );
   }

   A0 = A + LDA * ( N SHIFT );
/*
 * Generate random operands
 */
   Aseed = M * N + LDA;
   Xseed = M * aincX * 27 + 213;
   Yseed = N * aincY;

   Mjoin( PATL, gegen )( M, N, A,  LDA,   Aseed );
   Mjoin( PATL, gegen )( M, N, A0, LDA,   Aseed );
   Mjoin( PATL, gegen )( 1, M, X,  aincX, Xseed );
   Mjoin( PATL, gegen )( 1, N, Y,  aincY, Yseed );
/*
 * Compute the norm of A for later use in testing
 */
   if( TEST )
   {
       normA = Mjoin( PATL, genrm1 )( M, N, A, LDA );
#ifdef TREAL
      if( Mabs( ALPHA ) > ATL_rone ) normA *= Mabs( ALPHA );
#else
      if( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) > ATL_rone )
         normA *= ( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) );
#endif
      if( normA == ATL_rzero ) normA = ATL_rone;
   }
/*
 * Start cold cache timing operations for the trusted routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - M ) * INCX ) SHIFT );
   y = Y; if( INCY < 0 ) y = Y + ( ( ( 1 - N ) * INCY ) SHIFT );

   if( ROUT == GERC ) /* ( ROUT == GER ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_gerc( M, N, ALPHA, x, INCX, y, INCY, A0, LDA );
      ttrust = time00() - t0;
   }
   else /* if ( ROUT == GERU ) */ /* ( ROUT == GER  ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_geru( M, N, ALPHA, x, INCX, y, INCY, A0, LDA );
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 )
   { *TTRUST0 = ttrust; *MFTRUST0 = ops / ( ttrust * MEGA ); }
/*
 * Start cold cache timing operations for the tested routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - M ) * INCX ) SHIFT );
   y = Y; if( INCY < 0 ) y = Y + ( ( ( 1 - N ) * INCY ) SHIFT );

   if( ROUT == GERC ) /* ( ROUT == GER ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_gerc( M, N, ALPHA, x, INCX, y, INCY, A, LDA );
      ttest = time00() - t0;
   }
   else /* if ( ROUT == GERU ) */ /* ( ROUT == GER  ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_geru( M, N, ALPHA, x, INCX, y, INCY, A, LDA );
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 )
   { *TTEST0  = ttest;  *MFTEST0  = ops / ( ttest  * MEGA ); }
/*
 * if timing only, I am done ... so leave.
 */
   free( L2 );

   if( !( TEST ) ) { free( A ); free( X ); free( Y ); return( ATL_rzero ); }
/*
 * else perform error check
 */
   normX = Mjoin( PATL, infnrm )( M, X, aincX );
   if( normX == ATL_rzero ) normX = ATL_rone;
   free( X  );

   normY = Mjoin( PATL, infnrm )( N, Y, aincY );
   if( normY == ATL_rzero ) normY = ATL_rone;
   free( Y  );
/*
 * Ensure the difference of the output operands is relatively tiny enough
 */
   normD = Mjoin( PATL, gediffnrm1 )( M, N, A, LDA, A0, LDA );
   resid = normD / ( normA * normX * normY * eps * Mmax( M, N ) );

   if( ( resid > THRESH ) || ( resid != resid ) )
   {
      (void) fprintf( stderr,
      "ERROR:   resid=%f, normD=%f, normA=%f, normX=%f, normY=%f, eps=%e\n",
      resid, normD, normA, normX, normY, eps );
#ifdef ATLAS_DEBUG
      Mjoin( PATL, geprint )( "A_trusted", M, N, A0, LDA );
      Mjoin( PATL, geprint )( "A_test",    M, N, A,  LDA );
#endif
   }

   free( A  );

   return( resid );
}

TYPE sr1tst
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const TYPE                 ALPHA,
   const int                  INCX,
   const int                  LDA,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        i, lA;
   int                        Aseed, Xseed;
   const int                  aincX = Mabs( INCX ),
                              L2size = L2SIZE / sizeof( double );
   TYPE                       normA, normD, normX, resid;
   double                     ops, t0, ttest, ttrust;
   TYPE                       * A = NULL, * A0, * X = NULL, * x;
   double                     * L2 = NULL;

   *TTRUST0 = *TTEST0 = *MFTEST0 = *MFTRUST0 = 0.0;
   if( N == 0 ) { return( ATL_rzero ); }

   if( ROUT == SPR ) lA = ( N * ( N + 1 ) ) >> 1; /* ( ROUT == HER ) */
   else              lA = LDA * N;

   ops = opbl2( ROUT, N, N, 0, 0 );
/*
 * Allocate L2 cache space, A, A0 and X
 */
   L2 = (double *)malloc( L2size * sizeof( double )   );
   A  = (TYPE   *)malloc( ATL_MulBySize( lA ) * 2     );
   X  = (TYPE   *)malloc( ATL_MulBySize( N  ) * aincX );

   if( ( L2 == NULL ) || ( A == NULL ) || ( X == NULL ) )
   {
      if( L2 ) free( L2 );
      if( A  ) free( A  );
      if( X  ) free( X  );
      return( ATL_rnone );
   }

   A0 = A + ( lA SHIFT );
/*
 * Generate random operands
 */
   Aseed = N * N + LDA;
   Xseed = N * aincX * 27 + 213;

   if(      ROUT == SPR ) /* ( ROUT == HPR ) */
   {
      Mjoin( PATL, gegen )( lA, 1, A,  lA,  Aseed );
      Mjoin( PATL, gegen )( lA, 1, A0, lA,  Aseed );
   }
   else /* if( ROUT == SYR ) */ /* ( ROUT == HER ) */
   {
      Mjoin( PATL, gegen )( N,  N, A,  LDA, Aseed );
      Mjoin( PATL, gegen )( N,  N, A0, LDA, Aseed );
   }
   Mjoin( PATL, gegen )( 1, N, X,  aincX, Xseed );
/*
 * Compute the norm of A for later use in testing
 */
   if( TEST )
   {
#ifdef TREAL
      if(      ROUT == SPR )
         normA = Mjoin( PATL, spnrm )( UPLO, N, A      );
      else /* if( ROUT == SYR ) */
         normA = Mjoin( PATL, synrm )( UPLO, N, A, LDA );
#else
      if(      ROUT == SPR )
         normA = Mjoin( PATL, hpnrm )( UPLO, N, A      );
      else /* if( ROUT == SYR ) */
         normA = Mjoin( PATL, henrm )( UPLO, N, A, LDA );
#endif
      if( Mabs( ALPHA ) > ATL_rone ) normA *= Mabs( ALPHA );
      if( normA == ATL_rzero ) normA = ATL_rone;
   }
/*
 * Start cold cache timing operations for the trusted routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - N ) * INCX ) SHIFT );

   if( ROUT == SPR ) /* ( ROUT == HPR ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_spr( UPLO, N, ALPHA,  x, INCX, A0      );
      ttrust = time00() - t0;
   }
   else /* if( ROUT == SYR ) */ /* ( ROUT == HER ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_syr( UPLO, N, ALPHA,  x, INCX, A0, LDA );
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 )
   { *TTRUST0 = ttrust; *MFTRUST0 = ops / ( ttrust * MEGA ); }
/*
 * Start cold cache timing operations for the tested routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - N ) * INCX ) SHIFT );

   if( ROUT == SPR ) /* ( ROUT == HPR ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_spr( UPLO, N, ALPHA,  x, INCX, A      );
      ttest = time00() - t0;
   }
   else /* if( ROUT == SYR ) */ /* ( ROUT == HER ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_syr( UPLO, N, ALPHA,  x, INCX, A, LDA );
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 )
   { *TTEST0  = ttest;  *MFTEST0  = ops / ( ttest  * MEGA ); }
/*
 * if timing only, I am done ... so leave.
 */
   free( L2 );

   if( !( TEST ) ) { free( A ); free( X ); return( ATL_rzero ); }
/*
 * else perform error check
 */
   normX = Mjoin( PATL, infnrm )( N, X, aincX );
   if( normX == ATL_rzero ) normX = ATL_rone;
   free( X  );
/*
 * Ensure the difference of the output operands is relatively tiny enough
 */
   if(      ROUT == SPR ) /* ( ROUT == HPR ) */
      normD = Mjoin( PATL, gediffnrm1 )( lA, 1, A, lA,  A0, lA  );
   else /* if( ROUT == SYR ) */ /* ( ROUT == HER ) */
      normD = Mjoin( PATL, gediffnrm1 )( N,  N, A, LDA, A0, LDA );

   resid = normD / ( normA * normX * eps * N );

   if( ( resid > THRESH ) || ( resid != resid ) )
   {
      (void) fprintf( stderr,
      "ERROR:    resid=%f, normD=%f, normA=%f, normX=%f, eps=%e\n",
      resid, normD, normA, normX, eps );
#ifdef ATLAS_DEBUG
      if(      ROUT == SPR )
      {
         Mjoin( PATL, geprint )( "A_trusted", lA, 1, A0, lA  );
         Mjoin( PATL, geprint )( "A_test",    lA, 1, A,  lA  );
      }
      else /* if( ROUT == SYR ) */
      {
         Mjoin( PATL, geprint )( "A_trusted", N,  N, A0, LDA );
         Mjoin( PATL, geprint )( "A_test",    N,  N, A,  LDA );
      }
#endif
   }

   free( A  );

   return( resid );
}

TYPE sr2tst
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const SCALAR               ALPHA,
   const int                  INCX,
   const int                  INCY,
   const int                  LDA,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        i, lA;
   int                        Aseed, Xseed, Yseed;
   const int                  aincX = Mabs( INCX ), aincY = Mabs( INCY ),
                              L2size = L2SIZE / sizeof( double );
   TYPE                       normA, normD, normY, normX, resid;
   double                     ops, t0, ttest, ttrust;
   TYPE                       * A = NULL, * A0, * X = NULL, * Y = NULL, * x,
                              * y;
   double                     * L2 = NULL;

   *TTRUST0 = *TTEST0 = *MFTEST0 = *MFTRUST0 = 0.0;
   if( N == 0 ) { return( ATL_rzero ); }

   if( ROUT == SPR2 ) lA = ( N * ( N + 1 ) ) >> 1;
   else               lA = LDA * N;

   ops = opbl2( ROUT, N, N, 0, 0 );
/*
 * Allocate L2 cache space, A, A0, X and Y
 */
   L2 = (double *)malloc( L2size * sizeof( double )   );
   A  = (TYPE   *)malloc( ATL_MulBySize( lA ) * 2     );
   X  = (TYPE   *)malloc( ATL_MulBySize( N  ) * aincX );
   Y  = (TYPE   *)malloc( ATL_MulBySize( N  ) * aincY );

   if( ( L2 == NULL ) || ( A == NULL ) || ( X == NULL ) || ( Y == NULL ) )
   {
      if( L2 ) free( L2 );
      if( A  ) free( A  );
      if( X  ) free( X  );
      if( Y  ) free( Y  );
      return( ATL_rnone );
   }

   A0 = A + ( lA SHIFT );
/*
 * Generate random operands
 */
   Aseed = N * N + LDA;
   Xseed = N * aincX * 27 + 213;
   Yseed = N * aincY;

   if(      ROUT == SPR2 )
   {
      Mjoin( PATL, gegen )( lA, 1, A,  lA,  Aseed );
      Mjoin( PATL, gegen )( lA, 1, A0, lA,  Aseed );
   }
   else /* if( ROUT == SYR2 ) */
   {
      Mjoin( PATL, gegen )( N,  N, A,  LDA, Aseed );
      Mjoin( PATL, gegen )( N,  N, A0, LDA, Aseed );
   }
   Mjoin( PATL, gegen )( 1, N, X,  aincX, Xseed );
   Mjoin( PATL, gegen )( 1, N, Y,  aincY, Yseed );
/*
 * Compute the norm of A for later use in testing
 */
   if( TEST )
   {
#ifdef TREAL
      if(      ROUT == SPR2 )
         normA = Mjoin( PATL, spnrm )( UPLO, N, A      );
      else /* if( ROUT == SYR2 ) */
         normA = Mjoin( PATL, synrm )( UPLO, N, A, LDA );

      if( Mabs( ALPHA ) > ATL_rone ) normA *= Mabs( ALPHA );
#else
      if(      ROUT == SPR2 )
         normA = Mjoin( PATL, hpnrm )( UPLO, N, A      );
      else /* if( ROUT == HER2 ) */
         normA = Mjoin( PATL, henrm )( UPLO, N, A, LDA );

      if( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) > ATL_rone )
         normA *= ( Mabs( *ALPHA ) + Mabs( ALPHA[1] ) );
#endif
      if( normA == ATL_rzero ) normA = ATL_rone;
   }
/*
 * Start cold cache timing operations for the trusted routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - N ) * INCX ) SHIFT );
   y = Y; if( INCY < 0 ) y = Y + ( ( ( 1 - N ) * INCY ) SHIFT );

   if( ROUT == SPR2 )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_spr2( UPLO, N, ALPHA, x, INCX, y, INCY, A0      );
      ttrust = time00() - t0;
   }
   else /* if( ROUT == SYR2 ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0     = time00();
      trusted_syr2( UPLO, N, ALPHA, x, INCX, y, INCY, A0, LDA );
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 )
   { *TTRUST0 = ttrust; *MFTRUST0 = ops / ( ttrust * MEGA ); }
/*
 * Start cold cache timing operations for the tested routine
 */
   x = X; if( INCX < 0 ) x = X + ( ( ( 1 - N ) * INCX ) SHIFT );
   y = Y; if( INCY < 0 ) y = Y + ( ( ( 1 - N ) * INCY ) SHIFT );

   if( ROUT == SPR2 )
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_spr2( UPLO, N, ALPHA, x, INCX, y, INCY, A      );
      ttest = time00() - t0;
   }
   else /* if( ROUT == SYR2 ) */
   {
      ATL_dset( L2size, 0.0, L2, 1 );
      for( i = 0; i < L2size; i++ ) if( L2[i] != 0.0 ) exit( -1 );
      t0    = time00();
      test_syr2( UPLO, N, ALPHA, x, INCX, y, INCY, A, LDA );
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 )
   { *TTEST0  = ttest;  *MFTEST0  = ops / ( ttest  * MEGA ); }
/*
 * if timing only, I am done ... so leave.
 */
   free( L2 );

   if( !( TEST ) ) { free( A ); free( X ); free( Y ); return( ATL_rzero ); }
/*
 * else perform error check
 */
   normX = Mjoin( PATL, infnrm )( N, X, aincX );
   if( normX == ATL_rzero ) normX = ATL_rone;
   free( X  );

   normY = Mjoin( PATL, infnrm )( N, Y, aincY );
   if( normY == ATL_rzero ) normY = ATL_rone;
   free( Y  );
/*
 * Ensure the difference of the output operands is relatively tiny enough
 */
   if(      ROUT == SPR2 )
      normD = Mjoin( PATL, gediffnrm1 )( lA, 1, A, lA,  A0, lA  );
   else /* if( ROUT == SYR2 ) */
      normD = Mjoin( PATL, gediffnrm1 )( N,  N, A, LDA, A0, LDA );
   resid = normD / ( normA * normX * normY * eps * N );

   if( ( resid > THRESH ) || ( resid != resid ) )
   {
      (void) fprintf( stderr,
      "ERROR:    resid=%f, normD=%f, normA=%f, normX=%f, normY=%f, eps=%e\n",
      resid, normD, normA, normX, normY, eps );
#ifdef ATLAS_DEBUG
      if(      ROUT == SPR2 )
      {
         Mjoin( PATL, geprint )( "A_trusted", lA, 1, A0, lA  );
         Mjoin( PATL, geprint )( "A_test",    lA, 1, A,  lA  );
      }
      else /* if( ROUT == SYR2 ) */
      {
         Mjoin( PATL, geprint )( "A_trusted", N,  N, A0, LDA );
         Mjoin( PATL, geprint )( "A_test",    N,  N, A,  LDA );
      }
#endif
   }

   free( A  );

   return( resid );
}

int gmvcase
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  MFLOP,
   const enum ATLAS_TRANS     TRANS,
   const int                  M,
   const int                  N,
   const int                  KL,
   const int                  KU,
   const SCALAR               ALPHA,
   const int                  LDA,
   const int                  INCX,
   const SCALAR               BETA,
   const int                  INCY,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        aincX, aincY, inca, incx, incy, lX, lY, lA,
                              mA, nA, passed;
   int                        Aseed, Xseed, Yseed;
   unsigned long              ir, reps;
   TYPE                       resid = ATL_rzero;
   double                     flops, ttrust, ttest, mftrust, mftest, t0;
#ifdef TREAL
   TYPE                       bet,  beta,    nbeta;
#else
   TYPE                       *bet, beta[2], nbeta[2];
#endif
   TYPE                       * a, * stA, * stX, * stY, * x, * y, * A,
                              * A0 = NULL, * X, * X0 = NULL, * Y, * Y0 = NULL;

   if( ( MEGA * MFLOP <= ( flops = opbl2( ROUT, M, N, KL, KU ) ) ) || ( TEST ) )
   {
      resid = gmvtst( ROUT, TEST, TRANS, M, N, KL, KU, ALPHA, LDA, INCX,
                      BETA, INCY, TTRUST0, TTEST0, MFTRUST0, MFTEST0 );
      if( resid > THRESH ) (void) fprintf( stderr, "   resid=%f\n", resid );
   }
   if( resid < ATL_rzero ) passed = -1;
   else                    passed = ( resid < THRESH );

   if( MEGA * MFLOP <= flops ) return( passed );

   if( ROUT == GBMV )
   {
      if( TRANS == AtlasNoTrans ) { mA = KL + 1 + KU; nA = N; }
      else                        { mA = KL + 1 + KU; nA = M; }
   }
   else
   {
      if( TRANS == AtlasNoTrans ) { mA = M; nA = N; }
      else                        { mA = N; nA = M; }
   }

   incy = INCY * ( M  SHIFT ); aincY = Mabs( INCY );
   incx = INCX * ( N  SHIFT ), aincX = Mabs( INCX );
   inca = LDA  * ( nA SHIFT );

   lY = M * aincY * ( ( ATL_DivBySize( L2SIZE ) + M       - 1 ) / M           );
   lX = N * aincX * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lA =     inca  * ( ( ATL_DivBySize( L2SIZE ) + 2*mA*nA - 1 ) / ( mA * nA ) );

   Y0 = (TYPE *)malloc( ATL_MulBySize( lY ) );
   X0 = (TYPE *)malloc( ATL_MulBySize( lX ) );
   A0 = (TYPE *)malloc( ATL_MulBySize( lA ) );

   if( ( Y0 == NULL ) || ( X0 == NULL ) || ( A0 == NULL ) )
   {
      if( Y0 ) free( Y0 );
      if( X0 ) free( X0 );
      if( A0 ) free( A0 );
      return( -1 );
   }

   if( INCY < 1 ) { Y = Y0 + ( lY SHIFT ); stY = Y0; }
   else           { Y = Y0; stY = Y0 + ( lY SHIFT ); }
   if( INCX < 1 ) { X = X0 + ( lX SHIFT ); stX = X0; }
   else           { X = X0; stX = X0 + ( lX SHIFT ); }
   A = A0; stA = A0 + ( lA SHIFT );

#ifdef TREAL
   beta   =  BETA;
   nbeta  = -BETA;
#else
   *beta  =    *BETA; beta [1] =  BETA[1];
   *nbeta = -(*BETA); nbeta[1] = -BETA[1];
#endif

   Yseed = M * aincY;
   Xseed = N * aincX + 127 * 50 + 77;
   Aseed = N * M     + 513 *  7 + 90;

   reps  = ( MEGA * MFLOP ) / flops;
/*
 * Generate the random data and time the trusted routine
 */
   bet = beta; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if( ROUT == GBMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_gbmv( TRANS, M, N, KL, KU, ALPHA, (const TYPE *)(a), LDA,
                       (const TYPE *)(x), INCX, (const SCALAR)(bet),
                       (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_gemv( TRANS, M, N,         ALPHA, (const TYPE *)(a), LDA,
                       (const TYPE *)(x), INCX, (const SCALAR)(bet),
                       (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 ) mftrust = ( reps * flops ) / ( MEGA * ttrust );
   else               mftrust = 0.0;
   ttrust /= reps; *TTRUST0 = ttrust; *MFTRUST0 = mftrust;
/*
 * Generate the random data and time the tested routine
 */
   bet = beta; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if( ROUT == GBMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_gbmv( TRANS, M, N, KL, KU, ALPHA, (const TYPE *)(a), LDA,
                    (const TYPE *)(x), INCX, (const SCALAR)(bet),
                    (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_gemv( TRANS, M, N,         ALPHA, (const TYPE *)(a), LDA,
                    (const TYPE *)(x), INCX, (const SCALAR)(bet),
                    (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 ) mftest  = ( reps * flops ) / ( MEGA * ttest  );
   else               mftest  = 0.0;
   ttest  /= reps; *TTEST0  = ttest;  *MFTEST0  = mftest;
/*
 * release the memory and exit
 */
   free( Y0 );
   free( X0 );
   free( A0 );

   return( passed );
}

int smvcase
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  MFLOP,
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const int                  K,
   const SCALAR               ALPHA,
   const int                  LDA,
   const int                  INCX,
   const SCALAR               BETA,
   const int                  INCY,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        aincX, aincY, inca, incx, incy, lX, lY, lA,
                              mA, nA, passed;
   int                        Aseed, Xseed, Yseed;
   unsigned long              ir, reps;
   TYPE                       resid = ATL_rzero;
   double                     flops, ttrust, ttest, mftrust, mftest, t0;
#ifdef TREAL
   TYPE                       bet,   beta,    nbeta;
#else
   TYPE                       * bet, beta[2], nbeta[2];
#endif
   TYPE                       * a, * stA, * stX, * stY, * x, * y, * A,
                              * A0 = NULL, * X, * X0 = NULL, * Y, * Y0 = NULL;

   if( ( MEGA * MFLOP <= ( flops = opbl2( ROUT, N, N, K, K ) ) ) || ( TEST ) )
   {
      resid = smvtst( ROUT, TEST, UPLO, N, K, ALPHA, LDA, INCX, BETA,
                      INCY, TTRUST0, TTEST0, MFTRUST0, MFTEST0 );
      if( resid > THRESH ) (void) fprintf( stderr, "   resid=%f\n", resid );
   }
   if( resid < ATL_rzero ) passed = -1;
   else                    passed = ( resid < THRESH );

   if( MEGA * MFLOP <= flops ) return( passed );

   if(      ROUT == SBMV ) { mA = K + 1;                  nA = N; }
   else if( ROUT == SPMV ) { mA = ( N * ( N + 1 ) ) >> 1; nA = 1; }
   else                    { mA = N;                      nA = N; }

   incy = INCY * ( N  SHIFT ); aincY = Mabs( INCY );
   incx = INCX * ( N  SHIFT ); aincX = Mabs( INCX );
   if( ROUT == SPMV ) inca = ( mA SHIFT );
   else               inca = LDA  * ( nA SHIFT );

   lY = N * aincY * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lX = N * aincX * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lA =     inca  * ( ( ATL_DivBySize( L2SIZE ) + 2*mA*nA - 1 ) / ( mA * nA ) );

   Y0 = (TYPE *)malloc( ATL_MulBySize( lY ) );
   X0 = (TYPE *)malloc( ATL_MulBySize( lX ) );
   A0 = (TYPE *)malloc( ATL_MulBySize( lA ) );

   if( ( Y0 == NULL ) || ( X0 == NULL ) || ( A0 == NULL ) )
   {
      if( Y0 ) free( Y0 );
      if( X0 ) free( X0 );
      if( A0 ) free( A0 );
      return( -1 );
   }

   if( INCY < 1 ) { Y = Y0 + ( lY SHIFT ); stY = Y0; }
   else           { Y = Y0; stY = Y0 + ( lY SHIFT ); }
   if( INCX < 1 ) { X = X0 + ( lX SHIFT ); stX = X0; }
   else           { X = X0; stX = X0 + ( lX SHIFT ); }
   A = A0; stA = A0 + ( lA SHIFT );

#ifdef TREAL
   beta   =  BETA;
   nbeta  = -BETA;
#else
   *beta  =    *BETA; beta [1] =  BETA[1];
   *nbeta = -(*BETA); nbeta[1] = -BETA[1];
#endif

   Yseed = N * aincY;
   Xseed = N * aincX + 127 * 50 + 77;
   Aseed = N * N     + 513 *  7 + 90;

   reps = ( MEGA * MFLOP ) / flops;
/*
 * Generate the random data and time the trusted routine
 */
   bet = beta; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == SBMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_sbmv( UPLO, N, K, ALPHA, (const TYPE *)(a), LDA,
                       (const TYPE *)(x), INCX, (const SCALAR)(bet),
                       (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   else if( ROUT == SPMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_spmv( UPLO, N,    ALPHA, (const TYPE *)(a),
                       (const TYPE *)(x), INCX, (const SCALAR)(bet),
                       (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_symv( UPLO, N,    ALPHA, (const TYPE *)(a), LDA,
                       (const TYPE *)(x), INCX, (const SCALAR)(bet),
                       (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 ) mftrust = ( reps * flops ) / ( MEGA * ttrust );
   else               mftrust = 0.0;
   ttrust /= reps; *TTRUST0 = ttrust; *MFTRUST0 = mftrust;
/*
 * Generate the random data and time the tested routine
 */
   bet = beta; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == SBMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_sbmv( UPLO, N, K, ALPHA, (const TYPE *)(a), LDA,
                    (const TYPE *)(x), INCX, (const SCALAR)(bet),
                    (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   else if( ROUT == SPMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_spmv( UPLO, N,    ALPHA, (const TYPE *)(a),
                    (const TYPE *)(x), INCX, (const SCALAR)(bet),
                    (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_symv( UPLO, N,    ALPHA, (const TYPE *)(a), LDA,
                    (const TYPE *)(x), INCX, (const SCALAR)(bet),
                    (TYPE *)(y), INCY );
         y += incy;
         if( y == stY )
         { y = Y; if( bet == beta ) bet = nbeta; else bet = beta; }
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 ) mftest  = ( reps * flops ) / ( MEGA * ttest  );
   else               mftest  = 0.0;
   ttest  /= reps; *TTEST0  = ttest;  *MFTEST0  = mftest;
/*
 * release the memory and exit
 */
   free( Y0 );
   free( X0 );
   free( A0 );

   return( passed );
}

int tmvcase
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  MFLOP,
   const enum ATLAS_UPLO      UPLO,
   const enum ATLAS_TRANS     TRANS,
   const enum ATLAS_DIAG      DIAG,
   const int                  N,
   const int                  K,
   const int                  LDA,
   const int                  INCX,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        aincX, inca, incx, lX, lA, mA, nA, passed;
   int                        Aseed, Xseed;
   unsigned long              ir, reps;
   TYPE                       resid = ATL_rzero;
   double                     flops, ttrust, ttest, mftrust, mftest, t0;
   TYPE                       * a, * stA, * stX, * x, * A, * A0 = NULL, * X,
                              * X0 = NULL;

   if( ( MEGA * MFLOP <= ( flops = opbl2( ROUT, N, N, K, K ) ) ) || ( TEST ) )
   {
      resid = tmvtst( ROUT, TEST, UPLO, TRANS, DIAG, N, K, LDA, INCX,
                      TTRUST0, TTEST0, MFTRUST0, MFTEST0 );
      if( resid > THRESH ) (void) fprintf( stderr, "   resid=%f\n", resid );
   }
   if( resid < ATL_rzero ) passed = -1;
   else                    passed = ( resid < THRESH );

   if( MEGA * MFLOP <= flops ) return( passed );

   if(      ROUT == TBMV ) { mA = K + 1;                  nA = N; }
   else if( ROUT == TPMV ) { mA = ( N * ( N + 1 ) ) >> 1; nA = 1; }
   else                    { mA = N;                      nA = N; }

   incx = INCX * ( N  SHIFT ); aincX = Mabs( INCX );
   if( ROUT == TPMV ) inca = ( mA SHIFT );
   else               inca = LDA  * ( nA SHIFT );

   lX = N * aincX * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lA =     inca  * ( ( ATL_DivBySize( L2SIZE ) + 2*mA*nA - 1 ) / ( mA * nA ) );

   X0 = (TYPE *)malloc( ATL_MulBySize( lX ) );
   A0 = (TYPE *)malloc( ATL_MulBySize( lA ) );

   if( ( X0 == NULL ) || ( A0 == NULL ) )
   {
      if( X0 ) free( X0 );
      if( A0 ) free( A0 );
      return( -1 );
   }

   if( INCX < 1 ) { X = X0 + ( lX SHIFT ); stX = X0; }
   else           { X = X0; stX = X0 + ( lX SHIFT ); }
   A = A0; stA = A0 + ( lA SHIFT );

   Xseed = N * aincX + 127 * 50 + 77;
   Aseed = N * N     + 513 *  7 + 90;

   reps = ( MEGA * MFLOP ) / flops;
/*
 * Generate the random data and time the trusted routine
 */
   x = X; a = A;

   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == TBMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_tbmv( UPLO, TRANS, DIAG, N, K, (const TYPE *)(a), LDA,
                       (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   else if( ROUT == TPMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_tpmv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a),
                       (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_trmv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a), LDA,
                       (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 ) mftrust = ( reps * flops ) / ( MEGA * ttrust );
   else               mftrust = 0.0;
   ttrust /= reps; *TTRUST0 = ttrust; *MFTRUST0 = mftrust;
/*
 * Generate the random data and time the tested routine
 */
   x = X; a = A;

   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == TBMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_tbmv( UPLO, TRANS, DIAG, N, K, (const TYPE *)(a), LDA,
                    (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   else if( ROUT == TPMV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_tpmv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a),
                    (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_trmv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a), LDA,
                    (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 ) mftest  = ( reps * flops ) / ( MEGA * ttest  );
   else               mftest  = 0.0;
   ttest  /= reps; *TTEST0  = ttest;  *MFTEST0  = mftest;
/*
 * release the memory and exit
 */
   free( X0 );
   free( A0 );

   return( passed );
}

int tsvcase
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  MFLOP,
   const enum ATLAS_UPLO      UPLO,
   const enum ATLAS_TRANS     TRANS,
   const enum ATLAS_DIAG      DIAG,
   const int                  N,
   const int                  K,
   const int                  LDA,
   const int                  INCX,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        aincX, inca, incx, lX, lA, mA, nA, passed;
   int                        Aseed, Xseed;
   unsigned long              ir, reps;
   TYPE                       resid = ATL_rzero;
   double                     flops, ttrust, ttest, mftrust, mftest, t0;
   TYPE                       * a, * stA, * stX, * x, * A, * A0 = NULL, * X,
                              * X0 = NULL;

   if( ( MEGA * MFLOP <= ( flops = opbl2( ROUT, N, N, K, K ) ) ) || ( TEST ) )
   {
      resid = tsvtst( ROUT, TEST, UPLO, TRANS, DIAG, N, K, LDA, INCX,
                      TTRUST0, TTEST0, MFTRUST0, MFTEST0 );
      if( resid > THRESH ) (void) fprintf( stderr, "   resid=%f\n", resid );
   }
   if( resid < ATL_rzero ) passed = -1;
   else                    passed = ( resid < THRESH );

   if( MEGA * MFLOP <= flops ) return( passed );

   if(      ROUT == TBSV ) { mA = K + 1;                  nA = N; }
   else if( ROUT == TPSV ) { mA = ( N * ( N + 1 ) ) >> 1; nA = 1; }
   else                    { mA = N;                      nA = N; }

   incx = INCX * ( N  SHIFT ); aincX = Mabs( INCX );
   if( ROUT == TPSV ) inca = ( mA SHIFT );
   else               inca = LDA  * ( nA SHIFT );

   lX = N * aincX * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lA =     inca  * ( ( ATL_DivBySize( L2SIZE ) + 2*mA*nA - 1 ) / ( mA * nA ) );

   X0 = (TYPE *)malloc( ATL_MulBySize( lX ) );
   A0 = (TYPE *)malloc( ATL_MulBySize( lA ) );

   if( ( X0 == NULL ) || ( A0 == NULL ) )
   {
      if( X0 ) free( X0 );
      if( A0 ) free( A0 );
      return( -1 );
   }

   if( INCX < 1 ) { X = X0 + ( lX SHIFT ); stX = X0; }
   else           { X = X0; stX = X0 + ( lX SHIFT ); }
   A = A0; stA = A0 + ( lA SHIFT );

   Xseed = N * aincX + 127 * 50 + 77;
   Aseed = N * N     + 513 *  7 + 90;

   reps = ( MEGA * MFLOP ) / flops;
/*
 * Generate the random data and time the trusted routine
 */
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   a = A;

   if(      ROUT == TBSV )
   do { tbddom( UPLO, N, K, a, LDA ); a += inca; } while( a != stA );
   else if( ROUT == TPSV )
   do { tpddom( UPLO, N,    a      ); a += inca; } while( a != stA );
   else
   do { trddom( UPLO, N,    a, LDA ); a += inca; } while( a != stA );

   x = X; a = A;

   if(      ROUT == TBSV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_tbsv( UPLO, TRANS, DIAG, N, K, (const TYPE *)(a), LDA,
                       (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   else if( ROUT == TPSV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_tpsv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a),
                       (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_trsv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a), LDA,
                       (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 ) mftrust = ( reps * flops ) / ( MEGA * ttrust );
   else               mftrust = 0.0;
   ttrust /= reps; *TTRUST0 = ttrust; *MFTRUST0 = mftrust;
/*
 * Generate the random data and time the tested routine
 */
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   a = A;

   if(      ROUT == TBSV )
   do { tbddom( UPLO, N, K, a, LDA ); a += inca; } while( a != stA );
   else if( ROUT == TPSV )
   do { tpddom( UPLO, N,    a      ); a += inca; } while( a != stA );
   else
   do { trddom( UPLO, N,    a, LDA ); a += inca; } while( a != stA );

   x = X; a = A;

   if(      ROUT == TBSV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_tbsv( UPLO, TRANS, DIAG, N, K, (const TYPE *)(a), LDA,
                    (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   else if( ROUT == TPSV )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_tpsv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a),
                    (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_trsv( UPLO, TRANS, DIAG, N,    (const TYPE *)(a), LDA,
                    (TYPE *)(x), INCX );
         x += incx; if( x == stX ) { x = X; }
         a += inca; if( a == stA ) { a = A; }
      }
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 ) mftest  = ( reps * flops ) / ( MEGA * ttest  );
   else               mftest  = 0.0;
   ttest  /= reps; *TTEST0  = ttest;  *MFTEST0  = mftest;
/*
 * release the memory and exit
 */
   free( X0 );
   free( A0 );

   return( passed );
}

int gr1case
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  MFLOP,
   const int                  M,
   const int                  N,
   const SCALAR               ALPHA,
   const int                  INCX,
   const int                  INCY,
   const int                  LDA,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        aincX, aincY, inca, incx, incy, lX, lY, lA,
                              passed;
   int                        Aseed, Xseed, Yseed;
   unsigned long              ir, reps;
   TYPE                       resid = ATL_rzero;
   double                     flops, ttrust, ttest, mftrust, mftest, t0;
#ifdef TREAL
   TYPE                       alph,  alpha,    nalpha;
#else
   TYPE                       *alph, alpha[2], nalpha[2];
#endif
   TYPE                       * a, * stA, * stX, * stY, * x, * y, * A,
                              * A0 = NULL, * X, * X0 = NULL, * Y, * Y0 = NULL;

   if( ( MEGA * MFLOP <= ( flops = opbl2( ROUT, M, N, 0, 0 ) ) ) || ( TEST ) )
   {
      resid = gr1tst( ROUT, TEST, M, N, ALPHA, INCX, INCY, LDA, TTRUST0,
                      TTEST0, MFTRUST0, MFTEST0 );
      if( resid > THRESH ) (void) fprintf( stderr, "   resid=%f\n", resid );
   }
   if( resid < ATL_rzero ) passed = -1;
   else                    passed = ( resid < THRESH );

   if( MEGA * MFLOP <= flops ) return( passed );

   incy = INCY * ( N  SHIFT ); aincY = Mabs( INCY );
   incx = INCX * ( M  SHIFT ), aincX = Mabs( INCX );
   inca = LDA  * ( N SHIFT );

   lY = N * aincY * ( ( ATL_DivBySize( L2SIZE ) + N     - 1 ) / N         );
   lX = M * aincX * ( ( ATL_DivBySize( L2SIZE ) + M     - 1 ) / M         );
   lA =     inca  * ( ( ATL_DivBySize( L2SIZE ) + 2*M*N - 1 ) / ( M * N ) );

   Y0 = (TYPE *)malloc( ATL_MulBySize( lY ) );
   X0 = (TYPE *)malloc( ATL_MulBySize( lX ) );
   A0 = (TYPE *)malloc( ATL_MulBySize( lA ) );

   if( ( Y0 == NULL ) || ( X0 == NULL ) || ( A0 == NULL ) )
   {
      if( Y0 ) free( Y0 );
      if( X0 ) free( X0 );
      if( A0 ) free( A0 );
      return( -1 );
   }

   if( INCY < 1 ) { Y = Y0 + ( lY SHIFT ); stY = Y0; }
   else           { Y = Y0; stY = Y0 + ( lY SHIFT ); }
   if( INCX < 1 ) { X = X0 + ( lX SHIFT ); stX = X0; }
   else           { X = X0; stX = X0 + ( lX SHIFT ); }
   A = A0; stA = A0 + ( lA SHIFT );

#ifdef TREAL
   alpha  =  ALPHA;
   nalpha = -ALPHA;
#else
   *alpha  =  ALPHA[0]; alpha [1] =  ALPHA[1];
   *nalpha = -ALPHA[0]; nalpha[1] = -ALPHA[1];
#endif

   Yseed = N * aincY;
   Xseed = M * aincX + 127 * 50 + 77;
   Aseed = N * M     + 513 *  7 + 90;

   reps = ( MEGA * MFLOP ) / flops;
/*
 * Generate the random data and time the trusted routine
 */
   alph = alpha; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if( ROUT == GERU )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_geru( M, N, (const SCALAR)(alph), (const TYPE *)(x), INCX,
                       (const TYPE *)(y), INCY, (TYPE *)(a), LDA );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttrust = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_gerc( M, N, (const SCALAR)(alph), (const TYPE *)(x), INCX,
                       (const TYPE *)(y), INCY, (TYPE *)(a), LDA );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 ) mftrust = ( reps * flops ) / ( MEGA * ttrust );
   else               mftrust = 0.0;
   ttrust /= reps; *TTRUST0 = ttrust; *MFTRUST0 = mftrust;
/*
 * Generate the random data and time the tested routine
 */
   alph = alpha; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if( ROUT == GERU )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_geru( M, N, (const SCALAR)(alph), (const TYPE *)(x), INCX,
                    (const TYPE *)(y), INCY, (TYPE *)(a), LDA );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttest = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_gerc( M, N, (const SCALAR)(alph), (const TYPE *)(x), INCX,
                    (const TYPE *)(y), INCY, (TYPE *)(a), LDA );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 ) mftest  = ( reps * flops ) / ( MEGA * ttest  );
   else               mftest  = 0.0;
   ttest  /= reps; *TTEST0  = ttest;  *MFTEST0  = mftest;
/*
 * release the memory and exit
 */
   free( Y0 );
   free( X0 );
   free( A0 );

   return( passed );
}

int sr1case
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  MFLOP,
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const TYPE                 ALPHA,
   const int                  INCX,
   const int                  LDA,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        aincX, inca, incx, lX, lA, mA, nA, passed;
   int                        Aseed, Xseed;
   unsigned long              ir, reps;
   TYPE                       alph,  alpha, nalpha, resid = ATL_rzero;
   double                     flops, ttrust, ttest, mftrust, mftest, t0;
   TYPE                       * a, * stA, * stX, * x, * A, * A0 = NULL, * X,
                              * X0 = NULL;

   if( ( MEGA * MFLOP <= ( flops = opbl2( ROUT, N, N, 0, 0 ) ) ) || ( TEST ) )
   {
      resid = sr1tst( ROUT, TEST, UPLO, N, ALPHA, INCX, LDA, TTRUST0,
                      TTEST0, MFTRUST0, MFTEST0 );
      if( resid > THRESH ) (void) fprintf( stderr, "   resid=%f\n", resid );
   }
   if( resid < ATL_rzero ) passed = -1;
   else                    passed = ( resid < THRESH );

   if( MEGA * MFLOP <= flops ) return( passed );

   if( ROUT == SPR ) { mA = ( N * ( N + 1 ) ) >> 1; nA = 1; }
   else              { mA = N;                      nA = N; }

   incx = INCX * ( N  SHIFT ); aincX = Mabs( INCX );
   if( ROUT == SPR ) inca = ( mA SHIFT );
   else              inca = LDA  * ( nA SHIFT );

   lX = N * aincX * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lA =     inca  * ( ( ATL_DivBySize( L2SIZE ) + 2*mA*nA - 1 ) / ( mA * nA ) );

   X0 = (TYPE *)malloc( ATL_MulBySize( lX ) );
   A0 = (TYPE *)malloc( ATL_MulBySize( lA ) );

   if( ( X0 == NULL ) || ( A0 == NULL ) )
   {
      if( X0 ) free( X0 );
      if( A0 ) free( A0 );
      return( -1 );
   }

   if( INCX < 1 ) { X = X0 + ( lX SHIFT ); stX = X0; }
   else           { X = X0; stX = X0 + ( lX SHIFT ); }
   A = A0; stA = A0 + ( lA SHIFT );

   alpha  =  ALPHA;
   nalpha = -ALPHA;

   Xseed = N * aincX + 127 * 50 + 77;
   Aseed = N * N     + 513 *  7 + 90;

   reps = ( MEGA * MFLOP ) / flops;
/*
 * Generate the random data and time the trusted routine
 */
   alph = alpha; x = X; a = A;

   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == SPR )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
#ifdef TREAL
         trusted_spr( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                      INCX, (TYPE *)(a)      );
#else
         trusted_spr( UPLO, N, (const TYPE  )(alph), (const TYPE *)(x),
                      INCX, (TYPE *)(a)      );
#endif
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttrust = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
#ifdef TREAL
         trusted_syr( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                      INCX, (TYPE *)(a), LDA );
#else
         trusted_syr( UPLO, N, (const TYPE  )(alph), (const TYPE *)(x),
                      INCX, (TYPE *)(a), LDA );
#endif
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 ) mftrust = ( reps * flops ) / ( MEGA * ttrust );
   else               mftrust = 0.0;
   ttrust /= reps; *TTRUST0 = ttrust; *MFTRUST0 = mftrust;
/*
 * Generate the random data and time the tested routine
 */
   alph = alpha; x = X; a = A;

   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == SPR )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
#ifdef TREAL
         test_spr( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                   INCX, (TYPE *)(a)      );
#else
         test_spr( UPLO, N, (const TYPE  )(alph), (const TYPE *)(x),
                   INCX, (TYPE *)(a)      );
#endif
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttest = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
#ifdef TREAL
         test_syr( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                   INCX, (TYPE *)(a), LDA );
#else
         test_syr( UPLO, N, (const TYPE  )(alph), (const TYPE *)(x),
                   INCX, (TYPE *)(a), LDA );
#endif
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 ) mftest  = ( reps * flops ) / ( MEGA * ttest  );
   else               mftest  = 0.0;
   ttest  /= reps; *TTEST0  = ttest;  *MFTEST0  = mftest;
/*
 * release the memory and exit
 */
   free( X0 );
   free( A0 );

   return( passed );
}

int sr2case
(
   const enum LVL2_ROUT       ROUT,
   const int                  TEST,
   const int                  MFLOP,
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const SCALAR               ALPHA,
   const int                  INCX,
   const int                  INCY,
   const int                  LDA,
   double                     * TTRUST0,
   double                     * TTEST0,
   double                     * MFTRUST0,
   double                     * MFTEST0
)
{
   int                        aincX, aincY, inca, incx, incy, lX, lY, lA,
                              mA, nA, passed;
   int                        Aseed, Xseed, Yseed;
   unsigned long              ir, reps;
   TYPE                       resid = ATL_rzero;
   double                     flops, ttrust, ttest, mftrust, mftest, t0;
#ifdef TREAL
   TYPE                       alph,  alpha,    nalpha;
#else
   TYPE                       *alph, alpha[2], nalpha[2];
#endif
   TYPE                       * a, * stA, * stX, * stY, * x, * y, * A,
                              * A0 = NULL, * X, * X0 = NULL, * Y, * Y0 = NULL;

   if( ( MEGA * MFLOP <= ( flops = opbl2( ROUT, N, N, 0, 0 ) ) ) || ( TEST ) )
   {
      resid = sr2tst( ROUT, TEST, UPLO, N, ALPHA, INCX, INCY, LDA,
                      TTRUST0, TTEST0, MFTRUST0, MFTEST0 );
      if( resid > THRESH ) (void) fprintf( stderr, "   resid=%f\n", resid );
   }
   if( resid < ATL_rzero ) passed = -1;
   else                    passed = ( resid < THRESH );

   if( MEGA * MFLOP <= flops ) return( passed );

   if( ROUT == SPR2 ) { mA = ( N * ( N + 1 ) ) >> 1; nA = 1; }
   else               { mA = N;                      nA = N; }

   incy = INCY * ( N  SHIFT ); aincY = Mabs( INCY );
   incx = INCX * ( N  SHIFT ); aincX = Mabs( INCX );
   if( ROUT == SPR2 ) inca = ( mA SHIFT );
   else               inca = LDA  * ( nA SHIFT );

   lY = N * aincY * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lX = N * aincX * ( ( ATL_DivBySize( L2SIZE ) + N       - 1 ) / N           );
   lA =     inca  * ( ( ATL_DivBySize( L2SIZE ) + 2*mA*nA - 1 ) / ( mA * nA ) );

   Y0 = (TYPE *)malloc( ATL_MulBySize( lY ) );
   X0 = (TYPE *)malloc( ATL_MulBySize( lX ) );
   A0 = (TYPE *)malloc( ATL_MulBySize( lA ) );

   if( ( Y0 == NULL ) || ( X0 == NULL ) || ( A0 == NULL ) )
   {
      if( Y0 ) free( Y0 );
      if( X0 ) free( X0 );
      if( A0 ) free( A0 );
      return( -1 );
   }

   if( INCY < 1 ) { Y = Y0 + ( lY SHIFT ); stY = Y0; }
   else           { Y = Y0; stY = Y0 + ( lY SHIFT ); }
   if( INCX < 1 ) { X = X0 + ( lX SHIFT ); stX = X0; }
   else           { X = X0; stX = X0 + ( lX SHIFT ); }
   A = A0; stA = A0 + ( lA SHIFT );

#ifdef TREAL
   alpha   =  ALPHA;
   nalpha  = -ALPHA;
#else
   *alpha  =  (*ALPHA); alpha [1] =  ALPHA[1];
   *nalpha = -(*ALPHA); nalpha[1] = -ALPHA[1];
#endif

   Yseed = N * aincY;
   Xseed = N * aincX + 127 * 50 + 77;
   Aseed = N * N     + 513 *  7 + 90;

   reps = ( MEGA * MFLOP ) / flops;
/*
 * Generate the random data and time the trusted routine
 */
   alph = alpha; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == SPR2 )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_spr2( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                       INCX, (const TYPE *)(y), INCY, (TYPE *)(a)      );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttrust = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         trusted_syr2( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                       INCX, (const TYPE *)(y), INCY, (TYPE *)(a), LDA );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttrust = time00() - t0;
   }
   if( ttrust > 0.0 ) mftrust = ( reps * flops ) / ( MEGA * ttrust );
   else               mftrust = 0.0;
   ttrust /= reps; *TTRUST0 = ttrust; *MFTRUST0 = mftrust;
/*
 * Generate the random data and time the tested routine
 */
   alph = alpha; y = Y; x = X; a = A;

   Mjoin( PATL, gegen )( lY, 1, Y0, lY, Yseed );
   Mjoin( PATL, gegen )( lX, 1, X0, lX, Xseed );
   Mjoin( PATL, gegen )( lA, 1, A0, lA, Aseed );

   if(      ROUT == SPR2 )
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_spr2( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                    INCX, (const TYPE *)(y), INCY, (TYPE *)(a)      );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttest = time00() - t0;
   }
   else
   {
      t0 = time00();
      for( ir = reps; ir; ir-- )
      {
         test_syr2( UPLO, N, (const SCALAR)(alph), (const TYPE *)(x),
                    INCX, (const TYPE *)(y), INCY, (TYPE *)(a), LDA );
         y += incy; if( y == stY ) { y = Y; }
         x += incx; if( x == stX ) { x = X; }
         a += inca;
         if( a == stA )
         { a = A; if( alph == alpha ) alph = nalpha; else alph = alpha; }
      }
      ttest = time00() - t0;
   }
   if( ttest  > 0.0 ) mftest  = ( reps * flops ) / ( MEGA * ttest  );
   else               mftest  = 0.0;
   ttest  /= reps; *TTEST0  = ttest;  *MFTEST0  = mftest;
/*
 * release the memory and exit
 */
   free( Y0 );
   free( X0 );
   free( A0 );

   return( passed );
}

void RungbmCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NTRAN,
   enum ATLAS_TRANS           * TRANS,
   int                        M0,
   int                        MN,
   int                        MINC,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        KL0,
   int                        KLN,
   int                        KLINC,
   int                        KU0,
   int                        KUN,
   int                        KUINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NBETA,
   TYPE                       * BETAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       ctran;
   int                        al, be, ires, ix, iy, kkl, kku, lda, m, mm, n,
                              nn, ta, msame=0;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

   if( M0 == -1 ) { M0 = MN = MINC = NN; msame = 1; }

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "------------------------------------ ", "GBMV",
                   " -------------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# TR    M    N   KL   KU ALPHA  LDA INCX  BETA",
                   " INCY   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ==== ==== ===== ==== ==== =====",
                   " ==== ====== ===== ===== =====\n" );
form =
"%4d  %c %4d %4d %4d %4d %5.1f %4d %4d %5.1f %4d %6.2f %5.1f %5.2f %5s\n";
#else
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "---------------------------------------- ", "GBMV",
                   " -----------------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# TR    M    N   KL   KU     ALPHA  LDA INCX",
                   "      BETA INCY   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ==== ==== ==== ==== ==== ====",
                   " ==== ==== ==== ====== ===== ===== =====\n" );
form =
"%4d  %c %4d %4d %4d %4d %4.1f %4.1f %4d %4d %4.1f %4.1f %4d %6.2f %5.1f %5.2f %5s\n";
#endif

   for( mm = M0; mm <= MN; mm += MINC )
   {
      for( nn = N0; nn <= NN; nn += NINC )
      {
         n = nn;
         if( msame ) m = n;
         else        m = mm;

         for( kkl = KL0; kkl <= KLN; kkl += KLINC )
         {
            for( kku = KU0; kku <= KUN; kku += KUINC )
            {
               for( ta = 0; ta < NTRAN; ta++ )
               {
                  if(      TRANS[ta] == AtlasNoTrans   ) ctran = 'N';
                  else if( TRANS[ta] == AtlasTrans     ) ctran = 'T';
                  else if( TRANS[ta] == AtlasConjTrans ) ctran = 'C';

                  for( iy = 0; iy < NINCY; iy++ )
                  {
                     for( ix = 0; ix < NINCX; ix++ )
                     {
                        for( al = 0; al < NALPHA; al++ )
                        {
                           for( be = 0; be < NBETA; be++ )
                           {
                              if( LDA_IS_M ) lda = kkl+1+kku;
                              else           lda = KLN+1+KUN;
#ifdef TREAL
                              ires = gmvcase( ROUT, TEST, MFLOP, TRANS[ta],
                                              m, n, kkl, kku, ALPHAS[al], lda,
                                              INCXS[ix], BETAS[be], INCYS[iy],
                                              &ttrust, &ttest, &mftrust,
                                              &mftest );
#else
                              ires = gmvcase( ROUT, TEST, MFLOP, TRANS[ta],
                                              m, n, kkl, kku, ALPHAS+2*al, lda,
                                              INCXS[ix], BETAS+2*be, INCYS[iy],
                                              &ttrust, &ttest, &mftrust,
                                              &mftest );
#endif
                              if(     !( TEST ) ) pass = "SKIP ";
                              else if( ires < 0 ) pass = "NoMEM";
                              else if( ires     ) pass = "PASS ";
                              else                pass = "FAIL ";

                              if( ires > 0 ) (*NPASSED)++;

                              if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                                 t0 = mftest / mftrust;
                              else t0 = 0.0;
#ifdef TREAL
                              (void) fprintf( stdout, form, *NTESTS, ctran, m,
                                              n, kkl, kku, ALPHAS[al], lda,
                                              INCXS[ix], BETAS[be], INCYS[iy],
                                              ttrust, mftrust, 1.0, "-----" );
                              (void) fprintf( stdout, form, *NTESTS, ctran, m,
                                              n, kkl, kku, ALPHAS[al], lda,
                                              INCXS[ix], BETAS[be], INCYS[iy],
                                              ttest,  mftest,  t0,  pass );
#else
                              (void) fprintf( stdout, form, *NTESTS, ctran,
                                              m, n, kkl, kku, ALPHAS[2*al],
                                              ALPHAS[2*al+1], lda, INCXS[ix],
                                              BETAS[2*be], BETAS[2*be+1],
                                              INCYS[iy], ttrust, mftrust, 1.0,
                                              "-----" );
                              (void) fprintf( stdout, form, *NTESTS, ctran,
                                              m, n, kkl, kku, ALPHAS[2*al],
                                              ALPHAS[2*al+1], lda, INCXS[ix],
                                              BETAS[2*be], BETAS[2*be+1],
                                              INCYS[iy], ttest,  mftest,  t0,
                                              pass );
#endif
                              (*NTESTS)++;
                           }
                        }
                     }
                  }
               }
            }
         }
      }
   }
}

void RungemCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NTRAN,
   enum ATLAS_TRANS           * TRANS,
   int                        M0,
   int                        MN,
   int                        MINC,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NBETA,
   TYPE                       * BETAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       ctran;
   int                        al, be, ires, ix, iy, lda, m, mm, msame=0, n,
                              nn, ta;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

   if( M0 == -1 ) { M0 = MN = MINC = NN; msame = 1; }

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "------------------------------- ", "GEMV",
                   " --------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# TR    M    N ALPHA  LDA INCX  BETA INCY",
                   "   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ===== ==== ==== ===== ====",
                   " ====== ===== ===== =====\n" );
form = "%4d  %c %4d %4d %5.1f %4d %4d %5.1f %4d %6.2f %5.1f %5.2f %5s\n";
#else
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "----------------------------------- ", "GEMV",
                   " ------------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# TR    M    N     ALPHA  LDA INCX      BETA",
                   " INCY   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ==== ==== ==== ==== ==== ====",
                   " ==== ====== ===== ===== =====\n" );
form =
"%4d  %c %4d %4d %4.1f %4.1f %4d %4d %4.1f %4.1f %4d %6.2f %5.1f %5.2f %5s\n";
#endif

   for( mm = M0; mm <= MN; mm += MINC )
   {
      for( nn = N0; nn <= NN; nn += NINC )
      {
         n = nn;
         if( msame ) m = n;
         else        m = mm;

         for( ta = 0; ta < NTRAN; ta++ )
         {
            if(      TRANS[ta] == AtlasNoTrans   ) ctran = 'N';
            else if( TRANS[ta] == AtlasTrans     ) ctran = 'T';
            else if( TRANS[ta] == AtlasConjTrans ) ctran = 'C';

            for( iy = 0; iy < NINCY; iy++ )
            {
               for( ix = 0; ix < NINCX; ix++ )
               {
                  for( al = 0; al < NALPHA; al++ )
                  {
                     for( be = 0; be < NBETA; be++ )
                     {
                        if( TRANS[ta] == AtlasNoTrans )
                        { if( LDA_IS_M ) lda = m; else lda = MN; }
                        else
                        { if( LDA_IS_M ) lda = n; else lda = NN; }
#ifdef TREAL
                        ires = gmvcase( ROUT, TEST, MFLOP, TRANS[ta],
                                        m, n, 0, 0, ALPHAS[al], lda,
                                        INCXS[ix], BETAS[be], INCYS[iy],
                                        &ttrust, &ttest, &mftrust, &mftest );
#else
                        ires = gmvcase( ROUT, TEST, MFLOP, TRANS[ta],
                                        m, n, 0, 0, ALPHAS+2*al, lda,
                                        INCXS[ix], BETAS+2*be, INCYS[iy],
                                        &ttrust, &ttest, &mftrust, &mftest );
#endif
                        if(     !( TEST ) ) pass = "SKIP ";
                        else if( ires < 0 ) pass = "NoMEM";
                        else if( ires     ) pass = "PASS ";
                        else                pass = "FAIL ";

                        if( ires > 0 ) (*NPASSED)++;

                        if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                           t0 = mftest / mftrust;
                        else t0 = 0.0;
#ifdef TREAL
                        (void) fprintf( stdout, form, *NTESTS, ctran, m,
                                        n, ALPHAS[al], lda, INCXS[ix],
                                        BETAS[be], INCYS[iy], ttrust,
                                        mftrust, 1.0, "-----" );
                        (void) fprintf( stdout, form, *NTESTS, ctran, m,
                                        n, ALPHAS[al], lda, INCXS[ix],
                                        BETAS[be], INCYS[iy], ttest,
                                        mftest,  t0,  pass );
#else
                        (void) fprintf( stdout, form, *NTESTS, ctran, m,
                                        n, ALPHAS[2*al], ALPHAS[2*al+1],
                                        lda, INCXS[ix], BETAS[2*be],
                                        BETAS[2*be+1], INCYS[iy], ttrust,
                                        mftrust, 1.0, "-----" );
                        (void) fprintf( stdout, form, *NTESTS, ctran, m,
                                        n, ALPHAS[2*al], ALPHAS[2*al+1],
                                        lda, INCXS[ix], BETAS[2*be],
                                        BETAS[2*be+1], INCYS[iy], ttest,
                                        mftest,  t0,  pass );
#endif
                        (*NTESTS)++;
                     }
                  }
               }
            }
         }
      }
   }
}

void RunsbCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        KL0,
   int                        KLN,
   int                        KLINC,
   int                        KU0,
   int                        KUN,
   int                        KUINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NBETA,
   TYPE                       * BETAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       cuplo;
   int                        al, be, ires, ix, iy, kkl, kku, lda, nn, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "------------------------------- ", "SBMV",
                   " --------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UP    N    K ALPHA  LDA INCX  BETA INCY",
                   "   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ===== ==== ==== ===== ====",
                   " ====== ===== ===== =====\n" );
   form = "%4d  %c %4d %4d %5.1f %4d %4d %5.1f %4d %6.2f %5.1f %5.2f %5s\n";
#else
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "------------------------------------ ", "HBMV",
                   " -----------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UP    K    N     ALPHA  LDA INCX      BETA",
                   " INCY   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ==== ==== ==== ==== ==== ====",
                   " ==== ====== ===== ===== =====\n" );
form =
"%4d  %c %4d %4d %4.1f %4.1f %4d %4d %4.1f %4.1f %4d %6.2f %5.1f %5.2f %5s\n";
#endif
   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasLower )
         {
            cuplo = 'L';

            for( kkl = KL0; kkl <= KLN; kkl += KLINC )
            {
               for( ix = 0; ix < NINCX; ix++ )
               {
                  for( iy = 0; iy < NINCY; iy++ )
                  {
                     for( al = 0; al < NALPHA; al++ )
                     {
                        for( be = 0; be < NBETA; be++ )
                        {
                           if( LDA_IS_M ) lda = kkl+1;
                           else           lda = KLN+1;
#ifdef TREAL
                           ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                           kkl, ALPHAS[al], lda, INCXS[ix],
                                           BETAS[be], INCYS[iy], &ttrust,
                                           &ttest, &mftrust, &mftest );
#else
                           ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                           kkl, ALPHAS+2*al, lda, INCXS[ix],
                                           BETAS+2*be, INCYS[iy], &ttrust,
                                           &ttest, &mftrust, &mftest );
#endif
                           if(     !( TEST ) ) pass = "SKIP ";
                           else if( ires < 0 ) pass = "NoMEM";
                           else if( ires     ) pass = "PASS ";
                           else                pass = "FAIL ";

                           if( ires > 0 ) (*NPASSED)++;

                           if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                              t0 = mftest / mftrust;
                           else t0 = 0.0;
#ifdef TREAL
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kkl, ALPHAS[al], lda, INCXS[ix],
                                           BETAS[be], INCYS[iy], ttrust,
                                           mftrust, 1.0, "-----" );
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kkl, ALPHAS[al], lda, INCXS[ix],
                                           BETAS[be], INCYS[iy], ttest,
                                           mftest,  t0,  pass );
#else
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kkl, ALPHAS[2*al], ALPHAS[2*al+1],
                                           lda, INCXS[ix], BETAS[2*be],
                                           BETAS[2*be+1], INCYS[iy], ttrust,
                                           mftrust, 1.0, "-----" );
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kkl, ALPHAS[2*al], ALPHAS[2*al+1],
                                           lda, INCXS[ix], BETAS[2*be],
                                           BETAS[2*be+1], INCYS[iy], ttest,
                                           mftest,  t0,  pass );
#endif
                           (*NTESTS)++;
                        }
                     }
                  }
               }
            }
         }
         else
         {
            cuplo = 'U';

            for( kku = KU0; kku <= KUN; kku += KUINC )
            {
               for( ix = 0; ix < NINCX; ix++ )
               {
                  for( iy = 0; iy < NINCY; iy++ )
                  {
                     for( al = 0; al < NALPHA; al++ )
                     {
                        for( be = 0; be < NBETA; be++ )
                        {
                           if( LDA_IS_M ) lda = 1+kku;
                           else           lda = 1+KUN;
#ifdef TREAL
                           ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                           kku, ALPHAS[al], lda, INCXS[ix],
                                           BETAS[be], INCYS[iy], &ttrust,
                                           &ttest, &mftrust, &mftest );
#else
                           ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                           kku, ALPHAS+2*al, lda, INCXS[ix],
                                           BETAS+2*be, INCYS[iy], &ttrust,
                                           &ttest, &mftrust, &mftest );
#endif
                           if(     !( TEST ) ) pass = "SKIP ";
                           else if( ires < 0 ) pass = "NoMEM";
                           else if( ires     ) pass = "PASS ";
                           else                pass = "FAIL ";

                           if( ires > 0 ) (*NPASSED)++;

                           if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                              t0 = mftest / mftrust;
                           else t0 = 0.0;
#ifdef TREAL
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kku, ALPHAS[al], lda, INCXS[ix],
                                           BETAS[be], INCYS[iy], ttrust,
                                           mftrust, 1.0, "-----" );
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kku, ALPHAS[al], lda, INCXS[ix],
                                           BETAS[be], INCYS[iy], ttest,
                                           mftest,  t0,  pass );
#else
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kku, ALPHAS[2*al], ALPHAS[2*al+1],
                                           lda, INCXS[ix], BETAS[2*be],
                                           BETAS[2*be+1], INCYS[iy], ttrust,
                                           mftrust, 1.0, "-----" );
                           (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                           kku, ALPHAS[2*al], ALPHAS[2*al+1],
                                           lda, INCXS[ix], BETAS[2*be],
                                           BETAS[2*be+1], INCYS[iy], ttest,
                                           mftest,  t0,  pass );
#endif
                           (*NTESTS)++;
                        }
                     }
                  }
               }
            }
         }
      }
   }
}

void RunspCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NBETA,
   TYPE                       * BETAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       cuplo;
   int                        al, be, ires, ix, iy, nn, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "-------------------------- ", "SPMV",
                   " ---------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UP    N ALPHA INCX  BETA INCY",
                   "   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ===== ==== ===== ====",
                   " ====== ===== ===== =====\n" );
   form = "%4d  %c %4d %5.1f %4d %5.1f %4d %6.2f %5.1f %5.2f %5s\n";
#else
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "------------------------------ ", "HPMV",
                   " -------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UP    N     ALPHA INCX      BETA",
                   " INCY   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ==== ==== ==== ====",
                   " ==== ====== ===== ===== =====\n" );
   form = "%4d  %c %4d %4.1f %4.1f %4d %4.1f %4.1f %4d %6.2f %5.1f %5.2f %5s\n";
#endif
   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasLower ) cuplo = 'L';
         else                          cuplo = 'U';

         for( ix = 0; ix < NINCX; ix++ )
         {
            for( iy = 0; iy < NINCY; iy++ )
            {
               for( al = 0; al < NALPHA; al++ )
               {
                  for( be = 0; be < NBETA; be++ )
                  {
#ifdef TREAL
                     ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn, 0,
                                     ALPHAS[al], 1, INCXS[ix], BETAS[be],
                                     INCYS[iy], &ttrust, &ttest, &mftrust,
                                     &mftest );
#else
                     ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn, 0,
                                     ALPHAS+2*al, 1, INCXS[ix], BETAS+2*be,
                                     INCYS[iy], &ttrust, &ttest, &mftrust,
                                     &mftest );
#endif
                     if(     !( TEST ) ) pass = "SKIP ";
                     else if( ires < 0 ) pass = "NoMEM";
                     else if( ires     ) pass = "PASS ";
                     else                pass = "FAIL ";

                     if( ires > 0 ) (*NPASSED)++;

                     if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                        t0 = mftest / mftrust;
                     else t0 = 0.0;
#ifdef TREAL
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[al], INCXS[ix], BETAS[be],
                                     INCYS[iy], ttrust, mftrust, 1.0,
                                     "-----" );
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[al], INCXS[ix], BETAS[be],
                                     INCYS[iy], ttest, mftest, t0, pass );
#else
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[2*al], ALPHAS[2*al+1],
                                     INCXS[ix], BETAS[2*be], BETAS[2*be+1],
                                     INCYS[iy], ttrust, mftrust, 1.0,
                                     "-----" );
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[2*al], ALPHAS[2*al+1],
                                     INCXS[ix], BETAS[2*be], BETAS[2*be+1],
                                     INCYS[iy], ttest, mftest, t0, pass );
#endif
                     (*NTESTS)++;
                  }
               }
            }
         }
      }
   }
}

void RunsyCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NBETA,
   TYPE                       * BETAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       cuplo;
   int                        al, be, ires, ix, iy, lda, nn, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "----------------------------- ", "SYMV",
                   " -----------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UP    N ALPHA  LDA INCX  BETA INCY",
                   "   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ===== ==== ==== ===== ====",
                   " ====== ===== ===== =====\n" );
   form = "%4d  %c %4d %5.1f %4d %4d %5.1f %4d %6.2f %5.1f %5.2f %5s\n";
#else
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "--------------------------------- ", "HEMV",
                   " ---------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UP    N     ALPHA  LDA INCX      BETA",
                   " INCY   TIME MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== == ==== ==== ==== ==== ==== ==== ====",
                   " ==== ====== ===== ===== =====\n" );
form =
"%4d  %c %4d %4.1f %4.1f %4d %4d %4.1f %4.1f %4d %6.2f %5.1f %5.2f %5s\n";
#endif
   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasLower ) cuplo = 'L';
         else                          cuplo = 'U';

         for( ix = 0; ix < NINCX; ix++ )
         {
            for( iy = 0; iy < NINCY; iy++ )
            {
               for( al = 0; al < NALPHA; al++ )
               {
                  for( be = 0; be < NBETA; be++ )
                  {
                     if( LDA_IS_M ) lda = nn;
                     else           lda = NN;
#ifdef TREAL
                     ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                     0, ALPHAS[al], lda, INCXS[ix],
                                     BETAS[be], INCYS[iy], &ttrust,
                                     &ttest, &mftrust, &mftest );
#else
                     ires = smvcase( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                     0, ALPHAS+2*al, lda, INCXS[ix],
                                     BETAS+2*be, INCYS[iy], &ttrust,
                                     &ttest, &mftrust, &mftest );
#endif
                     if(     !( TEST ) ) pass = "SKIP ";
                     else if( ires < 0 ) pass = "NoMEM";
                     else if( ires     ) pass = "PASS ";
                     else                pass = "FAIL ";

                     if( ires > 0 ) (*NPASSED)++;

                     if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                        t0 = mftest / mftrust;
                     else t0 = 0.0;
#ifdef TREAL
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[al], lda, INCXS[ix],
                                     BETAS[be], INCYS[iy], ttrust,
                                     mftrust, 1.0, "-----" );
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[al], lda, INCXS[ix],
                                     BETAS[be], INCYS[iy], ttest,
                                     mftest,  t0,  pass );
#else
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[2*al], ALPHAS[2*al+1],
                                     lda, INCXS[ix], BETAS[2*be],
                                     BETAS[2*be+1], INCYS[iy], ttrust,
                                     mftrust, 1.0, "-----" );
                     (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                     ALPHAS[2*al], ALPHAS[2*al+1],
                                     lda, INCXS[ix], BETAS[2*be],
                                     BETAS[2*be+1], INCYS[iy], ttest,
                                     mftest,  t0,  pass );
#endif
                     (*NTESTS)++;
                  }
               }
            }
         }
      }
   }
}

void RuntbCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        NTRAN,
   enum ATLAS_TRANS           * TRANS,
   int                        NDIAG,
   enum ATLAS_DIAG            * DIAGS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        KL0,
   int                        KLN,
   int                        KLINC,
   int                        KU0,
   int                        KUN,
   int                        KUINC,
   int                        NINCX,
   int                        * INCXS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       ctran, cdiag, cuplo;
   int                        di, ires, ix, kkl, kku, lda, nn, ta, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

   if( ROUT == TBMV )
      (void) fprintf( stdout, "\n%s%s%s\n", "-----------------------------",
                      " TBMV ", "------------------------------" );
   else
      (void) fprintf( stdout, "\n%s%s%s\n", "-----------------------------",
                      " TBSV ", "------------------------------" );

   (void) fprintf( stdout, "%s%s",
                   "TST# UPLO TRAN DIAG    N    K  LDA INCX   TIME",
                   "  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== ==== ==== ==== ==== ==== ==== ==== ======",
                   " ====== ===== =====\n" );
   form = "%4d    %c    %c    %c %4d %4d %4d %4d %6.2f %6.1f %5.2f %5s\n";

   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasLower )
         {
            cuplo = 'L';

            for( kkl = KL0; kkl <= KLN; kkl += KLINC )
            {
               for( ta = 0; ta < NTRAN; ta++ )
               {
                  if(      TRANS[ta] == AtlasNoTrans   ) ctran = 'N';
                  else if( TRANS[ta] == AtlasTrans     ) ctran = 'T';
                  else if( TRANS[ta] == AtlasConjTrans ) ctran = 'C';

                  for( di = 0; di < NDIAG; di++ )
                  {
                     if( DIAGS[di] == AtlasUnit ) cdiag = 'U';
                     else                         cdiag = 'N';

                     for( ix = 0; ix < NINCX; ix++ )
                     {
                        if( LDA_IS_M ) lda = kkl+1;
                        else           lda = KLN+1;

                        if( ROUT == TBMV )
                           ires = tmvcase( ROUT, TEST, MFLOP, UPLOS[up],
                                           TRANS[ta], DIAGS[di], nn, kkl,
                                           lda, INCXS[ix], &ttrust, &ttest,
                                           &mftrust, &mftest );
                        else
                           ires = tsvcase( ROUT, TEST, MFLOP, UPLOS[up],
                                           TRANS[ta], DIAGS[di], nn, kkl,
                                           lda, INCXS[ix], &ttrust, &ttest,
                                           &mftrust, &mftest );

                        if(     !( TEST ) ) pass = "SKIP ";
                        else if( ires < 0 ) pass = "NoMEM";
                        else if( ires     ) pass = "PASS ";
                        else                pass = "FAIL ";

                        if( ires > 0 ) (*NPASSED)++;

                        if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                           t0 = mftest / mftrust;
                        else t0 = 0.0;

                        (void) fprintf( stdout, form, *NTESTS, cuplo, ctran,
                                        cdiag, nn, kkl, lda, INCXS[ix], ttrust,
                                        mftrust, 1.0, "-----" );
                        (void) fprintf( stdout, form, *NTESTS, cuplo, ctran,
                                        cdiag, nn, kkl, lda, INCXS[ix], ttest,
                                        mftest,  t0, pass );
                        (*NTESTS)++;
                     }
                  }
               }
            }
         }
         else
         {
            cuplo = 'U';

            for( kku = KU0; kku <= KUN; kku += KUINC )
            {
               for( ta = 0; ta < NTRAN; ta++ )
               {
                  if(      TRANS[ta] == AtlasNoTrans   ) ctran = 'N';
                  else if( TRANS[ta] == AtlasTrans     ) ctran = 'T';
                  else if( TRANS[ta] == AtlasConjTrans ) ctran = 'C';

                  for( di = 0; di < NDIAG; di++ )
                  {
                     if( DIAGS[di] == AtlasUnit ) cdiag = 'U';
                     else                         cdiag = 'N';

                     for( ix = 0; ix < NINCX; ix++ )
                     {
                        if( LDA_IS_M ) lda = 1+kku;
                        else           lda = 1+KUN;

                        if( ROUT == TBMV )
                           ires = tmvcase( ROUT, TEST, MFLOP, UPLOS[up],
                                          TRANS[ta], DIAGS[di], nn, kku,
                                           lda, INCXS[ix], &ttrust, &ttest,
                                           &mftrust, &mftest );
                        else
                           ires = tsvcase( ROUT, TEST, MFLOP, UPLOS[up],
                                           TRANS[ta], DIAGS[di], nn, kku,
                                           lda, INCXS[ix], &ttrust, &ttest,
                                           &mftrust, &mftest );

                        if(     !( TEST ) ) pass = "SKIP ";
                        else if( ires < 0 ) pass = "NoMEM";
                        else if( ires     ) pass = "PASS ";
                        else                pass = "FAIL ";

                        if( ires > 0 ) (*NPASSED)++;

                        if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                           t0 = mftest / mftrust;
                        else t0 = 0.0;

                        (void) fprintf( stdout, form, *NTESTS, cuplo, ctran,
                                        cdiag, nn, kku, lda, INCXS[ix], ttrust,
                                        mftrust, 1.0, "-----" );
                        (void) fprintf( stdout, form, *NTESTS, cuplo, ctran,
                                        cdiag, nn, kku, lda, INCXS[ix], ttest,
                                        mftest,  t0, pass );
                        (*NTESTS)++;
                     }
                  }
               }
            }
         }
      }
   }
}

void RuntpCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        NTRAN,
   enum ATLAS_TRANS           * TRANS,
   int                        NDIAG,
   enum ATLAS_DIAG            * DIAGS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NINCX,
   int                        * INCXS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       ctran, cdiag, cuplo;
   int                        nn, up, ta, di, ix, ires;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

   if( ROUT == TPMV )
      (void) fprintf( stdout, "\n%s%s%s\n", "------------------------",
                      " TPMV ", "-------------------------" );
   else
      (void) fprintf( stdout, "\n%s%s%s\n", "------------------------",
                      " TPSV ", "-------------------------" );
   (void) fprintf( stdout, "%s%s", "TST# UPLO TRAN DIAG    N INCX   TIME",
                   "  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s", "==== ==== ==== ==== ==== ==== ======",
                   " ====== ===== =====\n" );
   form = "%4d    %c    %c    %c %4d %4d %6.2f %6.1f %5.2f %5s\n";

   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasUpper ) cuplo = 'U';
         else                          cuplo = 'L';

         for( ta = 0; ta < NTRAN; ta++ )
         {
            if(      TRANS[ta] == AtlasNoTrans   ) ctran = 'N';
            else if( TRANS[ta] == AtlasTrans     ) ctran = 'T';
            else if( TRANS[ta] == AtlasConjTrans ) ctran = 'C';

            for( di = 0; di < NDIAG; di++ )
            {
               if( DIAGS[di] == AtlasUnit ) cdiag = 'U';
               else                         cdiag = 'N';

               for( ix = 0; ix < NINCX; ix++ )
               {
                  if( ROUT == TPMV )
                     ires = tmvcase( ROUT, TEST, MFLOP, UPLOS[up], TRANS[ta],
                                     DIAGS[di], nn, 0, 1, INCXS[ix], &ttrust,
                                     &ttest, &mftrust, &mftest );
                  else
                     ires = tsvcase( ROUT, TEST, MFLOP, UPLOS[up], TRANS[ta],
                                     DIAGS[di], nn, 0, 1, INCXS[ix], &ttrust,
                                     &ttest, &mftrust, &mftest );

                  if(     !( TEST ) ) pass = "SKIP ";
                  else if( ires < 0 ) pass = "NoMEM";
                  else if( ires     ) pass = "PASS ";
                  else                pass = "FAIL ";

                  if( ires > 0 ) (*NPASSED)++;

                  if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                     t0 = mftest / mftrust;
                  else t0 = 0.0;

                  (void) fprintf( stdout, form, *NTESTS, cuplo, ctran,
                                  cdiag, nn, INCXS[ix], ttrust, mftrust,
                                  1.0, "-----" );
                  (void) fprintf( stdout, form, *NTESTS, cuplo, ctran,
                                  cdiag, nn, INCXS[ix], ttest,  mftest,
                                  t0, pass );
                  (*NTESTS)++;
               }
            }
         }
      }
   }
}

void RuntrCase
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        NTRAN,
   enum ATLAS_TRANS           * TRANS,
   int                        NDIAG,
   enum ATLAS_DIAG            * DIAGS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NINCX,
   int                        * INCXS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       ctran, cdiag, cuplo;
   int                        nn, up, ta, di, ix, ires, lda;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

   if( ROUT == TRMV )
      (void) fprintf( stdout, "\n%s%s%s\n", "--------------------------",
                      " TRMV ", "----------------------------" );
   else
      (void) fprintf( stdout, "\n%s%s%s\n", "--------------------------",
                      " TRSV ", "----------------------------" );

   (void) fprintf( stdout, "%s%s", "TST# UPLO TRAN DIAG    N  LDA INCX   TIME",
                   "  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s", "==== ==== ==== ==== ==== ==== ==== ======",
                   " ====== ===== =====\n" );
   form = "%4d    %c    %c    %c %4d %4d %4d %6.2f %6.1f %5.2f %5s\n";

   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasUpper ) cuplo = 'U';
         else                          cuplo = 'L';

         for( ta = 0; ta < NTRAN; ta++ )
         {
            if(      TRANS[ta] == AtlasNoTrans   ) ctran = 'N';
            else if( TRANS[ta] == AtlasTrans     ) ctran = 'T';
            else if( TRANS[ta] == AtlasConjTrans ) ctran = 'C';

            for( di = 0; di < NDIAG; di++ )
            {
               if( DIAGS[di] == AtlasUnit ) cdiag = 'U';
               else                         cdiag = 'N';

               for( ix = 0; ix < NINCX; ix++ )
               {
                  if( LDA_IS_M ) lda = nn;
                  else           lda = NN;

                  if( ROUT == TRMV )
                     ires = tmvcase( ROUT, TEST, MFLOP, UPLOS[up], TRANS[ta],
                                     DIAGS[di], nn, 0, lda, INCXS[ix], &ttrust,
                                     &ttest, &mftrust, &mftest );
                  else
                     ires = tsvcase( ROUT, TEST, MFLOP, UPLOS[up], TRANS[ta],
                                     DIAGS[di], nn, 0, lda, INCXS[ix], &ttrust,
                                     &ttest, &mftrust, &mftest );

                  if(     !( TEST ) ) pass = "SKIP ";
                  else if( ires < 0 ) pass = "NoMEM";
                  else if( ires     ) pass = "PASS ";
                  else                pass = "FAIL ";

                  if( ires > 0 ) (*NPASSED)++;

                  if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                     t0 = mftest / mftrust;
                  else t0 = 0.0;

                  (void) fprintf( stdout, form, *NTESTS, cuplo, ctran, cdiag,
                                  nn, lda, INCXS[ix], ttrust, mftrust, 1.0,
                                  "-----" );
                  (void) fprintf( stdout, form, *NTESTS, cuplo, ctran, cdiag,
                                  nn, lda, INCXS[ix], ttest,  mftest,  t0,
                                  pass );
                  (*NTESTS)++;
               }
            }
         }
      }
   }
}

void Rungr1Case
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        M0,
   int                        MN,
   int                        MINC,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   int                        al, ires, ix, iy, lda, m, mm, msame=0, n, nn;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

   if( M0 == -1 ) { M0 = MN = MINC = NN; msame = 1; }

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "------------------------------ ", "GER",
                   " -----------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST#     M     N ALPHA INCX INCY   LDA   TIME",
                   "  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== ===== ===== ===== ==== ==== ===== ======",
                   " ====== ===== =====\n" );
   form = "%4d %5d %5d %5.1f %4d %4d %5d %6.2f %6.1f %5.2f %5s\n";
#else
   if( ROUT == GERC )
      (void) fprintf( stdout, "\n%s%s%s\n",
                      "-------------------------------- ", "GERC",
                      " --------------------------------" );
   else
      (void) fprintf( stdout, "\n%s%s%s\n",
                      "-------------------------------- ", "GERU",
                      " --------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST#     M     N       ALPHA INCX INCY   LDA",
                   "   TIME  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== ===== ===== ===== ===== ==== ==== =====",
                   " ====== ====== ===== =====\n" );
   form = "%4d %5d %5d %5.1f %5.1f %4d %4d %5d %6.2f %6.1f %5.2f %5s\n";
#endif
   for( mm = M0; mm <= MN; mm += MINC )
   {
      for( nn = N0; nn <= NN; nn += NINC )
      {
         n = nn;
         if( msame ) m = n;
         else        m = mm;

         for( iy = 0; iy < NINCY; iy++ )
         {
            for( ix = 0; ix < NINCX; ix++ )
            {
               for( al = 0; al < NALPHA; al++ )
               {
                  if( LDA_IS_M ) lda = m; else lda = MN;
#ifdef TREAL
                  ires = gr1case( ROUT, TEST, MFLOP, m, n, ALPHAS[al],
                                  INCXS[ix], INCYS[iy], lda, &ttrust,
                                  &ttest, &mftrust, &mftest );
#else
                  ires = gr1case( ROUT, TEST, MFLOP, m, n, ALPHAS+2*al,
                                  INCXS[ix], INCYS[iy], lda, &ttrust,
                                  &ttest, &mftrust, &mftest );
#endif
                  if(     !( TEST ) ) pass = "SKIP ";
                  else if( ires < 0 ) pass = "NoMEM";
                  else if( ires     ) pass = "PASS ";
                  else                pass = "FAIL ";

                  if( ires > 0 ) (*NPASSED)++;

                  if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                     t0 = mftest / mftrust;
                  else t0 = 0.0;
#ifdef TREAL
                  (void) fprintf( stdout, form, *NTESTS, m, n, ALPHAS[al],
                                  INCXS[ix], INCYS[iy], lda, ttrust,
                                  mftrust, 1.0, "-----" );
                  (void) fprintf( stdout, form, *NTESTS, m, n, ALPHAS[al],
                                  INCXS[ix], INCYS[iy], lda, ttest,
                                  mftest, t0, pass );
#else
                  (void) fprintf( stdout, form, *NTESTS, m, n, ALPHAS[2*al],
                                  ALPHAS[2*al+1], INCXS[ix], INCYS[iy],
                                  lda, ttrust, mftrust, 1.0, "-----" );
                  (void) fprintf( stdout, form, *NTESTS, m, n, ALPHAS[2*al],
                                  ALPHAS[2*al+1], INCXS[ix], INCYS[iy],
                                  lda, ttest, mftest, t0, pass );
#endif
                  (*NTESTS)++;
               }
            }
         }
      }
   }
}

void Runsp1Case
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NINCX,
   int                        * INCXS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       cuplo;
   int                        al, ires, ix, nn, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n", "----------------------- ", "SPR",
                   " ------------------------" );
#else
   (void) fprintf( stdout, "\n%s%s%s\n", "----------------------- ", "HPR",
                   " ------------------------" );
#endif
   (void) fprintf( stdout, "%s%s", "TST# UPLO     N ALPHA INCX   TIME",
                   "  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s", "==== ==== ===== ===== ==== ======",
                   " ====== ===== =====\n" );
   form = "%4d    %c %5d %5.1f %4d %6.2f %6.1f %5.2f %5s\n";

   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasUpper ) cuplo = 'U';
         else                          cuplo = 'L';

         for( ix = 0; ix < NINCX; ix++ )
         {
            for( al = 0; al < NALPHA; al++ )
            {
#ifdef TREAL
               ires = sr1case( ROUT, TEST, MFLOP, UPLOS[up], nn, ALPHAS[al],
                               INCXS[ix], 1, &ttrust, &ttest, &mftrust,
                               &mftest );
#else
               ires = sr1case( ROUT, TEST, MFLOP, UPLOS[up], nn, ALPHAS[2*al],
                               INCXS[ix], 1, &ttrust, &ttest, &mftrust,
                               &mftest );
#endif
               if(     !( TEST ) ) pass = "SKIP ";
               else if( ires < 0 ) pass = "NoMEM";
               else if( ires     ) pass = "PASS ";
               else                pass = "FAIL ";

               if( ires > 0 ) (*NPASSED)++;

               if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                  t0 = mftest / mftrust;
               else t0 = 0.0;
#ifdef TREAL
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[al],
                               INCXS[ix], ttrust, mftrust, 1.0, "-----" );
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[al],
                               INCXS[ix], ttest,  mftest,  t0,  pass );
#else
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[2*al],
                               INCXS[ix], ttrust, mftrust, 1.0, "-----" );
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[2*al],
                               INCXS[ix], ttest,  mftest,  t0,  pass );
#endif
               (*NTESTS)++;
            }
         }
      }
   }
}

void Runsr1Case
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NINCX,
   int                        * INCXS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       cuplo;
   int                        al, ires, ix, lda, nn, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n", "--------------------------- ", "SYR",
                   " --------------------------" );
#else
   (void) fprintf( stdout, "\n%s%s%s\n", "--------------------------- ", "HER",
                   " --------------------------" );
#endif
   (void) fprintf( stdout, "%s%s", "TST# UPLO     N ALPHA INCX   LDA   TIME",
                   "  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s", "==== ==== ===== ===== ==== ===== ======",
                   " ====== ===== =====\n" );
   form = "%4d    %c %5d %5.1f %4d %5d %6.2f %6.1f %5.2f %5s\n";

   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasUpper ) cuplo = 'U';
         else                          cuplo = 'L';

         for( ix = 0; ix < NINCX; ix++ )
         {
            for( al = 0; al < NALPHA; al++ )
            {
               if( LDA_IS_M ) lda = nn; else lda = NN;
#ifdef TREAL
               ires = sr1case( ROUT, TEST, MFLOP, UPLOS[up], nn, ALPHAS[al],
                               INCXS[ix], lda, &ttrust, &ttest, &mftrust,
                               &mftest );
#else
               ires = sr1case( ROUT, TEST, MFLOP, UPLOS[up], nn, ALPHAS[2*al],
                               INCXS[ix], lda, &ttrust, &ttest, &mftrust,
                               &mftest );
#endif
               if(     !( TEST ) ) pass = "SKIP ";
               else if( ires < 0 ) pass = "NoMEM";
               else if( ires     ) pass = "PASS ";
               else                pass = "FAIL ";

               if( ires > 0 ) (*NPASSED)++;

               if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                  t0 = mftest / mftrust;
               else t0 = 0.0;
#ifdef TREAL
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[al],
                               INCXS[ix], lda, ttrust, mftrust, 1.0,
                               "-----" );
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[al],
                               INCXS[ix], lda, ttest,  mftest,  t0,
                               pass );
#else
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[2*al],
                               INCXS[ix], lda, ttrust, mftrust, 1.0,
                               "-----" );
               (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[2*al],
                               INCXS[ix], lda, ttest,  mftest,  t0,
                               pass );
#endif
               (*NTESTS)++;
            }
         }
      }
   }
}

void Runsp2Case
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       cuplo;
   int                        al, ires, ix, iy, nn, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

#ifdef TREAL
         (void) fprintf( stdout, "\n%s%s%s\n",
                         "-------------------------- ", "SPR2",
                         " -------------------------" );
         (void) fprintf( stdout, "%s%s",
                         "TST# UPLO     N ALPHA INCX INCY   TIME",
                         "  MFLOP  SpUp  TEST\n" );
         (void) fprintf( stdout, "%s%s",
                         "==== ==== ===== ===== ==== ==== ======",
                         " ====== ===== =====\n" );
         form = "%4d    %c %5d %5.1f %4d %4d %6.2f %6.1f %5.2f %5s\n";
#else
         (void) fprintf( stdout, "\n%s%s%s\n",
                         "----------------------------- ", "HPR2",
                         " ----------------------------" );
         (void) fprintf( stdout, "%s%s",
                         "TST# UPLO     N       ALPHA INCX INCY",
                         "   TIME  MFLOP  SpUp  TEST\n");
         (void) fprintf( stdout, "%s%s",
                         "==== ==== ===== ===== ===== ==== ====",
                         " ====== ====== ===== =====\n");
         form = "%4d    %c %5d %5.1f %5.1f %4d %4d %6.2f %6.1f %5.2f %5s\n";
#endif
   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasUpper ) cuplo = 'U';
         else                          cuplo = 'L';

         for( iy = 0; iy < NINCY; iy++ )
         {
            for( ix = 0; ix < NINCX; ix++ )
            {
               for( al = 0; al < NALPHA; al++ )
               {
#ifdef TREAL
                  ires = sr2case( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                  ALPHAS[al], INCXS[ix], INCYS[iy],
                                  1, &ttrust, &ttest, &mftrust, &mftest );
#else
                  ires = sr2case( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                  ALPHAS+2*al, INCXS[ix], INCYS[iy],
                                  1, &ttrust, &ttest, &mftrust, &mftest );
#endif
                  if(     !( TEST ) ) pass = "SKIP ";
                  else if( ires < 0 ) pass = "NoMEM";
                  else if( ires     ) pass = "PASS ";
                  else                pass = "FAIL ";

                  if( ires > 0 ) (*NPASSED)++;

                  if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                     t0 = mftest / mftrust;
                  else t0 = 0.0;
#ifdef TREAL
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[al],
                                  INCXS[ix], INCYS[iy], ttrust, mftrust, 1.0,
                                  "-----" );
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn, ALPHAS[al],
                                  INCXS[ix], INCYS[iy], ttest, mftest, t0,
                                  pass );
#else
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                  ALPHAS[2*al], ALPHAS[2*al+1], INCXS[ix],
                                  INCYS[iy], ttrust, mftrust, 1.0, "-----" );
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                  ALPHAS[2*al], ALPHAS[2*al+1], INCXS[ix],
                                  INCYS[iy], ttest, mftest, t0, pass );
#endif
                  (*NTESTS)++;
               }
            }
         }
      }
   }
}

void Runsr2Case
(
   enum LVL2_ROUT             ROUT,
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        * NPASSED,
   int                        * NTESTS
)
{
   char                       cuplo;
   int                        al, ires, ix, iy, lda, nn, up;
   double                     t0, mftrust, mftest, ttrust, ttest;
   char                       * pass, * form;

#ifdef TREAL
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "---------------------------- ", "SYR2",
                   " -----------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UPLO     N ALPHA INCX INCY   LDA   TIME",
                   "  MFLOP  SpUp  TEST\n" );
   (void) fprintf( stdout, "%s%s",
                   "==== ==== ===== ===== ==== ==== ===== ======",
                   " ====== ===== =====\n" );
   form = "%4d    %c %5d %5.1f %4d %4d %5d %6.2f %6.1f %5.2f %5s\n";
#else
   (void) fprintf( stdout, "\n%s%s%s\n",
                   "------------------------------- ", "HER2",
                   " --------------------------------" );
   (void) fprintf( stdout, "%s%s",
                   "TST# UPLO     N       ALPHA INCX INCY   LDA",
                   "   TIME  MFLOP  SpUp  TEST\n");
   (void) fprintf( stdout, "%s%s",
                   "==== ==== ===== ===== ===== ==== ==== =====",
                   " ====== ====== ===== =====\n");
   form = "%4d    %c %5d %5.1f %5.1f %4d %4d %5d %6.2f %6.1f %5.2f %5s\n";
#endif
   for( nn = N0; nn <= NN; nn += NINC )
   {
      for( up = 0; up < NUPLO; up++ )
      {
         if( UPLOS[up] == AtlasUpper ) cuplo = 'U';
         else                          cuplo = 'L';

         for( iy = 0; iy < NINCY; iy++ )
         {
            for( ix = 0; ix < NINCX; ix++ )
            {
               for( al = 0; al < NALPHA; al++ )
               {
                  if( LDA_IS_M ) lda = nn; else lda = NN;
#ifdef TREAL
                  ires = sr2case( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                  ALPHAS[al], INCXS[ix], INCYS[iy], lda,
                                  &ttrust, &ttest, &mftrust, &mftest );
#else
                  ires = sr2case( ROUT, TEST, MFLOP, UPLOS[up], nn,
                                  ALPHAS+2*al, INCXS[ix], INCYS[iy], lda,
                                  &ttrust, &ttest, &mftrust, &mftest );
#endif
                  if(     !( TEST ) ) pass = "SKIP ";
                  else if( ires < 0 ) pass = "NoMEM";
                  else if( ires     ) pass = "PASS ";
                  else                pass = "FAIL ";

                  if( ires > 0 ) (*NPASSED)++;

                  if( ( mftrust > 0.0 ) && ( mftest > 0.0 ) )
                     t0 = mftest / mftrust;
                  else t0 = 0.0;
#ifdef TREAL
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                  ALPHAS[al], INCXS[ix], INCYS[iy],
                                  lda, ttrust, mftrust, 1.0, "-----" );
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                  ALPHAS[al], INCXS[ix], INCYS[iy],
                                  lda, ttest, mftest, t0, pass );
#else
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                  ALPHAS[2*al], ALPHAS[2*al+1], INCXS[ix],
                                  INCYS[iy], lda, ttrust, mftrust, 1.0,
                                  "-----" );
                  (void) fprintf( stdout, form, *NTESTS, cuplo, nn,
                                  ALPHAS[2*al], ALPHAS[2*al+1], INCXS[ix],
                                  INCYS[iy], lda, ttest, mftest, t0, pass );
#endif
                  (*NTESTS)++;
               }
            }
         }
      }
   }
}

void RunCases
(
   int                        TEST,
   int                        MFLOP,
   int                        LDA_IS_M,
   int                        NUPLO,
   enum ATLAS_UPLO            * UPLOS,
   int                        NTRAN,
   enum ATLAS_TRANS           * TRANS,
   int                        NDIAG,
   enum ATLAS_DIAG            * DIAGS,
   int                        M0,
   int                        MN,
   int                        MINC,
   int                        N0,
   int                        NN,
   int                        NINC,
   int                        KL0,
   int                        KLN,
   int                        KLINC,
   int                        KU0,
   int                        KUN,
   int                        KUINC,
   int                        NALPHA,
   TYPE                       * ALPHAS,
   int                        NBETA,
   TYPE                       * BETAS,
   int                        NINCX,
   int                        * INCXS,
   int                        NINCY,
   int                        * INCYS,
   int                        NROUT,
   enum LVL2_ROUT             * ROUTS
)
{
   int                        ro, ntests=0, np=0;

   for( ro = 0; ro < NROUT; ro++ )
   {
      if( ROUTS[ro] == GBMV  )
      {
         RungbmCase( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NTRAN, TRANS, M0, MN,
                     MINC, N0, NN, NINC, KL0, KLN, KLINC, KU0, KUN, KUINC,
                     NALPHA, ALPHAS, NBETA, BETAS, NINCX, INCXS, NINCY, INCYS,
                     &np, &ntests );
      }
      else if( ROUTS[ro] == GEMV  )
      {
         RungemCase( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NTRAN, TRANS, M0, MN,
                     MINC, N0, NN, NINC,
                     NALPHA, ALPHAS, NBETA, BETAS, NINCX, INCXS, NINCY, INCYS,
                     &np, &ntests );
      }
      else if( ROUTS[ro] == SBMV )
      {
         RunsbCase( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NUPLO, UPLOS, N0, NN,
                    NINC, KL0, KLN, KLINC, KU0, KUN, KUINC, NALPHA, ALPHAS,
                    NBETA, BETAS, NINCX, INCXS, NINCY, INCYS, &np, &ntests );
      }
      else if( ROUTS[ro] == SPMV )
      {
         RunspCase( ROUTS[ro], TEST, MFLOP,           NUPLO, UPLOS, N0, NN,
                    NINC,                                   NALPHA, ALPHAS,
                    NBETA, BETAS, NINCX, INCXS, NINCY, INCYS, &np, &ntests );
      }
      else if( ROUTS[ro] == SYMV )
      {
         RunsyCase( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NUPLO, UPLOS, N0, NN,
                    NINC,                                   NALPHA, ALPHAS,
                    NBETA, BETAS, NINCX, INCXS, NINCY, INCYS, &np, &ntests );
      }
      else if( ( ROUTS[ro] == TBMV ) || ( ROUTS[ro] == TBSV ) )
      {
         RuntbCase( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NUPLO, UPLOS, NTRAN,
                    TRANS, NDIAG, DIAGS, N0, NN, NINC, KL0, KLN, KLINC,
                    KU0, KUN, KUINC, NINCX, INCXS, &np, &ntests );
      }
      else if( ( ROUTS[ro] == TPMV ) || ( ROUTS[ro] == TPSV ) )
      {
         RuntpCase( ROUTS[ro], TEST, MFLOP,           NUPLO, UPLOS, NTRAN,
                    TRANS, NDIAG, DIAGS, N0, NN, NINC,
                                     NINCX, INCXS, &np, &ntests );
      }
      else if( ( ROUTS[ro] == TRMV ) || ( ROUTS[ro] == TRSV ) )
      {
         RuntrCase( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NUPLO, UPLOS, NTRAN,
                    TRANS, NDIAG, DIAGS, N0, NN, NINC,
                                     NINCX, INCXS, &np, &ntests );
      }
      else if( ( ROUTS[ro] == GERC ) || ( ROUTS[ro] == GERU ) )
      {
         Rungr1Case( ROUTS[ro], TEST, MFLOP, LDA_IS_M, M0, MN, MINC, N0, NN,
                     NINC, NALPHA, ALPHAS, NINCX, INCXS, NINCY, INCYS, &np,
                     &ntests );
      }
      else if( ROUTS[ro] == SPR )
      {
         Runsp1Case( ROUTS[ro], TEST, MFLOP,           NUPLO, UPLOS, N0, NN,
                     NINC, NALPHA, ALPHAS, NINCX, INCXS, &np, &ntests );
      }
      else if( ROUTS[ro] == SYR )
      {
         Runsr1Case( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NUPLO, UPLOS, N0, NN,
                     NINC, NALPHA, ALPHAS, NINCX, INCXS, &np, &ntests );
      }
      else if( ROUTS[ro] == SPR2 )
      {
         Runsp2Case( ROUTS[ro], TEST, MFLOP,           NUPLO, UPLOS, N0, NN,
                     NINC, NALPHA, ALPHAS, NINCX, INCXS, NINCY, INCYS, &np,
                     &ntests );
      }
      else if( ROUTS[ro] == SYR2 )
      {
         Runsr2Case( ROUTS[ro], TEST, MFLOP, LDA_IS_M, NUPLO, UPLOS, N0, NN,
                     NINC, NALPHA, ALPHAS, NINCX, INCXS, NINCY, INCYS, &np,
                     &ntests );
      }
   }

   if( TEST )
      (void) fprintf( stdout, "\n%d tests run, %d passed\n\n", ntests, np );
}

void PrintUsage( char * nam )
{
   (void) fprintf( stderr, "ATLAS usage:\n" );
   (void) fprintf( stderr, "    %s [-options ...]\n\n", nam );
   (void) fprintf( stderr, "where options include:\n" );

   (void) fprintf( stderr, "   -h                                   " );
   (void) fprintf( stderr, ". print this message                   \n" );

   (void) fprintf( stderr, "   -R <rout>                            " );
   (void) fprintf( stderr, ". select  one  or all routines to test.\n" );
   (void) fprintf( stderr, "                                        " );
#ifdef TREAL
   (void) fprintf( stderr, "  rout must be in {gbmv,gemv,sbmv,spmv,\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  symv,tbmv,tpmv,trmv,tbsv,tpsv,trsv,  \n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  ger,spr,syr,spr2,syr2, all}.         \n" );
#else
   (void) fprintf( stderr, "  rout must be in {gbmv,gemv,hbmv,hpmv,\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  hemv,tbmv,tpmv,trmv,tbsv,tpsv,trsv,  \n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  gerc,geru,hpr,her,hpr2,her2, all}.   \n" );
#endif
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Default: -R gemv.  Ex: -R tpmv       \n" );

   (void) fprintf( stderr, "   -R <nrout> <rout1> ... <routN>       " );
   (void) fprintf( stderr, ". same as above for more than one rout-\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  tine. Ex: -R 3 gemv tpmv tbsv        \n" );

   (void) fprintf( stderr, "   -U <nuplo>  L/U                      " );
   (void) fprintf( stderr, ". select values for the UPLO parameter.\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Default: -U 1 L. Ex: -U 2 L U        \n" );

   (void) fprintf( stderr, "   -A <ntrans> n/t/c                    " );
   (void) fprintf( stderr, ". select values of the TRANS parameter.\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Default: -A 1 n. Ex: -A 2 n t        \n" );

   (void) fprintf( stderr, "   -D <ndiags> N/U                      " );
   (void) fprintf( stderr, ". select values for the DIAG parameter.\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Default: -D 1 N. Ex: -Diag 2 N U     \n" );

   (void) fprintf( stderr, "   -m <m>                               " );
   (void) fprintf( stderr, ". select one value for the parameter M.\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Ex: -m 100                           \n" );

   (void) fprintf( stderr, "   -n <n>                               " );
   (void) fprintf( stderr, ". same as above for the parameter N.   \n" );

   (void) fprintf( stderr, "   -p <kl>                              " );
   (void) fprintf( stderr, ". same as above for the parameter KL.  \n" );

   (void) fprintf( stderr, "   -q <ku>                              " );
   (void) fprintf( stderr, ". same as above for the parameter KU.  \n" );

   (void) fprintf( stderr, "   -M <m1>  <mN>  <minc>                " );
   (void) fprintf( stderr, ". select the values of M, from m1 to mN\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  by increment of minc. m1 > 0.        \n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Ex: -M 100 1000 100                  \n" );

   (void) fprintf( stderr, "   -N <n1>  <nN>  <ninc>                " );
   (void) fprintf( stderr, ". same as above for the values of N.   \n" );

   (void) fprintf( stderr, "   -P <kl1> <kuN> <klinc>               " );
   (void) fprintf( stderr, ". same as above for the values of KL.  \n" );

   (void) fprintf( stderr, "   -Q <ku1> <kuN> <kuinc>               " );
   (void) fprintf( stderr, ". same as above for the values of KU.  \n" );

#ifdef TREAL
   (void) fprintf( stderr, "   -a <nalphas> <a1> ... <aN>           " );
   (void) fprintf( stderr, ". select the values of ALPHA.  Default:\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  -a 1 1.0. Ex: -a 3 -1.0 0.0 1.0      \n" );
#else
   (void) fprintf( stderr, "   -a <nalphas> <a1r> <a1i> ...  <aNi> " );
   (void) fprintf( stderr, ". select the values of ALPHA, where a1r\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  and  a1i  are the  real and imaginary\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  parts of a1. Default: -a 1 1.0 0.0   \n" );
#endif
   (void) fprintf( stderr, "   -b <nbetas>  <beta1>  ... <betaN>    " );
   (void) fprintf( stderr, ". same as above for the parameter BETA.\n" );

   (void) fprintf( stderr, "   -X <nincXs>  <incX0>  ... <incX1>    " );
   (void) fprintf( stderr, ". select  the  values  of the increment\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  INCX. Default: -X 1 1; Ex: -X 2 1 -1 \n" );

   (void) fprintf( stderr, "   -Y <nincYs>  <incY0>  ... <incYN>    " );
   (void) fprintf( stderr, ". same as above for the the values of  \n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  INCY.                                \n" );

   (void) fprintf( stderr, "   -d                                   " );
   (void) fprintf( stderr, ". use smallest possible leading  dimen-\n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  sion for the array A.                \n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Default: max( mN, nN ).              \n" );

   (void) fprintf( stderr, "   -T <0/1>                             " );
   (void) fprintf( stderr, ". disable/enable computational check.  \n" );
   (void) fprintf( stderr, "                                        " );
   (void) fprintf( stderr, "  Default: -T 1                        \n" );

   (void) fprintf( stderr, "   -F <mflops>                          " );
   (void) fprintf( stderr, ". perform at least mflops per measure. \n" );

   (void) fprintf( stderr, "\n" );
   exit( -1 );
}

void GetFlags
(
   int                        NARGS,
   char                       * ARGS[],
   int                        * NROUT,
   enum LVL2_ROUT             ** ROUTS,
   int                        * TEST,
   int                        * LDA_IS_M,
   int                        * MFLOP,
   int                        * NUPLO,
   enum ATLAS_UPLO            ** UPLOS,
   int                        * NTRAN,
   enum ATLAS_TRANS           ** TRANS,
   int                        * NDIAG,
   enum ATLAS_DIAG            ** DIAGS,
   int                        * M0,
   int                        * MN,
   int                        * MINC,
   int                        * N0,
   int                        * NN,
   int                        * NINC,
   int                        * KL0,
   int                        * KLN,
   int                        * KLINC,
   int                        * KU0,
   int                        * KUN,
   int                        * KUINC,
   int                        * NALPHA,
   TYPE                       ** ALPHAS,
   int                        * NBETA,
   TYPE                       ** BETAS,
   int                        * NINCX,
   int                        ** INCXS,
   int                        * NINCY,
   int                        ** INCYS
)
{
   char                       ch;
   int                        i = 1, j;
/*
 * Set up defaults
 */
   *NROUT    = -1;                         /* No routine to be tested */
   *TEST     = 1;                               /* Enable the testing */
   *LDA_IS_M = 0;    /* Leading dimension chosen as max testing value */
   *MFLOP    = 0;                /* smallest number of flops possible */
                                            /* Default bandwidth is 1 */
   *KL0      = *KU0   = *KLN   = *KUN   = *KLINC = *KUINC = 1;

   *NUPLO    = *NTRAN = *NDIAG = -1;
   *M0       = *N0    = -1;
   *NALPHA   = *NBETA = -1;
   *NINCX    = *NINCY = -1;

   for( i = 1; i < NARGS; )
   {
      if( ARGS[i][0] != '-' ) PrintUsage( ARGS[0] );

      switch( ARGS[i++][1] )
      {
         case 'h':
            PrintUsage( ARGS[0] );
            break;
         case 'T':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *TEST = atoi( ARGS[i++] );
            break;
         case 'd':
            *LDA_IS_M  = atoi( ARGS[i++] );
            break;
         case 'F':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *MFLOP = atoi( ARGS[i++] );
            if( *MFLOP < 0      ) PrintUsage( ARGS[0] );
            break;

         case 'U':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NUPLO = atoi( ARGS[i++] );
            if( *NUPLO <= 0     ) PrintUsage( ARGS[0] );
            *UPLOS = (enum ATLAS_UPLO *)malloc( *NUPLO *
                                                sizeof( enum ATLAS_UPLO ) );
            ATL_assert( *UPLOS );
            for( j = 0; j != *NUPLO; j++)
            {
               if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
               ch = *ARGS[i++];
               if(      ch == 'u' || ch == 'U' ) (*UPLOS)[j] = AtlasUpper;
               else if( ch == 'l' || ch == 'L' ) (*UPLOS)[j] = AtlasLower;
               else PrintUsage( ARGS[0] );
            }
            break;
         case 'D':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NDIAG = atoi( ARGS[i++] );
            if( *NDIAG <= 0     ) PrintUsage( ARGS[0] );
            *DIAGS = (enum ATLAS_DIAG *)malloc( *NDIAG *
                                                sizeof( enum ATLAS_DIAG ) );
            ATL_assert( *DIAGS );
            for( j = 0; j != *NDIAG; j++ )
            {
               if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
               ch = *ARGS[i++];
               if(      ch == 'u' || ch == 'U' ) (*DIAGS)[j] = AtlasUnit;
               else if( ch == 'n' || ch == 'N' ) (*DIAGS)[j] = AtlasNonUnit;
               else PrintUsage( ARGS[0] );
            }
            break;
         case 'A':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NTRAN = atoi(ARGS[i++]);
            if( *NTRAN <= 0     ) PrintUsage( ARGS[0] );
            *TRANS = (enum ATLAS_TRANS *)malloc( *NTRAN *
                                                 sizeof( enum ATLAS_TRANS ) );
            ATL_assert( *TRANS );
            for( j = 0; j != *NTRAN; j++ )
            {
               if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
               ch = *ARGS[i++];
               if(      ch == 'n' || ch == 'N' ) (*TRANS)[j] = AtlasNoTrans;
               else if( ch == 't' || ch == 'T' ) (*TRANS)[j] = AtlasTrans;
               else if( ch == 'c' || ch == 'C' ) (*TRANS)[j] = AtlasConjTrans;
               else PrintUsage( ARGS[0] );
            }
            break;

         case 'M':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *M0 = atoi( ARGS[i++] );
            if( *M0 < 0         ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *MN = atoi( ARGS[i++] );
            if( *MN < 0         ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *MINC = atoi( ARGS[i++] );
            if( *MINC <= 0      ) PrintUsage( ARGS[0] );
            break;
         case 'm':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *M0 = *MN = atoi( ARGS[i++] ); *MINC = 1;
            if( *M0 <= 0        ) PrintUsage( ARGS[0] );
            break;
         case 'N':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *N0 = atoi( ARGS[i++] );
            if( *N0 < 0         ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NN = atoi( ARGS[i++] );
            if( *NN < 0         ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NINC = atoi( ARGS[i++] );
            if( *NINC <= 0      ) PrintUsage( ARGS[0] );
            break;
         case 'n':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *N0 = *NN = atoi( ARGS[i++] ); *NINC = 1;
            if( *N0 < 0         ) PrintUsage( ARGS[0] );
            break;
         case 'P':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KL0 = atoi( ARGS[i++] );
            if( *KL0 < 0        ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KLN = atoi( ARGS[i++] );
            if( *KLN < 0        ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KLINC = atoi( ARGS[i++] );
            if( *KLINC <= 0     ) PrintUsage( ARGS[0] );
            break;
         case 'p':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KL0 = *KLN = atoi( ARGS[i++] ); *KLINC = 1;
            if( *KL0 <= 0       ) PrintUsage( ARGS[0] );
            break;
         case 'Q':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KU0 = atoi( ARGS[i++] );
            if( *KU0 < 0        ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KUN = atoi( ARGS[i++] );
            if( *KUN < 0        ) PrintUsage( ARGS[0] );
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KUINC = atoi( ARGS[i++] );
            if( *KUINC <= 0     ) PrintUsage( ARGS[0] );
            break;
         case 'q':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *KU0 = *KUN = atoi( ARGS[i++] ); *KUINC = 1;
            if( *KU0 <= 0       ) PrintUsage( ARGS[0] );
            break;

         case 'a':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NALPHA = atoi( ARGS[i++] );
            if( *NALPHA <= 0    ) PrintUsage( ARGS[0] );
            *ALPHAS = (TYPE *)malloc( ATL_MulBySize( *NALPHA ) );
            ATL_assert( *ALPHAS );
            for( j = 0; j < (*NALPHA SHIFT); j++ )
            {
               if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
               (*ALPHAS)[j] = (TYPE)atof( ARGS[i++] );
            }
            break;
         case 'b':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NBETA  = atoi( ARGS[i++] );
            if( *NBETA <= 0     ) PrintUsage( ARGS[0] );
            *BETAS  = (TYPE *)malloc( ATL_MulBySize( *NBETA ) );
            ATL_assert( *BETAS );
            for( j = 0; j < (*NBETA SHIFT); j++ )
            {
               if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
               (*BETAS)[j] = (TYPE)atof( ARGS[i++] );
            }
            break;

         case 'X':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NINCX = atoi( ARGS[i++] );
            if( *NINCX <= 0     ) PrintUsage( ARGS[0] );
            *INCXS = (int *)malloc( *NINCX * sizeof( int ) );
            ATL_assert( *INCXS );
            for( j = 0; j < *NINCX; j++ )
            {
               if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
               (*INCXS)[j] = atoi( ARGS[i++] );
            }
            break;
         case 'Y':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
            *NINCY = atoi( ARGS[i++] );
            if( *NINCY <= 0     ) PrintUsage( ARGS[0] );
            *INCYS = (int *)malloc( *NINCY * sizeof( int ) );
            ATL_assert( *INCYS );
            for( j = 0; j < *NINCY; j++ )
            {
               if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );
               (*INCYS)[j] = atoi( ARGS[i++] );
            }
            break;

         case 'R':
            if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );

            if( ( strcmp( ARGS[i], "ALL"  ) == 0 ) ||
                ( strcmp( ARGS[i], "all"  ) == 0 ) )
            {
#ifdef TREAL
               *NROUT = 16;
#else
               *NROUT = 17;
#endif
               *ROUTS = (enum LVL2_ROUT *)malloc( (*NROUT) *
                                                  sizeof( enum LVL2_ROUT ) );
               ATL_assert( *ROUTS );

               (*ROUTS)[ 0] = GBMV; (*ROUTS)[ 1] = GEMV; (*ROUTS)[ 2] = SBMV;
               (*ROUTS)[ 3] = SPMV; (*ROUTS)[ 4] = SYMV; (*ROUTS)[ 5] = TBMV;
               (*ROUTS)[ 6] = TPMV; (*ROUTS)[ 7] = TRMV; (*ROUTS)[ 8] = TBSV;
               (*ROUTS)[ 9] = TPSV; (*ROUTS)[10] = TRSV;
#ifdef TREAL
               (*ROUTS)[11] = GERC; (*ROUTS)[12] = SPR;  (*ROUTS)[13] = SYR;
               (*ROUTS)[14] = SPR2; (*ROUTS)[15] = SYR2;
#else
               (*ROUTS)[11] = GERC; (*ROUTS)[12] = GERU; (*ROUTS)[13] = SPR;
               (*ROUTS)[14] = SYR;  (*ROUTS)[15] = SPR2; (*ROUTS)[16] = SYR2;
#endif
               i++;
            }
            else
            {
               if( isdigit( *ARGS[i] ) ) { *NROUT = atoi( ARGS[i++] ); }
               else                      { *NROUT = 1;                 }
               *ROUTS = (enum LVL2_ROUT *)malloc( (*NROUT) *
                                                  sizeof( enum LVL2_ROUT ) );
               ATL_assert( *ROUTS );

               for( j = 0; j < *NROUT; j++ )
               {
                  if( ARGS[i] == NULL ) PrintUsage( ARGS[0] );

                  if(      ( strcmp( ARGS[i], "GBMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "gbmv" ) == 0 ) )
                     (*ROUTS)[j] = GBMV;
                  else if( ( strcmp( ARGS[i], "GEMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "gemv" ) == 0 ) )
                     (*ROUTS)[j] = GEMV;
#ifdef TREAL
                  else if( ( strcmp( ARGS[i], "SBMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "sbmv" ) == 0 ) )
                     (*ROUTS)[j] = SBMV;
                  else if( ( strcmp( ARGS[i], "SPMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "spmv" ) == 0 ) )
                     (*ROUTS)[j] = SPMV;
                  else if( ( strcmp( ARGS[i], "SYMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "symv" ) == 0 ) )
                     (*ROUTS)[j] = SYMV;
#else
                  else if( ( strcmp( ARGS[i], "HBMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "hbmv" ) == 0 ) )
                     (*ROUTS)[j] = SBMV;
                  else if( ( strcmp( ARGS[i], "HPMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "hpmv" ) == 0 ) )
                     (*ROUTS)[j] = SPMV;
                  else if( ( strcmp( ARGS[i], "HEMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "hemv" ) == 0 ) )
                     (*ROUTS)[j] = SYMV;
#endif
                  else if( ( strcmp( ARGS[i], "TBMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "tbmv" ) == 0 ) )
                     (*ROUTS)[j] = TBMV;
                  else if( ( strcmp( ARGS[i], "TPMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "tpmv" ) == 0 ) )
                     (*ROUTS)[j] = TPMV;
                  else if( ( strcmp( ARGS[i], "TRMV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "trmv" ) == 0 ) )
                     (*ROUTS)[j] = TRMV;

                  else if( ( strcmp( ARGS[i], "TBSV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "tbsv" ) == 0 ) )
                     (*ROUTS)[j] = TBSV;
                  else if( ( strcmp( ARGS[i], "TPSV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "tpsv" ) == 0 ) )
                     (*ROUTS)[j] = TPSV;
                  else if( ( strcmp( ARGS[i], "TRSV" ) == 0 ) ||
                           ( strcmp( ARGS[i], "trsv" ) == 0 ) )
                     (*ROUTS)[j] = TRSV;
#ifdef TREAL
                  else if( ( strcmp( ARGS[i], "GER"  ) == 0 ) ||
                           ( strcmp( ARGS[i], "ger"  ) == 0 ) )
                     (*ROUTS)[j] = GERU;
                  else if( ( strcmp( ARGS[i], "SPR"  ) == 0 ) ||
                           ( strcmp( ARGS[i], "spr"  ) == 0 ) )
                     (*ROUTS)[j] = SPR;
                  else if( ( strcmp( ARGS[i], "SYR"  ) == 0 ) ||
                           ( strcmp( ARGS[i], "syr"  ) == 0 ) )
                     (*ROUTS)[j] = SYR;
                  else if( ( strcmp( ARGS[i], "SPR2" ) == 0 ) ||
                           ( strcmp( ARGS[i], "spr2" ) == 0 ) )
                     (*ROUTS)[j] = SPR2;
                  else if( ( strcmp( ARGS[i], "SYR2" ) == 0 ) ||
                           ( strcmp( ARGS[i], "syr2" ) == 0 ) )
                     (*ROUTS)[j] = SYR2;
#else
                  else if( ( strcmp( ARGS[i], "GERU" ) == 0 ) ||
                           ( strcmp( ARGS[i], "geru" ) == 0 ) )
                     (*ROUTS)[j] = GERU;
                  else if( ( strcmp( ARGS[i], "GERC" ) == 0 ) ||
                           ( strcmp( ARGS[i], "gerc" ) == 0 ) )
                     (*ROUTS)[j] = GERC;
                  else if( ( strcmp( ARGS[i], "HPR"  ) == 0 ) ||
                           ( strcmp( ARGS[i], "hpr"  ) == 0 ) )
                     (*ROUTS)[j] = SPR;
                  else if( ( strcmp( ARGS[i], "HER"  ) == 0 ) ||
                           ( strcmp( ARGS[i], "her"  ) == 0 ) )
                     (*ROUTS)[j] = SYR;
                  else if( ( strcmp( ARGS[i], "HPR2" ) == 0 ) ||
                           ( strcmp( ARGS[i], "hpr2" ) == 0 ) )
                     (*ROUTS)[j] = SPR2;
                  else if( ( strcmp( ARGS[i], "HER2" ) == 0 ) ||
                           ( strcmp( ARGS[i], "her2" ) == 0 ) )
                     (*ROUTS)[j] = SYR2;
#endif
                  else PrintUsage( ARGS[0] );
                  i++;
               }
            }
            break;
         default:
            PrintUsage( ARGS[0] );
            break;
      }
   }
/*
 * Finish setting up defaults if the user has not selected
 */
   if( *NROUT == -1 )
   {
      *NROUT = 1;
      *ROUTS = (enum LVL2_ROUT *)malloc( sizeof( enum LVL2_ROUT ) );
      ATL_assert( *ROUTS );
      (*ROUTS)[0] = GEMV;
   }

   if( *NUPLO == -1 )
   {
      *NUPLO = 1;
      *UPLOS = (enum ATLAS_UPLO *)malloc( sizeof( enum ATLAS_UPLO ) );
      ATL_assert( *UPLOS );
      (*UPLOS)[0] = AtlasLower;
   }
   if( *NTRAN == -1 )
   {
      *NTRAN = 1;
      *TRANS = (enum ATLAS_TRANS *)malloc( sizeof( enum ATLAS_TRANS ) );
      ATL_assert( *TRANS );
      (*TRANS)[0] = AtlasNoTrans;
   }
   if( *NDIAG == -1 )
   {
      *NDIAG = 1;
      *DIAGS = (enum ATLAS_DIAG *)malloc( sizeof( enum ATLAS_DIAG ) );
      ATL_assert( *DIAGS );
      (*DIAGS)[0] = AtlasNonUnit;
   }

   if( *N0 == -1 ) { *N0 = 100; *NN = 1000; *NINC = 100; }

   if( *NALPHA == -1 )
   {
      *NALPHA = 1;
      *ALPHAS = (TYPE *)malloc( ATL_MulBySize( 1 ) );
      ATL_assert( *ALPHAS );
#ifdef TREAL
      (*ALPHAS)[0] = ATL_rone;
#else
      (*ALPHAS)[0] = ATL_rone;
      (*ALPHAS)[1] = ATL_rzero;
#endif
   }
   if( *NBETA == -1 )
   {
      *NBETA = 1;
      *BETAS = (TYPE *)malloc( ATL_MulBySize( 1 ) );
      ATL_assert( *BETAS );
#ifdef TREAL
      (*BETAS)[0] = ATL_rone;
#else
      (*BETAS)[0] = ATL_rone;
      (*BETAS)[1] = ATL_rzero;
#endif
   }

   if( *NINCX == -1 )
   {
      *NINCX = 1;
      *INCXS = (int *)malloc(sizeof(int));
      ATL_assert( *INCXS );
      (*INCXS)[0] = 1;
   }
   if( *NINCY == -1 )
   {
      *NINCY = 1;
      *INCYS = (int *)malloc( sizeof( int ) );
      ATL_assert( *INCYS );
      (*INCYS)[0] = 1;
   }
}

int main( int NARGS, char **ARGS )
{
   int                        klinc, klstart, klstop, kuinc, kustart,
                              kustop, ldaism, mflopmin, minc, mstart,
                              mstop, ninc, nstart, nstop, nalpha, nbeta,
                              ndiag, nincx, nincy, nrout, ntran, nuplo, test;
   int                        * incxs  = NULL, * incys = NULL;
   TYPE                       * alphas = NULL, * betas = NULL;
   enum LVL2_ROUT             * routs  = NULL;
   enum ATLAS_UPLO            * uplos  = NULL;
   enum ATLAS_TRANS           * trans  = NULL;
   enum ATLAS_DIAG            * diags  = NULL;

   GetFlags( NARGS, ARGS, &nrout, &routs, &test, &ldaism, &mflopmin, &nuplo,
             &uplos, &ntran, &trans, &ndiag, &diags, &mstart, &mstop, &minc,
             &nstart, &nstop, &ninc, &klstart, &klstop, &klinc, &kustart,
             &kustop, &kuinc, &nalpha, &alphas, &nbeta, &betas, &nincx,
             &incxs, &nincy, &incys );

   eps = EPS;

   RunCases( test, mflopmin, ldaism, nuplo, uplos, ntran, trans, ndiag,
             diags, mstart, mstop, minc, nstart, nstop, ninc, klstart,
             klstop, klinc, kustart, kustop, kuinc, nalpha, alphas, nbeta,
             betas, nincx, incxs, nincy, incys, nrout, routs );

   if( uplos  ) free( uplos  );
   if( trans  ) free( trans  );
   if( diags  ) free( diags  );
   if( alphas ) free( alphas );
   if( incxs  ) free( incxs  );
   if( incys  ) free( incys  );
   if( betas  ) free( betas  );
   if( routs  ) free( routs  );

   return( 0 );
}
