------------------------------------------------------------------------------
--                                                                          --
--             ASIS Tester And iNTerpreter (ASIStant) COMPONENTS            --
--                                                                          --
--             A S I S T A N T . B R O W S E R . I T E R A T O R            --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--          Copyright (c) 1997-1999, Free Software Foundation, Inc.         --
--                                                                          --
-- ASIStant is free software; you can redistribute it and/or modify it      --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software Foundation;  either version 2,  or  (at your option)  any later --
-- version. ASIStant is distributed  in the hope  that it will be useful,   --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MER-      --
-- CHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General  --
-- Public License for more details. You should have received a copy of the  --
-- GNU General Public License distributed with GNAT; see file COPYING. If   --
-- not, write to the Free Software Foundation, 59 Temple Place Suite 330,   --
-- Boston, MA 02111-1307, USA.                                              --
--                                                                          --
-- ASIStant is an evolution of ASIStint tool that was created by            --
-- Vasiliy Fofanov as part of a collaboration between Software Engineering  --
-- Laboratory of the Swiss Federal Institute of Technology in Lausanne,     --
-- Switzerland, and the Scientific Research Computer Center of the Moscow   --
-- University, Russia, supported by the Swiss National Science Foundation   --
-- grant #7SUPJ048247, "Development of ASIS for GNAT with industry quality" --
--                                                                          --
-- The original version of the Browser tool was created by Alain Le Guennec --
--                                                                          --
-- ASIStant is distributed as a part of the ASIS implementation for GNAT    --
-- (ASIS-for-GNAT) and is maintained by Ada Core Technologies Inc           --
-- (http ://www.gnat.com).                                                  --
------------------------------------------------------------------------------

with Asis.Elements;
with Asis.Iterator;

package body ASIStant.Browser.Iterator is

   type Operation is (Next, Previous, Up, Down);

   type State_Information is record
      Current_Operation : Operation;
      Current_Element   : Asis.Element;
      Current_Found     : Boolean;
      Result            : Asis.Element;
   end record;

   procedure Pre_Operation  (Element : in     Asis.Element;
                             Control : in out Asis.Traverse_Control;
                             State   : in out State_Information);
   --  ??? Should get documented

   procedure Post_Operation (Element : in     Asis.Element;
                             Control : in out Asis.Traverse_Control;
                                State   : in out State_Information);
   --  ??? Should get documented

   procedure Pre_Operation  (Element : in     Asis.Element;
                             Control : in out Asis.Traverse_Control;
                             State   : in out State_Information) is
   begin

      -------------------------------------------------------
      --  Was it already found just before in traversal ?  --
      -------------------------------------------------------
      if State.Current_Found then
         case State.Current_Operation is
            when Next     =>
               State.Result := Element;
               Control      := Asis.Terminate_Immediately;

            when Previous =>
               null;
               pragma Assert (False);

            when Up       =>
               null;
               pragma Assert (False);

            when Down     =>
               State.Result := Element;
               Control      := Asis.Terminate_Immediately;

         end case;

         return;
      end if;

      --------------------------------------------
      --  It wasn't found before in traversal.  --
      --------------------------------------------
      State.Current_Found :=
        Asis.Elements.Is_Equal (Element, State.Current_Element);


      -----------------------------------
      --  Have we just found it now ?  --
      -----------------------------------
      if State.Current_Found then
         case State.Current_Operation is
            when Next     =>
               Control := Asis.Abandon_Children;
               --  Result is the next sibling element in any,
               --  and will be set in the corresponding pre-op.
               --  If no such sibling exists, then we go back
               --  to the parent post-op, and terminates from there.

            when Previous =>
               Control := Asis.Terminate_Immediately;
               --  Result was already correctly set
               --  by the Post_Operation of the previous sibling, if any.
               --  If there is no such previous sibling, then
               --  the result is already a Nil_Element.

            when Up       =>
               Control := Asis.Abandon_Siblings;
               --  We found the current element.
               --  The result is its parent (enclosing_element), if any.
               --  The result will be set in the parent's Post_Op.
               --  If there is no such parent, the traversal ends,
               --  and the result will be a Nil_Element.

            when Down     =>
               --  The element down the tree, if any, will be the next one
               --  in the traversal. The result will be set in its Pre_Op.
               --  If there is no such element, we reach the Post_Op
               --  of the current element.
               Control := Asis.Continue;

         end case;
      end if;

   end Pre_Operation;

   procedure Post_Operation (Element : in     Asis.Element;
                             Control : in out Asis.Traverse_Control;
                                State   : in out State_Information) is
   begin
      if State.Current_Found then
         case State.Current_Operation is
            when Next     =>
               --  We are coming back to the parent of the element
               --  which means there was no next element (sibling).
               --  Result is already a Nil_Element.
               Control      := Asis.Terminate_Immediately;

            when Previous =>
               null;
               pragma Assert (False);

            when Up       =>
               State.Result := Element;
               Control      := Asis.Terminate_Immediately;

            when Down     =>
               --  If we are coming back here, it means that
               --  the current element has no children.
               --  We can terminate now, and result will be a Nil_Element.
               Control      := Asis.Terminate_Immediately;

         end case;

      elsif State.Current_Operation = Previous then
         State.Result := Element;
      end if;
   end Post_Operation;


   procedure Traverse
   is
     new Asis.Iterator.Traverse_Element (State_Information,
                                         Pre_Operation,
                                         Post_Operation);


   ------------
   --  Next  --
   ------------

   function Next     (E : Asis.Element) return Asis.Element is

      Parent  : Asis.Element;
      Control : Asis.Traverse_Control := Asis.Continue;
      State   : State_Information     :=
        (Current_Operation => Next,
         Current_Element   => E,
         Current_Found     => False,
         Result            => Asis.Nil_Element);
   begin
      Parent := Up (E);
      if not Asis.Elements.Is_Nil (Parent) then
         Traverse (Parent, Control, State);
         return State.Result;
      else
         return Asis.Nil_Element;
      end if;
   end Next;


   ----------------
   --  Previous  --
   ----------------

   function Previous (E : Asis.Element) return Asis.Element is

      Parent  : Asis.Element;
      Control : Asis.Traverse_Control := Asis.Continue;
      State   : State_Information :=
        (Current_Operation => Previous,
         Current_Element   => E,
         Current_Found     => False,
         Result            => Asis.Nil_Element);
   begin
      Parent := Up (E);
      if not Asis.Elements.Is_Nil (Parent) then
         Traverse (Parent, Control, State);
         return State.Result;
      else
         return Asis.Nil_Element;
      end if;
   end Previous;


   ----------
   --  Up  --
   ----------

   function Up (E : Asis.Element) return Asis.Element
      renames Asis.Elements.Enclosing_Element;

--  Legacy code; kept for the time being:

--      Enclosing_Unit  : Asis.Compilation_Unit;
--      Unit_As_Element : Asis.Element;

--      Control : Asis.Traverse_Control := Asis.Continue;
--      State   : State_Information :=
--        (Current_Operation => Up,
--         Current_Element   => E,
--         Current_Found     => False,
--         Result            => Asis.Nil_Element);

--      use type Asis.Element_Kinds;
--   begin
--      --
--      --  At the moment, Enclosing_Element is not reliable in ASIS for GNAT.
--      --  Handled by traversing down the whole compilation unit.
--      --
--      if Asis.Elements.Element_Kind (E) /= Asis.An_Expression then
--         return Asis.Elements.Enclosing_Element (E);
--      else
--         Enclosing_Unit  := Asis.Elements.Enclosing_Compilation_Unit (E);
--         Unit_As_Element := Asis.Elements.Unit_Declaration (Enclosing_Unit);

--         Traverse (Unit_As_Element, Control, State);
--         return State.Result;
--      end if;
--   end Up;


   ------------
   --  Down  --
   ------------

   function Down     (E : Asis.Element) return Asis.Element is

      Control : Asis.Traverse_Control := Asis.Continue;
      State   : State_Information :=
        (Current_Operation => Down,
         Current_Element   => E,
         Current_Found     => False,
         Result            => Asis.Nil_Element);

   begin
      Traverse (E, Control, State);
      return State.Result;
   end Down;

end ASIStant.Browser.Iterator;