/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#define GLOBAL					/* force declaration of global variables	*/

#include "loader.h"

/*--------------------------------------------------------------------------*/
/*	Map Data
 */

static const struct
{
    unsigned long	ID[2];
    unsigned long	Size;
    unsigned char	Data[0];

} Map __attribute__ ((section (".tail")))

 =	{
		{ LILO_ID, LILO_MAPDATAID }
	};

/*--------------------------------------------------------------------------*/
/* Read a Line from the Console
 *
 * Returns input length.
 */

int
read_line
(	int				mode,		/* echo mode								*/
	unsigned long	timeout
)
{	int				c;
    int				length    = 0;

	v->leader_adjust = 0;

	while (1)
	{
		if ((c = get_char(timeout)) == -1)
		{
			/* timeout */
			break;
		}

		switch (c)
		{
		    case '\r':
		    case '\n':
		    {
				v->input_buffer[length] = '\0';

				if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
				{
				    put_char('\n');
				}

				return length;
			}

		    case '\b':
		    case 0x7f:
		    {
				if (length)
				{
				    length--;
				    if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
				    {
						put_str("\b \b");
					}
				}
				break;
			}

			case 0x1b:
			{
			    if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
			    {
					while (length)
					{
					    length--;
						put_str("\b \b");
					}
				}
				break;
			}

		    case '\t':
		    {
		    	if (mode == ECHO_NORMAL)
		    	{
					while (length)
					{
					    length--;
						put_str("\b \b");
					}
				    length = put_str("List boot records\n") - 1;
				    strcpy(v->input_buffer, "\t");
				    return length;
				}
				break;
			}

		    default:
		    {
		    	if ((c >= ' ')
				&&	(c <= '~')
				&&	(length < sizeof(v->input_buffer) - 1))
				{
				    v->input_buffer[length++] = c;
				    switch (mode)
					{
						case ECHO_NORMAL:
						{
						    put_char(c);
						    break;
						}

						case ECHO_DOT:
						{
						    put_char('+');
						    break;
						}
				    }
				}
				break;
			}
		}
	}

	if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
	{
	    put_str(" ** timeout **\n");
	}

	/* timeout */
	v->input_buffer[0] = '\0';
	return -1;
}

/*--------------------------------------------------------------------------*/
/* Find the File Map for a File Path
 */

const FILEMAP *
find_file_map
(	const char		*path
)
{	const FILEDEF	*file;

	for (file = v->boot_files; file; file = file->next)
	{
		if (case_equal_strings(path, file->path))
		{
			return file->map;
		}
	}

	return NULL;
}

/*--------------------------------------------------------------------------*/
/* Get the Name for a Tag
 */

static struct {
    unsigned long	tagid;
    const char		*name;
    const char		*param;
    char			type;
#define TT_DAT	0
#define TT_STR	1
#define TT_OTH	2
} tag_names[] = {
	{ TAG_LILO,				"TAG_LILO",				NULL,			  TT_OTH },
	{ TAG_EOF,				"TAG_EOF",				NULL,			  TT_OTH },
	{ TAG_HEADER,			"TAG_HEADER",			NULL,			  TT_OTH },
	{ TAG_HEADER_END,		"TAG_HEADER_END",		NULL,			  TT_OTH },
	{ TAG_DEFAULT,			"TAG_DEFAULT",			"default",		  TT_STR },
	{ TAG_MESSAGE,			"TAG_MESSAGE",			"message",		  TT_STR },
	{ TAG_DELAY,			"TAG_DELAY",			"delay",		  TT_DAT },
	{ TAG_TIMEOUT,			"TAG_TIMEOUT",			"timeout",		  TT_DAT },
	{ TAG_DEBUG,			"TAG_DEBUG",			"debug",		  TT_DAT },
	{ TAG_PROMPT,			"TAG_PROMPT",			"prompt",		  TT_DAT },
	{ TAG_BOOT_RECORD,		"TAG_BOOT_RECORD",		"label",		  TT_STR },
	{ TAG_BOOT_RECORD_END,	"TAG_BOOT_RECORD_END",	NULL,			  TT_OTH },
	{ TAG_ALIAS,			"TAG_ALIAS",			"alias",		  TT_STR },
	{ TAG_KERNEL,			"TAG_KERNEL",			"kernel",		  TT_STR },
	{ TAG_ARGUMENTS,		"TAG_ARGUMENTS",		"cmdline",		  TT_STR },
	{ TAG_PASSWORD,			"TAG_PASSWORD",			"password",		  TT_STR },
	{ TAG_MASTER_PASSWORD,	"TAG_MASTER_PASSWORD",	"master-password",TT_STR },
	{ TAG_RESTRICTED,		"TAG_RESTRICTED",		"restricted",	  TT_DAT },
	{ TAG_RAMDISK,			"TAG_RAMDISK",			"ramdisk",		  TT_STR },
	{ TAG_SYMTAB,			"TAG_SYMTAB",			"symbols",		  TT_STR },
	{ TAG_MEMSIZE,			"TAG_MEMSIZE",			"memsize",		  TT_DAT },
	{ TAG_CALLMONITOR,		"TAG_CALLMONITOR",		"callmonitor",	  TT_DAT },
	{ TAG_ROOT,				"TAG_ROOT",				"root",			  TT_STR },
	{ TAG_CONSOLE,			"TAG_CONSOLE",			"console",		  TT_STR },
	{ TAG_READ_ONLY,		"TAG_READ_ONLY",		"read-only",	  TT_DAT },
	{ TAG_FILE_DEF,			"TAG_FILE_DEF",			NULL,			  TT_OTH },
	{ TAG_FILE_DEF_END,		"TAG_FILE_DEF_END",		NULL,			  TT_OTH },
	{ TAG_FILEMAP,			"TAG_FILEMAP",			NULL,			  TT_OTH }
};

void
show_tag
(	const TAGRECORD		*tr
)
{	unsigned			i;

    for (i = 0; i < arraysize(tag_names); i++)
    {
		if (tag_names[i].tagid == tr->tagid)
		{
			if (tag_names[i].type == TT_DAT)
			{
				printf(" %-10s = %ld\n", tag_names[i].param, *tr->data);
			}
			else if (tag_names[i].type == TT_STR)
			{
				printf(" %-10s = `%s'\n", tag_names[i].param, (char *)tr->data);
			}
			break;
		}
	}
}

/*--------------------------------------------------------------------------*/

const char *
get_tag_name
(	unsigned long	tag
)
{	unsigned		i;

    for (i = 0; i < arraysize(tag_names); i++)
    {
		if (tag_names[i].tagid == tag)
		{
		    return tag_names[i].name;
		}
	}

    return NULL;
}

/*--------------------------------------------------------------------------*/
/* Unexpected Tag
 */

void
unexpected_tag
(	const TAGRECORD		*tr
)
{	const char			*name;

    if ((name = get_tag_name(tr->tagid)) != NULL)
    {
		printf("Unexpected tag `%s' at offset 0x%08lx\n",
			name, v->map_offset);
	}
    else
    {
		printf("Unknown tag 0x%08lx at offset 0x%08lx\n",
			tr->tagid, v->map_offset);
	}
}

/*--------------------------------------------------------------------------*/
/* Get the first tag from the map data
 */

const TAGRECORD *
get_first_tag
(	void
)
{
	return (TAGRECORD *) &Map.Data[v->map_offset = 0];
}

/*--------------------------------------------------------------------------*/
/* Get the next tag from the map data
 */

const TAGRECORD *
get_next_tag
(	void
)
{	const TAGRECORD	*tr;

	/* get pointer to current tag */
	tr = (TAGRECORD *) &Map.Data[v->map_offset];

	/* get offset to next tag */
	v->map_offset += sizeof(TAGRECORD) + tr->size;

	/* sanity check */
	if (v->map_offset + sizeof(TAGRECORD) > Map.Size)
	{
		panic("Unexpected End of configuration data");
	}

	/* return pointer to next tag */
    return (TAGRECORD *) &Map.Data[v->map_offset];
}

/*--------------------------------------------------------------------------*/
/* Get the value of the debug tag (if it exists)
 */

void
get_debug_tag
(	void
)
{	const TAGRECORD	*tr;
    int				debug = FALSE;

	tr = get_first_tag();

	/* scan until end of boot options fields */	
	while (tr->tagid != TAG_HEADER_END)
	{
		if (tr->tagid == TAG_DEBUG)
		{
			debug = tr->data[0];
	   		break;
		}

		tr = get_next_tag();
	}

	v->debug_mode = debug;
}

/*--------------------------------------------------------------------------*/
/* Read the Map Data
 */

void
read_map_data
(	void
)
{	const TAGRECORD	*tr;
	BOOTRECORD		*record;
	BOOTRECORD		**p1		= &v->boot_records;
	FILEDEF			*file;
	FILEDEF			**p2 		= &v->boot_files;

#define CASE_TAG(tag, item)\
	case TAG_ ## tag:\
		item = (typeof(item)) tr->data;\
		if (v->debug_mode) show_tag(tr);\
		break

	/* File Identification */
	tr = get_first_tag();
	if (tr->tagid != TAG_LILO)
	{
		panic("Invalid configuration data");
	}

	/* Boot Options */
	if (v->debug_mode)
	{
		put_str("\n[Boot Options]\n");
	}

	tr = get_next_tag();
	if (tr->tagid != TAG_HEADER)
	{
		panic("Invalid configuration data");
	}

	for (tr = get_next_tag(); tr->tagid != TAG_HEADER_END; tr = get_next_tag())
	{
		switch (tr->tagid)
		{
			CASE_TAG( DEFAULT,			v->boot_options.boot_default	);
			CASE_TAG( MESSAGE,			v->boot_options.boot_message	);
			CASE_TAG( PROMPT,			v->boot_options.boot_prompt		);
			CASE_TAG( DELAY,			v->boot_options.boot_delay		);
			CASE_TAG( TIMEOUT,			v->boot_options.boot_timeout	);
			CASE_TAG( DEBUG,			v->boot_options.boot_debug		);
			CASE_TAG( MASTER_PASSWORD,  v->boot_options.boot_masterpswd	);
			default:
			{
				unexpected_tag(tr);
				break;
			}
		}
	}

	v->force_prompt = VALUE(v->boot_options.boot_prompt);

	/* Boot Records */
	if (v->debug_mode)
	{
		put_str("\n[Boot Records]");
	}

	for (tr = get_next_tag(); tr->tagid == TAG_BOOT_RECORD; tr = get_next_tag())
	{
		record        = (BOOTRECORD *) malloc(sizeof(BOOTRECORD));
		mem_clear(record, sizeof(BOOTRECORD));
		record->label = (const char *) tr->data;

		if (v->debug_mode)
		{
			put_char('\n');
			show_tag(tr);
		}

		for (tr = get_next_tag();
				tr->tagid != TAG_BOOT_RECORD_END;
					 tr = get_next_tag())
		{
			switch (tr->tagid)
			{
				CASE_TAG( ALIAS,       record->alias      );
				CASE_TAG( KERNEL,      record->kernel     );
				CASE_TAG( ARGUMENTS,   record->cmdline    );
				CASE_TAG( PASSWORD,    record->password   );
				CASE_TAG( RESTRICTED,  record->restricted );
				CASE_TAG( RAMDISK,     record->ramdisk    );
				CASE_TAG( MEMSIZE,     record->memsize    );
				CASE_TAG( SYMTAB,      record->symtab     );
				CASE_TAG( CALLMONITOR, record->callmonitor);
				CASE_TAG( APPEND,      record->append     );
				CASE_TAG( ROOT,        record->root       );
				CASE_TAG( CONSOLE,     record->console    );
				CASE_TAG( READ_ONLY,   record->read_only  );
				default:
				{
					unexpected_tag(tr);
					break;
				}
			}
		}

		*p1       = record;
		p1        = &record->next;
	}

	/* File Definitions */
	if (v->debug_mode)
	{
		put_str("\n[File Definitions]\n");
	}

	while (tr->tagid == TAG_FILE_DEF)
	{
		file = (FILEDEF *) malloc(sizeof(FILEDEF));
		mem_clear(file, sizeof(FILEDEF));
		file->path = (const char *) tr->data;

		for (tr = get_next_tag();
				tr->tagid != TAG_FILE_DEF_END;
					tr = get_next_tag())
		{
			switch (tr->tagid)
			{
				CASE_TAG( FILEMAP, file->map );
				default:
				{
					unexpected_tag(tr);
					break;
				}
			}
		}

		if (v->debug_mode)
		{
			printf("%s (", file->path);
			if (file->map)
			{
				printf("%ld bytes in %ld fragments)\n",
					MAP_FILESIZE(file->map), MAP_NUMFRAGS(file->map));
			}
			else
			{
				put_str("not available)\n");
			}
		}

		*p2 = file;
		p2  = &file->next;
		tr  = get_next_tag();
	}

	if (tr->tagid != TAG_EOF)
	{
		unexpected_tag(tr);
	}

	if (v->debug_mode)
	{
		put_char('\n');
	}
}

/*--------------------------------------------------------------------------*/
/* Find a Boot Record by Name
 */

const BOOTRECORD *
find_boot_record
(	char				*name
)
{	const BOOTRECORD	*record;
	char				*base = name;

    if (name == NULL)
    {
		name = (char *) v->boot_options.boot_default;
	}

    for (record = v->boot_records; record; record = record->next)
    {
		if (prefix_string(name, record->label))
		{
			name += strlen(record->label);
			break;
		}

		if (prefix_string(name, record->alias))
		{
			name += strlen(record->alias);
	    	break;
	    }
	}

	if (record && base)
	{
		/* skip white space */
		while (*name && *name < '!')
		{
			name++;
		}

		/* strip name from start of string */
		strcpy(base, name);
	}

    return record;
}

/*--------------------------------------------------------------------------*/
/* Check whether a Boot Image is available
 */

int
bootrecord_available
(	const BOOTRECORD	*record
)
{
	if (find_file_map(record->kernel) == NULL)
	{
		return FALSE;
	}

	if (record->ramdisk && (find_file_map(record->ramdisk) == NULL))
	{
	    return FALSE;
    }

    return TRUE;
}

/*--------------------------------------------------------------------------*/
/* List all available Boot Records
 */

void
list_records
(	void
)
{	const BOOTRECORD	*record;

	put_str("\nBoot images:\n");

	for (record = v->boot_records; record; record = record->next)
	{
		printf("    %c%-32s  %-10s",
				(record == v->default_boot_record) ? '*' : ' ',
				record->label,
				record->alias ? record->alias : ""
		);
		if (VALUE(record->restricted) && record->password)
		{
			put_str("  (restricted,password)");
		}
		else if (VALUE(record->restricted))
		{
			put_str("  (restricted)");
		}
		else if (record->password)
		{
			put_str("  (password)");
		}
		put_char('\n');
	}
	put_char('\n');
}

/*--------------------------------------------------------------------------*/

void
help
(	void
)
{
	put_str("\n"
"Commands:\n"
"     ? or help  - Display this message.\n"
"     <tab>      - List boot records.\n"
"     su         - Enter master mode.\n"
"\n"
"Master mode commands:\n"
"     exit       - Leave master mode.\n"
"     debug      - Toggle debug mode on and off.\n"
"     call       - Toggle call to ROM after loading kernel on and off.\n"
"     show       - Display debug mode settings.\n"
"     break      - Enter ROM debug monitor.\n"
"\n"
	);
}

/*--------------------------------------------------------------------------*/
/* Ask the User for a Boot Operating System
 */

const BOOTRECORD *
get_boot_record
(	unsigned long		timeout
)
{	const BOOTRECORD	*record;

	while (1)
	{
		put_str(v->master_mode ? "Master: " : "Boot: ");
		read_line(ECHO_NORMAL, timeout);

		/* timeout or no input */
		if (v->input_buffer[0] == '\0')
		{
			return v->default_boot_record;
		}

		if (v->input_buffer[0] == '\t')
		{
			list_records();
			continue;
		}

		if (prefix_string(v->input_buffer, "help")
		|| 	prefix_string(v->input_buffer, "?"   ))
		{
			help();
			continue;
		}

		if (prefix_string(v->input_buffer, "su"))
		{
			if (!v->master_mode
			&&	v->boot_options.boot_masterpswd
			&&	v->boot_options.boot_masterpswd[0])
			{
				put_str("Password: ");
				if (read_line(ECHO_DOT, timeout) == -1)
				{
					/* timeout */
					continue;
				}
	
				if (!case_equal_strings(v->input_buffer,
							v->boot_options.boot_masterpswd))
				{
					put_str("\nIncorrect password.\n\n");
					continue;
				}
			}

			v->master_mode = TRUE;
			continue;
		}

		if (v->master_mode && prefix_string(v->input_buffer, "exit"))
		{
			v->master_mode = FALSE;
			continue;
		}
		
		if (v->master_mode && prefix_string(v->input_buffer, "debug"))
		{
			v->debug_mode = !v->debug_mode;
			goto show_modes;
		}

		if (v->master_mode && prefix_string(v->input_buffer, "call"))
		{
			v->callmonitor = !v->callmonitor;
			goto show_modes;
		}

		if (v->master_mode && prefix_string(v->input_buffer, "show"))
		{
		show_modes:
			printf(
				"\n"
				"Debug mode %sabled\n"
				"Call to ROM monitor %sabled\n"
				"\n",
				v->debug_mode  ? "en" : "dis",
				v->callmonitor ? "en" : "dis"
			);
			continue;
		}

		if (v->master_mode && prefix_string(v->input_buffer, "break"))
		{
			call_bug();
			continue;
		}
		
		if ((record = find_boot_record(v->input_buffer)) == NULL)
		{
			printf("Nonexistent boot image `%s'.\n", v->input_buffer);
			continue;
		}

		if (!bootrecord_available(record))
		{
			printf("Boot image `%s' is not available.\n", record->label);
			continue;
		}

		/* preserve any extra command line parameters
		 * following the image name
		 */
		strcpy(v->cmdln_buffer, v->input_buffer);

		/* all done */
		return record;
	}
}

/*--------------------------------------------------------------------------*/

void
loading
(	const char	*what
)
{	int			n;

	printf("Loading %s %n", what, &n);

	if (v->debug_mode)
	{
		put_char('\n');
		return;
	}

	n = 40 - n - v->leader_adjust;
	v->leader_adjust = 0;

	while (n-- > 0)
	{
		put_char('.');
	}

	put_char(' ');
}

/*--------------------------------------------------------------------------*/

#if SYMBOL_SUPPORT

void
load_symbols
(	const char	*file_name
)
{
	char 			*iobuff;
	int				iocount;
	char			*p;
	int				len;
	unsigned long	sofar;

	if ((iobuff = malloc(IOBUF_SIZE)) == NULL)
	{
		put_str("\nNot enough memory to load symbols!\n");
		return;
	}

    stream_init();
    stream_push( &file_mod );
    stream_push( &gunzip_mod );

	loading("Symbols");

	if (sopen(file_name) < 0)
	{
		printf("\nUnable to open symbol file `%s'\n", file_name);
		goto bye;
	}

	/* get rid of all existing symbols */
	clear_symbols();

	p        = iobuff;
	iocount  = 0;
	sofar    = 0;

	while (1)
	{
		len = 0;
		while (1)
		{
			/* if buffer empty */
			if (iocount == 0)
			{
				/* read a chunk */
				if ((iocount = sread(iobuff, IOBUF_SIZE)) < 1)
				{
					/* error or end of file */
					if (iocount != 0)
					{
						printf("\nError reading symbol file `%s'\n",
								file_name);
						goto fail;
					}

					goto done;
				}

				/* reset empty pointer */
				p      = iobuff;

				/* count bytes read */
				sofar += iocount;
			}

			/* count byte examined */
			iocount--;

			/* end of line? */
			if (*p == 0x0a || *p == 0x0d)
			{
				/* skip it */
				p++;
				break;
			}

			/* save character */
			v->input_buffer[len++] = *p++;
		}

		/* null terminate */
		v->input_buffer[len] = '\0';

		/* if not empty line */
		if (len)
		{
			if (add_symbol(v->input_buffer))
			{
				goto fail;
			}
		}
	}

done:
	percent_term("OK\n");

fail:
	sclose();

bye:
	free(iobuff);
}

#endif /* SYMBOL_SUPPORT */

/*--------------------------------------------------------------------------*/

void
display_message_file
(	const char		*file_name
)
{	char 			*iobuff;
	int				iocount;
	char			*p;
	unsigned long	sofar;

	if ((iobuff = malloc(IOBUF_SIZE)) == NULL)
	{
		put_str("Not enough memory to display message file!\n");
		return;
	}

	if (file_open(file_name) != 0)
	{
		printf("Unable to open message file `%s'\n", file_name);
		goto bye;
	}

	p        = iobuff;
	iocount  = 0;
	sofar    = 0;

	while (1)
	{
		/* if buffer empty */
		if (iocount == 0)
		{
			/* read a chunk */
			if ((iocount = file_read(iobuff, IOBUF_SIZE)) < 1)
			{
				/* error or end of file */
				if (iocount)
				{
					printf("\nError reading message file `%s'\n", file_name);
				}

				goto done;
			}

			/* reset empty pointer */
			p      = iobuff;

			/* count bytes read */
			sofar += iocount;
		}

		/* count byte examined */
		iocount--;

		/* display it */
		put_char(*p++);
	}

done:
	file_close();

bye:
	free(iobuff);
}

/*--------------------------------------------------------------------------*/
/* Compare the Bootstrap and Kernel Versions
 */

int
check_bootinfo_version
(	const char					*memptr
)
{	const struct bootversion	*bv		= (struct bootversion *) memptr;
	unsigned long				version	= 0;
	int							i;
	int							kernel_major;
	int							kernel_minor;
	int							boots_major;
	int							boots_minor;
	unsigned long				machtype;

	if (bv->magic == BOOTINFOV_MAGIC)
	{
		machtype = get_machtype();

		for (i = 0; bv->machversions[i].machtype != 0; ++i)
		{
			if (bv->machversions[i].machtype == machtype)
			{
				version = bv->machversions[i].version;
				break;
			}
		}

#if BOOTINFO_COMPAT_1_0
		/* if no 2.1.xx version info */
		if (!version)
		{
			machtype = get_compat_machtype();

			/* look for 2.0.xx version info (has different machine type code) */
			for (i = 0; bv->machversions[i].machtype != 0; ++i)
			{
				if (bv->machversions[i].machtype == machtype)
				{
					version = bv->machversions[i].version;
				}
			}
		}
#endif
	}

	if (!version)
	{
		printf("Kernel has no bootinfo version info, assuming 1.0\n");
		version  = get_compat_booti_version();
	}

	kernel_major = BI_VERSION_MAJOR(version);
	kernel_minor = BI_VERSION_MINOR(version);
	boots_major	 = BI_VERSION_MAJOR(get_booti_version());
	boots_minor	 = BI_VERSION_MINOR(get_booti_version());

	printf("Bootstrap's bootinfo version : %u.%u\n",
		boots_major, boots_minor);

	printf("Kernel's bootinfo version    : %u.%u\n",
		kernel_major, kernel_minor);

	if (kernel_major == boots_major)
	{
	    if (kernel_minor > boots_minor)
		{
			printf(
			"Warning: Bootinfo version of bootstrap and kernel differ!\n"
			"         Certain features may not work.\n"
			);
	    }
	}

#if BOOTINFO_COMPAT_1_0
	else if (kernel_major == BI_VERSION_MAJOR(get_compat_booti_version()))
	{
	    printf("(using backwards compatibility mode)\n");
	}
#endif /* BOOTINFO_COMPAT_1_0 */

	else
	{
	    printf("\nThis bootstrap is too %s for this kernel!\n",
		   boots_major < kernel_major ? "old" : "new");
	   	return 0;
    }

    return kernel_major;
}

/*--------------------------------------------------------------------------*/
/* Add a Record to the Bootinfo Structure
 */

int
add_bi_record
(	unsigned short		tag,
	unsigned short		size,
	const void			*data
)
{	struct bi_record	*record;
    unsigned short		size2;

    size2 = (sizeof(struct bi_record) + size + 3) & -4;

    if (v->bi_size + size2 + sizeof(v->bi_union.record.tag) > MAX_BI_SIZE)
	{
		printf("Can't add bootinfo record. Size exceeded MAX_BI_SIZE (%u).\n",
						MAX_BI_SIZE);
		return FALSE;
    }

    record       = (struct bi_record *)
						((unsigned long)&v->bi_union.record + v->bi_size);
    record->tag  = tag;
    record->size = size2;
    mem_move(record->data, data, size);
    v->bi_size  += size2;

    return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Add a String Record to the Bootinfo Structure
 */

int
add_bi_string
(	unsigned short		tag,
	const unsigned char	*s
)
{
    return add_bi_record(tag, strlen(s) + 1, s);
}

/*--------------------------------------------------------------------------*/
/* Create the Bootinfo Structure
 */

int
create_bootinfo
(	void
)
{	int					i;
    struct bi_record	*record;

    /* initialization */
    v->bi_size = 0;

    /* Generic tags */
    if (!add_bi_record(BI_MACHTYPE, sizeof(v->bi.machtype), &v->bi.machtype))
		return FALSE;

    if (!add_bi_record(BI_CPUTYPE, sizeof(v->bi.cputype), &v->bi.cputype))
		return FALSE;

    if (!add_bi_record(BI_FPUTYPE, sizeof(v->bi.fputype), &v->bi.fputype))
		return FALSE;

    if (!add_bi_record(BI_MMUTYPE, sizeof(v->bi.mmutype), &v->bi.mmutype))
		return FALSE;

    for (i = 0; i < v->bi.num_memory; i++)
		if (!add_bi_record(BI_MEMCHUNK, sizeof(v->bi.memory[i]), &v->bi.memory[i]))
		    return FALSE;

    if (v->bi.ramdisk.size)
		if (!add_bi_record(BI_RAMDISK, sizeof(v->bi.ramdisk), &v->bi.ramdisk))
		    return FALSE;

    if (!add_bi_string(BI_COMMAND_LINE, v->bi.command_line))
		return FALSE;

	/* add VME specific bootinfo fields */
	if (!add_vme_bootinfo())
		return FALSE;

    /* Trailer */
    record      = (struct bi_record *)
						((unsigned long)&v->bi_union.record + v->bi_size);
    record->tag = BI_LAST;
	v->bi_size += sizeof(v->bi_union.record.tag);

    return TRUE;
}

/*--------------------------------------------------------------------------*/
/*  Create the Bootinfo structure for backwards compatibility mode
 */

#if BOOTINFO_COMPAT_1_0

int
create_compat_bootinfo
(	void
)
{	unsigned	i;

	v->compat_bootinfo.machtype = v->bi.machtype;

    if (v->bi.cputype & CPU_68020)
		v->compat_bootinfo.cputype = COMPAT_CPU_68020;
    else if (v->bi.cputype & CPU_68030)
		v->compat_bootinfo.cputype = COMPAT_CPU_68030;
    else if (v->bi.cputype & CPU_68040)
		v->compat_bootinfo.cputype = COMPAT_CPU_68040;
    else if (v->bi.cputype & CPU_68060)
		v->compat_bootinfo.cputype = COMPAT_CPU_68060;
    else
	{
		printf("CPU type 0x%08lx not supported by kernel.\n", v->bi.cputype);
		return FALSE;
    }

	if (v->bi.fputype & FPU_68881)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68881;
    else if (v->bi.fputype & FPU_68882)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68882;
    else if (v->bi.fputype & FPU_68040)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68040;
    else if (v->bi.fputype & FPU_68060)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68060;
    else if (v->bi.fputype)
	{
		printf("FPU type 0x%08lx not supported by kernel.\n", v->bi.fputype);
		return FALSE;
    }

    v->compat_bootinfo.num_memory = v->bi.num_memory;
    if (v->compat_bootinfo.num_memory > COMPAT_NUM_MEMINFO)
	{
		printf("Warning: using only first %d memory blocks.\n",
			       COMPAT_NUM_MEMINFO);
		v->compat_bootinfo.num_memory = COMPAT_NUM_MEMINFO;
    }

    for (i = 0; i < v->compat_bootinfo.num_memory; i++)
	{
		v->compat_bootinfo.memory[i].addr = v->bi.memory[i].addr;
		v->compat_bootinfo.memory[i].size = v->bi.memory[i].size;
    }

    if (v->bi.ramdisk.size)
	{
		v->compat_bootinfo.ramdisk_size = (v->bi.ramdisk.size + 1023) / 1024;
		v->compat_bootinfo.ramdisk_addr = v->bi.ramdisk.addr;
    }
	else
	{
		v->compat_bootinfo.ramdisk_size = 0;
		v->compat_bootinfo.ramdisk_addr = 0;
    }

    strncpy(v->compat_bootinfo.command_line,
					v->bi.command_line, COMPAT_CL_SIZE);
    v->compat_bootinfo.command_line[COMPAT_CL_SIZE - 1] = '\0';

    return TRUE;
}
#endif /* BOOTINFO_COMPAT_1_0 */

/*--------------------------------------------------------------------------*/
/* Call the copy-and-go-code
 */

void
start_kernel
(	void			(*startcode_entry)(void),
	unsigned long	kernel_dest_addr,
	char			*kernel_load_addr,
	unsigned long	kernel_mem_size,
	unsigned long	ramdisk_dest_addr,
	char			*ramdisk_load_addr,
	unsigned long	ramdisk_mem_size,
	int				call_monitor
)
{	register void (*a0)() __asm("a0") = startcode_entry;
	register long   a1    __asm("a1") = kernel_dest_addr;
	register char  *a2    __asm("a2") = kernel_load_addr;
	register long   a3    __asm("a3") = ramdisk_dest_addr;
	register char  *a4    __asm("a4") = ramdisk_load_addr;
	register long   d0    __asm("d0") = kernel_mem_size;
	register long   d1    __asm("d1") = ramdisk_mem_size;
	register long   d2    __asm("d2") = call_monitor;

	__asm __volatile ("jmp (%%a0)"
					  : /* no outputs */
					  : "r" (a0), "r" (a1), "r" (a2), "r" (a3),
						"r" (a4), "r" (d0), "r" (d1), "r" (d2)
	);

	/* fake a noreturn */
	for (;;);
}

/*--------------------------------------------------------------------------*/
/* load kernel image into a chunk of memory
 */

char *
load_kernel
(	const char		*file_name,
	unsigned long	mem_start,
	unsigned long	*kernel_size,
	unsigned long	extra_size
)
{	Elf32_Phdr		*kernel_phdrs     = NULL;
	char			*kernel_load_addr = NULL;
	int				opened            = FALSE;
	Elf32_Ehdr		kexec_elf;
	unsigned long	min_addr;
	unsigned long	max_addr;
	int				i;

    stream_init();
    stream_push( &file_mod   );
    stream_push( &gunzip_mod );

	/* open kernel executable and read exec header */
	if (sopen(file_name) < 0)
	{
		printf("\nUnable to open kernel file `%s'.\n", file_name);
		goto fail;
	}
	opened = TRUE;

	if (sread(&kexec_elf, sizeof(kexec_elf)) != sizeof(kexec_elf))
	{
		printf("\nCan't read ELF header of kernel image `%s'.\n",
				file_name);
		goto fail;
	}

	/* Allow images starting '0x00001000 0x00000800' as they are
	 * probably valid kernels on which 16xcfg has been run.
	 */
	if (mem_cmp(&kexec_elf.e_ident[EI_MAG0], ELFMAG, SELFMAG) != 0 &&
		mem_cmp(&kexec_elf.e_ident[EI_MAG0],
			"\000\000\020\000\000\000\010\000", 8) != 0)
	{
		printf("\nKernel image '%s' is not an ELF executable.\n",
				file_name);
		goto fail;
	}

	/* A few plausibility checks */
	if ((kexec_elf.e_type    != ET_EXEC   )
	||	(kexec_elf.e_machine != EM_68K    )
	||	(kexec_elf.e_version != EV_CURRENT))
	{
		printf("\nInvalid ELF header contents in kernel '%s'.\n",
				file_name);
		goto fail;
	}

	/* allocate memory for program headers */
	if ((kernel_phdrs = (Elf32_Phdr *)
		malloc(kexec_elf.e_phnum * sizeof(Elf32_Phdr))) == NULL)
	{
		goto no_mem;
	}

	/* Load the program headers */
	sseek(kexec_elf.e_phoff, SEEK_SET);
	if (sread(kernel_phdrs,
			kexec_elf.e_phnum * sizeof(*kernel_phdrs)) !=
				kexec_elf.e_phnum * sizeof(*kernel_phdrs))
	{
		printf("\nUnable to read program headers from kernel file '%s'.\n",
				file_name);
		goto fail;
	}

	/* calculate the total required amount of memory */
	min_addr = 0xffffffff;
	max_addr = 0;

	for (i = 0; i < kexec_elf.e_phnum; i++)
	{
		if (min_addr > kernel_phdrs[i].p_vaddr)
		{
			min_addr = kernel_phdrs[i].p_vaddr;
		}
		if (max_addr < kernel_phdrs[i].p_vaddr + kernel_phdrs[i].p_memsz)
		{
			max_addr = kernel_phdrs[i].p_vaddr + kernel_phdrs[i].p_memsz;
		}
	}

	/* This is needed for newer linkers that include the header in
	 * the first segment.
	 */
	if (min_addr == 0)
	{
		min_addr                  = PAGE_SIZE;
		kernel_phdrs[0].p_vaddr  += PAGE_SIZE;
		kernel_phdrs[0].p_offset += PAGE_SIZE;
		kernel_phdrs[0].p_filesz -= PAGE_SIZE;
		kernel_phdrs[0].p_memsz  -= PAGE_SIZE;
	}

	/* get size of kernel */
	*kernel_size = max_addr - min_addr;

	/* allocate memory for kernel and extras (bootinfo and ramdisk) */
	if ((kernel_load_addr = malloc(*kernel_size + extra_size)) == NULL)
	{
		goto no_mem;
	}

	/* zero initialise kernel memory */
	mem_clear(kernel_load_addr, *kernel_size);

	/* read the text and data segments from the kernel image */
	for (i = 0; i < kexec_elf.e_phnum; i++)
	{
		if (v->debug_mode)
		{
			printf("Kernel segment %u at 0x%08lx, size %lu, align %lu\n",
				i,
				mem_start + kernel_phdrs[i].p_vaddr - PAGE_SIZE,
				kernel_phdrs[i].p_memsz,
				kernel_phdrs[i].p_align
			);
		}

		if (sseek(kernel_phdrs[i].p_offset, SEEK_SET) == -1)
		{
			printf("\nFailed to seek to kernel segment %u.\n", i);
			goto fail;
		}

		if (sread(kernel_load_addr + kernel_phdrs[i].p_vaddr - PAGE_SIZE,
				 kernel_phdrs[i].p_filesz) != kernel_phdrs[i].p_filesz)
		{
			printf("\nFailed to read kernel segment %u.\n", i);
			goto fail;
		}
	}

 	if (v->debug_mode)
	{
		printf("Kernel entry is 0x%08lx.\n", kexec_elf.e_entry);
	}

	sclose();
	percent_term("OK\n");

	free(kernel_phdrs);

	return kernel_load_addr;

no_mem:
	put_str("\nNot enough memory to boot!\n");

fail:
	if (opened)
	{
		sclose();
	}

	free(kernel_phdrs    );
	free(kernel_load_addr);
	return NULL;
}

/*--------------------------------------------------------------------------*/
/* Boot the Linux/m68k Operating System
 */

void
boot_linux
(	const BOOTRECORD	*boot_record
)
{	int					i;
	unsigned long		mem_start;
	unsigned long		mem_size;
	unsigned long		mem_end;
	unsigned long		bootinfo_size;
	unsigned long		kernel_size;
	unsigned long		kernel_major;
	char				*kernel_load_addr  = NULL;
	char				*ramdisk_load_addr = NULL;
    void				*bi_ptr;

    mem_clear(&v->bi, sizeof(v->bi));

	/* get default (2.1.xx) machine type */
	v->bi.machtype = get_machtype();

	/* detect and configure cpu/fpu type */
	switch (v->cpu_type)
	{
		case 30:
		{
			v->bi.cputype = CPU_68030;
			break;
		}

		case 40:
		{
			v->bi.cputype = CPU_68040;
			break;
		}

		case 60:
		{
			v->bi.cputype = CPU_68060;
			break;
		}
	}
	v->bi.mmutype = v->bi.cputype;
	if (v->fpu_type)
		v->bi.fputype = v->bi.cputype;
	else
		v->bi.fputype = 0;

	if (v->debug_mode)
	{
		put_char('\n');
		print_model();
		printf(" CPU:680%d with%s FPU\n",
			v->cpu_type, v->fpu_type ? " internal" : "out");
	}

	/* if no ram size has been specified */
	if ((mem_size = VALUE(boot_record->memsize)) == 0)
	{
		/* use detected memory size */
		mem_size = v->detected_mem_size;
	}

	/* if too much ram has been specified */
	if (mem_size > v->detected_mem_size)
	{
		/* use detected memory size */
		mem_size = v->detected_mem_size;

		if (v->debug_mode)
		{
			printf("Specified memory size greater than detected size.\n"
				   "Using specified memory size %ldK.\n",
					mem_size >> 10
			);
		}
	}

	else if (v->debug_mode)
	{
		printf("Using %s memory size %ldK.\n",
				boot_record->memsize ? "specified" : "detected",
				mem_size >> 10
		);
	}

	v->bi.num_memory     = 1;
	v->bi.memory[0].addr = MEMORY_BASE_ADDR;
	v->bi.memory[0].size = mem_size;

	if (v->debug_mode)
	{
		printf("Found %u block%s of memory.\n", v->bi.num_memory,
			v->bi.num_memory > 1 ? "s" : "");

		for (i = 0; i < v->bi.num_memory; i++)
		{
			printf(" Block %d: 0x%08lx to 0x%08lx (%ldK)\n", i,
					v->bi.memory[i].addr,
					v->bi.memory[i].addr + v->bi.memory[i].size - 1,
					v->bi.memory[i].size >> 10
			);
		}
	}

	mem_start = v->bi.memory[0].addr;
	mem_size  = v->bi.memory[0].size;
	mem_end   = mem_start + mem_size;

	/* don't allow ramdisk to be moved over the startup code */
	if (mem_end > (unsigned long) v->startcode_entry)
		mem_end = (unsigned long) v->startcode_entry;

	/* Load the kernel at one page after start of memory */
	mem_start += PAGE_SIZE;

#if BOOTINFO_COMPAT_1_0
	/* make sure we have enough room for old bootinfo */
    if (sizeof(v->compat_bootinfo) > sizeof(v->bi_union))
		bootinfo_size = sizeof(v->compat_bootinfo);
	else
#endif /* BOOTINFO_COMPAT_1_0 */
		bootinfo_size = sizeof(v->bi_union);


	/* get size of ramdisk */
	if (boot_record->ramdisk)
	{
		if ((v->bi.ramdisk.size = file_size(boot_record->ramdisk)) == -1)
		{
			printf("Unable to find size of ramdisk file `%s'.\n",
				boot_record->ramdisk);
			goto fail;
		}

		/* allow a little extra to long word align after kernel and bootinfo */
		v->bi.ramdisk.size += 4;
	}


	/* load the kernel */
	loading(boot_record->label);
	if ((kernel_load_addr = load_kernel(boot_record->kernel, mem_start,
						&kernel_size, bootinfo_size + v->bi.ramdisk.size)) == NULL)
	{
		return;
	}

	/* load ramdisk */	
	if (boot_record->ramdisk)
	{	unsigned long	n;

		/* calculate ramdisk load address (after kernel and bootinfo) */
		n = (unsigned long) kernel_load_addr + kernel_size + bootinfo_size;

		/* align to long word boundry */
		if (n & 3) n = (n + 3) & ~3;

		/* remember aligned address */
		ramdisk_load_addr = (char *) n;

		/* remove alignment size added previously */
		v->bi.ramdisk.size -= 4;

		loading("Ramdisk Image");

		stream_init();
		stream_push(&file_mod);

		if (sopen(boot_record->ramdisk) < 0)
		{
			printf("\nUnable to open ramdisk file `%s'.\n",
					boot_record->ramdisk);
			goto fail;
		}

		if (sread(ramdisk_load_addr, v->bi.ramdisk.size) < 0)
		{
			sclose();
			printf("\nFailed to read ramdisk file\n");
			goto fail;
		}

		sclose();
		percent_term("OK\n");

		/* make sure ramdisk will sit on a 1K boundry at end of memory */
		v->bi.ramdisk.addr = mem_end - ((v->bi.ramdisk.size + 1023) & ~1023);
	}
	else
	{
		v->bi.ramdisk.size = 0;
		v->bi.ramdisk.addr = 0;
		ramdisk_load_addr  = NULL;
	}

#if SYMBOL_SUPPORT
	if (boot_record->symtab)
	{
		load_symbols(boot_record->symtab);
	}
#endif

    /* Check kernel's bootinfo version */
    kernel_major = check_bootinfo_version(kernel_load_addr);

	/* build kernel command line
	 */

	/* environment variable indicating loaded boot record */
	strncpy(v->bi.command_line, "BOOT_IMAGE=",      CL_SIZE);
	strcatn(v->bi.command_line, boot_record->label, CL_SIZE);

	/* add 'auto' flag if we didn't ask for input */
	if (!v->force_prompt)
	{
		strcatn(v->bi.command_line, " auto", CL_SIZE);
	}

	/* append boot record root device */
	if (boot_record->root)
	{
		strcatn(v->bi.command_line, " root=",          CL_SIZE);
   		strcatn(v->bi.command_line, boot_record->root, CL_SIZE);
	}

	/* append boot record read-only/read-write flag */
	if (boot_record->read_only)
	{
   		strcatn(v->bi.command_line,
				*boot_record->read_only ? " ro" : " rw", CL_SIZE);
	}

	/* append boot record console device, for kernels later than 2.0.xx */
	if (boot_record->console && (kernel_major > 1))
	{
		strcatn(v->bi.command_line, " console=",          CL_SIZE);
   		strcatn(v->bi.command_line, boot_record->console, CL_SIZE);
	}

   	/* append boot record command line arguments */
	if (boot_record->cmdline)
	{
		strcatn(v->bi.command_line, " ",                  CL_SIZE);
   		strcatn(v->bi.command_line, boot_record->cmdline, CL_SIZE);
	}

   	/* append user specified input */
   	if (v->cmdln_buffer[0])
   	{
		strcatn(v->bi.command_line, " ",             CL_SIZE);
   		strcatn(v->bi.command_line, v->cmdln_buffer, CL_SIZE);
   	}

   	/* append any additional arguments */
	if (boot_record->append)
	{
		strcatn(v->bi.command_line, " ",                 CL_SIZE);
   		strcatn(v->bi.command_line, boot_record->append, CL_SIZE);
   	}

   	if (v->debug_mode)
   	{
		printf("Using command line `%s'.\n", v->bi.command_line);
	}

   	if (kernel_major == BI_VERSION_MAJOR(get_booti_version()))
	{
	    /* create the bootinfo structure */
	    if (!create_bootinfo())
   		{
			printf("\nCouldn't create bootinfo.\n");
			goto fail;
		}

		bi_ptr = &v->bi_union.record;
	}

#if BOOTINFO_COMPAT_1_0
	else if (kernel_major == BI_VERSION_MAJOR(get_compat_booti_version()))
	{
		/* fix machine type for 2.0.xx kernels */
		v->bi.machtype = get_compat_machtype();

		if (!create_compat_bootinfo())
		{
    		printf("\nCouldn't create compatable bootinfo.\n");
    		goto fail;
    	}

		bi_ptr     = &v->compat_bootinfo;
		v->bi_size = sizeof(v->compat_bootinfo);
	}
#endif /* BOOTINFO_COMPAT_1_0 */

	else
	{
		printf("Kernel has unsupported bootinfo version.\n");
		goto fail;
   	}

	/* copy the bootinfo to the end of the kernel image */
	mem_move(kernel_load_addr + kernel_size, bi_ptr, v->bi_size);

 	if (v->debug_mode)
	{
		if (v->bi.ramdisk.size)
		{
			printf("RAM disk at 0x%08lx, size is %lu bytes.\n",
					(unsigned long) ramdisk_load_addr, v->bi.ramdisk.size
			);
		}

		printf("Boot info at 0x%08lx.\n", mem_start + kernel_size);

		if (!v->callmonitor)
		{
			put_str("\nPress a key to continue booting... ");
			get_char(NO_TIMEOUT);
			put_char('\n');
		}
	}

	/* move the copy-and-go code to it's proper place */
	mem_move((void *)v->startcode_entry, &startcode_beg,
								&startcode_end - &startcode_beg);
	disable_icache(v->cpu_type);

	/* execute the copy-and-go code (**never returns**) */
	start_kernel(
		v->startcode_entry,
		mem_start,          kernel_load_addr,  kernel_size + v->bi_size,
		v->bi.ramdisk.addr, ramdisk_load_addr, v->bi.ramdisk.size,
		v->callmonitor);

fail:
	free(kernel_load_addr);
}

/*--------------------------------------------------------------------------*/

void
start_loader
(	void
)
{	const BOOTRECORD	*boot_record;
	unsigned long		delay;
	unsigned long		timeout;
	char				save_force_prompt;
	int					restricted;

	/* determine CPU and FPU types */
	v->cpu_type = get_cpu_type();
	v->fpu_type = get_fpu_type();

	/* enable instruction cache */
	enable_icache(v->cpu_type);

	/* the kernel start code is to be placed at the bottom of
	 * the area we have reserved for the stack
	 */
	v->startcode_entry = (void (*)(void))(v->detected_mem_size - STACK_SIZE);

	get_debug_tag();

	/* get configuration */
	read_map_data();

	/* find the default boot record */
	if ((v->default_boot_record = find_boot_record(NULL)) == NULL)
	{
		panic("No boot Operating System!");
	}

	save_force_prompt = v->force_prompt;

	/* display boot message file */
	if (v->boot_options.boot_message)
	{
		display_message_file(v->boot_options.boot_message);
	}

restart:

	/* restore original debug modes */
	v->callmonitor   = FALSE;
	v->debug_mode    = VALUE(v->boot_options.boot_debug);

	v->leader_adjust = put_str("LILO ");

	if (!v->force_prompt)
	{
		/* get delay */
		delay = VALUE(v->boot_options.boot_delay);

		/* if a delay was specified */
		if (delay)
		{
			/* wait a while for a key press */
			if (get_char(delay) != -1)
			{
				/* key pressed, force boot prompt */
				v->force_prompt = TRUE;
			}
		}
	}

	/* get timeout */
	timeout = VALUE(v->boot_options.boot_timeout);

	while (1)
	{
		/* clear old command line */
		v->cmdln_buffer[0] = '\0';

		/* if interactive */
		if (v->force_prompt)
		{
			/* ask for boot record */
			boot_record = get_boot_record(timeout);
		}
		else
		{
			/* use default boot record */
			boot_record = v->default_boot_record;
		}

		v->callmonitor = v->callmonitor || VALUE(boot_record->callmonitor);
		restricted     = VALUE(boot_record->restricted);

		/* if there's a password */
		if (boot_record->password)
		{
			if (!restricted || (restricted && v->cmdln_buffer[0]))
			{
				put_str("Password: ");
				if (read_line(ECHO_DOT, timeout) == -1)
				{
					/* timeout */
					v->force_prompt = save_force_prompt;
					goto restart;
				}
	
				if (!case_equal_strings(v->input_buffer,
										boot_record->password))
				{
					put_str("\nIncorrect password.\n\n");
					goto again;
				}
			}
		}
		else
		{
			if (restricted && v->cmdln_buffer[0])
			{
				put_str(
	"Restricted boot configuration, extra command line options ignored.\n\n");
				v->cmdln_buffer[0] = '\0';
			}
		}

		/* go for it */
		boot_linux(boot_record);

		printf("\nCouldn't boot the %s configuration.\n\n",
				v->force_prompt ? "specified" : "default");

	again:
		v->force_prompt = TRUE;
	}
}

/*--------------------------------------------------------------------------*/

void
Main
(	CALLREGS			*regs
)
{	unsigned long		free_mem_start;
	unsigned long		detected_mem_size;

	/* do board specific initialisations */
	loader_init(regs);

	/* detect memory size */
	detected_mem_size = MEMORY_SIZE;
#if MEMORY_SIZE == 0
	while (ram_probe(MEMORY_BASE_ADDR + detected_mem_size))
	{
		detected_mem_size += MEMORY_SCAN_INCREMENT;
	}
#endif

	/* calculate start of free memory address */
	free_mem_start = ((unsigned long)&Map.Data[Map.Size] + 3) & ~3;

	/* initialise heap */
	malloc_init((void *)free_mem_start,
			detected_mem_size -  STACK_SIZE - free_mem_start);

	/* allocate and initialise global storage */
	v = malloc(sizeof(*v));
	mem_clear(v, sizeof(*v));

	/* remember detected memory size */
	v->detected_mem_size = detected_mem_size;
	
	/* move stack pointer to end of detected memory
	 * and execute: start_loader(void);
	 */
	asm volatile (	"movel	%0,%%sp\n"		/* switch stack					*/
					"jmp	(%1)\n"			/* jump to function				*/
					: /* no outputs */
					: "r" (detected_mem_size), "a" (&start_loader)
					: "sp", "memory"
	);
}

/*-----------------------------< end of file >------------------------------*/
