/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include "vmelilo.h"
#include "bvmbug.h"

#include "bvmeipl.data"
#include "bvmeloader.data"

/*--------------------------------------------------------------------------*/
/* IPL fragment list element
 */
typedef struct {
	u_short		count;			/* number of sectors						*/
	u_long		block;			/* sector address or zero for hole			*/

} IPLFRAG;

/*--------------------------------------------------------------------------*/

void
bvme_init_boot_block
(	void			*buf
)
{	BOOTBLOCK		*bb = (BOOTBLOCK *) buf;
	const FILEMAP	*map;
	IPLFRAG			*frp;
	int				i;
	u_long			maxfrags;
	u_long			n1;
	u_long			n2;

	/* zero initialise boot block */
	memset(bb, 0, sizeof(BOOTBLOCK));

	/* set boot block id */
	strcpy(bb->boot_id, BOOT_BLOCK_ID);

	/* say boot data holds IPL code */
	bb->boot_start_block = 0;
	bb->boot_block_count = 0;

	/* load and run IPL code at address 0x20000 */
	bb->boot_load_address = 0x20000;
	bb->boot_run_offset   = 0;

	/* get maximum number of fragments allowed in boot block */
	maxfrags = (sizeof(bb->boot_data) - (sizeof(bvmeipl_code) + 2))
					/ sizeof(IPLFRAG);

	if (MAP_NUMFRAGS(loader_map) > maxfrags)
	{
		n1 = MAP_NUMFRAGS(loader_map) - maxfrags;

		die("Boot loader file is too fragmented for boot block "
			"(%ld fragment%s too many)\n",
					n1, (n1 == 1) ? "" : "s");
	}

	if (f_verbose)
	{
		n1 = MAP_NUMFRAGS(loader_map);
		n2 = maxfrags - MAP_NUMFRAGS(loader_map);

		printf(	"Boot loader file has %ld fragment%s, "
				"enough space for %lu more\n",
				  	n1, (n1 == 1) ? "" : "s", n2);
	}

	/* put IPL code into boot block */
	memcpy(bb->boot_data, bvmeipl_code, sizeof(bvmeipl_code));

	/* get word aligned pointer to fragment list for the IPL */
	frp = (IPLFRAG *) &bb->boot_data[(sizeof(bvmeipl_code) + 1) & ~1];

	/* get pointer to first file fragment of the loader */
	map = MAP_FIRSTFRAG(loader_map);

	/* put file map of loader into boot block */
	i = MAP_NUMFRAGS(loader_map);
	while (i--)
	{
		frp->count = map->length;
		frp->block = map->offset;
		frp++;
		map++;
	}
}

/*--------------------------------------------------------------------------*/

int
bvme_valid_boot_block
(	void		*buf
)
{	BOOTBLOCK	*bb = (BOOTBLOCK *) buf;
	
	return (strcmp(bb->boot_id, BOOT_BLOCK_ID) == 0);
}

/*--------------------------------------------------------------------------*/

SUBARCH bvme_subarch = {
	"BVMBug",
	bvmeloader_code,
	sizeof(bvmeloader_code),
	sizeof(BOOTBLOCK),
	bvme_init_boot_block,
	bvme_valid_boot_block
};

/*-----------------------------< end of file >------------------------------*/
