/* savetab.c: saving and restoring the partition table
 *
 * Copyright (C) 1995-97 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *               1996-97 Michael Schlueter <schlue00@marvin.informatik.uni-dortmund.de>
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */

/* $Id: savetab.c,v 1.3 1997/06/22 10:30:57 rnhodek Exp $
 *
 * $Log: savetab.c,v $
 * Revision 1.3  1997/06/22 10:30:57  rnhodek
 * Add __attribute__((unused)) to cvid
 *
 * Revision 1.2  1997/06/21 20:47:50  rnhodek
 * Added RCS keywords
 *
 * Revision 1.1  1997/06/11 14:36:36  rnhodek
 * Initial revision
 * 
 * Revision 1.1.1.1  1997/06/11 14:36:36  rnhodek
 * Started using CVS for atafdisk
 *
 */

#ifndef lint
static char vcid[] __attribute__ ((unused)) =
"$Id: savetab.c,v 1.3 1997/06/22 10:30:57 rnhodek Exp $";
#endif /* lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "fdisk.h"
#include "disk.h"
#include "writepart.h"


/***************************** Prototypes *****************************/

static void ulong_to_chars( unsigned long u, char *uu );
static unsigned long chars_to_ulong( unsigned char *uu );
static int save_one_sector( unsigned long secnum, int fdout );

/************************* End of Prototypes **************************/


static void ulong_to_chars( unsigned long u, char *uu )
{
    int i;

    for( i = 0; i < 4; i++ ) {
	uu[i] = (u & 0xff);
	u >>= 8;
    }
}

static unsigned long chars_to_ulong( unsigned char *uu )
{
    int i;
    unsigned long u = 0;

    for( i=3; i >= 0; i-- )
	u = (u << 8) | uu[i];
    return u;
}

static int save_one_sector( unsigned long secnum, int fdout )
{
    char ss[516];

    ulong_to_chars( secnum, ss );
    
    if (!sseek( fd, secnum ))
	return 0;
    if (read( fd, ss+4, SECTOR_SIZE) != SECTOR_SIZE) {
	perror("read");
	fprintf( stderr, "read error on sector %lu\n", secnum );
	return 0;
    }
    if (write( fdout, ss, sizeof(ss)) != sizeof(ss)) {
	perror("write");
	fprintf( stderr, "write error on %s\n", save_sector_file );
	return 0;
    }
    return 1;
}

int save_sectors( void )
{
    int fdout, i;

    fdout = open(save_sector_file, O_WRONLY | O_CREAT, 0444);
    if (fdout < 0) {
	perror(save_sector_file);
	fprintf( stderr, "cannot open partition sector save file (%s)\n",
		 save_sector_file );
	return 0;
    }

    /* sector 0 always contains precious data */
    if (!save_one_sector( 0, fdout )) goto err;
    
    for( i = 0; i < partitions; ++i ) {
	if (IS_EXTENDED(part_table[i]))
	    if (!save_one_sector( part_table[i].rootsec, fdout ))
		goto err;
    }
    close( fdout );
    return 1;

  err:
    close( fdout );
    fprintf( stderr, "saving failed!\n" );
    return 0;
}

int restore_sectors( const char *dev )
{
    int fdin, fdout, ct;
    struct stat statbuf;
    char *ss0, *ss;
    unsigned long sno;

    if (stat( restore_sector_file, &statbuf ) < 0) {
	perror( restore_sector_file );
	fprintf( stderr, "cannot stat partition restore file (%s)\n",
		 restore_sector_file );
	return 0;
    }
    if (statbuf.st_size % 516) {
	fprintf( stderr, "partition restore file has wrong size - "
		 "not restoring\n" );
	return 0;
    }
    if (!(ss = (char *)malloc( statbuf.st_size ))) {
	fprintf( stderr, "out of memory?\n" );
	return 0;
    }
    if ((fdin = open( restore_sector_file, O_RDONLY )) < 0) {
	perror( restore_sector_file );
	fprintf( stderr, "cannot open partition restore file (%s)\n",
		 restore_sector_file );
	return 0;
    }
    if (read( fdin, ss, statbuf.st_size ) != statbuf.st_size) {
	perror( "read" );
	fprintf( stderr, "error reading %s\n", restore_sector_file );
	return 0;
    }
    close( fdin );
    
    fdout = open( dev, O_WRONLY );
    if (fdout < 0) {
	perror( dev );
	fprintf( stderr, "cannot open device %s for writing\n", dev );
	return 0;
    }

    ss0 = ss;
    ct = statbuf.st_size/516;
    while( ct-- ) {
	sno = chars_to_ulong( ss );
	if (!dangerous_write( fdout, ss+4, sno )) {
	    perror( dev );
	    fprintf( stderr, "error writing sector %lu on %s\n", sno, dev );
	    return 0;
	}
	ss += 516;
    }
    free(ss0);

    reread_disk_partition( fdout );
    close( fdout );
    
    return 1;
}

/* Local Variables: */
/* tab-width: 8     */
/* End:             */
