#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

/*
	Translate an IP number (ascii) into four number
	Return -1 if the IP number is invalid.
*/
static int ipnum_aip24 (const char *aip, int num4[4])
{
	int ret = 0;
	int i;
	for (i=0; i<4; i++){
		if (!isdigit(aip[0])){
			ret = -1;
			break;
		}else{
			int no = num4[i] = atoi(aip);
			if (no < 0 || no > 255) ret = -1;
			while (isdigit(*aip)) aip++;
			if (i < 3){
				if (*aip != '.'){
					ret = -1;
					break;
				}
				aip++;
			}
		}
	}
	if (*aip != '\0' && !isspace(*aip)) ret = -1;
	return ret;
}

/*
	Check is a string is valid IP number.
	if (ishost), check if the IP is not a broadcast or a network address.
*/
static int ipnum_validip(const char *aip, int ishost)
{
	int num4[4];
	int ret = 0;
	if (ipnum_aip24(aip,num4) != -1){
		ret = 1;
		if (ishost && (num4[3] == 0 || num4[3] == 255)){
			ret = 0;
		}
	}
	return ret;
}
/*
	Compute an IP number from a string.
*/
unsigned long ipnum_aip2l (const char *aip)
{
	unsigned long ret = 0xffffffffl;
	int n[4];
	if (ipnum_aip24(aip,n) != -1){
		ret = (n[0] << 24) + (n[1] << 16) + (n[2] << 8) + n[3];
	}
	return ret;
}

static int sockutil_sinconnect (struct sockaddr_in *sin, int nbretry)
{
	int ret = -1;
	int i;
	for (i=0; i<nbretry; i++){		
		int s;
		if ((s = socket (AF_INET, SOCK_STREAM, 0)) >= 0) {
			if (connect (s, (struct sockaddr *) sin, sizeof (*sin))
				== -1){
				if (i==0){
					fprintf (stderr,"Can't connect (%s)\n",strerror(errno));
				}
				close (s);
				sleep(1);
			}else{
				ret = s;
				break;
			}
		}else{
			/* logdebug (DBGALL,"socket"); */
		}
	}
	return ret;
}

/*
	Obtient le port de communication via /etc/service
*/
static int sockutil_getport(const char *service)
{
	int ret = -1;
	struct servent *serv = getservbyname (service,"tcp");
	if (serv == NULL){
		fprintf (stderr,"No service %s in /etc/service\n",service);
	}else{
		ret = ntohs(serv->s_port);
	}
	return ret;
}


int sockutil_connectn (const char *servname, int port, int nbretry)
{
	int ret = -1;
	struct sockaddr_in sin;
	sin.sin_family = AF_INET;
	sin.sin_port = htons(port);
	if (ipnum_validip (servname,1)){
		long addr = htonl(ipnum_aip2l (servname));
		memcpy  (&sin.sin_addr,&addr, sizeof(addr));
		ret = sockutil_sinconnect (&sin,nbretry);
	}else{
		struct hostent	*h = (struct hostent *) gethostbyname(servname);
		if (h == NULL){
			fprintf (stderr,"No server \"%s\" defined\n",servname);
		}else{
			memcpy  (&sin.sin_addr,h->h_addr, h->h_length);
			ret = sockutil_sinconnect (&sin,nbretry);
		}
	}
	return ret;
}
/*
	Etablie la connexion avec un serveur.
	Retourne -1 si erreur.
*/
int sockutil_connect (const char *servname, const char *portname, int nbretry)
{
	int port = sockutil_getport(portname);
	int ret = -1;
	if (port != -1){
		ret = sockutil_connectn (servname,port,nbretry);
	}
	return ret;
}

