/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "menus.h"
#include "preferences.h"
#include "sources.h"

// This file includes both the popup and main menus.
//  The data arg is always a pointer to the GAptPkgList

// Basically we are hiding all kinds of horrible grot in here.
// Menu code is unavoidably one big mess.

// Columns
static void name_column_cb         (GtkWidget* w, gpointer data);
static void status_column_cb       (GtkWidget* w, gpointer data);
static void installed_column_cb    (GtkWidget* w, gpointer data);
static void available_column_cb    (GtkWidget* w, gpointer data);
static void section_column_cb      (GtkWidget* w, gpointer data);
static void priority_column_cb     (GtkWidget* w, gpointer data);
static void description_column_cb  (GtkWidget* w, gpointer data);

// Order
static void alpha_order_cb        (GtkWidget* w, gpointer data);
static void section_order_cb      (GtkWidget* w, gpointer data);
static void status_order_cb       (GtkWidget* w, gpointer data);
static void priority_order_cb     (GtkWidget* w, gpointer data);

// Group
static void alpha_group_cb        (GtkWidget* w, gpointer data);
static void section_group_cb      (GtkWidget* w, gpointer data);
static void status_group_cb       (GtkWidget* w, gpointer data);
static void priority_group_cb     (GtkWidget* w, gpointer data);

// File
static void apt_prefs_cb   (GtkWidget* w, gpointer data);
static void sources_cb     (GtkWidget* w, gpointer data);
static void exit_cb        (GtkWidget* w, gpointer data);

// Actions
static void update_cb      (GtkWidget* w, gpointer data);
static void complete_run_cb (GtkWidget* w, gpointer data);
static void upgrade_cb      (GtkWidget* w, gpointer data);
static void smart_upgrade_cb(GtkWidget* w, gpointer data);
static void fix_broken_cb   (GtkWidget* w, gpointer data);

// Package
static void details_cb      (GtkWidget* w, gpointer data);
static void search_cb       (GtkWidget* w, gpointer data);
static void install_pkg_cb     (GtkWidget* w, gpointer data);
static void configure_pkg_cb   (GtkWidget* w, gpointer data);
static void upgrade_pkg_cb     (GtkWidget* w, gpointer data);
static void purge_pkg_cb       (GtkWidget* w, gpointer data);
static void remove_pkg_cb      (GtkWidget* w, gpointer data);
static void keep_pkg_cb        (GtkWidget* w, gpointer data);

// Advanced
static void export_cb      (GtkWidget* w, gpointer data);
static void import_cb      (GtkWidget* w, gpointer data);
static void bug_cb         (GtkWidget* w, gpointer data);

// Help
static void about_cb       (GtkWidget* w, gpointer data);
static void icons_cb       (GtkWidget* w, gpointer data);

static GnomeUIInfo columns_menu[] = {
#define COLUMNS_NAME 0
  {GNOME_APP_UI_TOGGLEITEM, N_("Package Name"), 
   N_("Whether to show the package name column"),
   name_column_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define COLUMNS_CURRENT 1
  {GNOME_APP_UI_TOGGLEITEM, N_("Installed Version"), 
   N_("Whether to show the installed version column"),
   installed_column_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define COLUMNS_AVAILABLE 2
  {GNOME_APP_UI_TOGGLEITEM, N_("Available Version"), 
   N_("Whether to show the available version column"),
   available_column_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define COLUMNS_SECTION 3
  {GNOME_APP_UI_TOGGLEITEM, N_("Package Section"), 
   N_("Whether to show the package section column"),
   section_column_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define COLUMNS_PRIORITY 4
  {GNOME_APP_UI_TOGGLEITEM, N_("Package Priority"), 
   N_("Whether to show the package priority column"),
   priority_column_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define COLUMNS_STATUS 5
  {GNOME_APP_UI_TOGGLEITEM, N_("Package Status"), 
   N_("Whether to show the package status column"),
   status_column_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define COLUMNS_DESCRIPTION 6
  {GNOME_APP_UI_TOGGLEITEM, N_("Description"), 
   N_("Whether to show the package description column"),
   description_column_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo order_items[] = {
#define ORDERS_ALPHA 0
  {GNOME_APP_UI_ITEM, N_("Alphabetical"), 
   N_("Show packages in alphabetical order"),
   alpha_order_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define ORDERS_SECTION 1
  {GNOME_APP_UI_ITEM, N_("By section"), 
   N_("Show sections together"),
   section_order_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define ORDERS_STATUS 2
  {GNOME_APP_UI_ITEM, N_("By status"), 
   N_("Show packages with the same status together"),
   status_order_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define ORDERS_PRIORITY 3
  {GNOME_APP_UI_ITEM, N_("By priority"), 
   N_("Show packages with the same priority together"),
   priority_order_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
  // FIXME
  GNOMEUIINFO_END
};

static GnomeUIInfo order_menu[] = {
  GNOMEUIINFO_RADIOLIST(order_items),
  GNOMEUIINFO_END
};


static GnomeUIInfo group_items[] = {
#define GROUP_ALPHA 0
  {GNOME_APP_UI_ITEM, N_("Alphabetical"), 
   N_("Show packages in alphabetical order"),
   alpha_group_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define GROUP_SECTION 1
  {GNOME_APP_UI_ITEM, N_("By section"), 
   N_("Show sections together"),
   section_group_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define GROUP_STATUS 2
  {GNOME_APP_UI_ITEM, N_("By status"), 
   N_("Show packages with the same status together"),
   status_group_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
#define GROUP_PRIORITY 3
  {GNOME_APP_UI_ITEM, N_("By priority"), 
   N_("Show packages with the same priority together"),
   priority_group_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
  // FIXME
  GNOMEUIINFO_END
};

static GnomeUIInfo group_menu[] = {
  GNOMEUIINFO_RADIOLIST(group_items),
  GNOMEUIINFO_END
};

static GnomeUIInfo popup_menu[] = {
  GNOMEUIINFO_SEPARATOR,
#define POPUP_REMOVE 1
  {GNOME_APP_UI_ITEM, N_("Delete"), 
   N_("Mark this package to be removed (preserve configuration files)"),
   remove_pkg_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },

#define POPUP_KEEP 2
  {GNOME_APP_UI_ITEM, N_("Keep"), 
   N_("Keep the currently selected package"),
   keep_pkg_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },

#define POPUP_INSTALL 3
  {GNOME_APP_UI_ITEM, N_("Install/upgrade"), 
   N_("Mark this package to be installed"),
   install_pkg_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },

  {GNOME_APP_UI_ITEM, N_("Search..."), 
   N_("Find a package by name or other criteria"),
   search_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SEARCH, '\0', 
   (GdkModifierType)0, NULL },  

  GNOMEUIINFO_SUBTREE (N_("Columns"), columns_menu),
  GNOMEUIINFO_SUBTREE (N_("Order"),   order_menu),
  GNOMEUIINFO_SUBTREE (N_("Group"),   group_menu),

  GNOMEUIINFO_END
};


static GnomeUIInfo view_menu[] = {
  {GNOME_APP_UI_ITEM, N_("Details..."), 
   N_("Show package information in a separate window"),
   details_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_OPEN, 'D', 
   (GdkModifierType)GDK_CONTROL_MASK, NULL },

  GNOMEUIINFO_SUBTREE (N_("Columns"), columns_menu),
  GNOMEUIINFO_SUBTREE (N_("Order"),   order_menu),
  GNOMEUIINFO_SUBTREE (N_("Group"),   group_menu),

  GNOMEUIINFO_END
};


static GnomeUIInfo file_menu[] = {
  {GNOME_APP_UI_ITEM, N_("General _preferences..."), 
   N_("Change general Apt preferences"),
   apt_prefs_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF,  '\0', 
   (GdkModifierType)0, NULL },
  {GNOME_APP_UI_ITEM, N_("_Sources..."), 
   N_("Choose where Apt looks for packages"),
   sources_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, '\0', 
   (GdkModifierType)0, NULL },
  GNOMEUIINFO_SEPARATOR,
  {GNOME_APP_UI_ITEM, N_("_Exit"), 
   N_("Quit the application"),
   exit_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 
   GNOME_KEY_NAME_EXIT, GNOME_KEY_MOD_EXIT,
   NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo actions_menu[] = {
  {GNOME_APP_UI_ITEM, N_("_Update"), 
   N_("Update the package list from chosen sources"),
   update_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH, 'U', 
   (GdkModifierType)GDK_CONTROL_MASK, NULL },

  {GNOME_APP_UI_ITEM, N_("_Complete run"), 
   N_("Install, remove, and delete packages as requested."),
   complete_run_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXEC, 'R', 
   (GdkModifierType)GDK_CONTROL_MASK, NULL },
  {GNOME_APP_UI_ITEM, N_("Mark _Upgrades"), 
   N_("Mark all out-of-date packages for upgrade, without removals or installs."),
   upgrade_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
  {GNOME_APP_UI_ITEM, N_("_Smart Mark Upgrades"), 
   N_("Mark upgrades, removing or installing packages if needed."),
   smart_upgrade_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
  {GNOME_APP_UI_ITEM, N_("_Fix Broken"), 
   N_("Fix broken packages, removing or installing packages if needed."),
   fix_broken_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 'F', 
   (GdkModifierType)GDK_CONTROL_MASK, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo package_menu[] = {
  {GNOME_APP_UI_ITEM, N_("Search..."), 
   N_("Find a package by name or other criteria"),
   search_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SEARCH, 'S', 
   (GdkModifierType)GDK_CONTROL_MASK, NULL },  
#define PACKAGE_REMOVE 1
  {GNOME_APP_UI_ITEM, N_("_Delete"), 
   N_("Uninstall package, keeping configuration."),
   remove_pkg_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 'D', 
   (GdkModifierType)0, NULL },
#define PACKAGE_KEEP 2
  {GNOME_APP_UI_ITEM, N_("_Keep"), 
   N_("Don't change this package."),
   keep_pkg_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 'K', 
   (GdkModifierType)0, NULL },
#define PACKAGE_INSTALL 3
  {GNOME_APP_UI_ITEM, N_("_Install/Upgrade"), 
   N_("Install this package."),
   install_pkg_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 'I', 
   (GdkModifierType)0, NULL },
  GNOMEUIINFO_END
};


static GnomeUIInfo advanced_menu[] = {
  {GNOME_APP_UI_ITEM, N_("_Export packagelist..."), 
   N_("Save the current state of the package list to disk for use on another system."),
   export_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, '\0', 
   (GdkModifierType)0, NULL },
  {GNOME_APP_UI_ITEM, N_("_Import packagelist..."), 
   N_("Load a package list."),
   import_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, '\0', 
   (GdkModifierType)0, NULL },
  {GNOME_APP_UI_ITEM, N_("_Bug report..."), 
   N_("File a bug against this package."),
   bug_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0', 
   (GdkModifierType)0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo help_menu[] = {
  GNOMEUIINFO_HELP(const_cast<char*>(APPNAME)),
  {GNOME_APP_UI_ITEM, N_("_About..."), 
   N_("Tell about this application"), 
   about_cb, NULL, NULL,
   GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT, '\0',  
   (GdkModifierType)0, NULL },
  {GNOME_APP_UI_ITEM, N_("_Icon Key"), 
   N_("Explain what all the icons in the tree view mean."), 
   icons_cb, NULL, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, '\0',  
   (GdkModifierType)0, NULL },
  GNOMEUIINFO_END
};

static GnomeUIInfo main_menu[] = {
  GNOMEUIINFO_SUBTREE (N_("_File"), file_menu),
  GNOMEUIINFO_SUBTREE (N_("_Actions"), actions_menu),
  GNOMEUIINFO_SUBTREE (N_("_Package"), package_menu),
  GNOMEUIINFO_SUBTREE (N_("_View"), view_menu),
  GNOMEUIINFO_SUBTREE (N_("A_dvanced"), advanced_menu),
  GNOMEUIINFO_SUBTREE (N_("_Help"), help_menu),
  GNOMEUIINFO_END
};


///////////////////////////////// 
// Menu-handling code

// this is a hack, because gnome-app-helper doesn't let us 
//  set state before connecting callbacks
static bool changing_state = false;


// God this is gruesome. This kind of fucked up crap should
//  not be necessary.
struct MenuWidgets {
  GtkWidget* popup_install;
  GtkWidget* popup_remove;
  GtkWidget* popup_keep;

  GtkWidget* main_install;
  GtkWidget* main_remove;
  GtkWidget* main_keep;

  map<GAptPkgTree::ColumnType,GtkWidget*> popup_column;
  map<GAptPkgTree::ColumnType,GtkWidget*> main_column;

  map<GAptPkgTree::CategoryType,GtkWidget*> popup_group;
  map<GAptPkgTree::CategoryType,GtkWidget*> main_group;

  map<GAptPkgTree::SortType,GtkWidget*> popup_order;
  map<GAptPkgTree::SortType,GtkWidget*> main_order;
  
};

static MenuWidgets widgets;

// Callback for button press on the package tree

static gint
tree_button_press(GtkWidget* da, GdkEventButton* event, gpointer data)
{
  if (event->button != 3) return 0;
  else 
    {
      GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

      g_return_val_if_fail(pkglist != 0, FALSE);

      // Select the package under the pointer
      pkglist->tree()->tree()->change_selection((gint)event->x, (gint)event->y);

      GtkWidget* popup = (GtkWidget*)gtk_object_get_data(GTK_OBJECT(da),
                                                         "popup");
      g_return_val_if_fail(popup != 0, FALSE);

      GtkWidget* pkgname = (GtkWidget*)gtk_object_get_data(GTK_OBJECT(popup),
                                                           "package_name");

      if (pkgname != NULL)
        {
          gtk_container_remove(GTK_CONTAINER(popup),
                               pkgname);
        }

      pkgCache::Package* pkg = pkglist->get_selection();

      const char* s = 0;

      if (pkg == 0)
        {
          s = _("No package selected");
        }
      else 
        {
          pkgCache::PkgIterator i(*(pkglist->tree()->cache()), pkg);
          s = i.Name();
        }

      g_return_val_if_fail(s != 0, FALSE);

      pkgname = gtk_menu_item_new_with_label(s);

      gtk_widget_set_name(GTK_BIN(pkgname)->child, "AptPopupTitle");

      gtk_signal_connect_object(GTK_OBJECT(pkgname),
                                "select",
                                GTK_SIGNAL_FUNC(gtk_item_deselect),
                                GTK_OBJECT(pkgname));

      gtk_menu_prepend(GTK_MENU(popup), pkgname);

      gtk_widget_show(pkgname);

      gtk_object_set_data(GTK_OBJECT(popup),
                          "package_name",
                          pkgname);
      
      gnome_popup_menu_do_popup(popup, NULL, NULL,
                                event, data);

      return 1;
    }
}

void 
gnome_apt_create_pkglist_menu(GAptPkgList* pkglist, GnomeAppBar* bar)
{
  g_return_if_fail(widgets.popup_column.empty());
  
  GtkWidget* popup = gnome_popup_menu_new(popup_menu);

  GAptPkgTree* tree = pkglist->tree();

  g_return_if_fail(tree != 0);

  changing_state = true;  

  // columns

  widgets.popup_column[GAptPkgTree::ColumnName] = 
    columns_menu[COLUMNS_NAME].widget;

  widgets.popup_column[GAptPkgTree::ColumnCurrent] = 
    columns_menu[COLUMNS_CURRENT].widget;

  widgets.popup_column[GAptPkgTree::ColumnAvailable] = 
    columns_menu[COLUMNS_AVAILABLE].widget;

  widgets.popup_column[GAptPkgTree::ColumnSection] = 
    columns_menu[COLUMNS_SECTION].widget;

  widgets.popup_column[GAptPkgTree::ColumnPriority] = 
    columns_menu[COLUMNS_PRIORITY].widget;

  widgets.popup_column[GAptPkgTree::ColumnStatus] = 
    columns_menu[COLUMNS_STATUS].widget;

  widgets.popup_column[GAptPkgTree::ColumnDescription] = 
    columns_menu[COLUMNS_DESCRIPTION].widget;

  map<GAptPkgTree::ColumnType,GtkWidget*>::iterator i = widgets.popup_column.begin();
  while (i != widgets.popup_column.end()) {
    
#ifdef GNOME_ENABLE_DEBUG
    if (i->second == 0) g_warning("widget == 0 in popup creation");
#endif

    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second),
                                  tree->is_visible(i->first));
    
    ++i;
  }

  // groups

  widgets.popup_group[GAptPkgTree::CategoryAlpha] = 
    group_items[GROUP_ALPHA].widget;

  widgets.popup_group[GAptPkgTree::CategoryStatus] = 
    group_items[GROUP_STATUS].widget;

  widgets.popup_group[GAptPkgTree::CategorySection] = 
    group_items[GROUP_SECTION].widget;

  widgets.popup_group[GAptPkgTree::CategoryPriority] = 
    group_items[GROUP_PRIORITY].widget;

  map<GAptPkgTree::CategoryType,GtkWidget*>::iterator j = 
    widgets.popup_group.find(pkglist->tree()->get_category());

#ifdef GNOME_ENABLE_DEBUG
  if (j == widgets.popup_group.end()) g_warning("category not found in popup menu");
    if (j->second == 0) g_warning("widget == 0 in popup creation");
#endif

    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(j->second),
                                  TRUE);
  
    // sort orders
    
    widgets.popup_order[GAptPkgTree::SortAlpha] = 
      order_items[ORDERS_ALPHA].widget;
    
    widgets.popup_order[GAptPkgTree::SortStatus] = 
      order_items[ORDERS_STATUS].widget;
    
    widgets.popup_order[GAptPkgTree::SortSection] = 
      order_items[ORDERS_SECTION].widget;
    
    widgets.popup_order[GAptPkgTree::SortPriority] = 
      order_items[ORDERS_PRIORITY].widget;
    
    map<GAptPkgTree::SortType,GtkWidget*>::iterator k = 
      widgets.popup_order.find(pkglist->tree()->get_sort());
    
#ifdef GNOME_ENABLE_DEBUG
    if (k == widgets.popup_order.end()) g_warning("Sort not found in popup menu");
    if (k->second == 0) g_warning("widget == 0 in popup creation");
#endif
    
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(k->second),
                                  TRUE);


  // package operations

  widgets.popup_install = popup_menu[POPUP_INSTALL].widget;
  widgets.popup_keep    = popup_menu[POPUP_KEEP].widget;
  widgets.popup_remove  = popup_menu[POPUP_REMOVE].widget;

  changing_state = false;

  gtk_object_set_data(GTK_OBJECT(pkglist->tree()->tree()->widget()),
                      "popup",
                      popup);

  gtk_signal_connect(GTK_OBJECT(pkglist->tree()->tree()->widget()),
                     "button_press_event",
                     GTK_SIGNAL_FUNC(tree_button_press),
                     pkglist);

  gnome_app_install_appbar_menu_hints(bar, popup_menu);
}

void 
gnome_apt_create_main_menu   (GnomeApp* app, GnomeAppBar* bar, GAptPkgList* pkglist)
{
  g_return_if_fail(widgets.main_column.empty());

  GAptPkgTree* tree = pkglist->tree();

  g_return_if_fail(tree != 0);

  gnome_app_create_menus_with_data(GNOME_APP(app), 
                                   main_menu,
                                   pkglist);

  changing_state = true;  

  // column visibility

  widgets.main_column[GAptPkgTree::ColumnName] = 
    columns_menu[COLUMNS_NAME].widget;

  widgets.main_column[GAptPkgTree::ColumnCurrent] = 
    columns_menu[COLUMNS_CURRENT].widget;

  widgets.main_column[GAptPkgTree::ColumnAvailable] = 
    columns_menu[COLUMNS_AVAILABLE].widget;

  widgets.main_column[GAptPkgTree::ColumnSection] = 
    columns_menu[COLUMNS_SECTION].widget;

  widgets.main_column[GAptPkgTree::ColumnPriority] = 
    columns_menu[COLUMNS_PRIORITY].widget;

  widgets.main_column[GAptPkgTree::ColumnStatus] = 
    columns_menu[COLUMNS_STATUS].widget;

  widgets.main_column[GAptPkgTree::ColumnDescription] = 
    columns_menu[COLUMNS_DESCRIPTION].widget;

  map<GAptPkgTree::ColumnType,GtkWidget*>::iterator i = widgets.main_column.begin();
  while (i != widgets.main_column.end()) {
    
#ifdef GNOME_ENABLE_DEBUG
    if (i->second == 0) g_warning("widget == 0 in main creation");
#endif

    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second),
                                  tree->is_visible(i->first));
    
    ++i;
  }

  // groups

  widgets.main_group[GAptPkgTree::CategoryAlpha] = 
    group_items[GROUP_ALPHA].widget;

  widgets.main_group[GAptPkgTree::CategoryStatus] = 
    group_items[GROUP_STATUS].widget;

  widgets.main_group[GAptPkgTree::CategorySection] = 
    group_items[GROUP_SECTION].widget;

  widgets.main_group[GAptPkgTree::CategoryPriority] = 
    group_items[GROUP_PRIORITY].widget;

  map<GAptPkgTree::CategoryType,GtkWidget*>::iterator j = 
    widgets.main_group.find(pkglist->tree()->get_category());

#ifdef GNOME_ENABLE_DEBUG
  if (j == widgets.main_group.end()) g_warning("category not found in main menu");
    if (j->second == 0) g_warning("widget == 0 in main creation");
#endif

    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(j->second),
                                  TRUE);

    // sort orders
    
    widgets.main_order[GAptPkgTree::SortAlpha] = 
      order_items[ORDERS_ALPHA].widget;
    
    widgets.main_order[GAptPkgTree::SortStatus] = 
      order_items[ORDERS_STATUS].widget;
    
    widgets.main_order[GAptPkgTree::SortSection] = 
      order_items[ORDERS_SECTION].widget;
    
    widgets.main_order[GAptPkgTree::SortPriority] = 
      order_items[ORDERS_PRIORITY].widget;
    
    map<GAptPkgTree::SortType,GtkWidget*>::iterator k = 
      widgets.main_order.find(pkglist->tree()->get_sort());
    
#ifdef GNOME_ENABLE_DEBUG
    if (k == widgets.main_order.end()) g_warning("Sort not found in main menu");
    if (k->second == 0) g_warning("widget == 0 in main creation");
#endif
    
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(k->second),
                                  TRUE);

  
  // package operations


  widgets.main_install = package_menu[PACKAGE_INSTALL].widget;
  widgets.main_keep    = package_menu[PACKAGE_KEEP].widget;
  widgets.main_remove  = package_menu[PACKAGE_REMOVE].widget;

  changing_state = false;

  gnome_app_install_appbar_menu_hints(bar, main_menu);
}

void 
gnome_apt_menus_selection_changed (GAptPkgList* pkglist)
{
  pkgCache::Package* p = pkglist->get_selection();

  bool install = false;
  bool keep    = false;
  bool remove  = false;

  if (p != 0) {
    install = Util::can_change_install(p,pkglist->tree());
    keep    = Util::can_change_keep   (p,pkglist->tree());
    remove  = Util::can_change_remove (p,pkglist->tree());
  }

  gtk_widget_set_sensitive(widgets.main_install,  install);
  gtk_widget_set_sensitive(widgets.popup_install, install);

  gtk_widget_set_sensitive(widgets.main_keep,  keep);
  gtk_widget_set_sensitive(widgets.popup_keep, keep);

  gtk_widget_set_sensitive(widgets.main_remove,  remove);
  gtk_widget_set_sensitive(widgets.popup_remove, remove);
}

////////////////////////////////////
// Menu callbacks

static void
showhide_column(GtkWidget* w, 
                GAptPkgList* pkglist, 
                GAptPkgTree::ColumnType ct)
{ 
  bool state = GTK_CHECK_MENU_ITEM(w)->active;

  pkglist->tree()->set_visible(ct, state);

  changing_state = true;
  
  // sync the item that didn't trigger the callback
  
  map<GAptPkgTree::ColumnType,GtkWidget*>::iterator i = widgets.popup_column.find(ct);
  g_return_if_fail (i != widgets.popup_column.end());
  
  if (i->second != w) 
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second), state);

  i = widgets.main_column.find(ct);

  g_return_if_fail (i != widgets.main_column.end());
  
  if (i->second != w) 
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second), state);

  changing_state = false;
}

static void
name_column_cb         (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);
  
  showhide_column(w, pkglist, GAptPkgTree::ColumnName);
}

static void
status_column_cb       (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  showhide_column(w, pkglist, GAptPkgTree::ColumnStatus);
}

static void
installed_column_cb    (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  showhide_column(w, pkglist, GAptPkgTree::ColumnCurrent);
}

static void
available_column_cb    (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  showhide_column(w, pkglist, GAptPkgTree::ColumnAvailable);
}

static void
section_column_cb      (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  showhide_column(w, pkglist, GAptPkgTree::ColumnSection);
}

static void
priority_column_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  showhide_column(w, pkglist, GAptPkgTree::ColumnPriority);
}


static void
description_column_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  showhide_column(w, pkglist, GAptPkgTree::ColumnDescription);
}


// Ordering

static void 
order_tree(GtkWidget* w, GAptPkgList* pkglist, GAptPkgTree::SortType st)
{
  pkglist->tree()->set_sort(st);

  // sync the item that didn't trigger the callback

  changing_state = true;
  
  map<GAptPkgTree::SortType,GtkWidget*>::iterator i = 
    widgets.popup_order.find(st);
  g_return_if_fail (i != widgets.popup_order.end());
  
  if (i->second != w) 
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second), TRUE);

  i = widgets.main_order.find(st);

  g_return_if_fail (i != widgets.main_order.end());
  
  if (i->second != w) 
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second), TRUE);  

  changing_state = false;
}

static void
alpha_order_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  order_tree(w, pkglist, GAptPkgTree::SortAlpha);
}

static void
section_order_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  order_tree(w, pkglist, GAptPkgTree::SortSection);
}

static void
status_order_cb         (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  order_tree(w, pkglist, GAptPkgTree::SortStatus);
}

static void
priority_order_cb       (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  order_tree(w, pkglist, GAptPkgTree::SortPriority);
}


// Grouping


static void 
group_tree(GtkWidget* w, GAptPkgList* pkglist, GAptPkgTree::CategoryType ct)
{
  pkglist->tree()->set_category(ct);

  changing_state = true;
  
  // sync the item that didn't trigger the callback
  
  map<GAptPkgTree::CategoryType,GtkWidget*>::iterator i = 
    widgets.popup_group.find(ct);
  g_return_if_fail (i != widgets.popup_group.end());
  
  if (i->second != w) 
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second), TRUE);

  i = widgets.main_group.find(ct);

  g_return_if_fail (i != widgets.main_group.end());
  
  if (i->second != w) 
    gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(i->second), TRUE);  

  changing_state = false;
}

static void
alpha_group_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  group_tree(w, pkglist, GAptPkgTree::CategoryAlpha);
}

static void
section_group_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  group_tree(w, pkglist, GAptPkgTree::CategorySection);
}


static void
status_group_cb         (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  group_tree(w, pkglist, GAptPkgTree::CategoryStatus);
}

static void
priority_group_cb       (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  group_tree(w, pkglist, GAptPkgTree::CategoryPriority);
}


// Other stuff

static void
description_cb  (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  gnome_ok_dialog("Not implemented");

  g_warning(__FUNCTION__);
}

// mark

static void
keep_cb         (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);
  
  pkglist->keep();
}

static void
install_cb      (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);
  
  pkglist->install();
}

static void
purge_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  g_warning(__FUNCTION__);
}

static void
remove_cb       (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);
  
  pkglist->remove();
}

static void
details_cb      (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);
  
  pkglist->details();
}

static void
search_cb      (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  pkglist->search();
}

///////////////////////////////// was in app.cc

static void
update_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf) {
    if (cf->UpdateAndReopen() == false) 
      {
        if (!_error->empty())
          gnome_apt_error_dialog(_("Update failed."));
      }
  }
  else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}

static void
apt_prefs_cb   (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  gnome_apt_preferences()->edit();
}

static void
sources_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  gnome_apt_sources()->edit();
}

static void
exit_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  gnome_apt_quit();
}


static void
complete_run_cb(GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf) {
    if (cf->Install() == false) 
      {
        if (!_error->empty())
          gnome_apt_error_dialog(_("Run failed."));
      }
  }
  else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}

static void
upgrade_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf) {
    cf->MarkUpgrades();
    pkglist->tree()->package_change_notify();
  }
  else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}


static void
smart_upgrade_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf) {
    cf->SmartMarkUpgrades();
    pkglist->tree()->package_change_notify();
  }
  else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}


static void
fix_broken_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf) {
    cf->Fix();
    pkglist->tree()->package_change_notify();
  }
  else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}

static void
install_pkg_cb     (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  pkglist->install();
}

static void
remove_pkg_cb      (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  pkglist->remove();
}

static void
keep_pkg_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);
  
  pkglist->keep();
}


static void
export_cb      (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  g_warning(__FUNCTION__ );
}

static void
import_cb      (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  g_warning(__FUNCTION__ );
}


static void
bug_cb        (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  g_warning(__FUNCTION__ );
}

static void
about_cb       (GtkWidget* w, gpointer data)
{
  if (changing_state) return;

  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

  static GtkWidget* about = NULL;

  if (about != NULL)
    {
      gdk_window_show(about->window);
      gdk_window_raise(about->window);
      return;
    }
 
  const gchar * authors [] = { 
    "Jason Gunthorpe <jgg@debian.org> - Project leader",
    "Manoj Srivastava <srivasta@datasync.com> - Dependency Expert",
    "Ben Gertzfield <che@debian.org> - Packaging and Releases",
    "Branden Robinson <branden@purdue.edu> - Man Page Documentation",
    "Adam Heath <doogie@debian.org> - FTP method author",
    "Havoc Pennington <hp@pobox.com> - GNOME frontend",
    "Diego Lages <dlages@dcc.ufrj.br> - GNOME frontend",
    "Olivier Gravel Aubin <ogaubin@ucalgary.ca> - GNOME frontend",
    "Mitch Blevins <mblevin@debian.org> - GNOME package maintainer",
    "",
    "Past Contributors:",
    "Brian White <bcwhite@verisim.com> - Project originator",
    "Tom Lees <tom@lpsg.demon.co.uk> - DPKG documentation and ideas",
    "Behan Webster <behanw@verisim.com> - Original GUI design",
    "Scott Ellis <storm@gate.net> - Original packaging and beta releases",
    NULL 
  };

  gchar * logo = NULL;// gnome_pixmap_file("aptlogo.xpm");

  gchar * text = 
    g_strconcat(_("Apt is a tool for managing the software packages installed"
                  " on your Debian GNU/Linux system.\n"),
                _("See http://www.debian.org/~hp/gnome-apt.html for more info."),
                // This legal text should not be translated
                " Apt is distributed in the hope that it will be useful, "
                "but WITHOUT ANY WARRANTY; without even the implied "
                "warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR "
                "PURPOSE. See the full text of the GPL for details.",
                NULL);

  about = 
    gnome_about_new("Apt: A Package Tool (GNOME Frontend)",
                    VERSION,
                    "Copyright (C) 1998 Individual contributors,"
                    " GNU General Public License.",
                    authors,
                    text,
                    logo );

  gnome_apt_setup_dialog(about);

  g_free(logo);
  g_free(text);
  
  // Zero the about pointer when the dialog is destroyed.
  gtk_signal_connect(GTK_OBJECT(about),
                     "destroy",
                     GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                     &about);

  gtk_widget_show(about);
}


static void
make_icon_help(GAptPkgTree::PixmapType p, 
               GtkWidget* dialog, 
               const gchar* text)
{
  GtkWidget* hbox;
  GtkWidget* pix;
  GtkWidget* label;

  hbox = gtk_hbox_new(FALSE, GNOME_PAD*12);

  pix = gnome_pixmap_new_from_xpm_d(const_cast<char**>(GAptPkgTree::pixmap_data(p)));

  gtk_box_pack_start(GTK_BOX(hbox), pix, TRUE, TRUE, 0);

  label = gtk_label_new(text);

  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);

  gtk_box_pack_end(GTK_BOX(hbox), label, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),
                     hbox, TRUE, TRUE, GNOME_PAD_SMALL);

  gtk_widget_show_all(hbox);
}

static void 
icons_cb       (GtkWidget* w, gpointer data)
{
  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);
  
  g_return_if_fail(pkglist != 0);
  
  static GtkWidget* icons = NULL;

  if (icons != NULL)
    {
      gdk_window_show(icons->window);
      gdk_window_raise(icons->window);
      return;
    }

  icons = gnome_dialog_new(_("Gnome Apt Icon Legend"),
                           GNOME_STOCK_BUTTON_OK,
                           NULL);
  gnome_dialog_set_close(GNOME_DIALOG(icons), TRUE);
  gnome_apt_setup_dialog(icons);
  gtk_signal_connect(GTK_OBJECT(icons),
                     "destroy",
                     GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                     &icons);
  
  
  make_icon_help(GAptPkgTree::PackagePixmap, icons, 
                 _("A package"));

  make_icon_help(GAptPkgTree::FolderPixmap, icons,
                 _("A group"));

  make_icon_help(GAptPkgTree::ConflictsPixmap, icons,
                 _("Conflicts")); 

  make_icon_help(GAptPkgTree::DependsPixmap, icons,
                 _("Depends"));

  make_icon_help(GAptPkgTree::RecommendsPixmap, icons,
                 _("Recommends"));

  make_icon_help(GAptPkgTree::SuggestsPixmap, icons,
                 _("Suggests"));

  make_icon_help(GAptPkgTree::ReplacesPixmap, icons,
                 _("Replaces"));
  
  gtk_widget_show(icons);
}
