/*
    ext2_block_relocator.c -- ext2 block relocator
    Copyright (C) 1998,99 Lennert Buytenhek <lbuijten@cs.leidenuniv.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"



struct ext2_block_entry
{
	blk_t num;
	blk_t dest;
	blk_t refblock;
	unsigned refoffset:16;
	unsigned isindirectblock:16;
};

struct ext2_block_relocator_state
{
	blk_t allocentries;
	blk_t usedentries;
	blk_t newsize;
	blk_t newallocoffset;
	struct ext2_block_entry *block;
};



static int compare_block_entries(const void *x0, const void *x1)
{
	const struct ext2_block_entry *b0;
	const struct ext2_block_entry *b1;

	b0 = (const struct ext2_block_entry *)x0;
	b1 = (const struct ext2_block_entry *)x1;

	if (b0->num < b1->num)
		return -1;

	if (b0->num > b1->num)
		return 1;

	return 0;
}

static int compare_block_entries_ref(const void *x0, const void *x1)
{
	const struct ext2_block_entry *b0;
	const struct ext2_block_entry *b1;

	b0 = (const struct ext2_block_entry *)x0;
	b1 = (const struct ext2_block_entry *)x1;

	if (b0->isindirectblock > b1->isindirectblock)
		return -1;

	if (b0->isindirectblock < b1->isindirectblock)
		return 1;

	if (b0->num < b1->num)
		return -1;

	if (b0->num > b1->num)
		return 1;

	return 0;
}

struct ext2_block_entry *findit(struct ext2_block_relocator_state *state, blk_t block)
{
	struct ext2_block_entry *b;
	struct ext2_block_entry key;

	key.num = block;

	b = bsearch(&key, state->block, state->usedentries, sizeof(struct ext2_block_entry), compare_block_entries);

	return b;
}

static __u32 getblocknumber(struct ext2_fs *fs, blk_t block, off_t offset)
{
	struct ext2_buffer_head *bh;
	__u32                    blocknumber;

	bh = ext2_bread(fs, block);
	blocknumber = *((__u32 *)(bh->data + offset));
	ext2_brelse(bh, 0);          /* FIXXXME might be indirect block */

	return blocknumber;
}


static void doblock(struct ext2_fs *fs, struct ext2_block_relocator_state *state, blk_t block, off_t offset, int indirect)
{
	__u32 blocknumber;
	struct ext2_block_entry *ent;

	if (!(blocknumber = getblocknumber(fs, block, offset)))
		return;

	if ((ent = findit(state, blocknumber)) == NULL)
		return;

	ent->refblock = block;
	ent->refoffset = offset;
	ent->isindirectblock = indirect;
}

static void doindblock(struct ext2_fs *fs, struct ext2_block_relocator_state *state, blk_t block, off_t offset)
{
	__u32 blocknumber;
	int   i;

	if (!(blocknumber = getblocknumber(fs, block, offset)))
		return;

	doblock(fs, state, block, offset, 1);
	for (i=0;i<fs->blocksize;i+=4)
		doblock(fs, state, blocknumber, i, 0);
}

static void dodindblock(struct ext2_fs *fs, struct ext2_block_relocator_state *state, blk_t block, off_t offset)
{
	__u32 blocknumber;
	int   i;

	if (!(blocknumber = getblocknumber(fs, block, offset)))
		return;

	doblock(fs, state, block, offset, 2);
	for (i=0;i<fs->blocksize;i+=4)
		doindblock(fs, state, blocknumber, i);
}

static void dotindblock(struct ext2_fs *fs, struct ext2_block_relocator_state *state, blk_t block, off_t offset)
{
	__u32 blocknumber;
	int   i;

	if (!(blocknumber = getblocknumber(fs, block, offset)))
		return;

	doblock(fs, state, block, offset, 3);
	for (i=0;i<fs->blocksize;i+=4)
		dodindblock(fs, state, blocknumber, i);
}



/* Ehrm.... sorry, pedanticists! :-) */
#define offsetof(type, field) ((int)(&(((type *)0)->field)))

static void doinode(struct ext2_fs *fs, struct ext2_block_relocator_state *state, int inode)
{
	struct ext2_inode buf;
	int		  i;
	off_t		  offset;

	ext2_read_inode(fs, inode, &buf);
	if (buf.i_blocks)
	{
		blk_t blk;

		offset = ext2_get_inode_offset(fs, inode, &blk);
		for (i=0;i<EXT2_NDIR_BLOCKS;i++)
			doblock(fs, state, blk, offset + offsetof(struct ext2_inode, i_block[i]), 0);

		doindblock(fs, state, blk, offset + offsetof(struct ext2_inode, i_block[EXT2_IND_BLOCK]));
		dodindblock(fs, state, blk, offset + offsetof(struct ext2_inode, i_block[EXT2_DIND_BLOCK]));
		dotindblock(fs, state, blk, offset + offsetof(struct ext2_inode, i_block[EXT2_TIND_BLOCK]));
	}
}

static void doscan(struct ext2_fs *fs, struct ext2_block_relocator_state *state)
{
	int i;

	for (i=0;i<fs->numgroups;i++)
	{
		struct ext2_buffer_head *bh;
		int			 j;
		int			 offset;

		bh = ext2_bread(fs, fs->gd[i].bg_inode_bitmap);
		offset = i * fs->sb.s_inodes_per_group + 1;

		for (j=0;j<fs->sb.s_inodes_per_group;j++)
			if (bh->data[j>>3] & _bitmap[j&7])
				doinode(fs, state, offset + j);

		ext2_brelse(bh, 0);
	}
}





static void ext2_block_relocator_copy(struct ext2_fs *fs, struct ext2_block_entry *block)
{
	ext2_copy_block(fs, block->num, block->dest);
}

static void ext2_block_relocator_ref(struct ext2_fs *fs, struct ext2_block_relocator_state *state, struct ext2_block_entry *block)
{
	struct ext2_buffer_head *bh;
	int i;

	if (!(block->refblock || block->refoffset))
		fprintf(stderr, "block %i has no reference? weird\n", block->num);

	bh = ext2_bread(fs, block->refblock);
	bh->dirty = 1;
	*((__u32 *)(bh->data + block->refoffset)) = block->dest;
	ext2_brelse(bh, 0);

	ext2_set_block_state(fs, block->dest, 1, 1);
	ext2_set_block_state(fs, block->num, 0, 1);

	/* FIXXXME slow */
	if (block->isindirectblock)
		for (i=0;i<state->usedentries;i++)
			if (state->block[i].refblock == block->num)
			{
				state->block[i].refblock = block->dest;
				if (block->isindirectblock - 1 != state->block[i].isindirectblock)
					fprintf(stderr,
						"hey! weird! block %i -> %i (%i) ref at %i %i (%i)\n",
						state->block[i].num, state->block[i].dest,
						state->block[i].isindirectblock,
						block->refblock, block->refoffset,
						block->isindirectblock);
			}
}

static int ext2_block_relocator_grab_blocks(struct ext2_fs *fs, struct ext2_block_relocator_state *state)
{
	int i;
	int ptr;

	ptr = 0;

	for (i=0;i<fs->numgroups;i++)
		if (fs->gd[i].bg_free_blocks_count)
		{
			struct ext2_buffer_head *bh;
			int j;
			int offset;

			bh = ext2_bread(fs, fs->gd[i].bg_block_bitmap);
			offset = i * fs->sb.s_blocks_per_group + fs->sb.s_first_data_block;

			for (j=state->newallocoffset;j<fs->sb.s_blocks_per_group;j++)
				if (!(bh->data[j>>3] & _bitmap[j&7]))
				{
					state->block[ptr++].dest = offset + j;

					if (ptr == state->usedentries)
					{
						ext2_brelse(bh, 0);
						return 1;
					}
				}

			ext2_brelse(bh, 0);
		}

	return 0;
}

static int ext2_block_relocator_flush(struct ext2_fs *fs, struct ext2_block_relocator_state *state)
{
	int i;
	int last;

	if (!state->usedentries)
		return 1;

#if 1
	fprintf(stderr, "ext2_block_relocator_flush\n");
#endif

#if 1
	qsort(state->block, state->usedentries, sizeof(struct ext2_block_entry), compare_block_entries);
#endif

	doscan(fs, state);

	if (!ext2_block_relocator_grab_blocks(fs, state))
		return 0;

	for (i=0;i<state->usedentries;i++)
		ext2_block_relocator_copy(fs, &state->block[i]);

	qsort(state->block, state->usedentries, sizeof(struct ext2_block_entry), compare_block_entries_ref);

	last = -1;
	for (i=0;i<state->usedentries;i++)
	{
		if (state->block[i].isindirectblock != last)
		{
			ext2_sync(fs);

			if (last != -1)
				printf("\n");

			last = state->block[i].isindirectblock;

			printf("relocating %s blocks",
			       (last==3?"triply indirect":
				(last==2?"doubly indirect":
				 (last==1?"singly indirect":
				  (last==0?"direct":"?")))));
			fflush(stdout);
		}

		ext2_block_relocator_ref(fs, state, &state->block[i]);
	}

	printf("\n");

	state->usedentries = 0;
	ext2_sync(fs);

	return 1;
}

static int ext2_block_relocator_mark(struct ext2_fs *fs, struct ext2_block_relocator_state *state, blk_t block)
{
	int i;

	if (!ext2_get_block_state(fs, block) || !ext2_is_data_block(fs, block))
	{
		fprintf(stderr, "bug! block %i shouldnt be marked!\n", block);
		fprintf(stderr, "please send a filesystem dump and your ext2resize version\n");
		fprintf(stderr, "number to <lbuijten@cs.leidenuniv.nl>\n");
		fprintf(stderr, "\n");
		fprintf(stderr, "aborting\n");

		return 0;
	}

	if (state->usedentries == state->allocentries - 1)
		if (!ext2_block_relocator_flush(fs, state))
			return 0;

	i = state->usedentries;
	state->block[i].num = block;
	state->block[i].dest = 0;
	state->block[i].refblock = 0;
	state->block[i].refoffset = 0;

	state->usedentries++;
	return 1;
}

static int ext2_block_relocate_grow(struct ext2_fs *fs, struct ext2_block_relocator_state *state, blk_t newsize)
{
	blk_t newgdblocks;
	blk_t newitoffset;
	int   i;

#if 1
	fprintf(stderr, "ext2_block_relocate_grow\n");
#endif

	newgdblocks = howmany(newsize - fs->sb.s_first_data_block, fs->sb.s_blocks_per_group);
	newgdblocks = howmany(newgdblocks * sizeof(struct ext2_group_desc), fs->blocksize);
	if (newgdblocks == fs->gdblocks)
		return 1;

	newitoffset = newgdblocks + 3;
	state->newallocoffset = newitoffset + fs->inodeblocks;

	for (i=0;i<fs->numgroups;i++)
	{
		struct ext2_buffer_head *bh;
		blk_t			 diff;
		blk_t			 j;
		blk_t			 start;
		int			 sparse;

		bh = ext2_bread(fs, fs->gd[i].bg_block_bitmap);
		start = (i * fs->sb.s_blocks_per_group) + fs->sb.s_first_data_block;
		sparse = ext2_is_group_sparse(fs, i);

		if (fs->gd[i].bg_inode_table < start + newitoffset
		    || (sparse && ((fs->gd[i].bg_block_bitmap < start + newitoffset - 2) ||
				   (fs->gd[i].bg_inode_bitmap < start + newitoffset - 1))))
		{
			diff = newitoffset - (fs->gd[i].bg_inode_table - start);

			for (j=0;j<diff;j++)
			{
				blk_t k;

				k = fs->itoffset + fs->inodeblocks + j;
				if (bh->data[k>>3] & _bitmap[k&7])
					if (!ext2_block_relocator_mark(fs, state, start + k))
					{
						ext2_brelse(bh, 0);
						return 0;
					}
			}
		}

		ext2_brelse(bh, 0);
	}

	if (!ext2_block_relocator_flush(fs, state))
		return 0;

	return 1;
}

static int ext2_block_relocate_shrink(struct ext2_fs *fs, struct ext2_block_relocator_state *state, blk_t newsize)
{
	int diff;
	int i;

#if 1
	fprintf(stderr, "ext2_block_relocate_shrink\n");
#endif

	diff = howmany(newsize - fs->sb.s_first_data_block, fs->sb.s_blocks_per_group);
	diff = howmany(diff * sizeof(struct ext2_group_desc), fs->blocksize);
	diff = fs->gdblocks - diff;

	state->newallocoffset = fs->itoffset + fs->inodeblocks;

	for (i=0;i<fs->numgroups;i++)
	{
		struct ext2_buffer_head *bh;
		blk_t			 groupsize;
		blk_t			 j;
		blk_t			 offset;
		int			 sparse;
		blk_t			 start;
		int			 type;

		offset = i * fs->sb.s_blocks_per_group + fs->sb.s_first_data_block;
		sparse = ext2_is_group_sparse(fs, i);

		if (newsize >= offset + fs->sb.s_blocks_per_group)
			continue;		/* group will survive */

		bh = ext2_bread(fs, fs->gd[i].bg_block_bitmap);

		if (newsize <= offset)
			type = 2;		/* group is fully chopped off */
		else
			type = 1;		/* group is partly chopped off */

		if (!sparse && type == 2)
		{
			for (j=fs->gd[i].bg_inode_bitmap+1;j<fs->gd[i].bg_inode_table;j++)
			{
				blk_t k;

				k = j - offset;
				if (bh->data[k>>3] & _bitmap[k&7])
					if (!ext2_block_relocator_mark(fs, state, j))
					{
						ext2_brelse(bh, 0);
						return 0;
					}
			}
		}

		start = newsize;
		if (type == 2)
			start = fs->gd[i].bg_inode_table + fs->inodeblocks;

		start -= offset;

		groupsize = fs->sb.s_blocks_per_group;
		if (offset + groupsize > fs->sb.s_blocks_count)
			groupsize = fs->sb.s_blocks_count - offset;

		for (j=start;j<groupsize;j++)
			if (bh->data[j>>3] & _bitmap[j&7])
				if (!ext2_block_relocator_mark(fs, state, offset + j))
				{
					ext2_brelse(bh, 0);
					return 0;
				}

		ext2_brelse(bh, 0);
	}

	return ext2_block_relocator_flush(fs, state);
}

int ext2_block_relocate(struct ext2_fs *fs, blk_t newsize)
{
	struct ext2_block_relocator_state state;

	fprintf(stderr, "ext2_block_relocate\n");

	state.allocentries = (ext2_relocator_pool_size << 10) / sizeof(struct ext2_block_entry);
	state.usedentries = 0;
	state.newsize = newsize;
	state.newallocoffset = 0;
	state.block = (struct ext2_block_entry *)fs->relocator_pool;

	if (newsize < fs->sb.s_blocks_count)
		return ext2_block_relocate_shrink(fs, &state, newsize);

	return ext2_block_relocate_grow(fs, &state, newsize);
}
