/*
    ext2.h -- ext2 header
    Copyright (C) 1998,99 Lennert Buytenhek <lbuijten@cs.leidenuniv.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _EXT2_H
#define _EXT2_H

#include <sys/types.h>

/* tunables */
extern int ext2_buffer_cache_pool_size;
extern int ext2_hash_bits;
extern int ext2_max_groups;
extern int ext2_relocator_pool_size;

#ifndef howmany
#define howmany(x,y)        (((x)+((y)-1))/(y))
#endif

#ifndef max
#define max(a,b) (((a)>(b))?(a):(b))
#endif

#ifndef min
#define min(a,b) (((a)<(b))?(a):(b))
#endif

extern unsigned char _bitmap[8];

typedef u_int32_t blk_t;

/* straight from the linux ext2 kernel sources... */

#define EXT2_SUPER_MAGIC    0xEF53
#define EXT2_MIN_BLOCK_SIZE 1024
#define EXT2_BLOCK_SIZE(s)  (EXT2_MIN_BLOCK_SIZE << (s)->s_log_block_size)
#define EXT2_NDIR_BLOCKS    12
#define EXT2_IND_BLOCK      EXT2_NDIR_BLOCKS
#define EXT2_DIND_BLOCK     (EXT2_IND_BLOCK + 1)
#define EXT2_TIND_BLOCK     (EXT2_DIND_BLOCK + 1)
#define EXT2_N_BLOCKS       (EXT2_TIND_BLOCK + 1)
#define EXT2_VALID_FS       0x0001
#define EXT2_ERROR_FS       0x0002

#define EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER 0x0001
#define EXT2_FEATURE_INCOMPAT_FILETYPE 0x0002

struct ext2_dir_entry
{
	__u32	inode;
	__u16	rec_len;
	__u8	name_len;
	__u8	file_type;
	char	name[255];
};

struct ext2_group_desc
{
	__u32  bg_block_bitmap;
	__u32  bg_inode_bitmap;
	__u32  bg_inode_table;
	__u16  bg_free_blocks_count;
	__u16  bg_free_inodes_count;
	__u16  bg_used_dirs_count;
	__u16  bg_pad;
	__u32  bg_reserved[3];
};

struct ext2_inode
{
	__u16   i_mode;         /* File mode */
	__u16   i_uid;          /* Owner Uid */
	__u32   i_size;         /* Size in bytes */
	__u32   i_atime;        /* Access time */
	__u32   i_ctime;        /* Creation time */
	__u32   i_mtime;        /* Modification time */
	__u32   i_dtime;        /* Deletion Time */
	__u16   i_gid;          /* Group Id */
	__u16   i_links_count;  /* Links count */
	__u32   i_blocks;       /* Blocks count */
	__u32   i_flags;        /* File flags */
	union {
		struct {
			__u32  l_i_reserved1;
		} linux1;
		struct {
			__u32  h_i_translator;
		} hurd1;
		struct {
			__u32  m_i_reserved1;
		} masix1;
	} osd1;                         /* OS dependent 1 */
	__u32   i_block[EXT2_N_BLOCKS];/* Pointers to blocks */
	__u32   i_version;      /* File version (for NFS) */
	__u32   i_file_acl;     /* File ACL */
	__u32   i_dir_acl;      /* Directory ACL */
	__u32   i_faddr;        /* Fragment address */
	union {
		struct {
			__u8    l_i_frag;       /* Fragment number */
			__u8    l_i_fsize;      /* Fragment size */
			__u16   i_pad1;
			__u32   l_i_reserved2[2];
		} linux2;
		struct {
			__u8    h_i_frag;       /* Fragment number */
			__u8    h_i_fsize;      /* Fragment size */
			__u16   h_i_mode_high;
			__u16   h_i_uid_high;
			__u16   h_i_gid_high;
			__u32   h_i_author;
		} hurd2;
		struct {
			__u8    m_i_frag;       /* Fragment number */
			__u8    m_i_fsize;      /* Fragment size */
			__u16   m_pad1;
			__u32   m_i_reserved2[2];
		} masix2;
	} osd2;                         /* OS dependent 2 */
};

struct ext2_super_block
{
	__u32   s_inodes_count;         /* Inodes count */
	__u32   s_blocks_count;         /* Blocks count */
	__u32   s_r_blocks_count;       /* Reserved blocks count */
	__u32   s_free_blocks_count;    /* Free blocks count */
	__u32   s_free_inodes_count;    /* Free inodes count */
	__u32   s_first_data_block;     /* First Data Block */
	__u32   s_log_block_size;       /* Block size */
	__s32   s_log_frag_size;        /* Fragment size */
	__u32   s_blocks_per_group;     /* # Blocks per group */
	__u32   s_frags_per_group;      /* # Fragments per group */
	__u32   s_inodes_per_group;     /* # Inodes per group */
	__u32   s_mtime;                /* Mount time */
	__u32   s_wtime;                /* Write time */
	__u16   s_mnt_count;            /* Mount count */
	__s16   s_max_mnt_count;        /* Maximal mount count */
	__u16   s_magic;                /* Magic signature */
	__u16   s_state;                /* File system state */
	__u16   s_errors;               /* Behaviour when detecting errors */
	__u16   s_minor_rev_level;      /* minor revision level */
	__u32   s_lastcheck;            /* time of last check */
	__u32   s_checkinterval;        /* max. time between checks */
	__u32   s_creator_os;           /* OS */
	__u32   s_rev_level;            /* Revision level */
	__u16   s_def_resuid;           /* Default uid for reserved blocks */
	__u16   s_def_resgid;           /* Default gid for reserved blocks */
	/*
	 * These fields are for EXT2_DYNAMIC_REV superblocks only.
	 *
	 * Note: the difference between the compatible feature set and
	 * the incompatible feature set is that if there is a bit set
	 * in the incompatible feature set that the kernel doesn't
	 * know about, it should refuse to mount the filesystem.
	 * 
	 * e2fsck's requirements are more strict; if it doesn't know
	 * about a feature in either the compatible or incompatible
	 * feature set, it must abort and not try to meddle with
	 * things it doesn't understand...
	 */
	__u32   s_first_ino;            /* First non-reserved inode */
	__u16   s_inode_size;           /* size of inode structure */
	__u16   s_block_group_nr;       /* block group # of this superblock */
	__u32   s_feature_compat;       /* compatible feature set */
	__u32   s_feature_incompat;     /* incompatible feature set */
	__u32   s_feature_ro_compat;    /* readonly-compatible feature set */
	__u32   s_reserved[230];        /* Padding to the end of the block */
};





struct ext2_buffer_cache
{
	struct ext2_buffer_head	 *cache;
	struct ext2_buffer_head  *heads;
	struct ext2_buffer_head **hash;
	struct ext2_fs		 *fs;

	int			  size;
	int			  numalloc;
	unsigned char		 *buffermem;
};

struct ext2_buffer_head
{
	struct ext2_buffer_head  *next;
	struct ext2_buffer_head  *prev;
	unsigned char		 *data;
	blk_t			  block;

	int			  usecount;
	int			  dirty;

	struct ext2_buffer_cache *bc;
	int			  alloc;
};

struct ext2_dev_ops
{
	void			(*close)(void *cookie);
	blk_t			(*get_size)(void *cookie);
	int			(*read)(void *cookie, void *ptr, blk_t block, blk_t num);
	void			(*set_blocksize)(void *cookie, int logsize);
	void			(*sync)(void *cookie);
	int			(*write)(void *cookie, void *ptr, blk_t block, blk_t num);
};

struct ext2_dev_handle
{
	struct ext2_dev_ops	*ops;
	void			*cookie;
};

struct ext2_fs
{      
	struct ext2_dev_handle		 *devhandle;

	struct ext2_super_block		  sb;
	struct ext2_group_desc		 *gd;
	int				  metadirty;			/* 0:all sb&gd copies clean
									   1:all sb&gd copies dirty
									   2:only first sb&gd copy clean */

	int				  sparse;			/* sparse superblocks */

	int				  blocksize;
	int				  logsize;
	int				  adminblocks;
	int				  gdblocks;
	int				  itoffset;
	int				  inodeblocks;
	int				  numgroups;
	int				  r_frac;			/* reserved % of blocks */

	struct ext2_buffer_cache	 *bc;

	unsigned char			 *relocator_pool;
	unsigned char			 *relocator_pool_end;
};



/* generic stuff */
void		ext2_copy_block			(struct ext2_fs *fs, blk_t from, blk_t to);
void		ext2_close			(struct ext2_fs *fs);
void		ext2_commit_metadata		(struct ext2_fs *fs, int all_copies);
off_t		ext2_get_inode_offset		(struct ext2_fs *fs, ino_t inode, blk_t *block);
int		ext2_get_inode_state		(struct ext2_fs *fs, ino_t inode);
int		ext2_is_group_sparse		(struct ext2_fs *fs, int group);
void		ext2_move_blocks		(struct ext2_fs *fs, blk_t src, blk_t num, blk_t dest);
struct ext2_fs *ext2_open			(struct ext2_dev_handle *handle);
int		ext2_read_blocks		(struct ext2_fs *fs, void *ptr, blk_t block, blk_t numblocks);
void		ext2_read_inode			(struct ext2_fs *fs, ino_t inode, struct ext2_inode *inodep);
void		ext2_set_inode_state		(struct ext2_fs *fs, ino_t inode, int state, int updatemetadata);
void		ext2_sync			(struct ext2_fs *fs);
int		ext2_write_blocks		(struct ext2_fs *fs, void *ptr, blk_t block, blk_t numblocks);
void		ext2_write_inode		(struct ext2_fs *fs, ino_t inode, const struct ext2_inode *inodep);
void		ext2_zero_blocks		(struct ext2_fs *fs, blk_t block, blk_t num);
void		ext2_zero_inode			(struct ext2_fs *fs, ino_t inode);

/* block related */
void		ext2_bgbitmap_cache_deinit	(struct ext2_fs *fs);
void		ext2_bgbitmap_cache_flush	(struct ext2_fs *fs);
void		ext2_bgbitmap_cache_init	(struct ext2_fs *fs);
int		ext2_get_block_state		(struct ext2_fs *, blk_t block);
void		ext2_set_block_state		(struct ext2_fs *, blk_t block, int state, int updatemetadata);

/* block relocator */
int		ext2_block_relocate		(struct ext2_fs *fs, blk_t newsize);

/* buffer */
void		ext2_bcache_deinit		(struct ext2_fs *fs);
void		ext2_bcache_dump		(struct ext2_fs *fs);
void		ext2_bcache_flush		(struct ext2_fs *fs, blk_t block);
void		ext2_bcache_flush_range		(struct ext2_fs *fs, blk_t first, blk_t last);
int 		ext2_bcache_init		(struct ext2_fs *fs);
void		ext2_bcache_sync		(struct ext2_fs *fs);
struct ext2_buffer_head *ext2_bcreate		(struct ext2_fs *fs, blk_t block);
struct ext2_buffer_head *ext2_bread		(struct ext2_fs *fs, blk_t block);
void		ext2_brelse			(struct ext2_buffer_head *bh, int forget);

/* inode relocator */
int		ext2_inode_relocate		(struct ext2_fs *fs, int newgroups);

/* metadata mover */
int		ext2_metadata_push		(struct ext2_fs *fs, blk_t newsize);

/* resize */
int		ext2_resize_fs			(struct ext2_fs *fs, blk_t newsize);

/* unix I/O */
struct ext2_dev_handle *ext2_make_dev_handle_from_file(char *dev);




static int __inline__ ext2_is_data_block(struct ext2_fs *fs, blk_t block)
{
	blk_t blk;
	int   group;

	if (block >= fs->sb.s_blocks_count)
		fprintf(stderr, "is_data_block(%i): block number invalid!\n", block);

	blk = block - fs->sb.s_first_data_block;

	group = blk / fs->sb.s_blocks_per_group;
	blk %= fs->sb.s_blocks_per_group;

	if (ext2_is_group_sparse(fs, group) && blk <= fs->gdblocks)
		return 0;

	if (block == fs->gd[group].bg_block_bitmap ||
	    block == fs->gd[group].bg_inode_bitmap)
		return 0;

	if (block >= fs->gd[group].bg_inode_table &&
	    block < fs->gd[group].bg_inode_table + fs->inodeblocks)
		return 0;

	return 1;
}

#endif
