/*
 * Description  Net-fetch module implementation.
 * CVS          $Id: net-fetch.c,v 1.38 2000/08/13 13:18:37 joey Exp $
 * Author       Marcel Harkema <marcel@debian.org>
 *
 * Copyright (C) 1999, 2000 Marcel Harkema
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/param.h>
#include <time.h>
#include <assert.h>

#include <newt.h>

#include "dbootstrap.h"
#include "net-fetch.h"
#include "lang.h"
#include "util.h"

/* Include headers of net-fetch modules. */
#include "http-fetch.h"

/*
 * Private data-structures and variables.
 */

struct _nf_endpoint {
  char                *hostname;
  unsigned short int   port;
};

struct _nf_state {
  char                 initialized;
  char                *method;
  struct _nf_endpoint  server;
  struct _nf_endpoint  proxy;
  char                *path;        /* debian/dists/potato/ ... */
};
static struct _nf_state nf_state;

struct _nf_transport {
  char                *url_prefix;
  unsigned short int   default_port;
  fetchfile_funtype    fetchfile_fn;
};

struct _nf_transport nf_transports[] = {
  { "http",  80, nf_http_fetchfile       },
//{ "http",  80, nf_snarf_http_fetchfile },
//{ "ftp",   21, nf_snarf_ftp_fetchfile  },
  { NULL,    0,  NULL                    }
};

#ifdef _TESTING_
extern char * get_device        (const char *);
extern void   release_device    (const char *);
extern int    install_floppy    (char *, const char *, const char *);
extern int    install_from_file (const char *, const char *);
extern int    extract_from_file (const char *, const char *);
extern void   setup_image_names (void);
extern void   get_kver          (void);
extern void   get_subarch_name  (void);
#endif


/*
 * Installation series.
 */

#define SERIES_OS    1
#define SERIES_BASE  2

static int nf_install(int series);

static int
nf_extract_kernel(const char *filename_)
{
  char *device;
  int   r;
  char  filename[MAXPATHLEN];
  char *old_Archive_Dir;

  /* get_device() puts the Archive_Dir in front the the diskimage
   * filename, so we point Archive_Dir to an empty string. */
  old_Archive_Dir = Archive_Dir;
  Archive_Dir = "";

  snprintf(filename, sizeof(filename), "%s", target_path(filename_));

  if ((device = get_device(filename)) != NULL) {
    r = install_floppy(device, "rescue", _("Rescue Floppy"));
    release_device(device);
  }
  else {
    r = 1;
  }

  Archive_Dir = old_Archive_Dir;

  if (r) {
    vaproblemBox(_("Floppy Error"),
                 _("The attempt to extract the %s failed."), _("Rescue Floppy"));
  }

  return r;
}


static int
nf_extract_drivers(const char *filename_)
{
  char filename[MAXPATHLEN];
  snprintf(filename, sizeof(filename), "%s", target_path(filename_));
  return install_from_file(filename, _("Drivers"));
}


int
nf_install_os(void)
{
  return nf_install(SERIES_OS);
}


static int
nf_extract_base(const char *filename_)
{
  char filename[MAXPATHLEN];
  snprintf(filename, sizeof(filename), "%s", target_path(filename_));
  return extract_from_file(filename, _("Base System"));
}


int
nf_install_base(void)
{
  return nf_install(SERIES_BASE);
}


static int
nf_method_lookup(char *method)
{
  int ti;
  for (ti = 0; nf_transports[ti].url_prefix != NULL; ti++)
    if (!strcmp(nf_state.method, nf_transports[ti].url_prefix))
      break;
  return (nf_transports[ti].url_prefix == NULL) ? -1 : ti;
}


/*
 * Widget for selecting the server to download from.
 */

static int
nf_parse_server_url(char *url)
{
  /* this should be replaced with a proper parse fun.  also, this
   * one has minimal error checking. */
  char *buf = strdup(url);
  char *p, *q;
  int method;

  /* parse 'method://hostname:port/path' */

  p = strstr(buf, "://");
  if (p) {
    *p = 0;
    q = p+3; /* start of next part */
    p = buf; 
  }
  else {
    q = buf; /* start of next part */
    p = "http"; /* default */
  }
  free(nf_state.method);
  nf_state.method = strdup(p);

  free(nf_state.server.hostname);
  free(nf_state.path);
  nf_state.path = NULL;

  p = strchr(q, ':');
  if (p) {
    *p = 0;
    nf_state.server.hostname = strdup(q);
    p++;
    nf_state.server.port = strtoul(p, &q, 10);
    if (q) nf_state.path = strdup((*q == 0) ? "" : q+1);
  }
  else {
    nf_state.server.port = 0; /* use method's default port */
    p = strchr(q, '/');
    if (p) {
      *p = 0;
      nf_state.server.hostname = strdup(q);
      nf_state.path = strdup(p+1);
    }
    else {
      nf_state.server.hostname = strdup(q);
    }
    if (nf_state.path == NULL)
      nf_state.path = strdup("/");
  }

  if ((method = nf_method_lookup(nf_state.method)) < 0) {
    snprintf(buf, sizeof buf - 1, _("Unknown protocol '%s' used in the URL."), nf_state.method);
    strcat(buf, "\n");
    problemBox(buf, "Unknown Protocol");
    free(buf);
    return 1;
  }

  if (nf_state.server.port == 0)
    nf_state.server.port = nf_transports[method].default_port;

  free(buf);
  return 0;
}


int
nf_select_server(void)
{
  char          *s[3];
  newtComponent  l[3];
  newtComponent  e[3];
  newtComponent  form, b1, b2, b3, textbox, ans;
  int            height;
  char           txtbuf[2048];
  int            ti;

  textbox = newtTextbox(1, 1, 72, 6, NEWT_FLAG_WRAP);
  newtTextboxSetText(textbox, _("Please provide the URL from which to download the installation files.  If you have a proxy server, fill that out as well; otherwise, leave it set to 'none'.\n"));
  height = newtTextboxGetNumLines(textbox);
  newtTextboxSetHeight(textbox, height);
  newtCenteredWindow(73, height + 7, _("Select Installation Server"));

  ti = nf_method_lookup(nf_state.method);
  if (ti >= 0
   && nf_transports[ti].url_prefix != NULL
   && nf_transports[ti].default_port == nf_state.server.port) {
    snprintf(txtbuf, sizeof txtbuf - 1, "%s://%s/%s",
             nf_state.method, nf_state.server.hostname, nf_state.path);
  }
  else {
    snprintf(txtbuf, sizeof txtbuf - 1, "%s://%s:%d/%s",
             nf_state.method, nf_state.server.hostname, nf_state.server.port, nf_state.path);
  }

  l[0] = newtLabel(1, height + 2, _("Download URL"));
  e[0] = newtEntry(14, height + 2, txtbuf, 58, &s[0], NEWT_FLAG_SCROLL);

  l[1] = newtLabel(1, height + 4, _("Proxy"));
  e[1] = newtEntry(14, height + 4, nf_state.proxy.hostname, 40, &s[1], NEWT_FLAG_SCROLL);
  l[2] = newtLabel(58, height + 4, _("Port"));
  snprintf(txtbuf, sizeof txtbuf - 1, "%d", nf_state.proxy.port);
  e[2] = newtEntry(66, height + 4, txtbuf, 6, &s[2], NEWT_FLAG_SCROLL);

  b1 = newtCompactButton(19, height + 6, _("Ok"));
  b2 = newtCompactButton(29, height + 6, _("Cancel"));
  b3 = newtCompactButton(42, height + 6, _("Help"));

  form = newtForm(NULL, NULL, 0);
  newtFormAddComponents(form, textbox, l[0], e[0], l[1], e[1], l[2], e[2], b1, b2, b3, NULL);

  do {
    ans = newtRunForm(form);
    free(nf_state.proxy.hostname);
    nf_state.proxy.hostname = strdup(s[1]);
    nf_state.proxy.port = atoi(s[2]);
    if (ans == b3) {
      problemBox(_("Debian supports installing drivers and the base system over the Internet.  Currently, only HTTP retrieval is supported.\nPlease enter some information so that the files can be retrieved.\n\nDownload URL: the URL of the directory containing the file to be downloaded.  The default location should work on Debian standard servers.\nProxy: a proxy server to use to fetch the URL, or \"none\" if you don't need one.\nPort: the port that the proxy server listens on."),
                 _("Help For Network Installation"));
    }
    else if (ans == b2 || nf_parse_server_url(s[0]) == 0) {
      break;
    } 
  }
  while (1);

  newtPopWindow();

  /* We cannot destroy the form until after we've used the value
   * from the entry widget. */
  newtFormDestroy(form);

  if (ans == b2)  /* cancel */
    return 1;

  return 0;
}


/*
 * Widget for displaying download progress.
 */

static int
nf_progress_scaleBox(const char *text,
                     const char *title,
		     const long value,
		     const int action)
{
  static newtComponent tx = NULL, scale = NULL, stopbtn = NULL, f = NULL;

  switch (action) {
    case SCALE_CREATE:
      if (f)
        return -1;
      newtCenteredWindow(72, 14, title);
      f = newtForm(NULL, NULL, 0);
      tx = newtTextbox(1, 1, 71, 8, NEWT_FLAG_WRAP);
      newtTextboxSetText(tx, text);
      scale = newtScale(4, 10, 62, value);
      stopbtn = newtCompactButton(33, 12, _("Stop"));
      newtFormAddComponents(f, tx, scale, stopbtn, NULL);
      newtDrawForm(f);
      newtRefresh();
      break;

    case SCALE_REFRESH:
      if (!f)
        return -1;
      newtTextboxSetText(tx, text);
      newtDrawForm(f);
      newtScaleSet(scale, value);
      newtRefresh();
      break;

    case SCALE_DELETE:
      if (!f)
        return -1;
      newtPopWindow();
      newtFormDestroy(f);
      f = NULL;
      break;
  }

  return 0;
}


static void
nf_progress_display(const char *basename, const off_t bytes)
{
  static char   txtbuf[2048];
  static int    len;
  static time_t starttime;
  static off_t  bytes_total;

  if (basename != NULL) {
    starttime = time(NULL);
    bytes_total = bytes;

    len = snprintf(txtbuf, sizeof txtbuf - 1,
                   _("Fetching    %s\n"
                     "Server      %s://%s:%d/\n"
                     "Path        %s\n"),
                   basename, nf_state.method, nf_state.server.hostname,
                   nf_state.server.port, nf_state.path);

    if (strcmp(nf_state.proxy.hostname, "")
        && strcmp(nf_state.proxy.hostname, "none")) {
      len += snprintf(txtbuf + len, sizeof txtbuf - len - 1,
                      _("Proxy       %s://%s:%d/\n"),
                      nf_state.method, nf_state.proxy.hostname, nf_state.proxy.port);
    }
    else {
      len += snprintf(txtbuf + len, sizeof txtbuf - len - 1, _("Proxy       none\n"));
    }

    len += snprintf(txtbuf + len, sizeof txtbuf - len - 1,
                    _("File size   %ld bytes\n"), bytes_total);

    nf_progress_scaleBox(txtbuf, _("Installing From Network"), bytes_total, SCALE_CREATE);
  }
  else {
    time_t seconds;
    char txteta[64];

    seconds = time(NULL) - starttime;

    if (seconds >= 2 && bytes > 0) {
      long lh, lm, ls;

      ls = (bytes_total - bytes) / (bytes / seconds);
      lm = ls / 60;
      ls = ls % 60;
      lh = lm / 60;
      lm = lm % 60;

      snprintf(txteta, sizeof txteta - 1,
               "%02ldH:%02ldM:%02ldS", lh, lm, ls);
    }
    else {
      snprintf(txteta, sizeof txteta - 1, "--H:--M:--S");
    }

    snprintf(txtbuf + len, sizeof txtbuf - len - 1,
             _("Bytes read  %ld bytes\n"
               "Rate        %ld bytes/sec\n"
               "ETA         %s\n"),
             bytes, seconds > 0 ? bytes / seconds : 0, txteta);

    nf_progress_scaleBox(txtbuf, NULL, bytes, SCALE_REFRESH);
  }
}


void
nf_progress_open(const char *basename, const off_t bytes_total)
{
  nf_progress_display(basename, bytes_total);
}


void
nf_progress_update(const off_t bytes_read)
{
  nf_progress_display(NULL, bytes_read);
}


void
nf_progress_close(void)
{
  nf_progress_scaleBox(NULL, NULL, 0, SCALE_DELETE);
}


/*
 * Initialize net-fetch.
 */

void
nf_initialize(void)
{
  if (nf_state.initialized)
    return;

  nf_state.initialized = 1;

  // http://http.us.debian.org/debian/dists/potato/main/disks-$arch/current/
  nf_state.method = strdup("http");
  nf_state.server.hostname = strdup("http.us.debian.org");
  nf_state.server.port = 80;
  nf_state.proxy.hostname = strdup("none");
  nf_state.proxy.port = 8080;
  snprintf(prtbuf, sizeof(prtbuf) - 1, "debian/" ARCHIVE_LOCATION "/", ARCHNAME);
  nf_state.path = strdup(prtbuf);

#if defined (_TESTING_)
  // nf_state.proxy.hostname = strdup("zuma.kawa.dhs.org");
  // nf_state.proxy.port = 8080;
  // nf_state.server.hostname = strdup("zuma.kawa.dhs.org");
  // nf_state.server.port = 7070;
  // nf_state.path = strdup("debian-boot/feb18");
#endif
}


void
nf_finalize(void)
{
  if (!nf_state.initialized)
    return;

  nf_state.initialized = 0;

  free(nf_state.method); nf_state.method = NULL;
  free(nf_state.server.hostname); nf_state.server.hostname = NULL;
  free(nf_state.proxy.hostname); nf_state.proxy.hostname = NULL;
  free(nf_state.path); nf_state.path = NULL;
}


static int
nf_fetchfile(char *remote_filename, char *local_filename)
{
  int ti;

  ti = nf_method_lookup(nf_state.method);

  assert(ti >= 0);
  assert(nf_transports[ti].fetchfile_fn != NULL);

  return nf_transports[ti].fetchfile_fn(nf_state.server.hostname, nf_state.server.port,
                                        nf_state.proxy.hostname, nf_state.proxy.port,
                                        nf_state.path,
                                        remote_filename, local_filename);
}


static int
nf_install(int series)
{
  char        txtbuf[2048];
  int         i;
  char        filename[MAXPATHLEN];
  struct stat statbuf;
  int         len;

  struct {
    char *local;
    char *remote;
    int   (*install_fn) (const char *);
    int   series;
  }
  files_to_fetch[] = {
    { "rescue.bin",   kernel_image_path, nf_extract_kernel,  SERIES_OS   },
    { "drivers.tgz",  drivers_path,      nf_extract_drivers, SERIES_OS   },
    { "base2_2.tgz",  BASETGZ,           nf_extract_base,    SERIES_BASE },
    { NULL,           NULL,              NULL,               0           }
  };

  if (!NAME_ISDIR(target_path("/tmp"), &statbuf)) {
    snprintf(txtbuf, sizeof txtbuf - 1, "mkdir %s", target_path("/tmp"));
    if (execlog(txtbuf, LOG_INFO)) {
      vaproblemBox(_("Directory creation failed"),
                   _("There was a problem creating the directory %s."),
                   target_path("/tmp"));
      return 1;
    }
  }

  len = snprintf(txtbuf, sizeof txtbuf - 1,
                 _("\nGoing to download the following files over a HTTP connection:\n"));
  for (i = 0; files_to_fetch[i].remote != NULL; i++) {
    if (series != files_to_fetch[i].series)
      continue;
    len += snprintf(txtbuf + len, sizeof txtbuf - len - 1, "  %s\n", files_to_fetch[i].remote);
  }
  wideMessageBox(txtbuf, _("Fetching installation files over the network"));

  nf_initialize();

  switch (nf_select_server()) {
    case 0:
      break;
    default:
    case 1:
      return 1;
      /* NOTREACHED */
  }

  if ( strcmp(nf_state.proxy.hostname, "none") != 0 ) {
    /* write the HTTP_PROXY user config, as used by base-config */
    snprintf(txtbuf, sizeof txtbuf, "http://%s:%d/", 
             nf_state.proxy.hostname, nf_state.proxy.port);
    write_userconfig("HTTP_PROXY", txtbuf);
  }

  /*
   * First fetch the required files.
   */
  for (i = 0; files_to_fetch[i].remote != NULL; i++) {
    if (series != files_to_fetch[i].series)
      continue;

    snprintf(filename, sizeof filename - 1, "/tmp/%s", files_to_fetch[i].local);

    INFOMSG("retrieving %s from %s", filename, files_to_fetch[i].remote);

    if (nf_fetchfile(files_to_fetch[i].remote, filename) != 0) {
      vaproblemBox(_("Problem"), _("Download failed for the file '%s'."),
                   files_to_fetch[i].remote);
      return 1;
    }

    INFOMSG("sucessfully installed %s", filename);
  }

  /*
   * Now install the required files.
   */
  for (i = 0; files_to_fetch[i].remote != NULL; i++) {
    if (series != files_to_fetch[i].series)
      continue;

    snprintf(filename, sizeof filename - 1, "/tmp/%s", files_to_fetch[i].local);

    if (files_to_fetch[i].install_fn(filename) != 0) {
      vaproblemBox(_("Problem"), _("Installation of the file '%s' failed."), filename);
      return 1;
    }
  }

  return 0;
}


#ifdef _TESTING_
/*
 * Source code for testing net-fetch.  This may require some tweaking on
 * your system.  Please don't remove this code.  (marcel@debian.org)
 */

int
install_from_file (const char *filename, const char *descr)
  {
    int          status;
    struct stat  statbuf;
    const char   tmpdir [] = "/tmp/notarget/tmp";
    char         instdir [128];

    snprintf (instdir, sizeof instdir - 1, "%s/drivers", tmpdir);
    if (!NAME_ISDIR (tmpdir, &statbuf))
      {
        if (!mkdir (tmpdir, 01777))
          {
            chown (tmpdir, 0, 3); /* root_uid= 0 , sys_gid= 3 */
          }
      }

    if (!NAME_ISDIR (instdir, &statbuf))
      {
        if (!mkdir (instdir, 01777))
          {
            chown (instdir, 0, 3); /* root_uid= 0 , sys_gid= 3 */
          }
      }

    if (!NAME_ISDIR (instdir, &statbuf))
      {
        vaproblemBox("Directory Error",
                     "Creation of the temporary directory '%s' failed.",
                     instdir);
        return 1;
      }

    sprintf (prtbuf, "Installing %s from %s...", descr, filename);
    pleaseWaitBox (prtbuf);
    chdir (instdir);

    sprintf (prtbuf, "zcat %s | tar -tf -", filename); /* list files, see system logs. */
    status = execlog (prtbuf, LOG_INFO);

    if (status)
      {
        boxPopWindow ();
        chdir ("/");
        vaproblemBox("File Error",
                     "There was a problem extracting the %s from the file '%s'.",
                     descr, filename);
        return 1;
      }
    else
      {
        status = 0;  // status = execlog ("./install.sh /target", LOG_INFO);
        chdir ("/");
        boxPopWindow ();
        if (status)
          {
            vaproblemBox("Install Script Error",
                         "There was a problem installing the %s: the install script, 'install.sh', failed.",
                         descr, filename);
            return 1;
          }
      }

    return 0;
  }


char *
get_device (const char *diskimage)
  {
    // char *device = find_unused_loop_device();
    // int   ro     = 1;
    //
    // sprintf (prtbuf, "%s/%s", Archive_Dir, diskimage);
    // if (set_loop(device,prtbuf,0,&ro))
    //   {
    //     syslog (LOG_ERR, "set_loop failed for %s on %s", prtbuf, device);
    //     return NULL;
    //   }
    // else
    //   {
    //     syslog (LOG_INFO, "set_loop for %s on %s", prtbuf, device);
    //     return device;
    //   }
    return "/dev/loop0";
  }


void
release_device (const char *device)
  {
    // del_loop(device);
  }


int
install_floppy (char *device, const char *type, const char *text)
  {
    int status = 0;

    // status = mount_and_check_floppy (device, type, text);
    // if (status == DLG_CANCEL)
    //   {
    //     return DLG_CANCEL;
    //   }
    //   else
    //   {
    //     return 1;
    //   }

    sprintf (prtbuf, "Installing the %s ", text);
    if (strncmp (device, "/dev/loop", 8))
      {
        strcat (prtbuf,"...");
      }
    else
      {
        strcat (prtbuf, "from floppy images on mounted medium ...");
      }

    pleaseWaitBox (prtbuf);

    sleep (3);
    // chdir ("/floppy");
    // status = execlog ("./install.sh /target", LOG_INFO);
    // chdir ("/");
    // execlog ("umount /floppy", LOG_DEBUG);

    boxPopWindow ();

    if (status)
      {
        return 1;
      }

    return 0;
  }


int
extract_from_file (const char *fil, const char *descr)
  {
    int status;

    sprintf (prtbuf, "The %s is being extracted from %s...", descr, fil);
    pleaseWaitBox (prtbuf);

    chdir (target_path (""));
    sprintf (prtbuf, "zcat %s | tar -tf -", fil); /* list files, see system logs. */
    status = execlog (prtbuf, LOG_INFO);
    chdir ("/");

    boxPopWindow ();

    if (status)
      {
        vaproblemBox("File error!",
                     "There was a problem extracting the %s from the file '%s'.",
                     descr, fil);
        return 1;
      }

    return 0;
  }


int
main (int argc, char *argv [])
  {
    struct stat statbuf;
    char        txtbuf [2048];

    assert (LOAD_TRMFILE ("test.trm") != 0);

    get_kver ();
    get_subarch_name ();

    if (bootargs.disksize == NULL)
      {
        bootargs.disksize = "1.44";
      }
    bootargs.flavor = "compact"; // compact, safe, standard
    bootargs.isquiet = 0;

    setup_image_names ();

    Archive_Dir = strdup ("netfetch");
    if (!NAME_ISDIR (target_path (""), &statbuf))
      {
        snprintf (txtbuf, sizeof txtbuf - 1, "mkdir %s", target_path (""));
        system (txtbuf);
        /* @@ Marcel TODO: check for errors here. */
      }

    boxInit ();

    wideMessageBox ("\nInstallation of operating system and modules via net-fetch.\n", "TEST");
    nf_install_os ();

    wideMessageBox ("\nInstallation of the base system via net-fetch.\n", "TEST");
    nf_install_base ();

    wideMessageBox ("\nNet-fetch tests completed.\n", "TEST");
    boxFinished ();

    return EXIT_SUCCESS;
  }
#endif /* _TESTING_ */
