/*
 Language Chooser Program
 This should be considered as a prototype for the final product.

 Copyright (C) 1999 The Software in the Public Interest (SPI)
 Written by Michael Sobolev <mss@transas.com>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <assert.h>

#include <locale.h>

#ifndef HAVE_UTF_CONSOLE
#include <slang.h>
#include <newt.h>
#else
#include <wchar.h>

#include "slang14/src/slang.h"
#include "newt-utf/newt.h"
#endif

#include "langs.h"

#ifndef NEWT_KEY_ESC
#define NEWT_KEY_ESC    0x1b
#endif

    void set_font (const char *, const char *);

    const struct language_definition *choose_language (const struct language_definition *);
    const struct language_item *choose_variant (const struct language_list *);
    void update_help (const char *, int);

#ifdef HAVE_UTF_CONSOLE
    int wcwidth (wchar_t);

static size_t
strwidth (const char *what)
{
    size_t res;
    int k;
    const char *p;
    wchar_t c;

    for (res = 0, p = what ; (k = mbtowc (&c, p, MB_LEN_MAX)) > 0 ; p += k)
        res += wcwidth (c);

    return res;
}

static void
wpadit (char *dest, const char *source, size_t width)
{
    wchar_t c;
    int k;
    const char *p;
    char *q;
    size_t len;

    for (len = 0, p = source, q = dest ; (k = mbtowc (&c, p, MB_LEN_MAX)) > 0 && len < width ; p += k)
    {
        len += wcwidth (c);

        q += wctomb (q, c);
    }

    if (k == 0)
        for (; len < width ; ++len)
        {
            *q++ = ' '; /* I hope this is ok */
        }

    *q = '\0';
}
#endif

// MAX is a macros, which is not good.
int
max (int a, int b)
{
    return a > b ? a : b;
}

void
doIt (newtComponent lb, void *data)
{
#ifndef HAVE_UTF_CONSOLE
    struct language_definition *lang = (struct language_definition *)newtListboxGetCurrent (lb);

    set_font (lang->font, lang->acm);

    update_help (lang->hint, 1);
#if 0
    newtRefresh ();
#else
#endif
#endif
}

int
main (int argc, char **argv)
{
    setlocale (LC_CTYPE, "");

    if (newtInit () == 0)
    {
        const struct language_definition *value = NULL;
        const struct language_item *lang = NULL;

        newtCls ();

        value = choose_language (available_languages ());

        if (value != NULL)
            lang = choose_variant (value->list);

        newtCls ();

        newtFinished ();

        if (value == NULL || lang == NULL)
            printf ("user canceled the action\n");
        else
        {
            printf ("chosen language: %s / %s\n", value->ename, lang->name);
            printf ("other params:\n");
            printf ("  locale: %s\n", lang->locale);
            printf ("  acm: %s\n", lang->acm);
            printf ("  font: %s\n", lang->font);
            printf ("  keymap: %s\n", lang->keymap);
            printf ("  messages: %s\n", lang->msgcat);

            printf ("The following command would be executed:\n");
            printf ("  echo %s > /tmp/kbdconf\n", lang->keymap);
            printf ("  zcat /etc/keymaps.tgz | tar -xOf - %s | loadkmap\n", lang->keymap);
            printf ("  zcat /etc/messages.tgz | tar -xOf - %s > /etc/messages.trm\n", lang->msgcat);
        }
    }
    else
        puts ("Unable to initialize newt");

    return 0;
}

const struct language_definition *
choose_language (const struct language_definition *_)
{
    struct language_definition *result;
    newtComponent form, lang_list;
    int i;
    struct newtExitStruct exit_code;
    size_t width;

    // Let's calculate the maximum length
    for (i = 0, width = 0 ; _[i].name != NULL ; ++i)
    {
        size_t l;
#ifdef HAVE_UTF_CONSOLE
        l = strwidth (_[i].hint);
#else
        l = strlen (_[i].ename);
#endif
        if (l > width)
            width = l;
    }

    // 2 -- main window borders
    // 2 -- listbox border
    // 4 -- spaces 
    width = max (25, width + 4);

    newtCenteredWindow (width, 14, "Choose The Language");
#ifndef HAVE_UTF_CONSOLE
    update_help ("Help here!", 1);
#endif
    form = newtForm (NULL, NULL, 0);

//  newtFormAddHotKey (form, NEWT_KEY_ESC);
    newtFormAddHotKey (form, NEWT_KEY_LEFT);
    newtFormAddHotKey (form, NEWT_KEY_RIGHT);

    lang_list = newtListbox (0, 0, 14, NEWT_FLAG_RETURNEXIT | NEWT_FLAG_SCROLL);

    width -= 4;

    {
        char *tempo = (char *)malloc (width * MB_LEN_MAX + 1);

        for (i = 0 ; _[i].name != NULL ; ++i)
        {
#ifdef HAVE_UTF_CONSOLE
            wpadit (tempo, _[i].hint, width);
#else
            sprintf (tempo, " %-*s", width, _[i].ename);
#endif
            newtListboxAppendEntry (lang_list, tempo, _ + i);
        }

        free (tempo);
    }

    newtComponentAddCallback (lang_list, doIt, NULL);

    newtFormAddComponents (form, lang_list, NULL);

    newtListboxSetCurrent (lang_list, 0);

    newtFormRun (form, &exit_code);

    switch (exit_code.reason)
    {
        case NEWT_EXIT_HOTKEY:
            switch (exit_code.u.key)
            {
                default:    // Unknown key???
                    ;

//              case NEWT_KEY_ESC:
                case NEWT_KEY_LEFT:
                    // User decided to cancel the action.
                    result = NULL;
                    break;

                case NEWT_KEY_RIGHT:
                case NEWT_KEY_ENTER:
                    result = (struct language_definition *)newtListboxGetCurrent (lang_list);
            }
            break;

        case NEWT_EXIT_COMPONENT:
            // Do something
            if (exit_code.u.co == lang_list)
                result = (struct language_definition *)newtListboxGetCurrent (lang_list);
            else
                result = NULL;
            break;

        default:    // Unknown reason.
            result = NULL;
    }

    newtFormDestroy (form);

    newtPopWindow ();

    return result;
}

const struct language_item *
choose_variant (const struct language_list *_)
{
    newtComponent form, variants;
    struct newtExitStruct exit_code;
    int i, width, done = 0;
    const struct language_item *result;
    const struct language_list_item *tempo;

    for (i = 0, width = 0 ; _->items[i].d != 0 ; ++i)
    {
        const char *name = _->items[i].d == 1 ? ((const struct language_item *)_->items[i].p)->name : ((const struct language_list *)_->items[i].p)->name;
#ifdef HAVE_UTF_CONSOLE
        size_t l = strwidth (name);
#else
        size_t l = strlen (name);
#endif
        if (l > width)
            width = l;
    }

#if HAVE_UTF_CONSOLE
    width = max (width + 8, strwidth (_->name) + 6);
#else
    width = max (width + 8, strlen (_->name) + 6);
#endif
    newtCenteredWindow (width, 14, _->name);

    form = newtForm (NULL, NULL, 0);

//  newtFormAddHotKey (form, NEWT_KEY_ESC);
    newtFormAddHotKey (form, NEWT_KEY_LEFT);
    newtFormAddHotKey (form, NEWT_KEY_RIGHT);

    variants = newtListbox (2, 1, 11, NEWT_FLAG_BORDER | NEWT_FLAG_RETURNEXIT | NEWT_FLAG_SCROLL);

    for (i = 0 ; _->items[i].d != 0 ; ++i)
    {
        const char *name = _->items[i].d == 1 ? ((const struct language_item *)_->items[i].p)->name : ((const struct language_list *)_->items[i].p)->name;

        newtListboxAddEntry (variants, name, _->items + i);
    }

    newtFormAddComponents (form, variants, NULL);

    do {
        newtFormRun (form, &exit_code);

        switch (exit_code.reason)
        {
            case NEWT_EXIT_HOTKEY:
                switch (exit_code.u.key)
                {
                    default:    // Unknown key???
                        ;

//                  case NEWT_KEY_ESC:  // User decided to retyrn to previous list
                    case NEWT_KEY_LEFT:
                        result = NULL;
                        done = 1;
                        break;

                    case NEWT_KEY_RIGHT:
                    case NEWT_KEY_ENTER:
                        tempo = (struct language_list_item *)newtListboxGetCurrent (variants);

                        if (tempo->d == 2)
                            result = choose_variant ((const struct language_list *)tempo->p);
                        else
                            result = (const struct language_item *)tempo->p;
                }
                break;

            case NEWT_EXIT_COMPONENT:
                // Do something
                if (exit_code.u.co == variants)
                {
                    tempo = (struct language_list_item *)newtListboxGetCurrent (variants);

                    if (tempo->d == 2)
                        result = choose_variant ((const struct language_list *)tempo->p);
                    else
                        result = (const struct language_item *)tempo->p;
                }
                else
                {
                    result = NULL;
                    done = 1;
                }
                break;

            default:    // Unknown reason.
                result = NULL;
                done = 1;
        }
    } while (result == NULL && !done);

    newtFormDestroy (form);

    newtPopWindow ();

    return result;
}

void
set_font (const char *font, const char *acm)
{
#if HAVE_UTF_CONSOLE
    ;   /* Do nothing as our console supports UTF */
#else
    char buffer[100];

    strcpy (buffer, "consolechars -f ");
    strcat (buffer, font);

    if (acm && acm[0] != '\0')
    {
        strcat (buffer, " -m ");
        strcat (buffer, acm);
    }
             
    newtSuspend ();

    system (buffer);

    newtResume ();
#endif
}

static int help_level = 0;

void
update_help (const char *_, int replace)
{
    int w, h, _len, len;
    char *buffer;

    newtGetScreenSize (&w, &h);

    if (replace && help_level)
    {
        newtPopHelpLine ();

        --help_level;
    }

    len = (w + (_len = strlen (_))) / 2 + 1;

    buffer = (char *)malloc (len);

    memset (buffer, ' ', len - 1);
    strcpy (buffer + len - _len - 1, _);

    newtPushHelpLine (buffer);

    free (buffer);

    ++help_level;
}

void
pop_help (void)
{
    if (help_level > 0)
    {
        newtPopHelpLine ();

        --help_level;
    }
}
