// pkg_ver_item.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Implementations of stuff in pkg_ver_item.h

#include "pkg_item.h"
#include "pkg_subtree.h"
#include "pkg_ver_item.h"

#include "pkg_description_treeitem.h"
#include "dep_item.h"

#include "apt.h"

#include <string>
#include <apt-pkg/configuration.h>

class pkg_grouppolicy_ver:public pkg_grouppolicy
{
public:
  pkg_grouppolicy_ver(pkg_subtree *_root):pkg_grouppolicy(_root) {}

  void add_package(pkgCache::PkgIterator pkg);
};

void setup_package_versions(pkgCache::PkgIterator pkg, pkg_vertree *tree)
{
  for(pkgCache::VerIterator i=pkg.VersionList(); !i.end(); i++)
    tree->add_child(new pkg_ver_item(i));

  for(pkgCache::PrvIterator i=pkg.ProvidesList(); !i.end(); i++)
    tree->add_child(new pkg_ver_item(i.OwnerVer(), true));
}

void setup_package_versions(pkgCache::PkgIterator pkg, pkg_vertree_generic *tree)
{
  for(pkgCache::VerIterator i=pkg.VersionList(); !i.end(); i++)
    tree->add_child(new pkg_ver_item(i));

  for(pkgCache::PrvIterator i=pkg.ProvidesList(); !i.end(); i++)
    tree->add_child(new pkg_ver_item(i.OwnerVer(), true));
}

void pkg_grouppolicy_ver::add_package(pkgCache::PkgIterator pkg)
{
  pkg_vertree *newtree=new pkg_vertree(pkg);

  get_root()->add_child(newtree);

  setup_package_versions(pkg, newtree);
}

pkg_grouppolicy *pkg_grouppolicy_ver_factory::instantiate(pkg_subtree *root)
{
  return new pkg_grouppolicy_ver(root);
}

int pkg_ver_item::get_normal_attr()
{
  pkgCache::PkgIterator pkg=version.ParentPkg();
  pkgDepCache::StateCache &state=(*apt_cache_file)[pkg];

  if(pkg.CurrentVer()==version &&
     (state.Delete() ||
      (state.Install() &&
       !state.InstBroken() &&
       state.InstVerIter(*apt_cache_file)!=version)))
    return COLOR_PAIR(get_color("pkg_to_remove", COLOR_BLACK, COLOR_MAGENTA));

  else if(state.NewInstall() ||
	  (state.Install() &&
	   !state.InstBroken() &&
	   state.InstVerIter(*apt_cache_file)==version) ||
	  (state.iFlags&pkgDepCache::ReInstall))
    return COLOR_PAIR(get_color("pkg_to_install", COLOR_BLACK, COLOR_GREEN));

  else if(state.Install() && state.InstBroken() && state.InstVerIter(*apt_cache_file)==version)
    return COLOR_PAIR(get_color("pkg_broken", COLOR_BLACK, COLOR_RED));

  else if(pkg.CurrentVer()!=version)
    return vs_treeitem::get_normal_attr();

  else if(state.NowBroken())
    return COLOR_PAIR(get_color("pkg_broken", COLOR_BLACK, COLOR_RED));

  else
    return A_BOLD|vs_treeitem::get_normal_attr();
}

void pkg_ver_item::display(vs_tree *win, int y)
{
  if(show_pkg_name)
    vs_treeitem::display(win, y, string(version.ParentPkg().Name())+" "+version.VerStr());
  else
    vs_treeitem::display(win, y, version.VerStr());
}

void pkg_ver_item::highlighted(vs_tree *win)
{
  win->set_status(apt_package_records->Lookup(version.FileList()).ShortDesc());
}

const char *pkg_ver_item::tag() const
{
  return version.VerStr();
}

void pkg_ver_item::select()
{
  if(version==version.ParentPkg().CurrentVer())
    {
      if(!(*apt_cache_file)[version.ParentPkg()].Keep())
	{
	  (*apt_cache_file)->MarkKeep(version.ParentPkg());
	  (*apt_cache_file)->get_ext_state(version.ParentPkg()).selection_state=pkgCache::State::Install;
	}
      else if(_config->FindB("Aptitude::Allow-ReInstall", true))
	{
	  (*apt_cache_file)->MarkInstall(version.ParentPkg(), _config->FindB("Aptitude::Auto-Install", false));
	  (*apt_cache_file)->SetReInstall(version.ParentPkg(), true);
	}
    }
  else if(version==(*apt_cache_file)[version.ParentPkg()].CandidateVerIter(*apt_cache_file))
    (*apt_cache_file)->MarkInstall(version.ParentPkg(), _config->FindB("Aptitude::Auto-Install", false));
  // FIXME: this should work even for other versions of the package (besides
  //       the 'target' and current versions)
}

void pkg_ver_item::remove()
{
  if(version==version.ParentPkg().CurrentVer())
    {
      if((*apt_cache_file)[version.ParentPkg()].iFlags&pkgDepCache::ReInstall)
	{
	  (*apt_cache_file)->SetReInstall(version.ParentPkg(), false);
	  (*apt_cache_file)->MarkKeep(version.ParentPkg());
	  (*apt_cache_file)->get_ext_state(version.ParentPkg()).selection_state=pkgCache::State::Install;
	}
      else
	(*apt_cache_file)->MarkDelete(version.ParentPkg());
    }
  else if(version==(*apt_cache_file)[version.ParentPkg()].CandidateVerIter(*apt_cache_file) && (*apt_cache_file)[version.ParentPkg()].Install())
    (*apt_cache_file)->MarkKeep(version.ParentPkg());
}

void pkg_ver_item::hold()
{
  if(version==version.ParentPkg().CurrentVer())
    {
      (*apt_cache_file)->MarkKeep(version.ParentPkg());
      (*apt_cache_file)->SetReInstall(version.ParentPkg(), false);
    }
}

void pkg_ver_item::purge()
{
}

pkg_ver_screen::pkg_ver_screen(pkgCache::PkgIterator pkg)
{
  pkg_vertree *newtree=new pkg_vertree(pkg, true);

  setup_package_versions(pkg, newtree);

  set_root(newtree,true);
  set_header("Available versions of "+string(pkg.Name()));

  prev=vscreen_show(this);
}

bool pkg_ver_item::dispatch_char(chtype ch)
{
  if(global_bindings.key_matches(ch, "PackageDependencies"))
    {
      new pkg_dep_screen(version);
      return true;
    }
  else if(global_bindings.key_matches(ch, "PackageDescription"))
    {
      new pkg_description_screen(version);
      return true;
    }
  else
    return pkg_tree_node::dispatch_char(ch);
}

void pkg_ver_screen::dispatch_char(chtype ch)
{
  if(global_bindings.key_matches(ch, "ExitScreen"))
    {
      vscreen_show(prev);
      vscreen_preparedelete(this);
    }
  else
    vs_tree::dispatch_char(ch);
}
