// pkg_tree.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  A package-tree displayer.

#include "pkg_tree.h"

#include "vs_progress.h"
#include "pkg_node.h"
#include "pkg_item.h"
#include "pkg_subtree.h"
#include "pkg_grouppolicy.h"
#include "download_screen.h"

#include "apt.h"

#include <apt-pkg/progress.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/algorithms.h>
#include <apt-pkg/error.h>
#include <apt-pkg/strutl.h>

list<pkg_tree *> active_trees;
// We need this because *all* trees have to rebuild their internal structure
// whenever something gets modified..

void rebuild_trees()
{
  for(list<pkg_tree *>::iterator i=active_trees.begin(); i!=active_trees.end(); i++)
    (*i)->build_tree();
}

class pkg_action_tree:public pkg_tree
// Displays which packages are being upgraded, installed, removed, etc.
// Closely tied to pkg_tree..
{
  pkg_tree *prev;
public:
  pkg_action_tree(pkg_grouppolicy_factory *_grouping, pkg_tree *_prev):pkg_tree(_grouping), prev(_prev)
  {
    set_header("Actions to be taken");
  }

  void dispatch_char(chtype ch)
  {
    if(global_bindings.key_matches(ch, "DoInstallRun"))
      {
	install_run();
	vscreen_show(prev);
	vscreen_preparedelete(this);
      }
    else if(global_bindings.key_matches(ch, "ExitScreen"))
      {
	vscreen_show(prev);
	vscreen_preparedelete(this);
      }
    else
      vs_tree::dispatch_char(ch);
  }
};

pkg_tree::pkg_tree(pkg_grouppolicy_factory *_grouping):vs_tree(), grouping(_grouping)
{
  if(grouping)
    build_tree();

  active_trees.push_back(this);
}

pkg_tree::~pkg_tree()
{
  active_trees.remove(this);
}

void pkg_tree::snarf_errs()
{
  while(pending_err())
    {
      add_status_widget(new vs_tree_msg(this,first_err(),COLOR_PAIR(get_color("pkgtree_error", COLOR_WHITE, COLOR_RED))|A_BOLD), false);
      pop_err();
    }
}

void pkg_tree::set_grouping(pkg_grouppolicy_factory *_grouping)
{
  if(grouping)
    delete grouping;

  grouping=_grouping;

  if(grouping)
    build_tree();
}

void pkg_tree::build_tree()
{
  set_root(NULL);
  // Clear the current tree.

  if(grouping && apt_cache_file)
    {
      pkg_subtree *mytree=new pkg_subtree("All Packages", true);
      pkg_grouppolicy *grouper=grouping->instantiate(mytree);

      mytree->set_depth(-1);

      for(pkgCache::PkgIterator i=**apt_cache_file; !i.end(); i++)
	grouper->add_package(i);

      mytree->sort();

      set_root(mytree);

      delete grouper;
    }
  snarf_errs();
  repaint();
}

void pkg_tree::update_pkglist()
{
  vs_progress progbar(this);
  add_status_widget(&progbar);
  do_pkglist_update(&progbar);
  snarf_errs();

  set_root(NULL);
  repaint();
  apt_reload_cache(&progbar);
  snarf_errs();
  remove_status_widget(&progbar);
  if(_config->FindB("Aptitude::Forget-New-On-Update", false))
    (*apt_cache_file)->forget_new();
  rebuild_trees();
  repaint();
}

void pkg_tree::install_run()
{
  vs_progress progbar(this);
  add_status_widget(&progbar);
  do_install_run(&progbar);
  snarf_errs();

  set_root(NULL);
  repaint();
  apt_reload_cache(&progbar);
  remove_status_widget(&progbar);
  snarf_errs();
  if(_config->FindB("Aptitude::Forget-New-On-Install", false))
    (*apt_cache_file)->forget_new();
  rebuild_trees();
  repaint();
}

void pkg_tree::dispatch_char(chtype ch)
{
  if(pending_err())
    {
      pop_err();
      show_status();
      repaint();
    }
  // Should I also respond to the character?  Right now I'm not, simply because
  // it's the least dangerous thing to do..
  else if(global_bindings.key_matches(ch, "UpdatePackageList"))
    update_pkglist();
  else if(global_bindings.key_matches(ch, "DoInstallRun"))
    {
      if(_config->FindB("Aptitude::Fix-Broken", true))
	{
	  bool founderr=false;

	  pkgProblemResolver fixer(*apt_cache_file);
	  for(pkgCache::PkgIterator i=**apt_cache_file; !i.end(); i++)
	    if(!i.CurrentVer().end() &&
	       (*apt_cache_file)->get_ext_state(i).selection_state==pkgCache::State::Hold)
	      fixer.Protect(i);
	    else
	      {
		pkgDepCache::StateCache &state=(*apt_cache_file)[i];
		if(state.InstBroken() || state.NowBroken())
		  (*apt_cache_file)->MarkInstall(i,true);
		else if(state.Delete())
		  fixer.Remove(i);
	      }
	  //fixer.InstallProtect();
	  if(!fixer.Resolve(true))
	    founderr=true;

	  if(founderr)
	    _error->Error("Unable to correct dependencies, some packages cannot be installed");
	}
      if(_config->FindB("Aptitude::Display-Planned-Action", true))
	vscreen_show(new pkg_action_tree(new pkg_grouppolicy_mode_factory(new pkg_grouppolicy_end_factory),
					 this));
      else
	install_run();
    }
  else if(global_bindings.key_matches(ch, "ForgetNewPackages"))
    {
      (*apt_cache_file)->forget_new();

      rebuild_trees();
      repaint();
    }
  else
    vs_tree::dispatch_char(ch);
}

void pkg_tree::paint_header()
{
  if(((aptitudeDepCache *) apt_cache_file) &&
     ((*apt_cache_file)->DebSize()!=0 || (*apt_cache_file)->UsrSize()!=0) &&
     _config->FindB("Aptitude::Display-SizeStats", true))
    {
      int width,height;
      getmaxyx(height,width);

      string sizeinf;
      if((*apt_cache_file)->DebSize()!=0)
	sizeinf+="DL Size: "+SizeToStr((*apt_cache_file)->DebSize())+"B";
      if((*apt_cache_file)->UsrSize()!=0 && (*apt_cache_file)->DebSize()!=0)
	sizeinf+="   ";
      if((*apt_cache_file)->UsrSize()>0)
	  sizeinf+="Will Use "+SizeToStr((*apt_cache_file)->UsrSize())+"B of disk space";
      else if((*apt_cache_file)->UsrSize()<0)
	sizeinf+="Will Free "+SizeToStr((*apt_cache_file)->UsrSize())+"B of disk space";

      unsigned int start_sizeinf=width-sizeinf.size();
      if(start_sizeinf<=get_header().size())
	start_sizeinf=get_header().size()+1;

      attrset(COLOR_PAIR(get_header_color())|A_BOLD);
      mvaddnstr(0, 0, get_header().c_str(), width);

      for(unsigned int i=get_header().size(); i<(unsigned) width && i<start_sizeinf; i++)
	// Yuck.  Let one unsigned into your code and you'll never hear the end
	// of it >=)
	addch(' ');

      if(start_sizeinf<(unsigned) width)
	{
	  mvaddnstr(0, start_sizeinf, sizeinf.c_str(), width-start_sizeinf);
	  for(unsigned int i=start_sizeinf+sizeinf.size(); i<(unsigned) width; i++)
	    addch(' ');
	}
    }
  else
    vs_tree::paint_header();
}
