/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Startup.h"

#include "Byteswapping.h"		// Canonical
#include "Hordes.h"				// Hordes::New
#include "Miscellaneous.h"		// IsExecutable
#include "Platform.h"			// ForceStartupScreen, QueryNewDocument
#include "PreferenceMgr.h"		// Preference
#include "Strings.r.h"			// kStr_Autoload, etc.

#include <algorithm>			// find()
#include <string>
#include <vector>

// Startup actions.
//static Bool				gAskWhatToDo;
static Bool				gCreateSession;
static Bool				gOpenSession;

// Post-startup actions.
static Bool				gStartNewHorde;

// Quit actions.
static Bool				gClose;
static Bool				gQuitOnExit;
static Bool				gQuit;

// Action-specific data.
static Configuration	gCfg;				// For CreateSession.
static FileReference	gSessionOpenRef;	// For OpenSession.
static HordeInfo		gHorde;				// For StartNewGremlin
static StringList		gHordeApps;			// For StartNewGremlin
static FileReference	gSessionCloseRef;	// For CloseSession

	// These are the files listed on the command line.
static string			gAutoRunApp;
static FileRefList		gAutoLoadFiles1;
//static FileRefList		gAutoRunFiles1;
//static FileRefList		gAutoRunAndQuitFiles1;

	// These are the files found in the Auto* directories.
static FileRefList		gAutoLoadFiles2;
static FileRefList		gAutoRunFiles2;
static FileRefList		gAutoRunAndQuitFiles2;


// These are the keys we use when tracking what the user has specified.

static const char		kOptPSF[]				= "psf";
static const char		kOptROM[]				= "rom";
static const char		kOptRAM[]				= "ram";
static const char		kOptDevice[]			= "device";
static const char		kOptSkin[]				= "skin";
static const char		kOptLoad[]				= "load";
static const char		kOptRun[]				= "run";
static const char		kOptQuitOnExit[]		= "quit_on_exit";
static const char		kOptLog[]				= "log_dir";
static const char		kOptHordeFirst[]		= "horde_first";
static const char		kOptHordeLast[]			= "horde_last";
static const char		kOptHordeApps[]			= "horde_apps";
static const char		kOptHordeSaveDir[]		= "horde_save_dir";
static const char		kOptHordeSaveFreq[]		= "horde_save_freq";
static const char		kOptHordeDepthMax[]		= "horde_depth_max";
static const char		kOptHordeDepthSwitch[]	= "horde_depth_switch";


// These are the options the user can specify on the command line.

static const struct
{
	const char*	option;
	const char*	optKey;
	int			optType;	// 0 == no parameter, 1 = has parameter
}
kOptionMap [] =
{
	{ "-psf",					kOptPSF,				1 },
	{ "-rom",					kOptROM,				1 },
	{ "-ram",					kOptRAM,				1 },
	{ "-ram_size",				kOptRAM,				1 },
	{ "-device",				kOptDevice,				1 },
	{ "-skin",					kOptSkin,				1 },
	{ "-silkscreen",			kOptSkin,				1 },
	{ "-load_apps",				kOptLoad,				1 },
	{ "-run_app",				kOptRun,				1 },
	{ "-quit_on_exit",			kOptQuitOnExit,			0 },
	{ "-log_save_dir",			kOptLog,				1 },
	{ "-horde",					kOptHordeFirst,			1 },
	{ "-horde_first",			kOptHordeFirst,			1 },
	{ "-horde_last",			kOptHordeLast,			1 },
	{ "-horde_apps",			kOptHordeApps,			1 },
	{ "-horde_save_dir",		kOptHordeSaveDir,		1 },
	{ "-horde_save_freq",		kOptHordeSaveFreq,		1 },
	{ "-horde_depth_max",		kOptHordeDepthMax,		1 },
	{ "-horde_depth_switch",	kOptHordeDepthSwitch,	1 }
};


// The list of devices users are allowed to specify for the
// -device switch, and the values they map to.

// -*- NEW DEVICE -*-

static const struct
{
	const char*	deviceName;
	DeviceType	deviceType;
}
kDevices[] =
{
	{ "Pilot",		kDevicePilot1000 },
	{ "Pilot1000",	kDevicePilot1000 },
	{ "Pilot5000",	kDevicePilot1000 },
	{ "PalmPilot",	kDevicePalmPilotPersonal },
	{ "PalmIII",	kDevicePalmIII },
	{ "PalmIIIx",	kDevicePalmIIIx },
	{ "PalmV",		kDevicePalmV },
	{ "PalmVII",	kDevicePalmVII },
	{ "ColorDevice",kDeviceAustin },
	{ "PalmVIIEZ",	kDevicePalmVIIEZ }

};

typedef StringStringMap	OptionList;


// Handy macro for helping us find and access options in the OptionList.
// For the option with the base name "name", this macro defines:
//
//		"iter<name>"	An iterator pointing to the named OptionList entry.
//		"have<name>"	A bool that says whether the iterator points to
//						something valid.
//		"opt<name>"		A string reference that refers to the option value
//						if "have<name>" is true.

#define DEFINE_VARS(name)												\
	OptionList::iterator	iter##name	= options.find(kOpt##name);		\
	Bool					have##name	= iter##name != options.end();	\
	string&					opt##name	= iter##name->second


/***********************************************************************
 *
 * FUNCTION:    PrvGetDatabaseInfosFromAppNames
 *
 * DESCRIPTION: Given a list of application names (as determined either
 *				by their 'tAIN' resources or their database names) and
 *				return AppInfos for all applications installed in the
 *				system with those names.
 *
 * PARAMETERS:  names - the StringList with the names of the applications
 *					to search for.  *All* executables are considered,
 *					includes .prc's and .pqa's.
 *
 *				results - the DatabaseInfoList to receive the DatabaseInfos of
 *					the executables with a name that appears in the
 *					"names" collection.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

static void PrvGetDatabaseInfosFromAppNames (StringList& names, DatabaseInfoList& results)
{
	if (!names.empty())
	{
		DatabaseInfoList	dbInfos;
		::GetDatabases (dbInfos, kApplicationsOnly);

		DatabaseInfoList::iterator	infoIter = dbInfos.begin();
		while (infoIter != dbInfos.end())
		{
			string					appInfoName(infoIter->name);
			StringList::iterator	nameIter = find (names.begin(), names.end(), appInfoName);
			if (nameIter != names.end())
			{
				results.push_back (*infoIter);
			}

			++infoIter;
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	PrvSetEmbeddedSessionPreferences
 *
 * DESCRIPTION:	Sets special preferences for sessions that are being
 *				run out of embedded resources. Namely, we turn off
 *				debugging and logging features. These preferences do not
 *				"persist" on the system--that is, they are not written
 *				out to the preferences file after execution, so that
 *				other copies of Emulator (perhaps without sessions bound
 *				as resources) will continue to run unaffected.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	none
 *
 ***********************************************************************/

static void PrvSetPrefToFalse(const char *name)
{
	Preference<bool> pref(name);
	pref = false;
}

static void PrvSetPrefToZero(const char *name)
{
	Preference<uae_u8> pref(name);
	pref = 0;
}

static void PrvSetEmbeddedSessionPreferences(void)
{
	PrvSetPrefToFalse(kPrefKeyRedirectNetLib);
	PrvSetPrefToFalse(kPrefKeyReportCorruptedHeap);
	PrvSetPrefToFalse(kPrefKeyReportCorruptedHeap);
	PrvSetPrefToFalse(kPrefKeyReportFreeChunkAccess);
	PrvSetPrefToFalse(kPrefKeyReportHardwareRegisterAccess);
	PrvSetPrefToFalse(kPrefKeyReportInvalidPC);
	PrvSetPrefToFalse(kPrefKeyReportLowMemoryAccess);
	PrvSetPrefToFalse(kPrefKeyReportLowStackAccess);
	PrvSetPrefToFalse(kPrefKeyReportMemMgrDataAccess);
	PrvSetPrefToFalse(kPrefKeyReportMemMgrSemaphore);
	PrvSetPrefToFalse(kPrefKeyReportScreenAccess);
	PrvSetPrefToFalse(kPrefKeyReportStackAlmostOverflow);
	PrvSetPrefToFalse(kPrefKeyReportStackOverflow);
	PrvSetPrefToFalse(kPrefKeyReportStorageHeapAccess);
	PrvSetPrefToFalse(kPrefKeyReportSysFatalAlert);
	PrvSetPrefToFalse(kPrefKeyReportSystemGlobalAccess);
	PrvSetPrefToFalse(kPrefKeyReportUnhandledException);
	PrvSetPrefToFalse(kPrefKeyReportUnimplementedTrap);
	PrvSetPrefToFalse(kPrefKeyReportUninitializedChunkAccess);
	PrvSetPrefToFalse(kPrefKeyReportUninitializedStackAccess);
	PrvSetPrefToFalse(kPrefKeyReportUnlockedChunkAccess);
	PrvSetPrefToFalse(kPrefKeyReportMemoryLeaks);
	PrvSetPrefToFalse(kPrefKeyReportLockedRecords);
	PrvSetPrefToFalse(kPrefKeyInterceptSysFatalAlert);
	PrvSetPrefToFalse(kPrefKeyFillNewBlocks);
	PrvSetPrefToFalse(kPrefKeyFillResizedBlocks);
	PrvSetPrefToFalse(kPrefKeyFillDisposedBlocks);
	PrvSetPrefToFalse(kPrefKeyFillStack);
	PrvSetPrefToZero(kPrefKeyLogErrorMessages);
	PrvSetPrefToZero(kPrefKeyLogWarningMessages);
	PrvSetPrefToZero(kPrefKeyLogGremlins);
	PrvSetPrefToZero(kPrefKeyLogCPUOpcodes);
	PrvSetPrefToZero(kPrefKeyLogEnqueuedEvents);
	PrvSetPrefToZero(kPrefKeyLogDequeuedEvents);
	PrvSetPrefToZero(kPrefKeyLogSystemCalls);
	PrvSetPrefToZero(kPrefKeyLogApplicationCalls);
	PrvSetPrefToZero(kPrefKeyLogSerial);
	PrvSetPrefToZero(kPrefKeyLogSerialData);
	PrvSetPrefToZero(kPrefKeyLogNetLib);
	PrvSetPrefToZero(kPrefKeyLogNetLibData);
	PrvSetPrefToZero(kPrefKeyLogExgMgr);
	PrvSetPrefToZero(kPrefKeyLogExgMgrData);
	PrvSetPrefToZero(kPrefKeyLogHLDebugger);
	PrvSetPrefToZero(kPrefKeyLogHLDebuggerData);
	PrvSetPrefToZero(kPrefKeyLogLLDebugger);
	PrvSetPrefToZero(kPrefKeyLogLLDebuggerData);
}


/***********************************************************************
 *
 * FUNCTION:    PrvDontUnderstand
 * FUNCTION:    PrvMissingArgument
 * FUNCTION:    PrvInvalidRAMSize
 *
 * DESCRIPTION: Display error messages concerning poor, misunderstood
 *				switches and their parameters.
 *
 * PARAMETERS:  The switch ("-foo") guy who caused the problem.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

static void PrvDontUnderstand (const char* arg)
{
	char	buffer[200];
	sprintf (buffer, "Don't understand the command line parameter \"%s\".", arg);
	Platform::CommonDialog ("Fatal Error", buffer, Errors::kOK);
}

static void PrvMissingArgument (const char* arg)
{
	char	buffer[200];
	sprintf (buffer, "The command line parameter \"%s\" needs to be followed by an "
		"argument (\"No, it doesn't.\" \"Yes, it does.\" \"No, it doesn't...\").", arg);
	Platform::CommonDialog ("Fatal Error", buffer, Errors::kOK);
}

static void PrvInvalidRAMSize (const char* arg)
{
	char	buffer[200];
	sprintf (buffer, "\"%s\" is an invalid RAM size. Specify 128, 256, 512, 1024, "
		"2048, 4096, or 8192.", arg);
	Platform::CommonDialog ("Fatal Error", buffer, Errors::kOK);
}


/***********************************************************************
 *
 * FUNCTION:    PrvCollectOptions
 *
 * DESCRIPTION: Breaks up the command line text into pairs of switches
 *				(the "-foo" part) and their optional parameters (anything
 *				that comes after the "-foo" part).  Switches are
 *				validated, and they and any parameter are added to
 *				the given OptionList.
 *
 * PARAMETERS:  argc, argv - standard C parameters describing the
 *					command line.
 *
 *				options - the OptionList to received the parsed and
 *					validated parameters.
 *
 * RETURNED:    True if everything went swimmingly.  If an error occured
 *				(for instance, an invalid switch was specified or a
 *				switch that needed a parameter didn't have one), the
 *				error is reported and the function returns false.
 *
 ***********************************************************************/

static Bool PrvCollectOptions (int argc, char** argv, OptionList& options)
{
	// Iterate over the command line arguments.

	for (int ii = 1; ii < argc;)
	{
		const char*	arg = argv[ii++];
		Bool		handledOption = false;

		// For each argument, see if it is a recognized switch.

		for (int jj = 0; jj < countof (kOptionMap); ++jj)
		{
			if (_stricmp (arg, kOptionMap[jj].option) == 0)
			{
				// It's recognized; see if we need to also collect a parameter.

				if (kOptionMap[jj].optType == 0)
				{
					// No parameter, just add the switch to our collection.

					options[kOptionMap[jj].optKey] = "";
					handledOption = true;
				}
				else if (kOptionMap[jj].optType == 1)
				{
					if (ii < argc)
					{
						// Add the switch and parameter to our collection.

						options[kOptionMap[jj].optKey] = argv[ii++];
						handledOption = true;
					}
					else
					{
						// Needed a parameter, but there wasn't one.
						::PrvMissingArgument (arg);
						return false;
					}
				}
			}
		}

		// Did not understand the argument.

		if (!handledOption)
		{
#if 1
			if ((arg[0] != '-') && (options.find(kOptPSF) != options.end()))
			{
				// For temporary backward compatibility, "bare" file names are treated
				// as paths to .psf files.

				options[kOptPSF] = arg;
			}
			else
#endif
			{
				::PrvDontUnderstand (arg);
				return false;
			}
		}
	}

	return true;
}


/***********************************************************************
 *
 * FUNCTION:    PrvConvertRAM
 *
 * DESCRIPTION: Convert a -ram parameter into a number and validate it.
 *
 * PARAMETERS:  str - the switch parameter.
 *
 *				ramSize - the RAMSizeType to receive the converted value.
 *
 * RETURNED:    True if everything went well.  If the resulting value
 *				was not a valid RAMSizeType value, then returns false.
 *
 ***********************************************************************/

static Bool PrvConvertRAM(const string& str, RAMSizeType& ramSize)
{
	ramSize = atol (str.c_str ());

	MemoryTextList	sizes;
	::GetMemoryTextList (sizes);

	MemoryTextList::iterator	iter = sizes.begin();
	while (iter != sizes.end())
	{
		if (ramSize == iter->first)
		{
			return true;
		}

		++iter;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:    PrvConvertDevice
 *
 * DESCRIPTION: Convert a -device parameter into a DeviceType value
 *				and validate it.
 *
 * PARAMETERS:  str - the switch parameter.
 *
 *				device - the DeviceType to receive the converted value.
 *
 * RETURNED:    True if everything went well.  If the resulting value
 *				was not a valid DeviceType value, then returns false.
 *
 ***********************************************************************/

static Bool PrvConvertDevice(const string& str, DeviceType& device)
{
	for (int ii = 0; ii < countof(kDevices); ++ii)
	{
		if (_stricmp(str.c_str(), kDevices[ii].deviceName) == 0)
		{
			device = kDevices[ii].deviceType;
			return true;
		}
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:    PrvParseFileList
 *
 * DESCRIPTION: Break up a comma-delimited list of files, returning the
 *				pieces in a FileRefList.
 *
 * PARAMETERS:  fileList - the FileRefList to receive the files from the
 *					comma-delimited list.  This collection is *not* first
 *					cleared out, so it's possible to add to the
 *					collection with this function.
 *
 *				option - the string containing the comma-delimited files.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

static void PrvParseFileList (FileRefList& fileList, string option)
{
	StringList	items;
	::SeperateList (items, option, ',');

	StringList::iterator	iter = items.begin();
	while (iter != items.end())
	{
		fileList.push_back (FileReference(*iter));
		++iter;
	}
}


/***********************************************************************
 *
 * FUNCTION:    PrvHandleOpenSessionParameters
 *
 * DESCRIPTION: Handle the following command line options:
 *
 *					kOptPSF
 *
 * PARAMETERS:  options - the OptionList containing the complete set
 *					of parsed switches and parameters.
 *
 * RETURNED:    True if everything when OK.  If there's something wrong
 *				with the specifications, this function displays an
 *				error message and return false.
 *
 ***********************************************************************/

static Bool PrvHandleOpenSessionParameters (OptionList& options)
{
	DEFINE_VARS(PSF);

	if (havePSF)
	{
		FileReference	psfRef = FileReference (optPSF);
		Startup::ScheduleOpenSession(psfRef);
	}

	return true;
}


/***********************************************************************
 *
 * FUNCTION:    PrvHandleCreateSessionParameters
 *
 * DESCRIPTION: Handle the following command line options:
 *
 *					kOptROM
 *					kOptRAM
 *					kOptDevice
 *
 * PARAMETERS:  options - the OptionList containing the complete set
 *					of parsed switches and parameters.
 *
 * RETURNED:    True if everything when OK.  If there's something wrong
 *				with the specifications, this function displays an
 *				error message and return false.
 *
 ***********************************************************************/

static Bool PrvHandleCreateSessionParameters (OptionList& options)
{
	DEFINE_VARS(ROM);
	DEFINE_VARS(RAM);
	DEFINE_VARS(Device);

	Bool					haveSome	= haveROM || haveRAM || haveDevice;
	Bool					haveAll		= haveROM && haveRAM && haveDevice;

	if (haveSome)
	{
		Configuration	cfg;

		// Check for a specified ROM file.

		if (haveROM)
		{
			cfg.fROMFile = FileReference (optROM);
		}

		// Check for a specified RAM size.

		if (haveRAM)
		{
			if (!::PrvConvertRAM(optRAM, cfg.fRAMSize))
			{
				::PrvInvalidRAMSize (optRAM.c_str ());
				return false;
			}
		}

		// Check for a specified device type.

		if (haveDevice)
		{
			if (!::PrvConvertDevice(optDevice, cfg.fDeviceType))
			{
				::PrvDontUnderstand (optDevice.c_str ());
				return false;
			}
		}

		// Try to start up with the specified parameters.
		// If the command line didn't specify all the required values, ask for them.

		if (!haveAll && !Platform::QueryNewDocument (cfg))
		{
			// User cancelled the "New Configuration" dialog.
			// Bring up the dialog with the New/Open/Download/Exit buttons.

			Startup::ScheduleAskWhatToDo();
		}
		else
		{
			Startup::ScheduleCreateSession(cfg);
		}
	}

	return true;
}


/***********************************************************************
 *
 * FUNCTION:    PrvHandleNewHordeParameters
 *
 * DESCRIPTION: Handle the following command line options:
 *
 *					kOptHordeFirst
 *					kOptHordeLast
 *					kOptHordeApps
 *					kOptHordeSaveDir
 *					kOptHordeSaveFreq
 *					kOptHordeDepthMax
 *					kOptHordeDepthSwitch
 *
 * PARAMETERS:  options - the OptionList containing the complete set
 *					of parsed switches and parameters.
 *
 * RETURNED:    True if everything when OK.  If there's something wrong
 *				with the specifications, this function displays an
 *				error message and return false.
 *
 ***********************************************************************/

static Bool PrvHandleNewHordeParameters (OptionList& options)
{
	DEFINE_VARS(HordeFirst);
	DEFINE_VARS(HordeLast);
	DEFINE_VARS(HordeApps);
	DEFINE_VARS(HordeSaveDir);
	DEFINE_VARS(HordeSaveFreq);
	DEFINE_VARS(HordeDepthMax);
	DEFINE_VARS(HordeDepthSwitch);

	if (haveHordeFirst ||
		haveHordeLast ||
		haveHordeApps ||
		haveHordeSaveDir ||
		haveHordeSaveFreq ||
		haveHordeDepthMax ||
		haveHordeDepthSwitch)
	{
		HordeInfo	info;
		StringList	appNames;

		if (haveHordeFirst)
		{
			info.fStartNumber = atoi (optHordeFirst.c_str());

			if (!haveHordeLast)
			{
				info.fStopNumber = info.fStartNumber;
			}
		}

		if (haveHordeLast)
		{
			info.fStopNumber = atoi (optHordeLast.c_str());

			if (!haveHordeFirst)
			{
				info.fStartNumber = info.fStopNumber;
			}
		}

		if (!haveHordeFirst && !haveHordeLast)
		{
			info.fStartNumber = 0;
			info.fStopNumber = 0;
		}

		if (haveHordeApps)
		{
			// Get the list of user-specified names.
			// We can't do much more with them now -- like look up
			// the applications on the device -- as the device may
			// not be up and running at this time.  We have to defer
			// doing that until we're ready to start the Gremlin.

			::SeperateList (appNames, optHordeApps, ',');

		}

		if (haveHordeSaveDir)
		{
			Hordes::SetGremlinsHome(optHordeSaveDir.c_str());
		}
		else
		{
			Hordes::SetGremlinsHomeToDefault();
		}

		if (haveHordeSaveFreq)
		{
			info.fSaveFrequency = atoi (optHordeSaveFreq.c_str());
		}

		if (haveHordeDepthMax)
		{
			info.fMaxDepth = atoi (optHordeDepthMax.c_str());
		}

		if (haveHordeDepthSwitch)
		{
			info.fSwitchDepth = atoi (optHordeDepthSwitch.c_str());
		}

		Startup::ScheduleNewHorde (info, appNames);
	}

	return true;
}


/***********************************************************************
 *
 * FUNCTION:    PrvHandleAutoLoadParameters
 *
 * DESCRIPTION: Handle the following command line options:
 *
 *					kOptLoad
 *					kOptRun
 *					kOptQuitOnExit
 *
 * PARAMETERS:  options - the OptionList containing the complete set
 *					of parsed switches and parameters.
 *
 * RETURNED:    True if everything when OK.  If there's something wrong
 *				with the specifications, this function displays an
 *				error message and return false.
 *
 ***********************************************************************/

static Bool PrvHandleAutoLoadParameters (OptionList& options)
{
	DEFINE_VARS(Load);
	DEFINE_VARS(Run);
	DEFINE_VARS(QuitOnExit);

	if (haveLoad)
	{
		::PrvParseFileList (gAutoLoadFiles1, optLoad);
	}

	if (haveRun)
	{
		gAutoRunApp = optRun;
	}

	if (haveQuitOnExit)
	{
		Startup::ScheduleQuitOnExit();
	}

	return true;
}


/***********************************************************************
 *
 * FUNCTION:    PrvParseCommandLine
 *
 * DESCRIPTION: Parse up the command line into its consituent parts,
 *				validate the parts, and act on the specifications.
 *
 * PARAMETERS:  argc, argv - standard C parameters describing the
 *					command line.
 *
 * RETURNED:    Nothing.  All actions to be taken based on the users
 *				specifications are recorded in global variables.
 *
 ***********************************************************************/

static void PrvParseCommandLine (int argc, char** argv)
{
	OptionList		options;

	// Convert the command line into a map of switch/parameter pairs.

	if (!::PrvCollectOptions(argc, argv, options))
		goto BadParameter;

	// Handle kOptPSF.

	if (!::PrvHandleOpenSessionParameters(options))
		goto BadParameter;

	// Handle kOptROM, kOptRAM, and kOptDevice.

	if (!::PrvHandleCreateSessionParameters(options))
		goto BadParameter;

	// Handle kOptHordeFirst, kOptHordeLast, kOptHordeApps, kOptHordeSaveDir,
	// kOptHordeSaveFreq, kOptHordeDepthMax, and kOptHordeDepthSwitch.

	if (!::PrvHandleNewHordeParameters(options))
		goto BadParameter;

	// Handle kOptLoad, kOptRun, and kOptQuitOnExit.

	if (!::PrvHandleAutoLoadParameters(options))
		goto BadParameter;

	// Handle kOptSkin
	// !!! TBD

	// Handle kOptLog
	// !!! TBD

	return;

BadParameter:
	// All bets are off.  Bring up the dialog with the
	// New/Open/Download/Exit buttons.

	Startup::ScheduleAskWhatToDo();
	return;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::DetermineStartupActions
 *
 * DESCRIPTION: Determine what to do at startup time, based on any
 *				command-line options, what the user was doing when they
 *				last ran Poser, and whether or not the CapsLock key
 *				is toggled.
 *
 *				In general, the startup rules are as follows:
 *
 *			 	1 If the Caps Lock key is toggled in the ON position, always bring
 *				  up the New/Open/... dialog.
 *
 *				2 Scan the command line for startup parameters.  If an error occurs
 *				  trying to scan the command line, the error is reported and the user
 *				  is presented with the New/Open/... dialog.
 *
 *				3 Use the .psf file if one is specified.  If an error occurs trying
 *				  to load the file, the error is reported and the user is presented
 *				  with the New/Open/... dialog.
 *
 *				4 If any of -rom, -ram, -device, or -silkscreen are specified, try
 *				  to start a new session based on those values.  If all are specified,
 *				  the new session is automatically created.  If any of those four
 *				  values are missing, the "New Configuration" dialog is displayed.
 *				  If the user cancels the dialog, or if there is an error creating
 *				  the new session, any error is reported and the user is presented
 *				  with the New/Open/... dialog.
 *
 *				5 If no command line options are specified, try re-opening the last
 *				  saved .psf file (this step is skipped if the user last created a
 *				  new session, but did NOT save that session to a file).  If an error
 *				  occurs trying to load the file, the error is reported and the user
 *				  is presented with the New/Open/... dialog.
 *
 *				6 Try creating a new session based on the settings the user last
 *				  specified when creating a session.  If there is an error creating
 *				  the new session, the error is reported and the user is presented
 *				  with the New/Open/... dialog.
 *
 *				7 Finally, if all else fails, present the user with the New/Open/...
 *				  dialog.
 *
 * PARAMETERS:  argc, argv - standard C parameters describing the
 *					command line.
 *
 * RETURNED:    True if we were able to determine a startup course of
 *				action.  If false is returned, something horrible happened.
 *				The caller should assume that an error was reported,
 *				and that the application should quit.
 *
 ***********************************************************************/

Bool Startup::DetermineStartupActions (int argc, char** argv)
{
	// By default, throw up our hands.

	Startup::ScheduleAskWhatToDo ();

	// See if the user wants to force the appearance of the Startup
	// screen (by holding down the ShiftLock key) and skip the auto-
	// loading of the previous session file.

	if (!Platform::ForceStartupScreen ())
	{
		Preference<Configuration>	pref1 (kPrefKeyLastConfiguration);
		Preference<FileReference>	pref2 (kPrefKeyLastPSF);

		Configuration	cfg = *pref1;
		FileReference	ramFileRef = *pref2;

		// See if there is embedded PSF resource. If so, use it.

		if (Platform::PSFResourcePresent())
		{
			// Bail if there is an embedded PSF without an embedded ROM.
			if (!Platform::ROMResourcePresent())
			{
				Platform::CommonDialog("Fatal Error",
					"An embedded PSF has been detected, but there is no "
					"corresponding embedded ROM file.  Palm OS Emulator "
					"cannot be run in this state and will quit.",
					Errors::kOK);

				return false;
			}
			
			::PrvSetEmbeddedSessionPreferences();

			Startup::ScheduleOpenSession (ramFileRef);	// Ref ignored...
		}
		
		// Else, see if there is an embedded ROM. If so, we will be creating a
		// new document based on it.

		else if (Platform::ROMResourcePresent())
		{
			// If we have an an embedded ROM without an embedded PSF, we should
			// have a resource configuration.

			if (!Platform::ConfigResourcePresent())
			{
				Platform::CommonDialog("Fatal Error",
					"An embedded ROM has been detected, but there is neither an "
					"embedded PSF now a device configuration.  Palm OS Emulator "
					"cannot be run in this state and will quit.",
					Errors::kOK);

				return false;
			}

			// Get the configuration settings from the embedded configuration.
			// If we can get them, create a new document based on them.

			if (Platform::GetEmbeddedDeviceType(cfg.fDeviceType) &&
				Platform::GetEmbeddedRAMSize(cfg.fRAMSize))
			{
				Startup::ScheduleCreateSession (cfg);	// ROM image comes from resource...
			}
		}

		// Else, see if there was a previously saved RAM file.	If so, open it.

		else if (ramFileRef.IsSpecified ())
		{
			Startup::ScheduleOpenSession (ramFileRef);
		}


		// Else, see if there was a previously created document.  If so,
		// create a new document based on its settings.

		else if (cfg.fDeviceType != kDeviceUnspecified &&
				cfg.fRAMSize != 0 &&
				cfg.fROMFile.IsSpecified () )
		{
			Startup::ScheduleCreateSession (cfg);
		}

		// Now that default actions have been established, let's see if
		// there's anything interesting on the command line -- UNLESS we have
		// an embedded session file.

		if (!Platform::PSFResourcePresent())
		{
			::PrvParseCommandLine (argc, argv);
		}
	}

	return true;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::GetAutoLoads
 *
 * DESCRIPTION: Return the list of applications that should automatically
 *				be loaded at startup time, regardless of whether a new
 *				session was created or an old one loaded.  The list of
 *				applications is collected from the command line and from
 *				the specially-named AutoLoad, AutoRun, and AutoRunAndQuit
 *				directories.
 *
 * PARAMETERS:  fileList - the FileRefList to receive the files to load.
 *					The collection is cleared out before the new items
 *					are added to it.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

static void PrvLookForAutoloads (void)
{
	gAutoLoadFiles2.clear();
	gAutoRunFiles2.clear();
	gAutoRunAndQuitFiles2.clear();

	Platform::GetLoadableFileList (Platform::GetString (kStr_Autoload), gAutoLoadFiles2);
	Platform::GetLoadableFileList (Platform::GetString (kStr_Autorun), gAutoRunFiles2);
	Platform::GetLoadableFileList (Platform::GetString (kStr_AutorunAndQuit), gAutoRunAndQuitFiles2);
}

static void PrvAppendFiles (FileRefList& list1, const FileRefList& list2)
{
	list1.insert(list1.end(), list2.begin(), list2.end());
}

void Startup::GetAutoLoads (FileRefList& fileList)
{
	fileList.clear();

	::PrvLookForAutoloads();

	::PrvAppendFiles (fileList, gAutoLoadFiles1);
//	::PrvAppendFiles (fileList, gAutoRunFiles1);
//	::PrvAppendFiles (fileList, gAutoRunAndQuitFiles1);

	::PrvAppendFiles (fileList, gAutoLoadFiles2);
	::PrvAppendFiles (fileList, gAutoRunFiles2);
	::PrvAppendFiles (fileList, gAutoRunAndQuitFiles2);

	if (/*gAutoRunAndQuitFiles1.size() +*/ gAutoRunAndQuitFiles2.size() > 0)
	{
		Startup::ScheduleQuitOnExit ();
	}
}


/***********************************************************************
 *
 * FUNCTION:    Startup::GetAutoRunApp
 *
 * DESCRIPTION: Returns the *database name* of the application to
 *				switch to at startup time.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    The name of the database of the executable.
 *
 ***********************************************************************/

static string PrvTryGetApp (const FileRefList& fileList)
{
	string	result;

	FileRefList::const_iterator	iter = fileList.begin();
	while (iter != fileList.end())
	{
		if ((*iter).IsPRC())	// egcs can't handle iter->IsPRC()!!!
		{
			try
			{
				FileHandle	openFile (*iter, kOpenExisting | kOpenRead);

				DatabaseHdrType	header;
				openFile.Read (sizeof (DatabaseHdrType), &header);
				Canonical(header);

				if (::IsExecutable (header.type, header.creator, header.attributes))
				{
					result = (char*) header.name;
					break;
				}
			}
			catch (...)
			{
			}
		}

		++iter;
	}

	return result;
}

string Startup::GetAutoRunApp (void)
{
	string	result;

	// Get the name the user specified on the command line (if any)
	// and get the DatabaseInfo for it, so that we can get the database name.

	if (!gAutoRunApp.empty())
	{
		StringList	appNames;
		appNames.push_back (gAutoRunApp);

		DatabaseInfoList	dbInfos;
		::PrvGetDatabaseInfosFromAppNames (appNames, dbInfos);

		if (dbInfos.size() > 0)
		{
			result = dbInfos.begin()->dbName;
		}
	}

	// If the user didn't specify an executable, or we couldn't find
	// that executable, then work from the files found in the AutoRun
	// and AutoRunAndQuit directories.

//	if (result.empty())
//		result = ::PrvTryGetApp (gAutoRunFiles1);

	if (result.empty())
		result = ::PrvTryGetApp (gAutoRunFiles2);

//	if (result.empty())
//		result = ::PrvTryGetApp (gAutoRunAndQuitFiles1);

	if (result.empty())
		result = ::PrvTryGetApp (gAutoRunAndQuitFiles2);

	return result;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::AskWhatToDo
 *
 * DESCRIPTION: Return whether or not we are supposed to ask the user
 *				what to do (that is, display the dialog box with the
 *				New, Open, Download, and Quit buttons).  In general,
 *				this function returns true if there's nothing else
 *				scheduled to be done (create a session, open a session,
 *				or quit the emulator).
 *
 * PARAMETERS:  None
 *
 * RETURNED:    True if so.
 *
 ***********************************************************************/

Bool Startup::AskWhatToDo (void)
{
//	Bool	result = gAskWhatToDo;
//	gAskWhatToDo = false;
//	return result;

	return !gCreateSession && !gOpenSession && !gQuit && !gClose;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::CreateSession
 *
 * DESCRIPTION: Return whether or not we are supposed to create a new
 *				session based on previously determined criteria.  This
 *				is a one-shot function: subsequent calls will return
 *				false until ScheduleCreateSession is called again.
 *
 * PARAMETERS:  cfg - the Configuration to receive the information to
 *					be used when creating the new session.
 *
 * RETURNED:    True if so.
 *
 ***********************************************************************/

Bool Startup::CreateSession (Configuration& cfg)
{
	Bool	result = gCreateSession;
	if (result)
		cfg = gCfg;
	gCreateSession = false;
	return result;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::OpenSession
 *
 * DESCRIPTION: Return whether or not we are supposed to open an old
 *				session based on previously determined criteria.  This
 *				is a one-shot function: subsequent calls will return
 *				false until ScheduleOpenSession is called again.
 *
 * PARAMETERS:  ref - the FileReference to receive the reference to
 *					the .psf file to be opened.
 *
 * RETURNED:    True if so.
 *
 ***********************************************************************/

Bool Startup::OpenSession (FileReference& ref)
{
	Bool	result = gOpenSession;
	if (result)
		ref = gSessionOpenRef;
	gOpenSession = false;
	return result;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::NewHorde
 *
 * DESCRIPTION: Return whether or not we are supposed to start a horde
 *				session based on previously determined criteria.  This
 *				is a one-shot function: subsequent calls will return
 *				false until ScheduleNewHorde is called again.
 *
 * PARAMETERS:  info - the HordeInfo to receive the information used to
 *					create the new horde.
 *
 * RETURNED:    True if so.
 *
 ***********************************************************************/

Bool Startup::NewHorde (HordeInfo* info)
{
	Bool	result = gStartNewHorde;

	if (info != NULL)
	{
		if (result)
		{
			*info = gHorde;

			// Find the AppInfos for the user-specified applications.

			::PrvGetDatabaseInfosFromAppNames (gHordeApps, info->fAppList);
		}

		gStartNewHorde = false;
	}

	return result;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::CloseSession
 *
 * DESCRIPTION: Return whether or not Poser is supposed to close the
 *				current session right now.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    True if so.
 *
 ***********************************************************************/

Bool Startup::CloseSession (FileReference& f)
{
	Bool	result = gClose;
	f = gSessionCloseRef;
	gClose = false;
	return result;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::QuitOnExit
 *
 * DESCRIPTION: Return whether or not Poser is supposed to quit when
 *				a particular application exits.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    True if so.
 *
 ***********************************************************************/

Bool Startup::QuitOnExit (void)
{
	Bool	result = gQuitOnExit;
//	gQuitOnExit = false;
	return result;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::Quit
 *
 * DESCRIPTION: Return whether or not it's time to quit.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    True if so.
 *
 ***********************************************************************/

Bool Startup::Quit (void)
{
	Bool	result = gQuit;
	gQuit = false;
	return result;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::Clear
 *
 * DESCRIPTION: Clear all settings.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::Clear (void)
{
	gCreateSession = false;
	gOpenSession = false;
	gStartNewHorde = false;
	gQuitOnExit = false;
	gQuit = false;

	gAutoRunApp = "";
	gAutoLoadFiles1.clear();
	gAutoLoadFiles2.clear();
	gAutoRunFiles2.clear();
	gAutoRunAndQuitFiles2.clear();
}


/***********************************************************************
 *
 * FUNCTION:    Startup::ScheduleAskWhatToDo
 *
 * DESCRIPTION: Schedule our "state machine" so that AskWhatToDo will
 *				return True.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::ScheduleAskWhatToDo (void)
{
	Startup::Clear ();
//	gAskWhatToDo = true;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::ScheduleCreateSession
 *
 * DESCRIPTION: Schedule our "state machine" so that CreateSession will
 *				return True.
 *
 * PARAMETERS:  cfg - the Configuration returned to the caller of
 *					CreatSession.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::ScheduleCreateSession (const Configuration& cfg)
{
	gCreateSession = true;
//	gAskWhatToDo = false;
	gCfg = cfg;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::ScheduleOpenSession
 *
 * DESCRIPTION: Schedule our "state machine" so that OpenSession will
 *				return True.
 *
 * PARAMETERS:  ref - the FileReference returned to the caller of
 *					OpenSession.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::ScheduleOpenSession (const FileReference& ref)
{
	gOpenSession = true;
//	gAskWhatToDo = false;
	gSessionOpenRef = ref;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::ScheduleNewHorde
 *
 * DESCRIPTION: Schedule our "state machine" so that NewHorde will
 *				return True.
 *
 * PARAMETERS:  info - the HordeInfo returned to the caller of NewHorde.
 *
 *				appNames - names of the applications to inflict the
 *					Gremlins on.  Normally, this information is
 *					specified in the fAppList field of the HordeInfo.
 *					However, at the time this function is called,
 *					we may not be able to generate the DatabaseInfoList
 *					(the emulator may be just starting up and not
 *					emulating a Palm OS environment, yet).  So we
 *					remember the names here.  When NewHorde is called,
 *					the appNames list is converted into an DatabaseInfoList.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::ScheduleNewHorde (const HordeInfo& info, const StringList& appNames)
{
	gStartNewHorde = true;
	gHorde = info;
	gHordeApps = appNames;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::ScheduleCloseSession
 *
 * DESCRIPTION: Schedule our "state machine" so that CloseSession will
 *				return True.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::ScheduleCloseSession (const FileReference& f)
{
	gSessionCloseRef = f;
	gClose = true;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::ScheduleQuitOnExit
 *
 * DESCRIPTION: Schedule our "state machine" so that QuitOnExit will
 *				return True.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::ScheduleQuitOnExit (void)
{
	gQuitOnExit = true;
}


/***********************************************************************
 *
 * FUNCTION:    Startup::ScheduleQuit
 *
 * DESCRIPTION: Schedule our "state machine" so that Quit will
 *				return True.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void Startup::ScheduleQuit (void)
{
//	Startup::Clear();
	gQuit = true;
//	gAskWhatToDo = false;
}


