/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "ErrorHandling.h"

#include "CPU_REG.h"			// SysTrapIndex
#include "Hordes.h"				// Hordes::IsOn
#include "Logging.h"			// LogWrite
#include "MetaMemory.h"			// MetaMemory
#include "Miscellaneous.h"		// LaunchCmdToString, SystemCallContext
#include "Platform.h"			// GetString, CommonDialog
#include "PreferenceMgr.h"		// Preference
#include "Strings.r.h"			// kStr_ values
#include "TrapPatches.h"		// Patches::GetCurrentAppInfo


// ===========================================================================
//		 Errors
// ===========================================================================

static ParamList		gUserParameters;
static unsigned long	gWarningFlags;


// ---------------------------------------------------------------------------
//		 Errors::Initialize
// ---------------------------------------------------------------------------

void Errors::Initialize (void)
{
}


// ---------------------------------------------------------------------------
//		 Errors::Reset
// ---------------------------------------------------------------------------

void Errors::Reset (void)
{
	ClearWarningFlags ();
}


// ---------------------------------------------------------------------------
//		 Errors::Save
// ---------------------------------------------------------------------------

void Errors::Save (SessionFile&)
{
}


// ---------------------------------------------------------------------------
//		 Errors::Load
// ---------------------------------------------------------------------------

void Errors::Load (SessionFile&)
{
}


// ---------------------------------------------------------------------------
//		 Errors::Dispose
// ---------------------------------------------------------------------------

void Errors::Dispose (void)
{
}


// ---------------------------------------------------------------------------
//		 Errors::ReportIfError
// ---------------------------------------------------------------------------
// Checks for the indicated error condition.  If there is an error, it
// displays a "Could not foo because bar." message in a dialog with an OK
// button.  If an optional recovery string is provided, the message is
// "Could no foo because bar.  Do this." message.  If "throwAfter" is true,
// Errors::Scram is called after the dialog is dismissed.
//
// "operation" and "recovery" are "kStr_Foo" values.
// "error" is Mac or Windows error number, or a kError_Foo value.

void Errors::ReportIfError (StrCode operation, ErrCode error, StrCode recovery, Bool throwAfter)
{
	if (error)
	{
		Errors::SetStandardParameters ();

		if (Errors::SetErrorParameters (operation, error, recovery))
		{
			Errors::DoDialog (kStr_OpErrorRecover, kOK);
		}
		else
		{
			Errors::DoDialog (kStr_OpError, kOK);
		}

		if (throwAfter)
		{
			Errors::Scram ();
		}
	}
}


// ---------------------------------------------------------------------------
//		 Errors::ReportIfPalmError
// ---------------------------------------------------------------------------
// Checks for the indicated error condition.  If there is an error, it
// displays a "Could not foo because bar." message in a dialog with an OK
// button.  If an optional recovery string is provided, the message is
// "Could no foo because bar.  Do this." message.  If "throwAfter" is true,
// Errors::Scram is called after the dialog is dismissed.
//
// "operation" and "recovery" are "kStr_Foo" values.
// "err" is a Palm OS error number.

void Errors::ReportIfPalmError (StrCode operation, Err err, StrCode recovery, Bool throwAfter)
{
	if (err)
	{
		Errors::ReportIfError (operation, ::ConvertFromPalmError (err), recovery, throwAfter);
	}
}


// ---------------------------------------------------------------------------
//		 Errors::ReportIfNULL
// ---------------------------------------------------------------------------
// Checks for the indicated error condition.  If there is an error, it
// displays a "Could not foo because bar." message in a dialog with an OK
// button.  If an optional recovery string is provided, the message is
// "Could no foo because bar.  Do this." message.  If "throwAfter" is true,
// Errors::Scram is called after the dialog is dismissed.
//
// "operation" and "recovery" are "kStr_Foo" values.
// "p" is a pointer to be tested.

void Errors::ReportIfNULL (StrCode operation, void* p, StrCode recovery, Bool throwAfter)
{
	if (p == NULL)
	{
		Errors::ReportIfError (operation, kError_OutOfMemory, recovery, throwAfter);
	}
}


// ---------------------------------------------------------------------------
//		 Errors::ReportPreventedAccess
// ---------------------------------------------------------------------------
// Some process just accessed some memory that it shouldn't have. "kind"
// holds a value indicating what part of memory was accessed (low memory,
// system globals, screen buffer, or hardware registers), and "forRead"
// tells if whether it was a read or write operation.
//
// !!! Eventually, we'll probably want to include the memory address so that
// we can report it.

int Errors::ReportPreventedAccess (uaecptr address, long size, Bool forRead,
							  EAccessType kind)
{
	UNUSED_PARAM(size)

	// See if we've already issued this warning for this application.
	// If so, just leave.

	if (AlreadyWarned (kind))
	{
		return kContinue;
	}

	// Generate the warning message.

	// Get the resource ID for the right error message template.

	const char*	prefKey = NULL;
	int	strIndex;

	switch (kind)
	{
		case kRegisterAccess:
			prefKey = kPrefKeyReportHardwareRegisterAccess;
			strIndex = kStr_RegisterAccess;
			break;

		case kLowMemAccess:
			prefKey = kPrefKeyReportLowMemoryAccess;
			if (address == UAE_NULL)
				strIndex = kStr_NULLMemAccess;
			else
				strIndex = kStr_LowMemAccess;
			break;

		case kGlobalVarAccess:
			prefKey = kPrefKeyReportSystemGlobalAccess;
			strIndex = kStr_GlobalsAccess;
			break;

		case kScreenAccess:
			prefKey = kPrefKeyReportScreenAccess;
			strIndex = kStr_ScreenAccess;
			break;

		case kMemMgrAccess:
			prefKey = kPrefKeyReportMemMgrDataAccess;
			strIndex = kStr_MemMgrAccess;
			break;

		case kUnlockedChunkAccess:
			prefKey = kPrefKeyReportUnlockedChunkAccess;
			strIndex = kStr_UnlockedChunkAccess;
			break;

		case kLowStackAccess:
			prefKey = kPrefKeyReportLowStackAccess;
			strIndex = kStr_LowStackAccess;
			break;

		case kUninitializedStackAccess:
			prefKey = kPrefKeyReportUninitializedStackAccess;
			strIndex = kStr_UninitializedStackAccess;
			break;

		case kFreeChunkAccess:
			prefKey = kPrefKeyReportFreeChunkAccess;
			strIndex = kStr_FreeChunkAccess;
			break;

		case kUninitializedChunkAccess:
			prefKey = kPrefKeyReportUninitializedChunkAccess;
			strIndex = kStr_UninitializedChunkAccess;
			break;

		case kStorageHeapAccess:
			prefKey = kPrefKeyReportStorageHeapAccess;
			strIndex = kStr_StorageHeapAccess;
			break;

		default: assert (false); break;
	}

	Preference<bool>	pref (prefKey);
	if (!*pref)
	{
		return kContinue;
	}

	// Get the application name and the operation text that need to
	// be inserted into the template.

	Errors::SetStandardParameters ();
	Errors::SetParameter ("%operation", Platform::GetString (forRead ? kStr_ReadFrom : kStr_WrittenTo));

	// Display the message with Continue and Debug buttons.

	int	button = Errors::DoDialog (strIndex, Errors::kContinueDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportUnimplementedTrap
// ---------------------------------------------------------------------------

int Errors::ReportUnimplementedTrap (uae_u16 trapWord, int resourceBase)
{
	if (!*(Preference<bool>(kPrefKeyReportUnimplementedTrap)))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	char	trapNumber[20];
	sprintf (trapNumber, "%04X", (int) trapWord);

	string	trapName (Platform::GetString (resourceBase + (trapWord & 0x7FF)));
	if (trapName[0] == '<')	// Start of "<Missing string...>"
	{
		trapName = Platform::GetString (kStr_UnknownTrapNumber);
	}

	Errors::SetParameter ("%trapNum", trapNumber);
	Errors::SetParameter ("%trapName", trapName);

	int	button = Errors::DoDialog (kStr_UnimplementedTrap, kDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportUnhandledException
// ---------------------------------------------------------------------------

int Errors::ReportUnhandledException (uae_s32 exceptionNumber)
{
	if (!*(Preference<bool>(kPrefKeyReportUnhandledException)))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();
	Errors::SetParameter ("%operation", Platform::GetString (kStr_ExceptionBase + exceptionNumber));

	int	button;
	if (exceptionNumber == kException_Trap0 + sysDbgTrapNum)
	{
		button = Errors::DoDialog (kStr_DebugBreak, kContinueDebugReset);
	}
	else
	{
		button = Errors::DoDialog (kStr_UnhandledException, kDebugReset);
	}

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportSANEUsage
// ---------------------------------------------------------------------------

int Errors::ReportSANEUsage ()
{
	if (!*(Preference<bool>(kPrefKeyReportUnhandledException)))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	int	button = Errors::DoDialog (kStr_SANEUsage, kDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportCorruptedHeap
// ---------------------------------------------------------------------------

int Errors::ReportCorruptedHeap (int corruptionType, uaecptr chunkHdr)
{
	if (!*(Preference<bool>(kPrefKeyReportCorruptedHeap)))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	int	strID;

	switch ((PalmHeap::EPalmHeapResult) corruptionType)
	{
		case PalmHeap::kCorruptedHeap_ChunkNotInHeap:
			strID = kStr_ChunkNotInHeap;
			break;

		case PalmHeap::kCorruptedHeap_ChunkTooLarge:
			strID = kStr_ChunkTooLarge;
			break;

		case PalmHeap::kCorruptedHeap_InvalidFlags:
			strID = kStr_InvalidFlags;
			break;

		case PalmHeap::kCorruptedHeap_HOffsetNotInMPT:
			strID = kStr_HOffsetNotInMPT;
			break;

		case PalmHeap::kCorruptedHeap_HOffsetNotBackPointing:
			strID = kStr_HOffsetNotBackPointing;
			break;

		case PalmHeap::kCorruptedHeap_InvalidLockCount:
			strID = kStr_InvalidLockCount;
			break;

		default:
			assert (false);
	}

	Errors::SetParameter ("%corruption_type", Platform::GetString (strID));

	char	address[20];
	sprintf (address, "0x%08lX", (long) chunkHdr);
	Errors::SetParameter ("%header_address", address);

	int	button = Errors::DoDialog (kStr_CorruptedHeap, kDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportStackOverflow
// ---------------------------------------------------------------------------

int Errors::ReportStackOverflow (Bool fatal)
{
	if (fatal)
	{
		if (!*(Preference<bool>(kPrefKeyReportStackOverflow)))
		{
			return kContinue;
		}
	}
	else
	{
		if (!*(Preference<bool>(kPrefKeyReportStackAlmostOverflow)))
		{
			return kContinue;
		}
	}

	// If this is just a warning, warn only once.

	if (!fatal && AlreadyWarned (kStackOverflowWarning))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	int	strID = fatal ? kStr_StackOverflowFatal : kStr_StackOverflowWarning;
	int	dlgID = fatal ? kDebugReset : kContinueDebugReset;

	int	button = Errors::DoDialog (strID, dlgID);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportLongMemSemaphore
// ---------------------------------------------------------------------------

int Errors::ReportLongMemSemaphore (unsigned long elapsedMSecs)
{
	if (!*(Preference<bool>(kPrefKeyReportMemMgrSemaphore)))
	{
		return kContinue;
	}

	if (AlreadyWarned (kLongMemSemaphore))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	char	elapsed[20];
	sprintf (elapsed, "%ld", elapsedMSecs);
	Errors::SetParameter ("%elapsed", elapsed);

	int	button = Errors::DoDialog (kStr_LongMemSemaphore, kContinueDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportSizelessFormObject
// ---------------------------------------------------------------------------

int Errors::ReportSizelessFormObject (Word id, const RectangleType& r)
{
//	if (!gPrefs->ReportMemMgrSemaphore ())
//	{
//		return kContinue;
//	}

	if (AlreadyWarned (kSizelessFormObject))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	char	temp[20];

	sprintf (temp, "%ld", (long) id);
	Errors::SetParameter ("%objID", temp);

	sprintf (temp, "%ld", (long) r.topLeft.x);
	Errors::SetParameter ("%left", temp);

	sprintf (temp, "%ld", (long) r.topLeft.y);
	Errors::SetParameter ("%top", temp);

	sprintf (temp, "%ld", (long) (r.topLeft.x + r.extent.x));
	Errors::SetParameter ("%right", temp);

	sprintf (temp, "%ld", (long) (r.topLeft.y + r.extent.y));
	Errors::SetParameter ("%bottom", temp);

	int	button = Errors::DoDialog (kStr_SizelessFormObject, kContinue);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportOffscreenFormObject
// ---------------------------------------------------------------------------

int Errors::ReportOffscreenFormObject (Word id, const RectangleType& r)
{
//	if (!gPrefs->ReportMemMgrSemaphore ())
//	{
//		return kContinue;
//	}

	if (AlreadyWarned (kOffscreenFormObject))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	char	temp[20];

	sprintf (temp, "%ld", (long) id);
	Errors::SetParameter ("%objID", temp);

	sprintf (temp, "%ld", (long) r.topLeft.x);
	Errors::SetParameter ("%left", temp);

	sprintf (temp, "%ld", (long) r.topLeft.y);
	Errors::SetParameter ("%top", temp);

	sprintf (temp, "%ld", (long) (r.topLeft.x + r.extent.x));
	Errors::SetParameter ("%right", temp);

	sprintf (temp, "%ld", (long) (r.topLeft.y + r.extent.y));
	Errors::SetParameter ("%bottom", temp);

	int	button = Errors::DoDialog (kStr_OffscreenFormObject, kContinue);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportA5Access
// ---------------------------------------------------------------------------

int Errors::ReportA5Access (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)

	Errors::SetStandardParameters ();
	Errors::SetParameter ("%operation", Platform::GetString (forRead ? kStr_ReadFrom : kStr_WrittenTo));

	EmuAppInfo	appInfo = Patches::GetCurrentAppInfo ();
	const char*	launchStr = ::LaunchCmdToString (appInfo.fCmd);

	Errors::SetParameter ("%launchCode", launchStr);

	int	button = Errors::DoDialog (kStr_A5Access, Errors::kDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportInvalidSystemCall
// ---------------------------------------------------------------------------

int Errors::ReportInvalidSystemCall (const SystemCallContext& context)
{
	uae_u32	errorBase;

	if (IsSystemTrap (context.fTrapWord))
	{
		errorBase = kStr_SysTrapBase;
	}
	else
	{
		errorBase = gLibErrorBase;	// !!! Hack.
	}

	int button = Errors::ReportUnimplementedTrap (context.fTrapWord, errorBase);

	// (Exception is thrown if button == kDebug or kReset).

	Emulator::HandleDlgButton (button, context.fPC);

	return button;	// Never reached.  HandleDlgButton will throw
					// an exception if the user selects Debug or
					// Reset, the only two buttons we offer them.
}


// ---------------------------------------------------------------------------
//		 Errors::ReportWroteToROM
// ---------------------------------------------------------------------------

int Errors::ReportWroteToROM (uaecptr address, long size, Bool forRead)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Errors::SetStandardParameters ();

	int	button = Errors::DoDialog (kStr_WroteToROM, Errors::kContinueDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::ReportInvalidPC
// ---------------------------------------------------------------------------

int Errors::ReportInvalidPC (uaecptr address, int reason)
{
	Errors::SetStandardParameters ();

	char	buffer[20];
	sprintf (buffer, "0x%08X", address);
	Errors::SetParameter ("%address", buffer);

	int	strID;

	switch (reason)
	{
		case kUnmappedAddress:
			strID = kStr_UnmappedAddress;
			break;

		case kNotInCodeSegment:
			strID = kStr_NotInCodeSegment;
			break;

		case kOddAddress:
			strID = kStr_OddAddress;
			break;

		default:
			assert (false);
	}

	Errors::SetParameter ("%reason", Platform::GetString (strID));

	int	button = Errors::DoDialog (kStr_InvalidPC, Errors::kDebugReset);

	// (Exception is thrown if button == kDebug or kReset).

	Emulator::HandleDlgButton (button, m68k_getpc());

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors:SysFatalAlert
// ---------------------------------------------------------------------------

int Errors::SysFatalAlert (const char* appMsg)
{
	if (!*(Preference<bool>(kPrefKeyReportSysFatalAlert)))
	{
		return kContinue;
	}

	Errors::SetStandardParameters ();

	// Format the entire message.  We use different messages depending on
	// whether or not Gremlins is running.

#if 0
	if (Hordes::IsOn ())
	{
		msg	= Platform::GetString (kStringErrDisplayWithGremlins);

		assert (msg != NULL);

		msg = Errors::ReplaceParameter (msg, "%message", appMsg);
		msg = Errors::ReplaceParameter (msg, "#gremlin_number", Hordes::GremlinNumber ());
		msg = Errors::ReplaceParameter (msg, "#gremlin_event", Hordes::EventNumber ());
	}
	else
#endif
	{
		Errors::SetParameter ("%message", appMsg);
	}

	int	button = Errors::DoDialog (kStr_SysError, Errors::kContinueDebugReset);

	return button;
}


// ---------------------------------------------------------------------------
//		 Errors::SetParameter
// ---------------------------------------------------------------------------
//	Add a user-supplied substitution rule. Parameters are deleted after they
//	are used (by making a call to ReportMessage), so they need to be
//	re-established after every call to that function.

void Errors::SetParameter (const string& key, const string& value)
{
	gUserParameters.push_back (key);
	gUserParameters.push_back (value);
}


// ---------------------------------------------------------------------------
//		 Errors::SetParameter
// ---------------------------------------------------------------------------
//	Add a user-supplied substitution rule. Parameters are deleted after they
//	are used (by making a call to ReportMessage), so they need to be
//	re-established after every call to that function.
//
//	This version of the function is a convenience for use on the Mac side,
//	where Pascal strings are used a lot.  (Actually, a string constructor
//	that took Pascal strings would be nice...).

void Errors::SetParameter (const string& key, const unsigned char* value)
{
	gUserParameters.push_back (key);
	gUserParameters.push_back (string ((char*) &value[1], value[0]));
}


// ---------------------------------------------------------------------------
//		 Errors::SetErrorParameters
// ---------------------------------------------------------------------------

Bool Errors::SetErrorParameters (StrCode operation, ErrCode error, StrCode recovery)
{
	// Add %operation, %reason, and %recovery entries to our parameter
	// list in such a way that they appear at the start of the list
	// in that order. That way, if any of the replacement strings
	// contain parameters to be replaced, they can be replaced by
	// user-defined values.

	string s;

	// Add the error number as a possible substitution value.

	ErrCode	errCode = error;
	if (::IsPalmError (errCode))
	{
		errCode = ::ConvertToPalmError (errCode);

		// For Palm OS errors, try coming up with a description of the error.

		s = Platform::GetString (kStr_PalmOSErrorBase + errCode);

		if (s[0] == '<')
		{
			s = Platform::GetString (kStr_UnknownErrorCode);
		}

		Errors::SetParameter ("%error_desc", s);
	}

	char	errCodeString[20];
	sprintf (errCodeString, "0x%04X", (int) errCode);
	Errors::SetParameter ("%error", errCodeString);

	// If the caller didn't provide a recovery string ID, let's try to get
	// appropriate for this error code.

	if (recovery == 0)
	{
		recovery = Errors::GetIDForRecovery (error);
	}

	// If we (now) have a recovery string ID, use it to get the string
	// and add it to our parameter list.

	if (recovery)
	{
		s = Platform::GetString (recovery);
		gUserParameters.push_front (s);
		gUserParameters.push_front ("%recovery");
	}

	// Get a string for the error code provided.  If we don't have a canned
	// string, create a generic string that includes the error number.

	StrCode	errID = Errors::GetIDForError (error);
	s = Platform::GetString (errID);
	gUserParameters.push_front (s);
	gUserParameters.push_front ("%reason");

	// Finally, add the operation string to the parameter list.

	s = Platform::GetString (operation);
	gUserParameters.push_front (s);
	gUserParameters.push_front ("%operation");

	return recovery != 0;	// Return whether or not there's a recovery string in
							// the parameter list.  The caller will want to know
							// this so that it can provide the right template for it.
}


// ---------------------------------------------------------------------------
//		 Errors::SetStandardParameters
// ---------------------------------------------------------------------------
//	Set the following three parameters for text substitution:
//
//		%Application = current application name, or "The current application"
//		%application = current application name, or "the current application"
//		%version = current application version, or "(unknown version)"

void Errors::SetStandardParameters (void)
{
	string	appNameUC;
	string	appNameLC;
	string	version;

	Errors::GetAppName (appNameUC, appNameLC);
	Errors::GetAppVersion (version);

	Errors::SetParameter ("%Application", appNameUC);
	Errors::SetParameter ("%application", appNameLC);
	Errors::SetParameter ("%version", version);
}


// ---------------------------------------------------------------------------
//		 Errors::DoDialog
// ---------------------------------------------------------------------------
//	Displays a dialog box with the given message and according to
//	the given flags.  Returns which button was clicked.

int Errors::DoDialog (StrCode messageID, int flags)
{
	string	msg (Platform::GetString (messageID));
	return Errors::DoDialog (NULL, msg.c_str (), flags);
}


// ---------------------------------------------------------------------------
//		 Errors::DoDialog
// ---------------------------------------------------------------------------
//	Displays a dialog box with the given message and according to
//	the given flags.  Returns which button was clicked.

int Errors::DoDialog (const char* title, const char* msg, int flags)
{
	string	titleStr;

	if (title)
		titleStr = title;

	else
		titleStr = Platform::GetString (kStr_AppName);

	string	msgStr (msg);
	msgStr = Errors::ReplaceParameters (msgStr, gUserParameters);

	if (Hordes::DoDialog (msgStr))
	{
		return kNextGremlin;
	}

	LogDump ();

	return Platform::CommonDialog (titleStr.c_str (), msgStr.c_str (), flags);
}


// ---------------------------------------------------------------------------
//		 Errors::ReplaceParameters
// ---------------------------------------------------------------------------
//	Take a string template (that is, a string containing text interspersed
//	with "parameters" (of the form "%parameterName") that need to be
//	replaced) and replace the parameters with their final values.

string Errors::ReplaceParameters (StrCode templateID)
{
	return Errors::ReplaceParameters (templateID, gUserParameters);
}


// ---------------------------------------------------------------------------
//		 Errors::ReplaceParameters
// ---------------------------------------------------------------------------
//	Take a string template (that is, a string containing text interspersed
//	with "parameters" (of the form "%parameterName") that need to be
//	replaced) and replace the parameters with their final values.

string Errors::ReplaceParameters (StrCode templateID, ParamList& params)
{
	string	templ = Platform::GetString (templateID);
	return Errors::ReplaceParameters (templ, params);
}


// ---------------------------------------------------------------------------
//		 Errors::ReplaceParameters
// ---------------------------------------------------------------------------
//	Take a string template (that is, a string containing text interspersed
//	with "parameters" (of the form "%parameterName") that need to be
//	replaced) and replace the parameters with their final values.
//
//	Parameters are replaced one at a time, according to their position in
//	the ParamList.  The first parameter is first fully replace in the
//	string template, in such a manner than recursion does not occur (that
//	is, if a parameter value itself include a parameter with the same
//	name, that embedded parameter is not also replace).
//
//	The next parameter is then pulled off of ParamList and treate the
//	same way. This time, any new parameters introduced during the previous
//	substitution pass can be replaced.
//
//	This approach allows for sequences like the following:
//
//	string template "Could not %operation because %reason."
//
//	parameter list:		"%operation"	"save the file %extra"
//						"%reason"		"the disk is full"
//						"%extra"		"FooBlitzky"
//
//	result: "Could not save the file FooBlitzky" because the disk is full."
//
//	If the "%extra" in "save the file %extra" had been "%operation", it
//	would have stayed that way and never been replaced (unless "%operation"
//	appeared in the parameter list again).

string Errors::ReplaceParameters (const string& templ, ParamList& params)
{
	string	result (templ);

	ParamList::iterator	paramIter;

	for (paramIter = params.begin (); paramIter != params.end (); )
	{
		string	key		= *paramIter++;
		string	value	= *paramIter++;

		string::size_type	pos = 0;

		for (;;)
		{
			pos = result.find (key, pos);
			if (pos == string::npos)
				break;
			result.replace (pos, key.size (), value);
			pos += value.size ();
		}
	}

	gUserParameters.clear ();

	return result;
}


// ---------------------------------------------------------------------------
//		 Errors::Throw
// ---------------------------------------------------------------------------

#ifndef Throw_
#define Throw_(x) throw x
#endif

void Errors::Throw (ErrCode error)
{
	Throw_ (error);
}


// ---------------------------------------------------------------------------
//		 Errors::ThrowIfError
// ---------------------------------------------------------------------------

void Errors::ThrowIfError (ErrCode error)
{
	if (error)
	{
		Errors::Throw (error);
	}
}


// ---------------------------------------------------------------------------
//		 Errors::ThrowIfPalmError
// ---------------------------------------------------------------------------

void Errors::ThrowIfPalmError (Err error)
{
	if (error)
	{
		Errors::Throw (::ConvertFromPalmError (error));
	}
}


// ---------------------------------------------------------------------------
//		 Errors::ThrowIfNULL
// ---------------------------------------------------------------------------

void Errors::ThrowIfNULL (void* p)
{
	if (!p)
	{
		Errors::Throw (kError_OutOfMemory);
	}
}


// ---------------------------------------------------------------------------
//		 Errors::Scram
// ---------------------------------------------------------------------------

void Errors::Scram (void)
{
	Errors::Throw (kError_NoError);
}


// ---------------------------------------------------------------------------
//		 Errors::ClearWarningFlags
// ---------------------------------------------------------------------------

void Errors::ClearWarningFlags (void)
{
	gWarningFlags = 0;
}


// ---------------------------------------------------------------------------
//		 Errors::GetIDForError
// ---------------------------------------------------------------------------

int Errors::GetIDForError (ErrCode error)
{
	if (::IsPalmError (error))
	{
		switch (::ConvertToPalmError (error))
		{
			case dmErrDatabaseOpen:			return kStr_DmErrDatabaseOpen;
			case memErrNotEnoughSpace:		return kStr_MemErrNotEnoughSpace;
			default:						return kStr_GenericPalmError;
		}
	}
	else if (::IsEmuError (error))
	{
		switch (error)
		{
			case kError_OutOfMemory:		return kStr_MemFull;
			case kError_BadROM:				return kStr_BadROM;
			case kError_WrongROMForType:	return kStr_WrongROMForType;
			case kError_UnsupportedROM:		return kStr_UnsupportedROM;
			case kError_CantDownloadROM_BadBaudRate:	return kStr_GenericError;
			case kError_CantDownloadROM_SerialPortBusy:	return kStr_GenericError;
			case kError_CantDownloadROM_Generic:		return kStr_GenericError;
			case kError_UnimplementedTrap:	return kStr_UnimplementedTrap;
			case kError_OnlySameType:		return kStr_OnlySameType;
			case kError_OnlyOnePSF:			return kStr_OnlyOnePSF;
			case kError_OnlyOneROM:			return kStr_OnlyOneROM;
			case kError_UnknownType:		return kStr_UnknownType;
			default:						return kStr_GenericError;
		}
	}

	return Platform::GetIDForError (error);
}


// ---------------------------------------------------------------------------
//		 Errors::GetIDForRecovery
// ---------------------------------------------------------------------------

int Errors::GetIDForRecovery (ErrCode error)
{
	if (::IsPalmError (error))
	{
		switch (::ConvertToPalmError (error))
		{
			default:
				return 0;
		}
	}
	else if (::IsEmuError (error))
	{
		switch (error)
		{
			default:
				return 0;
		}
	}

	return Platform::GetIDForRecovery (error);
}


// ---------------------------------------------------------------------------
//		 Errors::GetAppName
// ---------------------------------------------------------------------------
// Get the current application's name.  If the name is not known, return
// "Unknown application" and "unknown application".

void Errors::GetAppName (string& appNameUC, string& appNameLC)
{
	EmuAppInfo	appInfo = Patches::GetCurrentAppInfo ();

	if (strlen (appInfo.fName) > 0)
	{
		appNameUC = appInfo.fName;
		appNameLC = appInfo.fName;
	}
	else
	{
		appNameUC = Platform::GetString (kStr_CurrentAppUC);
		appNameLC = Platform::GetString (kStr_CurrentAppLC);
	}
}


// ---------------------------------------------------------------------------
//		 Errors::GetAppVersion
// ---------------------------------------------------------------------------
// Get the current application's version.  If the version cannot be
// determined, return "(unknown version)".

void Errors::GetAppVersion (string& appVersion)
{
	EmuAppInfo	appInfo = Patches::GetCurrentAppInfo ();

	if (strlen (appInfo.fVersion) > 0)
	{
		appVersion = appInfo.fVersion;
	}
	else
	{
		appVersion = Platform::GetString (kStr_UnknownVersion);
	}
}


// ---------------------------------------------------------------------------
//		 Errors::AlreadyWarned
// ---------------------------------------------------------------------------

Bool Errors::AlreadyWarned (EAccessType kind)
{
	// Always warn if Gremlins is on.  The point of warning only once is
	// to reduce the annoyance to the person trying to interactively use
	// a slightly bent program.  But when running Gremlins, we aren't
	// annoying the user by showing dialog boxes, so always warn.

	if (Hordes::IsOn ())
	{
		return false;
	}

	long	kindBit = 1 << ((long) kind);

	// See if we've already issued this warning for this application.
	// If so, just leave.

	if ((gWarningFlags & kindBit) != 0)
	{
		return true;
	}

	// Remember that we've issued a warning for this application.
	// Remember only certain kinds of warnings.  All the other kinds
	// are considered serious enough to warn EVERY time.

	const unsigned long	kMask = (1 << (long) kRegisterAccess)		|
								(1 << (long) kLowMemAccess)			|
								(1 << (long) kGlobalVarAccess)		|
								(1 << (long) kScreenAccess)			|
								(1 << (long) kStorageHeapAccess)	|
								(1 << (long) kStackOverflowWarning)	|
								(1 << (long) kSizelessFormObject)	|
								(1 << (long) kOffscreenFormObject)	|
								0;

	gWarningFlags |= kMask & kindBit;

	return false;
}
