/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Byteswapping.h"

#if BYTESWAP

#include "EmRPC.h"				// sysPktRPC2Cmd

#if defined (_MSC_VER) || defined (__GNUC__)

unsigned long	flipBits[] =
{
	/* 0000 -> 0000 */	0x00,
	/* 0001 -> 1000 */	0x08,
	/* 0010 -> 0100 */	0x04,
	/* 0011 -> 1100 */	0x0C,
	/* 0100 -> 0010 */	0x02,
	/* 0101 -> 1010 */	0x0A,
	/* 0110 -> 0110 */	0x06,
	/* 0111 -> 1110 */	0x0E,
	/* 1000 -> 0001 */	0x01,
	/* 1001 -> 1001 */	0x09,
	/* 1010 -> 0101 */	0x05,
	/* 1011 -> 1101 */	0x0D,
	/* 1100 -> 0011 */	0x03,
	/* 1101 -> 1011 */	0x0B,
	/* 1110 -> 0111 */	0x07,
	/* 1111 -> 1111 */	0x0F
};

#endif

void Byteswap (BreakpointType& bp)
{
	Byteswap (bp.addr);

	Byteswap (bp.enabled);
	Byteswap (bp.installed);
}


void Byteswap (CardHeaderType& cardHdr)
{
	Byteswap (cardHdr.initStack);
	Byteswap (cardHdr.resetVector);
	Byteswap (cardHdr.signature);
	Byteswap (cardHdr.hdrVersion);
	Byteswap (cardHdr.flags);
	Byteswap (cardHdr.version);
	Byteswap (cardHdr.creationDate);
	Byteswap (cardHdr.numRAMBlocks);
	Byteswap (cardHdr.blockListOffset);
	Byteswap (cardHdr.readWriteParmsOffset);
	Byteswap (cardHdr.readWriteParmsSize);
	Byteswap (cardHdr.readOnlyParmsOffset);
	Byteswap (cardHdr.bigROMOffset);
	Byteswap (cardHdr.checksumBytes);
	Byteswap (cardHdr.checksumValue);
	Byteswap (cardHdr.version);
}


void Byteswap (DatabaseDirType& db)
{
	// !!! Note: this byteswapper works only in
	// one direction.  Because numDatabases is
	// converted early on, this function is assuming
	// that it is in Byteswap form.  Thus, this function
	// can only be used to convert from Byteswap to
	// Host form.

	Byteswap (db.nextDatabaseListID);
	Byteswap (db.numDatabases);

	for (Word ii = 0; ii < db.numDatabases; ++ii)
	{
		Byteswap (db.databaseID[ii]);
	}
}


void Byteswap (DatabaseHdrType& db)
{
	Byteswap (db.attributes);
	Byteswap (db.version);
	Byteswap (db.creationDate);
	Byteswap (db.modificationDate);
	Byteswap (db.lastBackupDate);
	Byteswap (db.modificationNumber);
	Byteswap (db.appInfoID);
	Byteswap (db.sortInfoID);
	Byteswap (db.type);
	Byteswap (db.creator);
	Byteswap (db.uniqueIDSeed);
	Byteswap (db.recordList.nextRecordListID);
	Byteswap (db.recordList.numRecords);
}


void Byteswap (DmAccessType& db)
{
	Byteswap (db.next);
	Byteswap (db.mode);
	Byteswap (db.openP);
	Byteswap (db.savedModNum);
}


void Byteswap (DmOpenInfoType& info)
{
	Byteswap (info.next);
	Byteswap (info.openCount);
	Byteswap (info.ownerTaskID);
//	Byteswap (info.exclusive);	// !!! sort these bitfields out
//	Byteswap (info.writeAccess);
//	Byteswap (info.resDB);
	Byteswap (info.hdrID);
	Byteswap (info.hdrH);
	Byteswap (info.hdrP);
	Byteswap (info.cardNo);
	Byteswap (info.numRecords);
	Byteswap (info.hdrMasterP);
	Byteswap (info.handleTableP);
}


void Byteswap (DmSearchStateType& state)
{
	// This is an opaque type.  We don't have to byteswap its
	// contents, as nothing on the host will access them.  Indeed,
	// the internal structure of this data type varies from system
	// to system, and so we *can't* byteswap it without knowing
	// what version of the system we're running.
}


void Byteswap (EventType& event)
{
	// Compile time assertion
	{
		char	_dummy1[offsetof(EventType, eType) == 0 ? 1 : 0];
		char	_dummy2[offsetof(EventType, penDown) == 2 ? 1 : 0];
		char	_dummy3[offsetof(EventType, screenX) == 4 ? 1 : 0];
		char	_dummy4[offsetof(EventType, screenY) == 6 ? 1 : 0];
		char	_dummy5[offsetof(EventType, data) == 8 ? 1 : 0];
		(void)_dummy1;
		(void)_dummy2;
		(void)_dummy3;
		(void)_dummy4;
		(void)_dummy5;
	}

	Byteswap (event.eType);
	Byteswap (event.penDown);
	Byteswap (event.screenX);
	Byteswap (event.screenY);

	switch(event.eType)
	{
		case nilEvent:
			break;

		case penDownEvent:
			Byteswap (event.data.penUp.start);
			Byteswap (event.data.penUp.end);
			break;

		case penUpEvent:
			Byteswap (event.data.penUp.start);
			Byteswap (event.data.penUp.end);
			break;

		case penMoveEvent:
			Byteswap (event.data.penUp.start);
			Byteswap (event.data.penUp.end);
			break;

		case keyDownEvent:
			Byteswap (event.data.keyDown.chr);
			Byteswap (event.data.keyDown.keyCode);
			Byteswap (event.data.keyDown.modifiers);
			break;

		case winEnterEvent:
			Byteswap (event.data.winEnter.enterWindow);
			Byteswap (event.data.winEnter.exitWindow);
			break;

		case winExitEvent:
			Byteswap (event.data.winExit.enterWindow);
			Byteswap (event.data.winExit.exitWindow);
			break;

		case ctlEnterEvent:
			Byteswap (event.data.ctlEnter.controlID);
			Byteswap (event.data.ctlEnter.pControl);
			break;

		case ctlExitEvent:
			break;

		case ctlSelectEvent:
			Byteswap (event.data.ctlSelect.controlID);
			Byteswap (event.data.ctlSelect.pControl);
			Byteswap (event.data.ctlSelect.on);
			break;

		case ctlRepeatEvent:
			Byteswap (event.data.ctlRepeat.controlID);
			Byteswap (event.data.ctlRepeat.pControl);
			Byteswap (event.data.ctlRepeat.time);
			break;

		case lstEnterEvent:
			Byteswap (event.data.lstEnter.listID);
			Byteswap (event.data.lstEnter.pList);
			Byteswap (event.data.lstEnter.selection);
			break;

		case lstSelectEvent:
			Byteswap (event.data.lstSelect.listID);
			Byteswap (event.data.lstSelect.pList);
			Byteswap (event.data.lstSelect.selection);
			break;

		case lstExitEvent:
			Byteswap (event.data.lstExit.listID);
			Byteswap (event.data.lstExit.pList);
			break;

		case popSelectEvent:
			Byteswap (event.data.popSelect.controlID);
			Byteswap (event.data.popSelect.controlP);
			Byteswap (event.data.popSelect.listID);
			Byteswap (event.data.popSelect.listP);
			Byteswap (event.data.popSelect.selection);
			Byteswap (event.data.popSelect.priorSelection);
			break;

		case fldEnterEvent:
			Byteswap (event.data.fldEnter.fieldID);
			Byteswap (event.data.fldEnter.pField);
			break;

		case fldHeightChangedEvent:
			Byteswap (event.data.fldHeightChanged.fieldID);
			Byteswap (event.data.fldHeightChanged.pField);
			Byteswap (event.data.fldHeightChanged.newHeight);
			Byteswap (event.data.fldHeightChanged.currentPos);
			break;

		case fldChangedEvent:
			Byteswap (event.data.fldChanged.fieldID);
			Byteswap (event.data.fldChanged.pField);
			break;

		case tblEnterEvent:
			Byteswap (event.data.tblEnter.tableID);
			Byteswap (event.data.tblEnter.pTable);
			Byteswap (event.data.tblEnter.row);
			Byteswap (event.data.tblEnter.column);
			break;

		case tblSelectEvent:
			Byteswap (event.data.tblEnter.tableID);
			Byteswap (event.data.tblEnter.pTable);
			Byteswap (event.data.tblEnter.row);
			Byteswap (event.data.tblEnter.column);
			break;

		case daySelectEvent:
			Byteswap (event.data.daySelect.pSelector);
			Byteswap (event.data.daySelect.selection);
			Byteswap (event.data.daySelect.useThisDate);
			break;

		case menuEvent:
			Byteswap (event.data.menu.itemID);
			break;

		case appStopEvent:
			break;

		case frmLoadEvent:
			Byteswap (event.data.frmLoad.formID);
			break;

		case frmOpenEvent:
			Byteswap (event.data.frmOpen.formID);
			break;

		case frmGotoEvent:
			Byteswap (event.data.frmGoto.formID);
			Byteswap (event.data.frmGoto.recordNum);
			Byteswap (event.data.frmGoto.matchPos);
			Byteswap (event.data.frmGoto.matchLen);
			Byteswap (event.data.frmGoto.matchFieldNum);
			Byteswap (event.data.frmGoto.matchCustom);
			break;

		case frmUpdateEvent:
			Byteswap (event.data.frmUpdate.formID);
			Byteswap (event.data.frmUpdate.updateCode);
			break;

		case frmSaveEvent:
			break;

		case frmCloseEvent:
			Byteswap (event.data.frmClose.formID);
			break;

		case frmTitleEnterEvent:
			Byteswap (event.data.frmTitleEnter.formID);
			break;

		case frmTitleSelectEvent:
			Byteswap (event.data.frmTitleSelect.formID);
			break;

		case tblExitEvent:
			Byteswap (event.data.tblExit.tableID);
			Byteswap (event.data.tblExit.pTable);
			Byteswap (event.data.tblExit.row);
			Byteswap (event.data.tblExit.column);
			break;

		case sclEnterEvent:
			Byteswap (event.data.sclEnter.scrollBarID);
			Byteswap (event.data.sclEnter.pScrollBar);
			break;

		case sclExitEvent:
			Byteswap (event.data.sclExit.scrollBarID);
			Byteswap (event.data.sclExit.pScrollBar);
			Byteswap (event.data.sclExit.value);
			Byteswap (event.data.sclExit.newValue);
			break;

		case sclRepeatEvent:
			Byteswap (event.data.sclRepeat.scrollBarID);
			Byteswap (event.data.sclRepeat.pScrollBar);
			Byteswap (event.data.sclRepeat.value);
			Byteswap (event.data.sclRepeat.newValue);
			Byteswap (event.data.sclRepeat.time);
			break;
	}
}


void Byteswap (FieldAttrType& attr)
{
	Byteswap (*(unsigned short*) &attr);

#if defined (_MSC_VER) || defined (__GNUC__)
	unsigned short	v = *(unsigned short*) &attr;

	/*
		======== BITFIELD LAYOUT CHEAT-SHEET ========

		typedef struct {
			Word usable			:1;
			Word visible		:1;
			Word editable		:1;
			Word singleLine		:1;
			Word hasFocus		:1;
			Word dynamicSize	:1;
			Word insPtVisible	:1;
			Word dirty			:1;
			Word underlined		:2;
			Word justification	:2;
			Word autoShift		:1;
			Word hasScrollBar	:1;
			Word numeric		:1;
		} FieldAttrType;

		// On the Mac:

		|---------- high byte ----------|---------- low byte -----------|

		 15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0  
		  7   6   5   4   3   2   1   0   7   6   5   4   3   2   1   0
		+---+---+---+---+---+---+---+---+-------+-------+---+---+---+---+
		| u | v | e | s | h | d | i | d |   u   |   j   | a | h | n | * |
		+---+---+---+---+---+---+---+---+-------+-------+---+---+---+---+


		// On Windows (in-register representation):

		|---------- high byte ----------|---------- low byte -----------|

		 15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0  
		  7   6   5   4   3   2   1   0   7   6   5   4   3   2   1   0
		+---+---+---+---+-------+-------+---+---+---+---+---+---+---+---+
		| * | n | h | a |   j   |   u   | d | i | d | h | s | e | v | u |
		+---+---+---+---+-------+-------+---+---+---+---+---+---+---+---+
	*/

	v = (unsigned short) (
		(flipBits[ (v >>  0) & 0x0F ] << 12) |
		(flipBits[ (v >>  4) & 0x0F ] <<  8) |
		(flipBits[ (v >>  8) & 0x0F ] <<  4) |
		(flipBits[ (v >> 12) & 0x0F ] <<  0));

	*(unsigned short*) &attr = v;
#endif
}

void Byteswap (HwrDBallEZType& regs)
{
	Byteswap (regs.swID);

	Byteswap (regs.csAGroupBase);
	Byteswap (regs.csBGroupBase);
	Byteswap (regs.csCGroupBase);
	Byteswap (regs.csDGroupBase);

	Byteswap (regs.csASelect);
	Byteswap (regs.csBSelect);
	Byteswap (regs.csCSelect);
	Byteswap (regs.csDSelect);

	Byteswap (regs.emuCS);

	Byteswap (regs.pllControl);
	Byteswap (regs.pllFreqSel);
	Byteswap (regs.pllTest);

	Byteswap (regs.intControl);
	Byteswap (regs.intMaskHi);
	Byteswap (regs.intMaskLo);
	Byteswap (regs.intStatusHi);
	Byteswap (regs.intStatusLo);
	Byteswap (regs.intPendingHi);
	Byteswap (regs.intPendingLo);

	Byteswap (regs.pwmControl);

	Byteswap (regs.tmr1Control);
	Byteswap (regs.tmr1Prescaler);
	Byteswap (regs.tmr1Compare);
	Byteswap (regs.tmr1Capture);
	Byteswap (regs.tmr1Counter);
	Byteswap (regs.tmr1Status);

	Byteswap (regs.spiMasterData);
	Byteswap (regs.spiMasterControl);

	Byteswap (regs.uControl);
	Byteswap (regs.uBaud);
	Byteswap (regs.uReceive);
	Byteswap (regs.uTransmit);
	Byteswap (regs.uMisc);
	Byteswap (regs.uNonIntPresc);

	Byteswap (regs.lcdStartAddr);
	Byteswap (regs.lcdScreenWidth);
	Byteswap (regs.lcdScreenHeight);
	Byteswap (regs.lcdCursorXPos);
	Byteswap (regs.lcdCursorYPos);
	Byteswap (regs.lcdCursorWidthHeight);
	Byteswap (regs.lcdContrastControlPWM);

	Byteswap (regs.rtcHourMinSec);
	Byteswap (regs.rtcAlarm);
	Byteswap (regs.rtcWatchDog);
	Byteswap (regs.rtcControl);
	Byteswap (regs.rtcIntStatus);
	Byteswap (regs.rtcIntEnable);
	Byteswap (regs.stopWatch);
	Byteswap (regs.rtcDay);
	Byteswap (regs.rtcDayAlarm);

	Byteswap (regs.dramConfig);
	Byteswap (regs.dramControl);

	Byteswap (regs.emuAddrCompare);   
	Byteswap (regs.emuAddrMask);
	Byteswap (regs.emuControlCompare);
	Byteswap (regs.emuControlMask);
	Byteswap (regs.emuControl);
	Byteswap (regs.emuStatus);
}

void Byteswap (HwrDBallType& regs)
{
	Byteswap (regs.csAGroupBase);
	Byteswap (regs.csBGroupBase);
	Byteswap (regs.csCGroupBase);
	Byteswap (regs.csDGroupBase);

	Byteswap (regs.csAGroupMask);
	Byteswap (regs.csBGroupMask);
	Byteswap (regs.csCGroupMask);
	Byteswap (regs.csDGroupMask);

	Byteswap (regs.csASelect0);
	Byteswap (regs.csASelect1);
	Byteswap (regs.csASelect2);
	Byteswap (regs.csASelect3);

	Byteswap (regs.csBSelect0);
	Byteswap (regs.csBSelect1);
	Byteswap (regs.csBSelect2);
	Byteswap (regs.csBSelect3);

	Byteswap (regs.csCSelect0);
	Byteswap (regs.csCSelect1);
	Byteswap (regs.csCSelect2);
	Byteswap (regs.csCSelect3);

	Byteswap (regs.csDSelect0);
	Byteswap (regs.csDSelect1);
	Byteswap (regs.csDSelect2);
	Byteswap (regs.csDSelect3);

	Byteswap (regs.csDebug);

	Byteswap (regs.pllControl);
	Byteswap (regs.pllFreqSel);
	Byteswap (regs.pllTest);

	Byteswap (regs.intControl);
	Byteswap (regs.intMaskHi);
	Byteswap (regs.intMaskLo);
	Byteswap (regs.intWakeupEnHi);
	Byteswap (regs.intWakeupEnLo);
	Byteswap (regs.intStatusHi);
	Byteswap (regs.intStatusLo);
	Byteswap (regs.intPendingHi);
	Byteswap (regs.intPendingLo);

	Byteswap (regs.pwmControl);
	Byteswap (regs.pwmPeriod);
	Byteswap (regs.pwmWidth);
	Byteswap (regs.pwmCounter);

	Byteswap (regs.tmr1Control);
	Byteswap (regs.tmr1Prescaler);
	Byteswap (regs.tmr1Compare);
	Byteswap (regs.tmr1Capture);
	Byteswap (regs.tmr1Counter);
	Byteswap (regs.tmr1Status);
	
	Byteswap (regs.tmr2Control);
	Byteswap (regs.tmr2Prescaler);
	Byteswap (regs.tmr2Compare);
	Byteswap (regs.tmr2Capture);
	Byteswap (regs.tmr2Counter);
	Byteswap (regs.tmr2Status);
	
	Byteswap (regs.wdControl);
	Byteswap (regs.wdReference);
	Byteswap (regs.wdCounter);

	Byteswap (regs.spiSlave);

	Byteswap (regs.spiMasterData);
	Byteswap (regs.spiMasterControl);

	Byteswap (regs.uControl);
	Byteswap (regs.uBaud);
	Byteswap (regs.uReceive);
	Byteswap (regs.uTransmit);
	Byteswap (regs.uMisc);

	Byteswap (regs.lcdStartAddr);
	Byteswap (regs.lcdScreenWidth);
	Byteswap (regs.lcdScreenHeight);
	Byteswap (regs.lcdCursorXPos);
	Byteswap (regs.lcdCursorYPos);
	Byteswap (regs.lcdCursorWidthHeight);
	Byteswap (regs.lcdGrayPalette);

	Byteswap (regs.rtcHourMinSec);
	Byteswap (regs.rtcAlarm);
	Byteswap (regs.rtcReserved);
	Byteswap (regs.rtcControl);
	Byteswap (regs.rtcIntStatus);
	Byteswap (regs.rtcIntEnable);
	Byteswap (regs.stopWatch);
}

void Byteswap (M68KRegsType& debuggerRegs)
{
	Byteswap (debuggerRegs.d[0]);
	Byteswap (debuggerRegs.d[1]);
	Byteswap (debuggerRegs.d[2]);
	Byteswap (debuggerRegs.d[3]);
	Byteswap (debuggerRegs.d[4]);
	Byteswap (debuggerRegs.d[5]);
	Byteswap (debuggerRegs.d[6]);
	Byteswap (debuggerRegs.d[7]);

	Byteswap (debuggerRegs.a[0]);
	Byteswap (debuggerRegs.a[1]);
	Byteswap (debuggerRegs.a[2]);
	Byteswap (debuggerRegs.a[3]);
	Byteswap (debuggerRegs.a[4]);
	Byteswap (debuggerRegs.a[5]);
	Byteswap (debuggerRegs.a[6]);
//	Byteswap (debuggerRegs.a[7]);

	Byteswap (debuggerRegs.usp);
	Byteswap (debuggerRegs.ssp);
	Byteswap (debuggerRegs.sr);
	Byteswap (debuggerRegs.pc);
}


void Byteswap (Mem1ChunkHeaderType& p)
{
	Byteswap (p.size);
	Byteswap (p.lockOwner);
	Byteswap (p.flags);
	Byteswap (p.hOffset);
}


void Byteswap (Mem1HeapHeaderType& p)
{
	Byteswap (p.flags);
	Byteswap (p.size);
	Byteswap (p.mstrPtrTbl);
}


void Byteswap (Mem1MstrPtrTableType& p)
{
	Byteswap (p.numEntries);
	Byteswap (p.nextTblOffset);
//	Byteswap (p.mstrP);	// !!! Swap the table?
}


void Byteswap (Mem2HeapHeaderType& p)
{
	Byteswap (p.flags);
	Byteswap (p.size);
	Byteswap (p.mstrPtrTbl);
}


void Byteswap (MemChunkHeaderType& p)
{
	DWord	size = p.size << 8;
	Byteswap (size);
	p.size = size;

	DWord	hOffset = p.hOffset << 8;
	Byteswap (hOffset);	// offset/2 to handle for movables
	p.hOffset = hOffset;
}


void Byteswap (MemMstrPtrTableType& p)
{
	Byteswap (p.numEntries);
	Byteswap (p.nextTblOffset);
//	Byteswap (p.mstrP);	// !!! Swap the table?
}


void Byteswap (PenBtnInfoType& p)
{
	Byteswap (p.boundsR);
	Byteswap (p.asciiCode);
	Byteswap (p.keyCode);
	Byteswap (p.modifiers);
}


void Byteswap (PointType& p)
{
	Byteswap (p.x);
	Byteswap (p.y);
}


void Byteswap (RecordListType& recordList)
{
	Byteswap (recordList.nextRecordListID);
	Byteswap (recordList.numRecords);
	Byteswap (recordList.firstEntry);
}


void Byteswap (RecordEntryType& entry)
{
	Byteswap (entry.localChunkID);
	Byteswap (entry.attributes);
//	Byteswap (entry.uniqueID);	// array of 3 bytes
}


void Byteswap (RectangleType& r)
{
	Byteswap (r.topLeft);
	Byteswap (r.extent);
}


void Byteswap (RsrcEntryType& entry)
{
	Byteswap (entry.type);
	Byteswap (entry.id);
	Byteswap (entry.localChunkID);
}


void Byteswap (SlkPktHeaderType& hdr)
{
	Byteswap (hdr.signature1);
	Byteswap (hdr.signature2);
	Byteswap (hdr.dest);
	Byteswap (hdr.src);
	Byteswap (hdr.type);
	Byteswap (hdr.bodySize);
	Byteswap (hdr.transId);
	Byteswap (hdr.checksum);
}


void Byteswap (SlkPktFooterType& ftr)
{
	Byteswap (ftr.crc16);
}


void Byteswap (SysAppInfoType& info)
{
	Byteswap (info.cmd);
	Byteswap (info.cmdPBP);
	Byteswap (info.launchFlags);
	Byteswap (info.taskID);
	Byteswap (info.codeH);
	Byteswap (info.dbP);
	Byteswap (info.stackP);
	Byteswap (info.globalsChunkP);
	Byteswap (info.memOwnerID);
	Byteswap (info.dmAccessP);
	Byteswap (info.dmLastErr);
	Byteswap (info.errExceptionP);
	Byteswap (info.a5Ptr);
	Byteswap (info.stackEndP);
	Byteswap (info.globalEndP);
	Byteswap (info.rootP);
	Byteswap (info.extraP);
}


void Byteswap (SysKernelInfoType& info)
{
	// Compile time assertion
	{
		char	_dummy[sizeof (info.selector) == 1 ? 1 : 0];
		(void)_dummy;
	}

	Byteswap (info.selector);
	Byteswap (info.id);

	switch (info.selector)
	{
		case sysKernelInfoSelCurTaskInfo:
		case sysKernelInfoSelTaskInfo:
			Byteswap (info.param.task.id);
			Byteswap (info.param.task.nextID);
			Byteswap (info.param.task.tag);
			Byteswap (info.param.task.status);
			Byteswap (info.param.task.timer);
			Byteswap (info.param.task.timeSlice);
			Byteswap (info.param.task.priority);
			Byteswap (info.param.task.attributes);
			Byteswap (info.param.task.pendingCalls);
			Byteswap (info.param.task.senderTaskID);
			Byteswap (info.param.task.msgExchangeID);
			Byteswap (info.param.task.tcbP);
			Byteswap (info.param.task.stackP);
			Byteswap (info.param.task.stackStart);
			Byteswap (info.param.task.stackSize);
			break;

		case sysKernelInfoSelSemaphoreInfo:
			Byteswap (info.param.semaphore.id);
			Byteswap (info.param.semaphore.nextID);
			Byteswap (info.param.semaphore.tag);
			Byteswap (info.param.semaphore.initValue);
			Byteswap (info.param.semaphore.curValue);
			Byteswap (info.param.semaphore.nestLevel);
			Byteswap (info.param.semaphore.ownerID);
			break;

		case sysKernelInfoSelTimerInfo:
			Byteswap (info.param.timer.id);
			Byteswap (info.param.timer.nextID);
			Byteswap (info.param.timer.tag);
			Byteswap (info.param.timer.ticksLeft);
			Byteswap (info.param.timer.period);
			Byteswap (info.param.timer.proc);
			break;
	}
}


void Byteswap (SysNVParamsType& params)
{
	Byteswap (params.rtcHours);
	Byteswap (params.rtcHourMinSecCopy);
}


void Byteswap (SysPktBodyType& packet)
{
	// Compile time assertion
	{
		char	_dummy[sizeof (packet.command) == 1 ? 1 : 0];
		(void)_dummy;
	}

	Byteswap (packet.command);

	switch (packet.command)
	{
		case sysPktStateCmd:
			Byteswap ((SysPktStateCmdType&) packet);
			break;

		case sysPktStateRsp:
			Byteswap ((SysPktStateRspType&) packet);
			break;

		case sysPktReadMemCmd:
			Byteswap ((SysPktReadMemCmdType&) packet);
			break;

		case sysPktReadMemRsp:
			Byteswap ((SysPktReadMemRspType&) packet);
			break;

		case sysPktWriteMemCmd:
			Byteswap ((SysPktWriteMemCmdType&) packet);
			break;

		case sysPktWriteMemRsp:
			Byteswap ((SysPktWriteMemRspType&) packet);
			break;

		case sysPktSingleStepCmd:
			// I don't think there's anything to do here.  I think that
			// single-stepping is performed by the debugger setting the
			// trace bit in the SR and sending a sysPktContinueCmd.
			break;

		case sysPktGetRtnNameCmd:
			Byteswap ((SysPktRtnNameCmdType&) packet);
			break;

		case sysPktGetRtnNameRsp:
			Byteswap ((SysPktRtnNameRspType&) packet);
			break;

		case sysPktReadRegsCmd:
			Byteswap ((SysPktReadRegsCmdType&) packet);
			break;

		case sysPktReadRegsRsp:
			Byteswap ((SysPktReadRegsRspType&) packet);
			break;

		case sysPktWriteRegsCmd:
			Byteswap ((SysPktWriteRegsCmdType&) packet);
			break;

		case sysPktWriteRegsRsp:
			Byteswap ((SysPktWriteRegsRspType&) packet);
			break;

		case sysPktContinueCmd:
			Byteswap ((SysPktContinueCmdType&) packet);
			break;

		case sysPktRPCCmd:
		case sysPktRPCRsp:
			Byteswap ((SysPktRPCType&) packet);
			break;

		case sysPktRPC2Cmd:
		case sysPktRPC2Rsp:
			Byteswap ((SysPktRPC2Type&) packet);
			break;

		case sysPktGetBreakpointsCmd:
			Byteswap ((SysPktGetBreakpointsCmdType&) packet);
			break;

		case sysPktGetBreakpointsRsp:
			Byteswap ((SysPktGetBreakpointsRspType&) packet);
			break;

		case sysPktSetBreakpointsCmd:
			Byteswap ((SysPktSetBreakpointsCmdType&) packet);
			break;

		case sysPktSetBreakpointsRsp:
			Byteswap ((SysPktSetBreakpointsRspType&) packet);
			break;

//		case sysPktRemoteUIUpdCmd:
//			// Sent TO debugger; never received FROM debugger.
//			break;

//		case sysPktRemoteEvtCmd:
//			// Sent TO debugger; never received FROM debugger.
//			break;

		case sysPktDbgBreakToggleCmd:
			Byteswap ((SysPktDbgBreakToggleCmdType&) packet);
			break;

		case sysPktDbgBreakToggleRsp:
			Byteswap ((SysPktDbgBreakToggleRspType&) packet);
			break;

		case sysPktFlashCmd:
		case sysPktFlashRsp:
			Byteswap ((SysPktFlashWriteType&) packet);
			break;

		case sysPktCommCmd:
			Byteswap ((SysPktCommCmdType&) packet);
			break;

		case sysPktCommRsp:
			Byteswap ((SysPktCommRspType&) packet);
			break;

		case sysPktGetTrapBreaksCmd:
			Byteswap ((SysPktGetTrapBreaksCmdType&) packet);
			break;

		case sysPktGetTrapBreaksRsp:
			Byteswap ((SysPktGetTrapBreaksRspType&) packet);
			break;

		case sysPktSetTrapBreaksCmd:
			Byteswap ((SysPktSetTrapBreaksCmdType&) packet);
			break;

		case sysPktSetTrapBreaksRsp:
			Byteswap ((SysPktSetTrapBreaksRspType&) packet);
			break;

		case sysPktGremlinsCmd:
			Byteswap ((SysPktGremlinsCmdType&) packet);
			break;

		case sysPktFindCmd:
			Byteswap ((SysPktFindCmdType&) packet);
			break;

		case sysPktFindRsp:
			Byteswap ((SysPktFindRspType&) packet);
			break;

		case sysPktGetTrapConditionsCmd:
			Byteswap ((SysPktGetTrapConditionsCmdType&) packet);
			break;

		case sysPktGetTrapConditionsRsp:
			Byteswap ((SysPktGetTrapConditionsRspType&) packet);
			break;

		case sysPktSetTrapConditionsCmd:
			Byteswap ((SysPktSetTrapConditionsCmdType&) packet);
			break;

		case sysPktSetTrapConditionsRsp:
			Byteswap ((SysPktSetTrapConditionsRspType&) packet);
			break;

		case sysPktChecksumCmd:
		case sysPktChecksumRsp:
			Byteswap ((SysPktChecksumType&) packet);
			break;

		case sysPktExecFlashCmd:
		case sysPktExecFlashRsp:
			Byteswap ((sysPktExecFlashType&) packet);
			break;

		case sysPktRemoteMsgCmd:
			Byteswap ((SysPktRemoteMsgCmdType&) packet);
			break;

		default:
			break;
	}
}


void Byteswap (SysPktChecksumType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.firstAddr);
	Byteswap (packet.numBytes);
	Byteswap (packet.seed);
	Byteswap (packet.checksum);
}


void Byteswap (SysPktCommCmdType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.set);
	Byteswap (packet.baudRate);
	Byteswap (packet.flags);
}


void Byteswap (SysPktCommRspType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.baudRate);
	Byteswap (packet.flags);
}


void Byteswap (SysPktContinueCmdType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.regs);
	Byteswap (packet.stepSpy);
	Byteswap (packet.ssAddr);
	Byteswap (packet.ssCount);
	Byteswap (packet.ssCheckSum);
}


void Byteswap (SysPktDbgBreakToggleCmdType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktDbgBreakToggleRspType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.newState);
}


void Byteswap (sysPktExecFlashType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.numBytes);
	Byteswap (packet.flashBase);
	Byteswap (packet.srcAddr);
	Byteswap (packet.dstAddr);
	Byteswap (packet.codeAddr);
	Byteswap (packet.flashResID);
	Byteswap (packet.manuf);
	Byteswap (packet.device);
	Byteswap (packet.devInfo);
	Byteswap (packet.remoteCmd);
	Byteswap (packet.response);
	Byteswap (packet.spareByte);
	Byteswap (packet.spareDWord);
}


void Byteswap (SysPktFindCmdType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.firstAddr);
	Byteswap (packet.lastAddr);
	Byteswap (packet.numBytes);
	Byteswap (packet.caseInsensitive);
}


void Byteswap (SysPktFindRspType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.addr);
	Byteswap (packet.found);
}


void Byteswap (SysPktFlashWriteType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.numBytes);
	Byteswap (packet.flashBase);
	Byteswap (packet.srcAddr);
	Byteswap (packet.dstAddr);
	Byteswap (packet.manuf);
	Byteswap (packet.device);
	Byteswap (packet.width);
	Byteswap (packet.filler);
	Byteswap (packet.ramSpaceP);

}


void Byteswap (SysPktGetBreakpointsCmdType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktGetBreakpointsRspType& packet)
{
	SysPktGetBreakpointsRspType2&	packet2 = (SysPktGetBreakpointsRspType2&) packet;

	Byteswap (packet2.command);

	for (int ii = 0; ii < dbgTotalBreakpoints; ++ii)
		Byteswap (packet2.bp[ii]);
}


void Byteswap (SysPktGetTrapBreaksCmdType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktGetTrapBreaksRspType& packet)
{
	SysPktGetTrapBreaksRspType2&	packet2 = (SysPktGetTrapBreaksRspType2&) packet;

	Byteswap (packet2.command);

	for (int ii = 0; ii < dbgTotalTrapBreaks; ++ii)
		Byteswap (packet2.trapBP[ii]);
}


void Byteswap (SysPktGetTrapConditionsCmdType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktGetTrapConditionsRspType& packet)
{
	SysPktGetTrapConditionsRspType2&	packet2 = (SysPktGetTrapConditionsRspType2&) packet;

	Byteswap (packet2.command);

	for (int ii = 0; ii < dbgTotalTrapBreaks; ++ii)
		Byteswap (packet2.trapParam[ii]);
}


void Byteswap (SysPktGremlinsCmdType& packet)
{
	Byteswap (packet.command);
	Byteswap (packet.action);

	// We can't really decode the data unless we know the
	// action number.  And we don't know if we know the
	// action number because we don't know if it is in
	// host or canonical format.
}


void Byteswap (SysPktReadMemCmdType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.address);
	Byteswap (packet.numBytes);
}


void Byteswap (SysPktReadMemRspType& packet)
{
	Byteswap (packet.command);

	// Variable length data follows.  This is all just a
	// stream of bytes; it can't be byteswapped without
	// knowing the format that the requester will apply
	// to those bytes.
}


void Byteswap (SysPktReadRegsCmdType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktReadRegsRspType& packet)
{
	SysPktReadRegsRspType2&	packet2 = (SysPktReadRegsRspType2&) packet;

	Byteswap (packet2.command);

	Byteswap (packet2.reg);
}


void Byteswap (SysPktRemoteMsgCmdType& packet)
{
	Byteswap (packet.command);

	// The rest of this guy is just straight text.
}


void Byteswap (SysPktRPCType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.trapWord);
	Byteswap (packet.resultD0);
	Byteswap (packet.resultA0);
	Byteswap (packet.numParams);

	// Nothing really to do with the "param" field.  "byRef" and "size"
	// are bytes, so nothing to swap there, and the "data" field is
	// of indeterminate size, so we can't bytewap it either.  That means
	// that we must take care to always treat it as a big-endian value.	
}


void Byteswap (SysPktRPC2Type& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.trapWord);
	Byteswap (packet.resultD0);
	Byteswap (packet.resultA0);
	Byteswap (packet.resultException);
	Byteswap (packet.DRegMask);
	Byteswap (packet.ARegMask);

	// Byteswap any values in the variable-length Regs
	// array.  Note that in doing this, we don't care
	// *which* registers are extant; we just care how
	// many.  Therefore, it doesn't matter if DRegMask
	// and ARegMask are in Big-Endian or Little-Endian
	// format (not that they could change; they're bytes!).

	UInt32*	regs = packet.Regs;
	int		mask = 0x01;

	while (mask < 0x100)
	{
		if ((mask & packet.DRegMask) != 0)
		{
			Canonical (*regs++);
		}

		mask <<= 1;
	}

	mask = 0x01;
	while (mask < 0x100)
	{
		if ((mask & packet.ARegMask) != 0)
		{
			Canonical (*regs++);
		}

		mask <<= 1;
	}

	UInt16*	numParams = (UInt16*) regs;
	Byteswap (*numParams);

	// Nothing really to do with the "param" field.  "byRef" and "size"
	// are bytes, so nothing to swap there, and the "data" field is
	// of indeterminate size, so we can't bytewap it either.  That means
	// that we must take care to always treat it as a big-endian value.	
}


void Byteswap (SysPktRtnNameCmdType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.address);
}


void Byteswap (SysPktRtnNameRspType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.address);
	Byteswap (packet.startAddr);
	Byteswap (packet.endAddr);
}


void Byteswap (SysPktSetBreakpointsCmdType& packet)
{
	SysPktSetBreakpointsCmdType2&	packet2 = (SysPktSetBreakpointsCmdType2&) packet;

	Byteswap (packet2.command);

	for (int ii = 0; ii < dbgTotalBreakpoints; ++ii)
		Byteswap (packet2.bp[ii]);
}


void Byteswap (SysPktSetBreakpointsRspType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktSetTrapBreaksCmdType& packet)
{
	SysPktSetTrapBreaksCmdType2&	packet2 = (SysPktSetTrapBreaksCmdType2&) packet;

	Byteswap (packet2.command);

	for (int ii = 0; ii < dbgTotalTrapBreaks; ++ii)
		Byteswap (packet2.trapBP[ii]);
}


void Byteswap (SysPktSetTrapBreaksRspType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktSetTrapConditionsCmdType& packet)
{
	SysPktSetTrapConditionsCmdType2&	packet2 = (SysPktSetTrapConditionsCmdType2&) packet;

	Byteswap (packet2.command);

	for (int ii = 0; ii < dbgTotalTrapBreaks; ++ii)
		Byteswap (packet2.trapParam[ii]);
}


void Byteswap (SysPktSetTrapConditionsRspType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktStateCmdType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktStateRspType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.resetted);
	Byteswap (packet.exceptionId);
	Byteswap (packet.reg);

	int ii;
	for (ii = 0; ii < dbgTotalBreakpoints; ++ii)
		Byteswap (packet.bp[ii]);

	for (ii = 0; ii < sysPktStateRspInstWords; ++ii)
		Byteswap (packet.inst[ii]);

	Byteswap (packet.startAddr);
	Byteswap (packet.endAddr);
	Byteswap (packet.trapTableRev);	
}


void Byteswap (SysPktWriteRegsCmdType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.reg);
}


void Byteswap (SysPktWriteRegsRspType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (SysPktWriteMemCmdType& packet)
{
	Byteswap (packet.command);

	Byteswap (packet.address);
	Byteswap (packet.numBytes);
}


void Byteswap (SysPktWriteMemRspType& packet)
{
	Byteswap (packet.command);
}


void Byteswap (regstruct& p)
{
	Byteswap (p.regs[0]);
	Byteswap (p.regs[1]);
	Byteswap (p.regs[2]);
	Byteswap (p.regs[3]);
	Byteswap (p.regs[4]);
	Byteswap (p.regs[5]);
	Byteswap (p.regs[6]);
	Byteswap (p.regs[7]);

	Byteswap (p.regs[8]);
	Byteswap (p.regs[9]);
	Byteswap (p.regs[10]);
	Byteswap (p.regs[11]);
	Byteswap (p.regs[12]);
	Byteswap (p.regs[13]);
	Byteswap (p.regs[14]);
	Byteswap (p.regs[15]);

	Byteswap (p.usp);
	Byteswap (p.isp);
	Byteswap (p.msp);
	Byteswap (p.sr);
	Byteswap (p.t1);
	Byteswap (p.t0);
	Byteswap (p.s);
	Byteswap (p.m);
	Byteswap (p.stopped);
	Byteswap (p.intmask);
	Byteswap (p.pc);
	Byteswap (p.pc_p);
	Byteswap (p.pc_oldp);
	Byteswap (p.pc_meta_oldp);
	Byteswap (p.vbr);
	Byteswap (p.sfc);
	Byteswap (p.dfc);

#if 0	// we don't support FP
	Byteswap (fp.regs[0]);
	Byteswap (fp.regs[1]);
	Byteswap (fp.regs[2]);
	Byteswap (fp.regs[3]);
	Byteswap (fp.regs[4]);
	Byteswap (fp.regs[5]);
	Byteswap (fp.regs[6]);
	Byteswap (fp.regs[7]);

	Byteswap (p.fpcr);
	Byteswap (p.fpsr);
	Byteswap (p.fpiar);
#endif

	Byteswap (p.spcflags);
	Byteswap (p.kick_mask);
	Byteswap (p.prefetch);
}


void Byteswap (SED1375RegsType& p)
{
	Byteswap (p.productRevisionCode);
	Byteswap (p.mode0);
	Byteswap (p.mode1);
	Byteswap (p.mode2);
	Byteswap (p.horizontalPanelSize);
	Byteswap (p.verticalPanelSizeLSB);
	Byteswap (p.verticalPanelSizeMSB);
	Byteswap (p.FPLineStartPosition);
	Byteswap (p.horizontalNonDisplayPeriod);
	Byteswap (p.FPFRAMEStartPosition);
	Byteswap (p.verticalNonDisplayPeriod);
	Byteswap (p.MODRate);
	Byteswap (p.screen1StartAddressLSB);
	Byteswap (p.screen1StartAddressMSB);
	Byteswap (p.screen1StartAddressMSBit);
	Byteswap (p.screen2StartAddressLSB);
	Byteswap (p.screen2StartAddressMSB);
	Byteswap (p.screen2StartAddressMSBit);
	Byteswap (p.memoryAddressOffset);
	Byteswap (p.screen1VerticalSizeLSB);
	Byteswap (p.screen1VerticalSizeMSB);
	Byteswap (p.lookUpTableAddress);
	Byteswap (p.unused1);
	Byteswap (p.lookUpTableData);
	Byteswap (p.GPIOConfigurationControl);
	Byteswap (p.GPIOStatusControl);
	Byteswap (p.scratchPad);
	Byteswap (p.portraitMode);
	Byteswap (p.lineByteCountRegister);
	Byteswap (p.unused2);
	Byteswap (p.unusual);
	Byteswap (p.testMode);
}

void Byteswap (HwrJerryPLDType& p)
{
	Byteswap (p.dspOn);
	Byteswap (p.unused1);
	Byteswap (p.chargeOn);
	Byteswap (p.unused2);
	Byteswap (p.refOn);
	Byteswap (p.unused3);
	Byteswap (p.pipaBiasEnable);
	Byteswap (p.unused4);
	Byteswap (p.dspReset);
	Byteswap (p.unused5);
	Byteswap (p.ezToDsl);
	Byteswap (p.unused6);
	Byteswap (p.rs232Shdn);
	Byteswap (p.unused7);
	Byteswap (p.spareOut);
	Byteswap (p.unused8);
	Byteswap (p.dTo3Sln);
	Byteswap (p.iXtrnl2);
}

#endif	// BYTESWAP
