/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Bank_MC68681.h"

#include "CPU_REG.h"			// Software::BusError
#include "RAM_ROM.h"			// Memory::InitializeBanks


// ===========================================================================
//		 MC68681 Bank Accessors
// ===========================================================================
// These functions provide fetch and store access to the emulated M68681
// serial port.

static AddressBank	gAddressBank =
{
	MC68681Bank::GetLong,
	MC68681Bank::GetWord,
	MC68681Bank::GetByte,
	MC68681Bank::SetLong,
	MC68681Bank::SetWord,
	MC68681Bank::SetByte,
	MC68681Bank::GetRealAddress,
	MC68681Bank::ValidAddress,
	NULL,
	NULL
};

static const uae_u32	kMemoryStart = hwrDuartBase;

static HwrDuartType		gDuart;

/***********************************************************************
 *
 * FUNCTION:	MC68681Bank::Initialize
 *
 * DESCRIPTION:	Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68681Bank::Initialize (void)
{
	// Memory bank 0x10E0 is managed by the functions in MC68681Bank.

	Memory::InitializeBanks (	gAddressBank,
								bankindex (kMemoryStart), 1);
}


/***********************************************************************
 *
 * FUNCTION:	MC68681Bank::Reset
 *
 * DESCRIPTION:	Standard reset function.  Sets the sub-system to a
 *				default state.  This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68681Bank::Reset (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	MC68681Bank::Save
 *
 * DESCRIPTION:	Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68681Bank::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	MC68681Bank::Load
 *
 * DESCRIPTION:	Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68681Bank::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	MC68681Bank::Dispose
 *
 * DESCRIPTION:	Standard dispose function.  Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void MC68681Bank::Dispose (void)
{
}


// ---------------------------------------------------------------------------
//		 MC68681Bank::GetLong
// ---------------------------------------------------------------------------

uae_u32 MC68681Bank::GetLong (uaecptr iAddress)
{
	InvalidAccess (iAddress, 4, true);

	return 0;
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::GetWord
// ---------------------------------------------------------------------------

uae_u32 MC68681Bank::GetWord (uaecptr iAddress)
{
	InvalidAccess (iAddress, 2, true);

	return 0;
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::GetByte
// ---------------------------------------------------------------------------

uae_u32 MC68681Bank::GetByte (uaecptr iAddress)
{
	return ((uae_u8*) &gDuart)[iAddress - kMemoryStart];
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::SetLong
// ---------------------------------------------------------------------------

void MC68681Bank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
	UNUSED_PARAM(iLongValue)

	InvalidAccess (iAddress, 4, false);
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::SetWord
// ---------------------------------------------------------------------------

void MC68681Bank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
	UNUSED_PARAM(iWordValue)

	InvalidAccess (iAddress, 2, false);
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::SetByte
// ---------------------------------------------------------------------------

void MC68681Bank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
	((uae_u8*) &gDuart)[iAddress - kMemoryStart] = iByteValue;
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::ValidAddress
// ---------------------------------------------------------------------------

int MC68681Bank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	UNUSED_PARAM(iSize)

	const uaecptr	offset = iAddress - kMemoryStart;

	int	result = (offset >= 0) && (offset < sizeof (gDuart));

	assert (result);

	return result;
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* MC68681Bank::GetRealAddress (uaecptr iAddress)
{
	return &(((uae_u8*) &gDuart)[iAddress - kMemoryStart]);
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::InvalidAccess
// ---------------------------------------------------------------------------

void MC68681Bank::InvalidAccess (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Software::BusError ();
}

// ---------------------------------------------------------------------------
//		 MC68681Bank::PreventedAccess
// ---------------------------------------------------------------------------

void MC68681Bank::PreventedAccess (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Software::BusError ();
}


