/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl.functions;
import com.kvisco.xsl.*;
import com.kvisco.util.*;
import org.w3c.dom.Node;
import com.kvisco.xml.XMLUtil;

/**
 * A class that represents the XSL local-part(NodeSetExpr) function call
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class XMLNamesFunctionCall extends FunctionCall {
    
    public static final short LOCAL_PART   = 1;
    public static final short NAME         = 2;
    public static final short NAMESPACE    = 3;
    
    private short fType = NAME;
    
    /**
     * Creates a new XMLNames Function Call
    **/
    public XMLNamesFunctionCall() {
        super(Names.NAME_FN);
    } //-- XMLNamesFunctionCall
    
    /**
     * Creates a new XMLNames Function Call
    **/
    public XMLNamesFunctionCall(short type) {
        super(getFunctionName(type));
        this.fType = type;
    } //-- XMLNamesFunctionCall
    
    /**
     * Evalutes this FunctionCall using the given Node as
     * the context. This method will always return a StringResult
     * @exception InvalidExprException when the expression being
     * evaluated has an error.
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        QuickStack stack = ps.getNodeSetStack();
        NodeSet nodeSet = null;
        if (stack.empty()) {
            nodeSet = new NodeSet();
            nodeSet.add(context);
        }
        else nodeSet = (NodeSet)stack.peek();

        String nodeName = null;
        List params = super.getParameterList();
        if (params.size() == 1) {
            Expr expr = (Expr)params.get(0);
            ExprResult result = expr.evaluate(context, ps);
            if (result.getResultType() == ExprResult.NODE_SET) {
                NodeSet nodes = (NodeSet)result;
                if (nodes.size() > 0) 
                    nodeName = nodes.get(0).getNodeName();
            }
        }
        else nodeName = context.getNodeName();
        
        switch(fType) {
            case LOCAL_PART:
                return new StringResult(XMLUtil.getLocalPart(nodeName));
            case NAMESPACE:
                return new StringResult(XMLUtil.getNameSpace(nodeName));
            default:
                return new StringResult(nodeName);
        }
    } //-- evaluate
    
    /**
     * 
    **/
    private static String getFunctionName(short type) {
        switch(type) {
            case LOCAL_PART:
                return Names.LOCAL_PART_FN;
            case NAMESPACE:
                return Names.NAMESPACE_FN;
            default:
                return Names.NAME_FN;
        }
    } //-- getFunctionName
    
} //-- XMLNamesFunctionCall