/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl.functions;
import com.kvisco.xsl.*;
import com.kvisco.util.*;
import org.w3c.dom.Node;

/**
 * A class that represents the XSL Sum(Expr) function call
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class SumFunctionCall extends FunctionCall {
    
    /**
     * Creates a new SumFunctionCall
    **/
    public SumFunctionCall() {
        super(Names.SUM_FN);
    } //-- SumFunctionCall
    
    /**
     * Evalutes this FunctionCall using the given Node as
     * the context. This method will always return a NumberResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        List params = super.getParameterList();
        
        if (params.size() != 1)
            throw new InvalidExprException(INVALID_NUMBER_PARAMS+this);
            
        Expr expr = (Expr) params.get(0);
        if (expr != null)  {
            ExprResult exprResult = expr.evaluate(context, ps);
            
            if (exprResult.getResultType() != ExprResult.NODE_SET) {
                
                throw new InvalidExprException(INVALID_RESULT+this+
                    " a node-set was expected");
            }
            NodeSet nodeSet = (NodeSet)exprResult;
            return new NumberResult(computeSum(nodeSet));
        }
        return new NumberResult(Double.NaN);
    } //-- evaluate
    
    public double computeSum(NodeSet nodeSet) {
        double result = 0;
        for (int i = 0; i < nodeSet.size(); i++) {
            result += getNodeValue(nodeSet.get(i));
        }
        return result;
    } //-- computeSum
    
    private double getNodeValue(Node node) {
        String strValue = XSLObject.getNodeValue(node);
        double result = 0;
        try {
            return Double.valueOf(strValue).doubleValue();
        }
        catch(NumberFormatException nfe) {};
        return result;
        
    } //-- getNodeValue
    
} //-- SumFunctionCall