/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl.functions;
import com.kvisco.xsl.*;
import com.kvisco.util.*;
import org.w3c.dom.Node;

/**
 * A class that represents the following FunctionCalls:<BR /> 
 * number(); floor(); round(); ceiling();
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class NumberFunctionCall extends FunctionCall {
    
    public static final short NUMBER   = 0;
    public static final short ROUND    = 1;
    public static final short FLOOR    = 2;
    public static final short CEILING  = 3;
    
    short functionType = NUMBER;
    
    
    /**
     * Creates a new NumberFunctionCall
    **/
    public NumberFunctionCall() {
        super(Names.NUMBER_FN);
    } //-- NumberFunctionCall
    
    /**
     * Creates a new NumberFunctionCall using the given type
    **/
    public NumberFunctionCall(short type) {
        super(NumberFunctionCall.getFunctionName(type));
        this.functionType = type;
    } //-- NumberFunctionCall
    
    /**
     * Evalutes this FunctionCall using the given Node as
     * the context. This method will always return a NumberResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        List params = super.getParameterList();
        
        if (params.size() != 1)
            throw new InvalidExprException(INVALID_NUMBER_PARAMS+this);
            
        Expr expr = (Expr) params.get(0);
        if (expr != null)  {
            
            double value = expr.evaluate(context, ps).numberValue();
            
            switch (functionType) {
                case FLOOR:
                    value = Math.floor(value);
                    break;
                case ROUND:
                    value = Math.round(value);
                    break;
                case CEILING:
                    value = Math.ceil(value);
                    break;
                default: //-- NUMBER
                    break;
            }
            return new NumberResult(value);
        }
        return new NumberResult(Double.NaN);
    } //-- evaluate
    
    private static String getFunctionName(short type) {
        switch (type) {
            case FLOOR:
                return Names.FLOOR_FN;
            case ROUND:
                return Names.ROUND_FN;
            case CEILING:
                return Names.CEILING_FN;
            default:
                return Names.NUMBER_FN;
        }
    } //-- getFunctionName
    
} //-- NumberFunctionCall