/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl.functions;
import com.kvisco.xsl.*;
import com.kvisco.util.List;
import org.w3c.dom.Node;
import java.text.DecimalFormat;

/**
 * A implementation of the "format-number()" function call.
 * <BR />
 * <I>method signature:</I><BR />
 *   String format-number(number, string, string?) <BR />
 * The 3rd argument which is optional (string?) is currently not
 * handled.
 * <BR />
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class FormatNumber extends FunctionCall {
    
    /**
     * Creates a new FormatNumber FunctionCall
    **/
    public FormatNumber() {
        super(Names.FORMAT_NUMBER_FN);
    } //-- FormatNumber
    
    /**
     * Evalutes this FunctionCall using the given Node as
     * the context. This method will always return a StringResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        List params = getParameterList();
        
        int nparams = params.size();
        if ((nparams < 2) || (nparams > 3))
            throw new InvalidExprException(INVALID_NUMBER_PARAMS+this);
            
        ExprResult exprResult = ((Expr)params.get(0)).evaluate(context, ps);
        double dbl = exprResult.numberValue();
        
        if (Double.isNaN(dbl)) return new StringResult("NaN");
        if (Double.isInfinite(dbl))  return new StringResult("Infinite");
        
        //-- get number format
        exprResult = ((Expr)params.get(1)).evaluate(context, ps);
        String format = StringExpr.toStringResult(exprResult).toString();
        
        String locale = null;
        //-- handle locale
        if (nparams == 3) {
            exprResult = ((Expr)params.get(2)).evaluate(context, ps);
            locale = StringExpr.toStringResult(exprResult).toString();
        }
        DecimalFormat df = new DecimalFormat(format);
        
        return new StringResult(df.format(dbl));
        
    } //-- evaluate
    
} //-- FormatNumber

