/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.*;

import com.kvisco.util.QuickStack;

import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * This class represents a SelectExpr
 * <PRE>
 * [1] SelectExpr ::= UnionExpr
 * </PRE>
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
class SelectExpr implements Expr {
    
    /**
     * The UnionExpr for this SelectExpr
    **/
    private UnionExpr unionExpr = null;
    
      //----------------/
     //- Constructors -/
    //----------------/

    /**
     * Creates a new SelectExpr. This SelectExpr will select the 
     * empty node set by default.
    **/
    public SelectExpr() {
        super();
    } //-- SelectExpr


    public String toString() {
        if (this.unionExpr != null) return unionExpr.toString();
        else return "";
    } //-- toString
    
    
    public short getExprType() {
        return Expr.NODE_SET;
    } //-- getExprType
    
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        return selectNodes(context, ps);
    } //-- evaluate
    
      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    protected void getUnionExpr(UnionExpr unionExpr) {
        this.unionExpr = unionExpr;
    } //-- setUnionExpr
    
    protected NodeSet selectNodes(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        if (unionExpr == null) return new NodeSet(0);
        
        NodeSet nodes = new NodeSet();

        // Select all nodes for each PathExpr in the UnionExpr
        NodeSet tmpNodes = null;
        PathExpr pathExpr = null;
        for (int i = 0; i < unionExpr.size(); i++) {
            pathExpr = (PathExpr)unionExpr.get(i);
            tmpNodes = (NodeSet)pathExpr.evaluate(context, ps);
            for (int j = 0; j < tmpNodes.size(); j++) {
                nodes.add(tmpNodes.get(j));
            }
            tmpNodes.clear();
        }
        
        nodes = reorder(nodes, ps);
        
        return nodes;
    } // -- selectNodes
    
    
    protected void setUnionExpr(UnionExpr unionExpr) {
        this.unionExpr = unionExpr;
    } //-- setUnionExpr

      //-------------------/
     //- Private Methods -/
    //-------------------/

    
    /**
     * Compares the document orders.
    **/
    private int compareOrders(int[] orderA, int[] orderB) {
        int idx = 0;
        while ((idx < orderA.length) && (idx < orderB.length)) {
            if (orderA[idx] < orderB[idx]) return -1;
            else if (orderB[idx] < orderA[idx]) return 1;
            ++idx;
        }
        if (orderA.length < orderB.length) return -1;
        else if (orderB.length < orderA.length) return 1;
        else return 0;
    } //-- compareOrders
    
    /**
     * Reorders the array of Nodes so that the nodes are in Document Order.
    **/
    private NodeSet reorder(NodeSet nodes, ProcessorState ps) {
        if (nodes.size() < 2) return nodes;
        
        NodeSet sorted = new NodeSet(nodes.size());
        sorted.add(nodes.get(0));
        
        for (int i = 1; i < nodes.size(); i++) {
            Node node = nodes.get(i);
            
            int[] order0 = ps.getDocumentOrder(node);
            for (int j = 0; j < sorted.size(); j++) {
                int[] order1 = ps.getDocumentOrder(sorted.get(j));
                if (compareOrders(order0, order1) < 0) {
                    sorted.add(j, node);
                    break;
                }
                else if (j == sorted.size()-1) {
                    sorted.add(node);
                    break;
                }
            }
        }
        return sorted;
    } //-- reorder
        
} //-- SelectExpr
