/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.xsl;

import java.util.Hashtable;

import org.w3c.dom.*;

/**
 * Represents an EqualityExpr
 * @author Keith Visco (kvisco@ziplink.net)
**/
public class EqualityExpr implements Expr {

    public static final short EQUAL            = 0;
    public static final short LESS_THAN        = 1;
    public static final short GREATER_THAN     = 2;
    public static final short LT_OR_EQUAL      = 4;
    public static final short GT_OR_EQUAL      = 5;
    public static final short NOT_EQUAL        = 6;
    
    
    private Expr leftExpr = null;
    private Expr rightExpr = null;
    
    private short compareOp = EQUAL;
    
    private static Hashtable relationalOps = null;
    static {
        relationalOps = new Hashtable(5);
        relationalOps.put(Names.EQUALS_OP,       new Short(EQUAL));
        relationalOps.put(Names.LESS_THAN_OP,    new Short(LESS_THAN));
        relationalOps.put(Names.GREATER_THAN_OP, new Short(GREATER_THAN));
        relationalOps.put(Names.LT_OR_EQUAL_OP,  new Short(LT_OR_EQUAL));
        relationalOps.put(Names.GT_OR_EQUAL_OP,  new Short(GT_OR_EQUAL));
        relationalOps.put(Names.NOT_EQUAL_OP,    new Short(NOT_EQUAL));
    }
    
      //---------------/
     //- Constructor -/
    //---------------/
    
    /**
     * Creates a new EqualityExpr using the default operator
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this EqualityExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this EqualityExpr
     * <BR><B>Note:</B> the default operator is EqualityExpr.EQUALS
    **/
    public EqualityExpr(Expr leftSideExpr, Expr rightSideExpr) {
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
    } //-- EqualityExpr 
    
    /**
     * Creates a new EqualityExpr
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this EqualityExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this EqualityExpr
     * @param compareOp the comparison operator for this EqualityExpr
     * @exception InvalidExprException when the comparison operator is 
     * invalid 
    **/
    public EqualityExpr
        (Expr leftSideExpr, Expr rightSideExpr, short compareOp) 
        throws InvalidExprException
    {
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
        if ((compareOp < 0) || (compareOp > relationalOps.size()))
            throw new InvalidExprException("invalid operator for relational expression");
        this.compareOp = compareOp;
    } //-- EqualityExpr 
    
    /**
     * Creates a new EqualityExpr
     * @param leftSideExpr the Expr that is to be evaluated as
     * the left side of this EqualityExpr
     * @param rightSideExpr the Expr that is to be evaluated as
     * the right side of this EqualityExpr
     * @param compareOp the comparison operator for this EqualityExpr
     * @exception InvalidExprException when the comparison operator is 
     * invalid 
    **/
    public EqualityExpr
        (Expr leftSideExpr, Expr rightSideExpr, String compareOp) 
        throws InvalidExprException
    {
        this.compareOp = -1;
        if (compareOp != null) {
            Short sval = (Short)relationalOps.get(compareOp);
            if (sval != null) this.compareOp = sval.shortValue();
        }
        if ((this.compareOp < 0) || (this.compareOp > relationalOps.size()))
            throw new InvalidExprException("invalid operator for relational expression");
        leftExpr = leftSideExpr;
        rightExpr = rightSideExpr;
    } //-- EqualityExpr
        
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Returns the type of Expr this Expr represents
     * @return the type of Expr this Expr represents
    **/
    public short getExprType() {
        return Expr.BOOLEAN;
    } //-- getExprType
    
    /**
     * Evaluates this Expr using the given context Node and ProcessorState
     * @param context the current context Node
     * @param ps the ProcessorState that contains the current processing 
     * environment
     * @return the ExprResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        if ((leftExpr == null) || (rightExpr == null))
            return BooleanResult.FALSE_RESULT;

        ExprResult rResult = null;
        ExprResult lResult = null;
            
        lResult = leftExpr.evaluate(context, ps);
        rResult = rightExpr.evaluate(context, ps);
        short lType = lResult.getResultType();
        short rType = rResult.getResultType();
        
        boolean evalResult = false;
        
        switch (compareOp) {
            
            case EQUAL:
                //-- See section 6.2.3 Booleans of XSL WD 19990421
                if (lType != rType) {
                    
                    // Check for Boolean First
                    if  ((lType == ExprResult.BOOLEAN) || 
                        (rType == ExprResult.BOOLEAN)) 
                    {
                        evalResult =
                            (rResult.booleanValue() == lResult.booleanValue());
                    }
                    // Then Check for Numbers
                    else if ((lType == ExprResult.NUMBER) ||
                            (rType == ExprResult.NUMBER) )
                    {
                        evalResult =
                            (rResult.numberValue() == lResult.numberValue());
                    }
                    // otherwise compare as Strings
                    else {
                        rResult = StringExpr.toStringResult(rResult);
                        lResult = StringExpr.toStringResult(lResult);
                        evalResult = lResult.equals(rResult);
                    }
                }
                else if (lType == ExprResult.NODE_SET) {
                    rResult = StringExpr.toStringResult(rResult);
                    lResult = StringExpr.toStringResult(lResult);
                    evalResult = lResult.equals(rResult);
                }
                else evalResult = lResult.equals(rResult);
                
                break;
            case LESS_THAN:
                evalResult = (lResult.numberValue() < rResult.numberValue());
                break;
            case GREATER_THAN:
                evalResult = (lResult.numberValue() > rResult.numberValue());
                break;
            case LT_OR_EQUAL:
                evalResult = (lResult.numberValue() >= rResult.numberValue());
                break;
            case GT_OR_EQUAL:
                evalResult = (lResult.numberValue() >= rResult.numberValue());
                break;
            default:
                evalResult = false;
        } //-- switch
    
        return new BooleanResult(evalResult);
    } //-- evaluate
    
    public static boolean isRelationalOperator(String operator) {
        if (operator == null) return false;
        return (relationalOps.get(operator) != null);
    } //-- isRelationalOperator
    
    /**
     * Returns the String representation of this EqualityExpr
     * @return the String representation of this EqualityExpr
    **/
    public String toString() {
        
        
        StringBuffer sb = new StringBuffer();
        if (leftExpr != null)
            sb.append(leftExpr.toString());
        else
            sb.append("null");
        sb.append(" ");
        
        switch (compareOp) {
            case EQUAL:
                sb.append(Names.EQUALS_OP);
                break;
            case LESS_THAN:
                sb.append(Names.LESS_THAN_OP);
                break;
            case GREATER_THAN:
                sb.append(Names.GREATER_THAN_OP);
                break;
            case LT_OR_EQUAL:
                sb.append(Names.LT_OR_EQUAL_OP);
                break;
            case GT_OR_EQUAL:
                sb.append(Names.GT_OR_EQUAL_OP);
                break;
            case NOT_EQUAL:
                sb.append(Names.NOT_EQUAL_OP);
                break;
            default:
                break;
        }
        
        sb.append(" ");
        if (rightExpr != null)
            sb.append(rightExpr.toString());
        else sb.append("null");
        
        return sb.toString();
    } //-- toString
} //-- Expr