/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */

package com.kvisco.xsl;

import org.w3c.dom.*;

/**
 * Axis Identifier Functions
 * @author <a href="kvisco@ziplink.net">Keith Visco</a>
**/
public class AxisIdentifier {

    /**
     * Selects all the Ancestors of the context node that match
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the ancestors of the context node
     * that match the given MatchExpr
    **/
    public static NodeSet fromAncestors
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        Node parent = null;
        if (context != null) parent = context.getParentNode();
        return fromAncestorsOrSelf(matchExpr, parent, ps);
    } //-- fromAncestors
    
    /**
     * Selects all the Ancestors of the context node, including the
     * context node that match the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the ancestors of the context node
     * and including the context node, that match the given MatchExpr
    **/
    public static NodeSet fromAncestorsOrSelf
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        Node parent = context;
        while (parent != null) {
            if (matchExpr.matches(parent, context, ps)) {
                nodeSet.add(parent);
            }
            parent = parent.getParentNode();
        }
        return nodeSet;
        
    } //-- fromAncestorsOrSelf
    
    /**
     * Selects all the attributes of the context node that match
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the attributes of the context node
     * that match the given MatchExpr
    **/
    
    public static NodeSet fromAttributes
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        if (context.getNodeType() != Node.ELEMENT_NODE) return nodeSet;
        NamedNodeMap atts = ((Element)context).getAttributes();
        if (atts != null) {
            for (int i = 0; i < atts.getLength(); i++) {
                if (matchExpr.matches(atts.item(i), context, ps))
                    nodeSet.add(atts.item(i));
            }
        }
        return nodeSet;
    } //-- fromAttributes
     
    /**
     * Selects all the children of the context node that match
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the children of the context node
     * that match the given MatchExpr
    **/
    public static NodeSet fromChildren 
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        NodeSet nodeSet = new NodeSet();
        NodeList nodeList = context.getChildNodes();
        for (int i = 0; i < nodeList.getLength(); i++) {
            Node node = nodeList.item(i);
            if (matchExpr.matches(node, context, ps))
                nodeSet.add(node);
        }
        return nodeSet;
    } //-- fromChildren


    /**
     * Selects all the descendants of the context node that match
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the descendants of the context node
     * that match the given MatchExpr
     * <BR />
     * <B>Note</B> this could be an expensive operation
    **/
    public static NodeSet fromDescendants
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        
        if (context == null) return nodeSet;
        
        NodeList nl = context.getChildNodes();
        
        for (int i = 0; i < nl.getLength(); i++) {
            Node child = nl.item(i);
            if (matchExpr.matches(child, context, ps))
                nodeSet.add(child);
            
            //-- check childs descendants
            if (child.hasChildNodes()) {
                NodeSet temp = fromDescendants(matchExpr, child, ps);
                nodeSet.add(temp);
            }
        }
        return nodeSet;
        
    } //-- fromDecendants

    /**
     * Selects all the descendants of the context node, including the
     * context node, that match the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the descendants of the context node,
     * including the context node, that match the given MatchExpr
     * <BR />
     * <B>Note</B> this could be an expensive operation
    **/
    public static NodeSet fromDescendantsOrSelf
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        
        if (context == null) return new NodeSet();
        NodeSet nodeSet = fromDescendants(matchExpr, context, ps);
        if (matchExpr.matches(context, context, ps))
            nodeSet.add(0, context);
        
        return nodeSet;
        
    } //-- fromDecendantsOrSelf
    
    /**
     * Selects all the following nodes of the context node that match
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the following nodes of the 
     * context node that match the given MatchExpr
     * <BR />
     * <B>Note</B> this could be an expensive operation
    **/
    public static NodeSet fromFollowing
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        
        if (context == null) return nodeSet;
        
        Node node = context.getNextSibling();
        while (node != null) {
            if (matchExpr.matches(node, context, ps))
                nodeSet.add(node);
                
            if (node.hasChildNodes()) {
                NodeSet temp = fromDescendants(matchExpr, node, ps);
                nodeSet.add(temp);
            }
            
            Node tmpNode = node.getNextSibling();
            if (tmpNode == null) {
                node = node.getParentNode();
                if ((node != null) && 
                    (node.getNodeType() != Node.DOCUMENT_NODE))
                    node = node.getNextSibling();
            }
            else node = tmpNode;
        }
        return nodeSet;
    } //-- fromFollowing
    
    /**
     * Selects all the following siblings of the context node that match
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the following siblings of the 
     * context node that match the given MatchExpr
    **/
    public static NodeSet fromFollowingSiblings
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        
        if (context == null) return nodeSet;
        
        Node node = context.getNextSibling();
        while (node != null) {
            if (matchExpr.matches(node, context, ps))
                nodeSet.add(node);
            node = node.getNextSibling();
        }
        return nodeSet;
    } //-- fromFollowingSiblings
    
    /**
     * Selects the parent of the context node if it matches
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing the parent of the context node
     * if it matches the given MatchExpr
    **/
    public static NodeSet fromParent
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        Node parent = null;
        if (context.getNodeType() == Node.ATTRIBUTE_NODE)
            parent = ps.findParent((Attr)context);
        else
            parent = context.getParentNode();
            
        if (matchExpr.matches(parent, context, ps))
            nodeSet.add(parent);
        return nodeSet;
    } //-- fromParent

    
    /**
     * Selects all the preceding siblings, in reverse document order,
     * of the context node that match the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the preceding siblings of the 
     * context node that match the given MatchExpr
    **/
    public static NodeSet fromPreceding
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
    
        NodeSet nodeSet = new NodeSet();
        
        if (context == null) return nodeSet;
        
        Node node = context.getPreviousSibling();
        if (node == null) node = context.getParentNode();
        
        while (node != null) {
            if (matchExpr.matches(node, context, ps))
                nodeSet.add(node);
            
            Node temp = node.getPreviousSibling();
            if (temp == null) node = node.getParentNode();
            else node = temp;
        }
        return nodeSet;
    } //-- fromPreceding
    
    /**
     * Selects all the preceding nodes, in reverse document order,
     * of the context node that match
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing all the preceding nodes of the 
     * context node (in reverse document order)
     * that match the given MatchExpr
    **/
    public static NodeSet fromPrecedingSiblings
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
    
        NodeSet nodeSet = new NodeSet();
        
        if (context == null) return nodeSet;
        
        Node node = context.getPreviousSibling();
        
        while (node != null) {
            if (matchExpr.matches(node, context, ps))
                nodeSet.add(node);
            
            node = node.getPreviousSibling();
        }
        return nodeSet;
    } //-- fromPrecedingSiblings
    
    
    /**
     * Selects the the context node if it matches
     * the given MatchExpr
     * @param the MatchExpr to evaluate
     * @param context the Node to start matching from
     * @param ps the current ProcessorState
     * @return a NodeSet containing the context node
     * if it matches the given MatchExpr
    **/
    public static NodeSet fromSelf
        (MatchExpr matchExpr, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        if (matchExpr.matches(context, context, ps))
            nodeSet.add(context);
        return nodeSet;
    } //-- fromSelf
    
} //-- AxisIdentifier
