/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.xml;

import org.w3c.dom.*;

import java.io.PrintWriter;
import java.util.Hashtable;

/**
 * A class for printing XML nodes where the output document is
 * HTML. The result document will not be a well-formed XML document.
 * Certain elements are printed with no ending tags
 *<PRE>
 *  The following elements will have no ending tag
 *  BR, HR, IMG, LI, and P (if it has no child nodes)
 *</PRE>
 * @author Keith Visco (kvisco@ziplink.net)
**/
public class HTMLPrinter extends XMLPrinter {

    public static final String BREAK             = "BR";
    public static final String PARAGRAPH         = "P";
    public static final String HORIZONTAL_RULE   = "HR";
    public static final String IMAGE             = "IMG";
    public static final String LIST_ITEM         = "LI";
    public static final String META              = "META";
    public static final String SCRIPT            = "SCRIPT";
    // casa: other empty tags
    public static final String AREA              = "AREA";
    public static final String BASE              = "BASE";
    public static final String BASEFONT          = "BASEFONT";
    public static final String FRAME             = "FRAME";
    public static final String INPUT             = "INPUT";
    public static final String ISINDEX           = "ISINDEX";
    public static final String LINK              = "LINK";
    public static final String PARAM             = "PARAM";

    //private final char indentChar = ' ';

    private PrintWriter pw = null;
    
    private Hashtable specialElements = null;
    
    private boolean useFormat = false;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new HTML Printer using the given PrintWriter
     * for output
     * @param writer the PrintWriter to use for output
    **/
    public HTMLPrinter(PrintWriter writer) {
        this(writer, XMLPrinter.DEFAULT_INDENT);
    } //-- HTMLPrinter

    /**
     * Creates a new HTML Printer using the given PrintWriter
     * for output, and nodes are indenting using the specified
     * indent size
     * @param writer the PrintWriter to use for output
     * @param indent the number of spaces to indent
    **/
    public HTMLPrinter (PrintWriter writer, int indent) {
        super(writer,indent);
        super.setUnescapeCDATA(false);
        super.setUseEmptyElementShorthand(false);
        this.pw = writer;
        specialElements = getSpecialElements();
    } //-- HTMLPrinter

   
    public void setUseFormat(boolean useFormat) {
        super.setUseFormat(useFormat);
        this.useFormat = useFormat;
    } //-- setUseFormat
    
      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    /**
     * Overrides XMLPrinter#print
    **/
    protected boolean print(Node node, String currentIndent) {

        if (node == null) return false;
        
        switch(node.getNodeType()) {
            
            case Node.DOCUMENT_NODE:
                Document doc = (Document)node;
                NodeList nl = doc.getChildNodes();
                for (int i = 0; i < nl.getLength(); i++) {
                    print(nl.item(i),currentIndent);
                }
                break;
            case Node.ELEMENT_NODE:
                Element element = (Element) node;
                String tagName = element.getNodeName().toUpperCase();
                if (element.hasChildNodes() || 
                    (specialElements.get(tagName) == null)) 
                {
                    return super.print(node, currentIndent);
                }
                else { //-- handle special HTML elements

                    super.printStartTag(element.getNodeName(),
                        element.getAttributes());
		            if (useFormat) {
                        Node sibling = node.getNextSibling();
                        if ((sibling == null) ||
                            (sibling.getNodeType() != Node.TEXT_NODE))
                        {
                            pw.println();
                            return true;
                        }
                    }
                }
		        break;
		    case Node.CDATA_SECTION_NODE:
		        //-- added based on suggestions from Chris Bennett
		        Node parent = node.getParentNode();
		        if (parent.getNodeType() == Node.ELEMENT_NODE) {
		            if (parent.getNodeName().equalsIgnoreCase(SCRIPT)) {
		                String data = ((CDATASection)node).getData();
		                super.printUTF8Chars(data.toCharArray());
		                return false;
		            }
		        }
		        super.print(node, currentIndent);
		        break;
		    default:
		        return super.print(node,currentIndent);
		} //-- switch
		return false;
    } //-- print
    
    private static Hashtable getSpecialElements() {
        Hashtable ht = new Hashtable();
        
        Object notNull = new Boolean(true);
        
        ht.put(BREAK,           notNull);
        ht.put(HORIZONTAL_RULE, notNull);
        ht.put(IMAGE,           notNull);
        ht.put(META,            notNull);
        ht.put(LIST_ITEM,       notNull);
        ht.put(PARAGRAPH,       notNull);
	    // casa: other empty tags
	    ht.put(AREA,            notNull);
	    ht.put(BASE,            notNull);
	    ht.put(BASEFONT,        notNull);
	    ht.put(FRAME,           notNull);
	    ht.put(INPUT,           notNull);
	    ht.put(ISINDEX,         notNull);
	    ht.put(LINK,            notNull);
	    ht.put(PARAM,           notNull);
        
        return ht;
    } //getSpecialElements
    
} //-- HTMLPrinter