#include <unistd.h>
#include <cstdlib>
#include <strstream>
#include <gtk--/main.h>
#include <gtk--/box.h>
#include <gtk--/button.h>
#include "AboutWindow.h"
#include "CWGlob.h"
#include "CWText.h"
#include "CWWindow.h"

////////////////////////////////////////////////////////////////////////
// public
////////////////////////////////////////////////////////////////////////

CWWindow::CWWindow() : Gtk_Window()
{
  set_border_width(5);

  _vbox = new Gtk_VBox();
  _vbox->set_spacing(5);
  add(Gtk_ObjectHandle<Gtk_VBox>(_vbox));
  _vbox->show();

  _text = new CWText(_vbox);
  _text->set_editable(false);
  _text->set_word_wrap(false);
  _text->set_line_wrap(true);
  _text->show();

  _buttonBox = new Gtk_HBox();
  _buttonBox->set_spacing(5);
  _vbox->pack_start(Gtk_ObjectHandle<Gtk_HBox>(_buttonBox), false, false);
  _buttonBox->show();

  _aboutButton = new Gtk_Button("About");
  _buttonBox->pack_start(Gtk_ObjectHandle<Gtk_Button>(_aboutButton));
  connect_to_method(_aboutButton->clicked, this, &CWWindow::AboutCB);
  _aboutButton->show();

  _rewButton = new Gtk_Button("<<");
  _buttonBox->pack_start(Gtk_ObjectHandle<Gtk_Button>(_rewButton));
  connect_to_method(_rewButton->clicked, this, &CWWindow::RewCB);
  _rewButton->show();

  _prevButton = new Gtk_Button("<");
  _buttonBox->pack_start(Gtk_ObjectHandle<Gtk_Button>(_prevButton));
  connect_to_method(_prevButton->clicked, this, &CWWindow::PrevCB);
  _prevButton->show();

  _nextButton = new Gtk_Button(">");
  _buttonBox->pack_start(Gtk_ObjectHandle<Gtk_Button>(_nextButton));
  connect_to_method(_nextButton->clicked, this, &CWWindow::NextCB);
  _nextButton->show();

  _ffButton = new Gtk_Button(">>");
  _buttonBox->pack_start(Gtk_ObjectHandle<Gtk_Button>(_ffButton));
  connect_to_method(_ffButton->clicked, this, &CWWindow::FfCB);
  _ffButton->show();

  _exitButton = new Gtk_Button("Exit");
  _buttonBox->pack_start(Gtk_ObjectHandle<Gtk_Button>(_exitButton));
  connect_to_method(_exitButton->clicked, this, &CWWindow::ExitCB);
  _exitButton->show();

  // Hard code this for now.
  _baseStr = "log.";

  GetCraftyLogPath();

  // Because files can be loaded with "waitUntilAtEnd == true",
  // there can be a delay between when this class asks the CWText
  // class to load a file and when it is actually loaded.
  connect_to_method(_text->FileLoaded, this, &CWWindow::FileLoadedCB);

  // It's difficult to insert text prior to the Gtk_Text widget being
  // realized in the normal course.  So, just wait until after
  // everything has popped up.
  connect_to_method(Gtk_Main::timeout(0), this, &CWWindow::InitialLoadFileCB);

  // Set up a timeout to make periodic updates.
  _tick = connect_to_method(Gtk_Main::timeout(1000), this, &CWWindow::TickCB); 
}
  
CWWindow::~CWWindow()
{
  // empty
}


////////////////////////////////////////////////////////////////////////
// protected
////////////////////////////////////////////////////////////////////////

void CWWindow::AboutCB()
{
  AboutWindow* tmp = new AboutWindow;
  tmp->show();
}


void CWWindow::RewCB()
{
  if( _n <= _nMin )
    {
      // Should also ring the bell.
      return;
    }

  _n = _nMin;
  string fileName = CreateFileName(_n);
  while( access(fileName.c_str(), F_OK) != 0 && ++_n <= _nMax )
    fileName = CreateFileName(_n);

  if( _n > _nMax )
    RugPulledOutFromUnder();
  else
    {
      _nMin = _n;
      LoadFile(fileName);
    }
}

void CWWindow::PrevCB()
{
  if( _n <= _nMin )
    {
      // Should also ring the bell.
      return;
    }

  string fileName = CreateFileName(--_n);
  while( access(fileName.c_str(), F_OK) != 0 && --_n >= _nMin )
    fileName = CreateFileName(_n);

  if( _n < _nMin )
    RugPulledOutFromUnder();
  else
    LoadFile(fileName);
}

void CWWindow::NextCB()
{
  if( _n >= _nMax )
    {
      // Should also ring the bell.
      return;
    }

  string fileName = CreateFileName(++_n);
  while( access(fileName.c_str(), F_OK) != 0 && ++_n <= _nMax )
    fileName = CreateFileName(_n);

  if( _n > _nMax )
    RugPulledOutFromUnder();
  else
    LoadFile(fileName);
}


void CWWindow::FfCB()
{
  if( _n > _nMax )
    return;

  _n = _nMax;
  string fileName = CreateFileName(_n);
  while( access(fileName.c_str(), F_OK) != 0 && --_n >= _nMin )
    fileName = CreateFileName(_n);

  if( _n < _nMin )
    RugPulledOutFromUnder();
  else
    {
      _nMax = _n;
      LoadFile(fileName);
    }
}


void CWWindow::ExitCB()
{
  Gtk_Main::instance()->quit();
}


void CWWindow::FileLoadedCB(const string& fileName)
{
  string tmp = "Crafty Watcher:  " + GetBaseName(fileName);
  set_title(tmp.c_str());
  if( GetFileNumber(fileName) == _nMax )
    _text->GotoEnd();
}


gint CWWindow::delete_event_impl(GdkEventAny* p1)
{
  Gtk_Main::instance()->quit(); 
  return 0;
}


////////////////////////////////////////////////////////////////////////
// private
////////////////////////////////////////////////////////////////////////

void CWWindow::GetCraftyLogPath()
{
  char* tmpStr = getenv("CRAFTY_LOG_PATH");
  if( tmpStr )
    _craftyLogPath = tmpStr;
  else
    _craftyLogPath = ".";
}


void CWWindow::GlobForMaxMin()
{
  // Set _nMax=-1 and _nMin=-1 if no regular file matches the glob
  // pattern.  Otherwise, set _nMax and _nMin to the maximum and
  // minimum respectively.
  try
    {
      _globPattern = _craftyLogPath + "/" + _baseStr + "*";
      CWGlob glob(_globPattern, true);

      _nMax = 0;
      _nMin = 0;
      if( glob.size() > 0 )
        {
          _nMax = GetFileNumber(glob[0]);
          _nMin = _nMax;
          for( int i = 1 ; i < glob.size() ; i++ )
            {
              int n = GetFileNumber(glob[i]);
              if( n == -1 )
                continue;
              _nMax = max(_nMax, n);
              _nMin = min(_nMin, n);
            }
        }
    }
  catch(int globErr)
    {
      // You can't call Gtk_Main::quit() yet because you haven't
      // yet called Gtk_Main::run().
      switch( globErr )
        {
        case GLOB_NOSPACE:
          cerr << "Error: unable to allocate memory for glob." << endl;
          exit(1);
          break;
      
        case GLOB_ABORTED:
          cerr << "Error: unable to read glob pattern." << endl;
          exit(1);
          break;
        };
    }
}


int CWWindow::GetFileNumber(const string& fileName)
{
  // Because "fileName" will be the results of a "glob", the invariant
  // holds that all fileNames will start with the value of baseStr.

  // Return -1 if the fileName points to a directory.  I have set up
  // glob to return a trailing '/' if the file is a directory.
  string::const_reverse_iterator rtmp = fileName.rbegin();
  if( *rtmp == '/' )
    return -1;

  istrstream istrm(fileName.c_str() + _globPattern.size() - 1 );
  int n;
  if( !(istrm >> n) )
    return -1;

  return n;
}


gint CWWindow::InitialLoadFileCB()
{
  GlobForMaxMin();
  _n = _nMax;

  // Prepare the name of the "new" file that you will be polling for
  // whenever the _tick timeout expires.
  _newFileName = CreateFileName(_nMax + 1);

  // Load the first file (if possible).
  string fileName = CreateFileName(_n);
  LoadFile(fileName);

  // Needed to disable the initial timeout.  (See the constructor.)
  return 0;
}


void CWWindow::LoadFile(const string& fileName, bool waitUntilAtEnd)
{
  if( _n == 0 )
    return;

  _text->LoadFile(fileName, waitUntilAtEnd);
}


gint CWWindow::TickCB()
{
  if( _n == _nMax )
    CheckForNewFile();
  return 1;
}


void CWWindow::CheckForNewFile()
{
  if( access(_newFileName.c_str(), F_OK) == 0 )
    {
      _n = ++_nMax;
      if( _nMin == 0 )
        _nMin = 1;
      LoadFile(_newFileName, true);
      _newFileName = CreateFileName(_nMax + 1);
    }
}


void CWWindow::RugPulledOutFromUnder()
{
  _text->Clear();

  ostrstream ostrm;
  ostrm << "\nWarning: The rug has been pulled out from under Crafty Watcher.\n"
        << "         No log file(s) from " << _nMin << " to " << _nMax << " "
        << "exist.\n" 
        << "         Crafty Watcher will rescan \"" 
        << _craftyLogPath << "\".\n" << ends;
  _text->insert(ostrm.str(), -1);
  ostrm.rdbuf()->freeze(0);

  set_title("Crafty Watcher");
  InitialLoadFileCB();
}


string CWWindow::CreateFileName(int n)
{
  // Create the file name from "n".
  ostrstream ostrm;
  ostrm << _craftyLogPath << "/" << _baseStr;
  if( n < 100 )
    ostrm << '0';
  if( n < 10 )
    ostrm << '0';
  ostrm << n << ends;

  string tmp(ostrm.str());
  ostrm.rdbuf()->freeze(0);
  return tmp;
}


string CWWindow::GetBaseName(const string& baseName)
{
  string::const_iterator i = baseName.end();
  for( ; ; )
    if( --i < baseName.begin() || *i == '/' )
      break;

  if( i == baseName.begin() && *i != '/' )
    return baseName;

  return string(++i, baseName.end());
}
