;;; jde-help.el
;; $Revision: 1.8 $ 

;; Author: Paul Kinnucan <paulk@mathworks.com>, Phillip Lord <plord@hgmp.mrc.ac.uk>
;; Maintainer: Paul Kinnucan
;; Keywords: java, tools

;; Copyright (C) 1999 Paul Kinnucan.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

(require 'beanshell)

(defcustom jde-help-docsets nil
  "*Lists collections of HTML files documenting Java classes. 
This list is used by the `jde-help-class' command to find help for 
a class. You can specify the following information for each docset:

Docset type

  The following types are valid: 

  * javadoc. 

    Collections generated by the javadoc command.

  * Other

    Collections of HTML class documentation files generated by some
    other means.

Docset directory

   Directory containing the collection, e.g., d:/jdk1.2/docs/api.

Doc lookup function

   Should specify a function that accepts a fully qualified class name, 
   e.g., java.awt.String, and a docset directory and returns a path to 
   an HTML file that documents that class, e.g., 
   d:/jdk1.2/docs/api/java/awt/String.html. This field must be specified
   for non-javadoc collections. It is ignored for javadoc colletions.
"
  :group 'jde-project
  :type '(repeat 
	  (list
	   (radio-button-choice
	    :format "%t \n%v"
	    :tag "Docset type:"
	    (const "javadoc")
	    (const "Other"))
	   (string :tag "Docset directory")
	   (function :tag "Doc lookup function:"))))

(defun jde-help-docset-get-type (docset)
  (nth 0 docset))

(defun jde-help-docset-get-dir (docset)
  (nth 1 docset))

(defun jde-help-docset-get-lookup-function (docset)
  (nth 2 docset))


(defun jde-help-normalize-dir-path (path)
"Appends a slash to the end of a path, if it does not already
end in a slash." 
  (if (let ((last-char (aref path (- (length path) 1))))
	(or (equal last-char ?/)
	    (equal last-char ?\))))
      path
    (concat path "/")))

(defun jde-help-lookup-java1-javadoc (class, docset-dir) 
  (let ((doc-path
	 (concat 
	  (jde-help-normalize-dir-path docset-dir)
	  class 
	  ".html")))
    (if (file-exists-p doc-path) 
	doc-path)))

(defun jde-help-lookup-java2-javadoc (class, docset-dir) 
  (let ((doc-path
	 (concat 
	  (jde-help-normalize-dir-path docset-dir)
	  (substitute ?/ ?. class) 
	  ".html")))
    (if (file-exists-p doc-path) 
	doc-path)))


(defun jde-help-get-doc (class) 
"Gets path to the HTML file for CLASS where CLASS is a 
qualified class name."
  (if class
      (cond 
       ((not jde-help-docsets)
	(error "%s" "Help error: No docsets available. See jde-help-docsets."))
       (t
	(let ((paths
	       (mapcar
		(lambda (docset)
		  (cond
		   ((string= (jde-help-docset-get-type docset) "javadoc")
		    (or 
		     (jde-help-lookup-java1-javadoc
		      class
		      (jde-help-docset-get-dir docset)) 
		     (jde-help-lookup-java2-javadoc
		      class
		      (jde-help-docset-get-dir docset))))
		   (t
		    (apply
		     (jde-help-docset-get-lookup-function docset)
		     class
		     (jde-help-docset-get-dir docset)))))
		jde-help-docsets)))
	  (setq paths (delq nil paths))
	  ;; Return first file found.
	  (if paths (car paths) paths))))))

(defun jde-help-class ()
"Displays help on the class whose name appears at point in the current
Java buffer. This command uses `jde-help-javadoc-dirs' to find 
documentation on the class. It uses `browse-url' to display the 
documentation."
  (interactive)
  (condition-case err
      (let* ((unqualified-name (thing-at-point 'symbol))
	     (class-names 
	      ;;expand the names into full names, or a list of names
	      (bsh-eval-r 
	       (concat "jde.util.JdeUtilities.getQualifiedName(\"" unqualified-name "\");")))
	     (doc-files (mapcar 'jde-help-get-doc class-names)))
	;;Check return value of QualifiedName
	(if (eq nil class-names)
	    (error "Cannot find %s" unqualified-name))
	;;Remove any annoying nils from the returned values
	(setq doc-files (delq nil doc-files))
	(if (eq nil doc-files) 
	    (error "Cannot find documentation for %s" unqualified-name))
	;;If the list is only one long
	(if (eq 1 (length doc-files))
	    ;;then show it
	    (jde-help-show-document (car doc-files))
	  ;;else let the user choose
	  (jde-help-choose-document doc-files)))
    (error
     (message "%s" (error-message-string err)))))


(defun jde-help-show-document (doc-file)
  "Actually displays the document."
  (if (not (eq doc-file nil))
      (browse-url (format "file://%s" doc-file))))

(defun jde-help-choose-document(doc-files)
  "Allows the user to select which of the possible documentation files they wish to view."
  (let ((buf (get-buffer-create "*Select Class*" )))
    (setq jde-help-documentation-files doc-files)
    (setq jde-help-selected-documentation-file (car doc-files))
    (set-buffer buf)
    (widget-insert "Several documentation files match your class.\n")
    (widget-insert "Select the one you want to view.\n")
    (widget-insert "Then click the OK button.\n\n" )
    (let ((args (list
		 'radio-button-choice
		 :value (car doc-files)
		 :notify (lambda (widget &rest ignore)
			   (setq jde-help-selected-documentation-file (widget-value widget))
			   (message "You selected: %s"
				    (widget-value widget))))))
	  (setq args (nconc
		      args
		       (mapcar (lambda (x) (list 'item x)) doc-files)))
	  (apply 'widget-create args))
    (widget-insert "\n")
    (widget-create 'push-button
		   :notify (lambda (&rest ignore)
			     (let ((dialog-buffer
				    (current-buffer)))
			       (delete-window)
			       (kill-buffer dialog-buffer)
			       (jde-help-show-document jde-help-selected-documentation-file)
			       (message "Viewing initiated.")))
		   "Ok")
    (use-local-map widget-keymap)
    (widget-setup)
    (pop-to-buffer buf)))

(provide 'jde-help)

;; $Log: jde-help.el,v $
;; Revision 1.8  1999/09/30 04:46:10  paulk
;; Fixed typo spotted by David Biesack.
;;
;; Revision 1.7  1999/09/18 03:26:39  paulk
;; Now prepends "file://" to doc file when invoking browse-url. Hopefully
;; this will fix the problem reported by one user where the browser
;; prepends http://www to doc file path.
;;
;; Revision 1.6  1999/08/20 00:44:43  paulk
;; Corrected spelling of Phillip Lord's name.
;;
;; Revision 1.5  1999/06/26 00:00:12  paulk
;; Type javadoc now sufficient to specify both Java 1 and Java 2 javadoc docsets.
;;
;; Revision 1.4  1999/06/25 01:38:17  paulk
;; Enhanced to support doc collections of any type.
;;
;; Revision 1.3  1999/06/17 22:27:33  paulk
;; Bug fix.
;;
;; Revision 1.2  1999/06/17 21:53:05  paulk
;; Eliminated separate customization group for help variables.
;;
;; Revision 1.1  1999/06/17 21:47:15  paulk
;; Initial revision
;;

;; End of jde-help.el
