/* `rsff-f9t.c' -- Routines to construct the generator polynomial and
   some other look-up tables for the Reed-Solomon (255, 249) code over
   the Galois Field (2^8).

   Copyright (C) 1996 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Written by Fung-Chai Lim (fclim@acm.org).

   This file is part of GNU ecc.  */

#ifdef HAVE_CONFIG_H
#if defined (CONFIG_BROKETS)
/* We use <config.h> instead of "config.h" so that a compilation
   using -I. -I$srcdir will use ./config.h rather than $srcdir/config.h
   (which it would do because it found this file in $srcdir).  */
#include <config.h>
#else
#include "config.h"
#endif
#endif

#include <stdio.h>

#include "gf256.h"

static void make_generator _PROTO_ ((void));
static void make_gen_roots _PROTO_ ((void));
static void make_gen_poly _PROTO_ ((void));
static void make_quadratic_roots _PROTO_ ((void));
static void make_square_roots _PROTO_ ((void));
static void make_cubic_roots _PROTO_ ((void));

static void print_table _PROTO_ ((const GF256 *const table, FILE *file));
static void print_quadratic_roots _PROTO_ ((FILE *file));
static void print_cubic_roots _PROTO_ ((const GF256 table[][3], FILE *file));

/* The roots of the generating polynomial.  */
static GF256 gen_root_0_times[256];
static GF256 gen_root_1_times[256];
static GF256 gen_root_2_times[256];
static GF256 gen_root_3_times[256];
static GF256 gen_root_4_times[256];
static GF256 gen_root_5_times[256];

/* The coefficients of the generating polynomial.  */
static GF256 gen_0_times[256];
static GF256 gen_1_times[256];
static GF256 gen_2_times[256];
static GF256 gen_3_times[256];
static GF256 gen_4_times[256];
static GF256 gen_5_times[256];

/* The roots of the quadratic polynomial x^2 + x + b, where B ranges over
   the non-zero elements of GF(256).  If the polynomial is irreducible, or
   if one of its roots is 0, or it has repeated roots, the array element is
   set to 0.  */
static GF256 quadratic_roots[256][2];

/* The square roots of the elements in GF(256).  */
static GF256 square_roots[256];

/* The roots of the cubic polynomial x^3 + b, where B ranges over the
   non-zero elements of GF(256).  If the polynomial is irreducible, or
   if one of its roots is 0, or it has repeated roots, the array element
   is set to 0.  */
static GF256 x3_x_b_roots[256][3];

/* The roots of the cubic polynomial x^3 + x + b, where B ranges over the
   non-zero elements of GF(256).  If the polynomial is irreducible, or if
   one of its roots is 0, or it has repeated roots, the array element is
   set to 0.  */
static GF256 x3_b_roots[256][3];

     void
make_tables ()
{
  make_generator ();
  make_quadratic_roots ();
  make_square_roots ();
  make_cubic_roots ();
}

     int
substitute (string, file)
     const char *const string;
     FILE *file;
{
  if (strcmp (string, "GEN_0_TIMES") == 0)
    print_table (gen_0_times, file);
  else if (strcmp (string, "GEN_1_TIMES") == 0)
    print_table (gen_1_times, file);
  else if (strcmp (string, "GEN_2_TIMES") == 0)
    print_table (gen_2_times, file);
  else if (strcmp (string, "GEN_3_TIMES") == 0)
    print_table (gen_3_times, file);
  else if (strcmp (string, "GEN_4_TIMES") == 0)
    print_table (gen_4_times, file);
  else if (strcmp (string, "GEN_5_TIMES") == 0)
    print_table (gen_5_times, file);

  else if (strcmp (string, "GEN_ROOT_0_TIMES") == 0)
    print_table (gen_root_0_times, file);
  else if (strcmp (string, "GEN_ROOT_1_TIMES") == 0)
    print_table (gen_root_1_times, file);
  else if (strcmp (string, "GEN_ROOT_2_TIMES") == 0)
    print_table (gen_root_2_times, file);
  else if (strcmp (string, "GEN_ROOT_3_TIMES") == 0)
    print_table (gen_root_3_times, file);
  else if (strcmp (string, "GEN_ROOT_4_TIMES") == 0)
    print_table (gen_root_4_times, file);
  else if (strcmp (string, "GEN_ROOT_5_TIMES") == 0)
    print_table (gen_root_5_times, file);

  else if (strcmp (string, "QUADRATIC_ROOTS") == 0)
    print_quadratic_roots (file);
  else if (strcmp (string, "X3_B_ROOTS") == 0)
    print_cubic_roots (x3_b_roots, file);
  else if (strcmp (string, "X3_X_B_ROOTS") == 0)
    print_cubic_roots (x3_x_b_roots, file);
  else if (strcmp (string, "SQUARE_ROOTS") == 0)
    print_table (square_roots, file);

  else
    return 1;
  return 0;
}

/* The Generator Polynomial for the Reed-Solomon (255, 249) code is
   defined as G(x) = (x - 2) (x - 2^2) (x - 2^3) ... (x - 2^6) =
   x^6 + G[5] x^5 + G[4] x^4 + ... + G[1] x + G[0].  */

     static void
make_generator ()
{
  /* Must call make_gen_roots() first before make_gen_poly().  */
  make_gen_roots ();
  make_gen_poly ();
}

     static void
make_gen_roots ()
{
  int i;
  GF256 g0, g1, g2, g3, g4, g5;

  i = 1;
  g0 = gf256_power_of_2 (i); i++;
  g1 = gf256_power_of_2 (i); i++;
  g2 = gf256_power_of_2 (i); i++;
  g3 = gf256_power_of_2 (i); i++;
  g4 = gf256_power_of_2 (i); i++;
  g5 = gf256_power_of_2 (i);

  for (i = 0; i < 256; i++)
    {
      gen_root_0_times[i] = gf256_mul (i, g0);
      gen_root_1_times[i] = gf256_mul (i, g1);
      gen_root_2_times[i] = gf256_mul (i, g2);
      gen_root_3_times[i] = gf256_mul (i, g3);
      gen_root_4_times[i] = gf256_mul (i, g4);
      gen_root_5_times[i] = gf256_mul (i, g5);
    }
}

     static void
make_gen_poly ()
{
  int i, j;
  GF256 power, product;
  GF256 gen_roots[6];
  GF256 gen[6];

  gen_roots[0] = gen_root_0_times[1];
  gen_roots[1] = gen_root_1_times[1];
  gen_roots[2] = gen_root_2_times[1];
  gen_roots[3] = gen_root_3_times[1];
  gen_roots[4] = gen_root_4_times[1];
  gen_roots[5] = gen_root_5_times[1];

  gen[0] = gen_roots[0];
  for (i = 1; i < 6; i++)
    {
      power = gen_roots[i];
      gen[i] = 1;
      for (j = i; j > 0; j--)
	{
	  product = gf256_mul (power, gen[j]);
	  gen[j] = gf256_add (product, gen[j-1]);
	}
      gen[0] = gf256_mul (power, gen[0]);
    }

  for (i = 0; i < 256; i++)
    {
      gen_0_times[i] = gf256_mul (i, gen[0]);
      gen_1_times[i] = gf256_mul (i, gen[1]);
      gen_2_times[i] = gf256_mul (i, gen[2]);
      gen_3_times[i] = gf256_mul (i, gen[3]);
      gen_4_times[i] = gf256_mul (i, gen[4]);
      gen_5_times[i] = gf256_mul (i, gen[5]);
    }
}

     static void
make_quadratic_roots ()
{
  int i;
  GF256 x1, x0;
  GF256 z1, z0;

  for (i = 0; i < 256; i++)
    {
      quadratic_roots[i][0] = 0;
      quadratic_roots[i][1] = 0;
    }

  /* If x^2 + z1*x + z0 = (x - x1) (x - x0), then
       z1 = x1 + x0
       z0 = x1 * x0
     Thus, we have
       x0 = z1 - x1.
     Therefore, by letting x1 range over GF(256), we will have the roots of
     x^2 + x + b, if we let z1 = 1 and b = x1*x0.  */

  z1 = 1;
  for (i = 0; i < 256; i++)
    {
      x1 = i;
      x0 = gf256_subtract (z1, x1);
      if (x0 == 0 || x1 == 0)
	continue;
      z0 = gf256_mul (x1, x0);
      quadratic_roots[z0][0] = x0;
      quadratic_roots[z0][1] = x1;
    }
}

     static void
make_square_roots ()
{
  int i;

#define square(x)	(gf256_exp ((x), 2))

  for (i = 0; i < 256; i++)
    square_roots [square (i)] = i;

#undef square
}

/* If the polynomial x^2 + Z1*x + Z0 is irreducible in GF(256), or if one of
   its roots is 0, or if the polynomial has repeated roots, return 1;
   otherwise, store the roots in X1 and X0, and return 0.  */

     static int
solve_quad_poly (z1, z0, x1, x0)
     GF256 z1, z0;
     GF256 *x1, *x0;
{
  GF256 prod;

  /* One of the roots is 0.  */
  if (z0 == 0)
    return 1;
  /* Repeated roots.  */
  if (z1 == 0)
    return 1;

  /* The algorithm used here is described in ``Theory of Error Correcting
     Codes'' by MacWilliams and Sloane.

     Let x = z1*y.  Then
       x^2 + z1*x + z0 = z1^2*y^2 + z1^2*y + z0 = z1^2*(y^2 + y + z0/z1^2).
     If y1 and y0 are the roots of the transformed quadratic, then
       x1 = z1*y1 and x0 = z1*y0
     are the roots of the original.  */

  prod = gf256_exp (z1, 2);
  z0 = gf256_div (z0, prod);
  if (quadratic_roots[z0][0] == 0)
    return 1;
  *x1 = gf256_mul (z1, quadratic_roots[z0][1]);
  *x0 = gf256_mul (z1, quadratic_roots[z0][0]);
  return 0;
}

     static void
make_cubic_roots ()
{
  int i;
  GF256 x2, x1, x0;
  GF256 z2, z1, z0;
  GF256 x10;		/* Represents (x1 + x0).  */
  GF256 x1x0;		/* Represents (x1 * x0).  */
  GF256 prod;

  for (i = 0; i < 256; i++)
    {
      x3_b_roots[i][0] = 0;
      x3_b_roots[i][1] = 0;
      x3_b_roots[i][2] = 0;
      x3_x_b_roots[i][0] = 0;
      x3_x_b_roots[i][1] = 0;
      x3_x_b_roots[i][2] = 0;
    }

  /* If x^3 + z2*x^2 + z1*x + z0 = (x - x2) (x - x1) (x - x0), then
       z2 = x2 + x1 + x0,
       z1 = x2*x1 + x2*x0 + x1*x0 = x2*(x1+x0) + x1*x0,
       z0 = x2*x1*x0
     Thus, we have
       x1+x0 = z2 - x2,
       x1*x0 = z1 - x2*(x1+x0)
     Therefore, by letting x2 range over GF(256), and solving for x1 and x0,
     we will have the roots of (x^3 + b) and (x^3 + x + b), if we let
     (z2 = z1 = 0) and (z2 = 0, z1 = 1), respectively, and b = x2*x1*x0.  */
  z2 = 0;
  z1 = 0;
  for (i = 0; i < 256; i++)
    {
      x2 = i;
      x10 = gf256_subtract (z2, x2);
      prod = gf256_mul (x2, x10);
      x1x0 = gf256_subtract (z1, prod);
      if (x10 == 0 || x1x0 == 0)
	continue;
      if (solve_quad_poly (x10, x1x0, &x1, &x0))
	continue;
      z0 = gf256_product (x2, x1, x0);
      x3_b_roots[z0][0] = x0;
      x3_b_roots[z0][1] = x1;
      x3_b_roots[z0][2] = x2;
    }

  z2 = 0;
  z1 = 1;
  for (i = 0; i < 256; i++)
    {
      x2 = i;
      x10 = gf256_subtract (z2, x2);
      prod = gf256_mul (x2, x10);
      x1x0 = gf256_subtract (z1, prod);
      if (x10 == 0 || x1x0 == 0)
	continue;
      if (solve_quad_poly (x10, x1x0, &x1, &x0))
	continue;
      z0 = gf256_product (x2, x1, x0);
      x3_x_b_roots[z0][0] = x0;
      x3_x_b_roots[z0][1] = x1;
      x3_x_b_roots[z0][2] = x2;
    }
}

     static void
print_table (table, file)
     const GF256 *const table;
     FILE *file;
{
  int i;

  for (i = 0; i < 256; i++)
    {
      if ((i % 8) == 0)
	fprintf (file, "\n  ");
      fprintf (file, "%4d", table[i]);
      if (i != 255)
	fputc (',', file);
    }
  fputc ('\n', file);
}

     static void
print_quadratic_roots (file)
     FILE *file;
{
  int i;

  for (i = 0; i < 256; i++)
    {
      if ((i % 4) == 0)
	fprintf (file, "\n  ");
      fprintf (file, "{%4d,%4d }",
	       quadratic_roots [i][0], quadratic_roots [i][1]);
      if (i != 255)
	fprintf (file, ", ");
    }
  fputc ('\n', file);
}

     static void
print_cubic_roots (table, file)
     const GF256 table[][3];
     FILE *file;
{
  int i;

  for (i = 0; i < 256; i++)
    {
      if ((i % 4) == 0)
	fprintf (file, "\n  ");
      fprintf (file, "{%4d,%4d,%4d }", table [i][0], table[i][1], table [i][2]);
      if (i != 255)
	fprintf (file, ", ");
    }
  fputc ('\n', file);
}
