//========================================================================
//
// Page.h
//
// Copyright 1996 Derek B. Noonburg
//
//========================================================================

#ifndef PAGE_H
#define PAGE_H

#ifdef __GNUC__
#pragma interface
#endif

#include "Object.h"

class Dict;
class XRef;
class OutputDev;
class Links;
class Catalog;

// Used to store information about page boundaries 
struct Rectangle {
    double x1, y1, x2, y2;

    Rectangle () {
        x1 = x2 = y1 = y2 = 0;
        };
        
    Rectangle (double x1, double y1, double x2, double y2) {
        Rectangle::x1 = x1;
        Rectangle::x2 = x2;
        Rectangle::y1 = y1;
        Rectangle::y2 = y2;
        }

    // has the Rectangle meaningful information?
    bool isValid () {
        return x1 || x2 || y1 || y2;
        }
    };

//------------------------------------------------------------------------
// PageAttrs
//------------------------------------------------------------------------

class PageAttrs {
public:

  // Construct a new PageAttrs object by merging a dictionary
  // (of type Pages or Page) into another PageAttrs object.  If
  // <attrs> is NULL, uses defaults.
  PageAttrs(PageAttrs *attrs, Dict *dict);

  // Destructor.
  ~PageAttrs();

  // Accessors.
  double getX1() { return limitToCropBox ? cropBox.x1 : mediaBox.x1; }
  double getY1() { return limitToCropBox ? cropBox.y1 : mediaBox.y1; }
  double getX2() { return limitToCropBox ? cropBox.x2 : mediaBox.x2; }
  double getY2() { return limitToCropBox ? cropBox.y2 : mediaBox.y2; }
  GBool isCropped() { return cropBox.x2 > cropBox.x1; }
  double getCropX1() { return cropBox.x1; }
  double getCropY1() { return cropBox.y1; }
  double getCropX2() { return cropBox.x2; }
  double getCropY2() { return cropBox.y2; }
  // these functions return the Boxes as specified in the pdf;
  // Box.isValid () may be false.
  Rectangle getCropBox () {return cropBox; }
  Rectangle getBleedBox () {return bleedBox; }
  Rectangle getTrimBox () {return trimBox; }
  Rectangle getArtBox () {return artBox; }
  // these functions return the Boxes as specified in the pdf or
  // with meaningfull default values (see p.525 of the PDF ref
  // v1.3); Box.isValid () is always true
  Rectangle getLogicalMediaBox () {return mediaBox; }
  Rectangle getLogicalCropBox ()  {return cropBox.isValid ()  ? cropBox  : getLogicalMediaBox (); }
  Rectangle getLogicalBleedBox () {return bleedBox.isValid () ? bleedBox : getLogicalCropBox (); }
  Rectangle getLogicalTrimBox ()  {return trimBox.isValid ()  ? trimBox  : getLogicalCropBox (); }
  Rectangle getLogicalArtBox ()   {return artBox.isValid ()   ? artBox   : getLogicalCropBox (); }
  int getRotate() { return rotate; }
  Dict *getResourceDict()
    { return resources.isDict() ? resources.getDict() : (Dict *)NULL; }

private:

  Rectangle mediaBox, cropBox, artBox, trimBox, bleedBox;
  GBool limitToCropBox;
  int rotate;
  Object resources;
};

//------------------------------------------------------------------------
// Page
//------------------------------------------------------------------------

class Page {
public:

  // Constructor.
  Page(int num1, Dict *pageDict, PageAttrs *attrs1);

  // Destructor.
  ~Page();

  // Is page valid?
  GBool isOk() { return ok; }

  // Get page parameters.
  double getX1() { return attrs->getX1(); }
  double getY1() { return attrs->getY1(); }
  double getX2() { return attrs->getX2(); }
  double getY2() { return attrs->getY2(); }
  GBool isCropped() { return attrs->isCropped(); }
  double getCropX1() { return attrs->getCropX1(); }
  double getCropY1() { return attrs->getCropY1(); }
  double getCropX2() { return attrs->getCropX2(); }
  double getCropY2() { return attrs->getCropY2(); }
  double getWidth() { return attrs->getX2() - attrs->getX1(); }
  double getHeight() { return attrs->getY2() - attrs->getY1(); }
  int getRotate() { return attrs->getRotate(); }

  // Get resource dictionary.
  Dict *getResourceDict() { return attrs->getResourceDict(); }

  // Get annotations array.
  Object *getAnnots(Object *obj) { return annots.fetch(obj); }

  // Get contents.
  Object *getContents(Object *obj) { return contents.fetch(obj); }

  // Display a page.
  void display(OutputDev *out, double dpi, int rotate,
	       Links *links, Catalog *catalog);

  // these functions return the Boxes as specified in the pdf or
  // with meaningfull default values (see p.525 of the PDF ref
  // v1.3); Box.isValid () is always true
  Rectangle getLogicalMediaBox () {return attrs->getLogicalMediaBox (); }
  Rectangle getLogicalCropBox ()  {return attrs->getLogicalCropBox (); }
  Rectangle getLogicalBleedBox () {return attrs->getLogicalBleedBox (); }
  Rectangle getLogicalTrimBox ()  {return attrs->getLogicalTrimBox (); }
  Rectangle getLogicalArtBox ()   {return attrs->getLogicalArtBox (); }
  
private:

  int num;			// page number
  PageAttrs *attrs;		// page attributes
  Object annots;		// annotations array
  Object contents;		// page contents
  GBool ok;			// true if page is valid
};

#endif
