/*************************************************************************** 
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              * 
 *                                                                         * 
 *   This program is free software; you can redistribute it and/or modify  * 
 *   it under the terms of the GNU General Public License as published by  * 
 *   the Free Software Foundation; either version 2 of the License, or     * 
 *   (at your option) any later version.                                   * 
 *                                                                         * 
 *   This program is distributed in the hope that it will be useful,       * 
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        * 
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         * 
 *   GNU General Public License for more details.                          * 
 *                                                                         * 
 *   You should have received a copy of the GNU General Public License     * 
 *   along with this program; if not, write to the                         * 
 *   Free Software Foundation, Inc.,                                       * 
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             * 
 *                                                                         * 
 *   Licensed under the GNU GPL                                            * 
 *   Original code supplied under license from RaLink Inc, 2003.           * 
 ***************************************************************************/ 

 /*************************************************************************** 
 *      Module Name: auth_rsp.c 
 *              
 *      Abstract: Authentication State Machine 
 *              
 *      Revision History: 
 *      Who             When            What 
 *      --------        -----------     ----------------------------- 
 *      MarkW           9th  Feb 04     Baseline of code
 *      defekt          13th Feb 04     ATOMIC kmalloc fix
 ***************************************************************************/ 


#include "rt_config.h"

/*
    ==========================================================================
    Description:
        authentication state machine init procedure
    Parameters:
        Sm - the state machine
    Note:
        the state machine looks like the following 
        
                                        AUTH_RSP_IDLE                   AUTH_RSP_WAIT_CHAL
    MT2_AUTH_CHALLENGE_TIMEOUT    auth_rsp_challenge_timeout_action    auth_rsp_challenge_timeout_action
    MT2_PEER_AUTH_ODD        peer_auth_at_auth_rsp_idle_action peer_auth_at_auth_rsp_wait_action
    MT2_PEER_DEAUTH                 peer_deauth_action                 peer_deauth_action
    ==========================================================================
 */
VOID AuthRspStateMachineInit(
    IN PRTMP_ADAPTER pAd, 
    IN PSTATE_MACHINE Sm, 
    IN STATE_MACHINE_FUNC Trans[]) 
{
    LARGE_INTEGER        NOW;

    StateMachineInit(Sm, (STATE_MACHINE_FUNC*)Trans, MAX_AUTH_RSP_STATE, MAX_AUTH_RSP_MSG, (STATE_MACHINE_FUNC)Drop, AUTH_RSP_IDLE, AUTH_RSP_MACHINE_BASE);

    // column 1
//  StateMachineSetAction(Sm, AUTH_RSP_IDLE, MT2_AUTH_CHALLENGEG_TIMEOUT, (STATE_MACHINE_FUNC)AuthRspChallengeTimeoutAction);
//  StateMachineSetAction(Sm, AUTH_RSP_IDLE, MT2_PEER_AUTH_ODD, (STATE_MACHINE_FUNC)PeerAuthAtAuthRspIdleAction);
    StateMachineSetAction(Sm, AUTH_RSP_IDLE, MT2_PEER_DEAUTH, (STATE_MACHINE_FUNC)PeerDeauthAction);

    // column 2
//  StateMachineSetAction(Sm, AUTH_RSP_WAIT_CHAL, MT2_PEER_AUTH_ODD, (STATE_MACHINE_FUNC)PeerAuthAtAuthRspWaitAction);
//  StateMachineSetAction(Sm, AUTH_RSP_WAIT_CHAL, MT2_AUTH_CHALLENGE_TIMEOUT, (STATE_MACHINE_FUNC)AuthRspChallengeTimeoutAction);
    StateMachineSetAction(Sm, AUTH_RSP_WAIT_CHAL, MT2_PEER_DEAUTH, (STATE_MACHINE_FUNC)PeerDeauthAction);

    // initialize the timer
    init_timer(&pAd->Mlme.AuthRspAux.AuthRspTimer);
    pAd->Mlme.AuthRspAux.AuthRspTimer.data = (unsigned long)pAd;
    pAd->Mlme.AuthRspAux.AuthRspTimer.function = &AuthRspChallengeTimeout;				/* timer handler */

    // initialize the random number generator
    NOW = jiffies;
    LfsrInit(pAd, NOW);
}


/*
    ==========================================================================
    Description:
        challenge time out, called by timer thread
    ==========================================================================
 */
VOID AuthRspChallengeTimeout(
    IN unsigned long data) 
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;
    
    DBGPRINT(RT_DEBUG_TRACE,"AUTH_RSP - AuthRspChallengeTimeout \n");
    MlmeEnqueue(&pAd->Mlme.Queue, AUTH_RSP_STATE_MACHINE, MT2_AUTH_CHALLENGE_TIMEOUT, 0, NULL);
    MlmeHandler(pAd);
}


/*
    ==========================================================================
    Description:
    ==========================================================================
*/
VOID PeerAuthAtAuthRspIdleAction(
    IN PRTMP_ADAPTER pAd, 
    IN PMLME_QUEUE_ELEM Elem) 
{
    USHORT          Seq, Alg, RspReason, i, Status;
    MACADDR         Addr2;
    MACHDR         *pRcvHdr, AuthHdr;
    CHAR            Chtxt[CIPHER_TEXT_LEN];
    CHAR           *OutBuffer = NULL;
    UINT            FrameLen = 0;

    if (PeerAuthSanity(pAd, Elem->Msg, Elem->MsgLen, &Addr2, &Alg, &Seq, &Status, Chtxt)) 
    {
        if (!MAC_ADDR_EQUAL(&pAd->PortCfg.Bssid, &Addr2)) 
        {
            DBGPRINT(RT_DEBUG_TRACE, "AUTH_RSP - receive AUTH request but not from my AP, ignored\n");
            return;
        }
        
        pRcvHdr = (MACHDR *)(Elem->Msg);
        if (Seq == 1) 
        {
            if ((Alg == AUTH_MODE_OPEN) && 
                ((pAd->PortCfg.AuthAlgorithm == AUTH_MODE_OPEN) || (pAd->PortCfg.AuthAlgorithm == AUTH_MODE_AUTO_SWITCH) )) 
            {
                 PeerAuthSimpleRspGenAndSend(pAd, pRcvHdr, Alg, Seq + 1, MLME_SUCCESS, AUTH_MODE_OPEN);
            } 
            else if ((Alg == AUTH_MODE_KEY) && 
                ((pAd->PortCfg.AuthAlgorithm == AUTH_MODE_KEY) || (pAd->PortCfg.AuthAlgorithm == AUTH_MODE_AUTO_SWITCH) )) 
            {
                del_timer_sync(&pAd->Mlme.AuthRspAux.AuthRspTimer);

                for(i = 0; i < CIPHER_TEXT_LEN; i++) 
                    pAd->Mlme.AuthRspAux.Challenge[i] = RandomByte(pAd);

                RspReason = 0;
                Seq++;
                COPY_MAC_ADDR(&pAd->Mlme.AuthRspAux.Addr, &Addr2);

                OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_ATOMIC);
                if(OutBuffer == NULL)
                    return;  // if no memory, can't do anything

                DBGPRINT(RT_DEBUG_TRACE, "AUTH_RSP - Send AUTH seq#2 (Challenge) to AP...\n");
                MgtMacHeaderInit(pAd, &AuthHdr, SUBTYPE_AUTH, 0, &Addr2, &pAd->PortCfg.Bssid);
                // AuthHdr.Wep = 1; // challenge should be in clear text ?????????????
                MakeOutgoingFrame(OutBuffer, &FrameLen,
                                  sizeof(MACHDR), (PVOID)&AuthHdr,
                                  2, &Alg,
                                  2, &Seq,
                                  2, &RspReason,
                                  CIPHER_TEXT_LEN, pAd->Mlme.AuthRspAux.Challenge,
                                  NULL);
                MiniportMMRequest(pAd, OutBuffer, FrameLen);

                // start a response timer
                pAd->Mlme.AuthRspAux.AuthRspTimer.expires = jiffies + (pAd->PortCfg.AuthRspTimeout * HZ)/1000;
                add_timer(&pAd->Mlme.AuthRspAux.AuthRspTimer);	// in mSec

                pAd->Mlme.AuthRspMachine.CurrState = AUTH_RSP_WAIT_CHAL;
            } 
            else 
            {
                // wrong algorithm
                 PeerAuthSimpleRspGenAndSend(pAd, pRcvHdr, Alg, Seq + 1, MLME_ALG_NOT_SUPPORT, AUTH_MODE_DEAUTH);
            }
        } 
        else 
        {
            // wrong sequence number
            PeerAuthSimpleRspGenAndSend(pAd, pRcvHdr, Alg, Seq + 1, MLME_SEQ_NR_OUT_OF_SEQUENCE, AUTH_MODE_DEAUTH);
        }
    }
    else
    {
        DBGPRINT(RT_DEBUG_TRACE,"AUTH_RSP - PeerAuthAtAuthRspIdleAction() sanity check fail\n");
    }
}

/*
    ==========================================================================
    Description:
    ==========================================================================
*/
VOID PeerAuthSimpleRspGenAndSend(
    IN PRTMP_ADAPTER pAd, 
    IN PMACHDR Hdr, 
    IN USHORT Alg, 
    IN USHORT Seq, 
    IN USHORT Reason, 
    IN USHORT Status) 
{
    MACHDR            AuthHdr;
    UINT              FrameLen = 0;
    UCHAR            *OutBuffer = NULL;

    // set station status first
    // StaState(pAd, &Hdr->Addr2, AUTH_MODE, Status);

    OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_ATOMIC);
    if(OutBuffer == NULL)
        return;

    if (Reason == MLME_SUCCESS)
    {
        DBGPRINT(RT_DEBUG_TRACE, "Send AUTH response (seq#2)...\n");
        MgtMacHeaderInit(pAd, &AuthHdr, SUBTYPE_AUTH, 0, &Hdr->Addr2, &pAd->PortCfg.Bssid);
        MakeOutgoingFrame(OutBuffer, &FrameLen, sizeof(MACHDR), (PVOID)&AuthHdr, 2, &Alg, 2, &Seq, 2, &Reason, NULL);
        MiniportMMRequest(pAd, OutBuffer, FrameLen);
    }
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, "Peer AUTH fail...\n");
    }
}

/*
    ==========================================================================
    Description:
    ==========================================================================
*/
VOID PeerAuthAtAuthRspWaitAction(
    IN PRTMP_ADAPTER pAd, 
    IN PMLME_QUEUE_ELEM Elem) 
{
    USHORT      Seq, Alg, Status, Wep;
    UCHAR       Ectxt[128];
    MACADDR     Addr2;
    PMACHDR     Hd;

// TODO: need to re-examine this routine
    DBGPRINT(RT_DEBUG_TRACE, ">>>PeerAuthAtAuthRspWaitAction()\n");
    if(PeerAuthSanity(pAd, Elem->Msg, Elem->MsgLen, &Addr2, &Alg, &Seq, &Status, Ectxt)) 
    {
        Hd = (PMACHDR)Elem->Msg;
        Wep = Hd->Wep;
        if(Seq == 1) 
        {
            if (!MAC_ADDR_EQUAL(&Addr2, &pAd->Mlme.AuthRspAux.Addr) && 
                (Alg == AUTH_MODE_OPEN))
//              && (pAd->PortCfg.AuthAlgorithm == AUTH_MODE_OPEN) || (pAd->PortCfg.AuthAlgorithm == AUTH_MODE_AUTO_SWITCH) ))
            {
                // definitely we support open authentication
                PeerAuthSimpleRspGenAndSend(pAd, Hd, Alg, Seq + 1, MLME_SUCCESS, AUTH_MODE_OPEN);
            } 
            else 
            {
                // something wrong in the auth req frame
                PeerAuthSimpleRspGenAndSend(pAd, Hd, Alg, Seq + 1, MLME_UNSPECIFY_FAIL, AUTH_MODE_DEAUTH);
            }
        } 
        else if (Seq == 3) 
        {
            // sequence number == 3
            if(MAC_ADDR_EQUAL(&Addr2, &(pAd->Mlme.AuthRspAux.Addr))) 
            {
                // delete the timer
                del_timer_sync(&pAd->Mlme.AuthRspAux.AuthRspTimer);
                if(Wep == 1 && RTMPEqualMemory(Ectxt, pAd->Mlme.AuthRspAux.Challenge, CIPHER_TEXT_LEN) == 0) 
                {
                    // Successful
                    PeerAuthSimpleRspGenAndSend(pAd, Hd, Alg, Seq + 1, MLME_SUCCESS, AUTH_MODE_KEY);
                } 
                else 
                {
                    // wep bit is not set or challenge text is not equal
                    PeerAuthSimpleRspGenAndSend(pAd, Hd, Alg, Seq + 1, MLME_REJ_CHALLENGE_FAILURE, AUTH_MODE_DEAUTH);
                }
                pAd->Mlme.AuthRspMachine.CurrState = AUTH_RSP_IDLE;
            } 
            else 
            {
                // fail for unknown reason
                PeerAuthSimpleRspGenAndSend(pAd, Hd, Alg, Seq + 1, MLME_UNSPECIFY_FAIL, AUTH_MODE_DEAUTH);
            }
        } 
        else 
        {
            // sequence number incorrect
            PeerAuthSimpleRspGenAndSend(pAd, Hd, Alg, Seq + 1, MLME_UNSPECIFY_FAIL, AUTH_MODE_DEAUTH);
        }
    } 
    DBGPRINT(RT_DEBUG_TRACE, "<<<PeerAuthAtAuthRspWaitAction()\n");
}

/*
    ==========================================================================
    Description:
    ==========================================================================
*/
VOID PeerDeauthAction(
    IN PRTMP_ADAPTER pAd, 
    IN PMLME_QUEUE_ELEM Elem) 
{
    MACADDR     Addr2;
    USHORT      Reason;

    if (PeerDeauthSanity(pAd, Elem->Msg, Elem->MsgLen, &Addr2, &Reason)) 
    {
        // StaState(pAd, &Addr2, AUTH_MODE, AUTH_MODE_DEAUTH);
        if (INFRA_ON(pAd) && MAC_ADDR_EQUAL(&Addr2, &pAd->PortCfg.Bssid)) 
        {
            del_timer_sync(&pAd->Mlme.AuthRspAux.AuthRspTimer);
            DBGPRINT(RT_DEBUG_TRACE,"AUTH_RSP - receive DE-AUTH from our AP\n");
            LinkDown(pAd);
        }
    }
    else
    {
        DBGPRINT(RT_DEBUG_TRACE,"AUTH_RSP - PeerDeauthAction() sanity check fail\n");
    }

}

/*
    ==========================================================================
    Description:
    ==========================================================================
*/
VOID AuthRspChallengeTimeoutAction(
    IN PRTMP_ADAPTER pAd, 
    IN MLME_QUEUE_ELEM *Elem) 
{
    DBGPRINT(RT_DEBUG_TRACE, "AUTH_RSP - AuthRspChallengeTimeoutAction\n");
    // StaState(pAd, &pAd->Mlme.AuthRspAux.Addr, AUTH_MODE, AUTH_MODE_DEAUTH);
    pAd->Mlme.AuthRspMachine.CurrState = AUTH_RSP_IDLE;
}

