"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRoutes = defineRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

// TODO: consider to extract entity CRUD operations and put it into a client class
function defineRoutes(router, dataSourceEnabled) {
  const internalUserSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    password: _configSchema.schema.maybe(_configSchema.schema.string()),
    backend_roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    attributes: _configSchema.schema.any({
      defaultValue: {}
    })
  });
  const actionGroupSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    allowed_actions: _configSchema.schema.arrayOf(_configSchema.schema.string())
    // type field is not supported in legacy implementation, comment it out for now.
    // type: schema.oneOf([
    //   schema.literal('cluster'),
    //   schema.literal('index'),
    //   schema.literal('opensearch_dashboards'),
    // ]),
  });

  const roleMappingSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    backend_roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    hosts: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    users: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  });
  const roleSchema = _configSchema.schema.object({
    description: _configSchema.schema.maybe(_configSchema.schema.string()),
    cluster_permissions: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    tenant_permissions: _configSchema.schema.arrayOf(_configSchema.schema.any(), {
      defaultValue: []
    }),
    index_permissions: _configSchema.schema.arrayOf(_configSchema.schema.any(), {
      defaultValue: []
    })
  });
  const tenantSchema = _configSchema.schema.object({
    description: _configSchema.schema.string()
  });
  const accountSchema = _configSchema.schema.object({
    password: _configSchema.schema.string(),
    current_password: _configSchema.schema.string()
  });
  const schemaMap = {
    internalusers: internalUserSchema,
    actiongroups: actionGroupSchema,
    rolesmapping: roleMappingSchema,
    roles: roleSchema,
    tenants: tenantSchema,
    account: accountSchema
  };
  function validateRequestBody(resourceName, requestBody) {
    const inputSchema = schemaMap[resourceName];
    if (!inputSchema) {
      throw new Error(`Unknown resource ${resourceName}`);
    }
    inputSchema.validate(requestBody); // throws error if validation fail
  }

  function validateEntityId(resourceName) {
    if (!(0, _common.isValidResourceName)(resourceName)) {
      return 'Invalid entity name or id.';
    }
  }

  /**
   * Lists resources by resource name.
   *
   * The response format is:
   * {
   *   "total": <total_entity_count>,
   *   "data": {
   *     "entity_id_1": { <entity_structure> },
   *     "entity_id_2": { <entity_structure> },
   *     ...
   *   }
   * }
   *
   * e.g. when listing internal users, response may look like:
   * {
   *   "total": 2,
   *   "data": {
   *     "api_test_user2": {
   *       "hash": "",
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "attributes": {},
   *       "description": "",
   *       "static": false
   *     },
   *     "api_test_user1": {
   *       "hash": "",
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "attributes": {},
   *       "static": false
   *     }
   * }
   *
   * when listing action groups, response will look like:
   * {
   *   "total": 2,
   *   "data": {
   *     "read": {
   *       "reserved": true,
   *       "hidden": false,
   *       "allowed_actions": ["indices:data/read*", "indices:admin/mappings/fields/get*"],
   *       "type": "index",
   *       "description": "Allow all read operations",
   *       "static": false
   *     },
   *     "cluster_all": {
   *       "reserved": true,
   *       "hidden": false,
   *       "allowed_actions": ["cluster:*"],
   *       "type": "cluster",
   *       "description": "Allow everything on cluster level",
   *       "static": false
   *     }
   * }
   *
   * role:
   * {
   *   "total": 2,
   *   "data": {
   *     "opensearch_dashboards_user": {
   *       "reserved": true,
   *       "hidden": false,
   *       "description": "Provide the minimum permissions for a opensearch_dashboards user",
   *       "cluster_permissions": ["cluster_composite_ops"],
   *       "index_permissions": [{
   *         "index_patterns": [".opensearch_dashboards", ".opensearch_dashboards-6", ".opensearch_dashboards_*"],
   *         "fls": [],
   *         "masked_fields": [],
   *         "allowed_actions": ["read", "delete", "manage", "index"]
   *       }, {
   *         "index_patterns": [".tasks", ".management-beats"],
   *         "fls": [],
   *         "masked_fields": [],
   *         "allowed_actions": ["indices_all"]
   *       }],
   *       "tenant_permissions": [],
   *       "static": false
   *     },
   *     "all_access": {
   *       "reserved": true,
   *       "hidden": false,
   *       "description": "Allow full access to all indices and all cluster APIs",
   *       "cluster_permissions": ["*"],
   *       "index_permissions": [{
   *         "index_patterns": ["*"],
   *         "fls": [],
   *         "masked_fields": [],
   *         "allowed_actions": ["*"]
   *       }],
   *       "tenant_permissions": [{
   *         "tenant_patterns": ["*"],
   *         "allowed_actions": ["opensearch_dashboards_all_write"]
   *       }],
   *       "static": false
   *     }
   *   }
   * }
   *
   * rolesmapping:
   * {
   *   "total": 2,
   *   "data": {
   *     "security_manager": {
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "hosts": [],
   *       "users": ["zengyan", "admin"],
   *       "and_backend_roles": []
   *     },
   *     "all_access": {
   *       "reserved": false,
   *       "hidden": false,
   *       "backend_roles": [],
   *       "hosts": [],
   *       "users": ["zengyan", "admin", "indextest"],
   *       "and_backend_roles": []
   *     }
   *   }
   * }
   *
   * tenants:
   * {
   *   "total": 2,
   *   "data": {
   *     "global_tenant": {
   *       "reserved": true,
   *       "hidden": false,
   *       "description": "Global tenant",
   *       "static": false
   *     },
   *     "test tenant": {
   *       "reserved": false,
   *       "hidden": false,
   *       "description": "tenant description",
   *       "static": false
   *     }
   *   }
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    let esResp;
    try {
      esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.listResource', {
        resourceName: request.params.resourceName
      });
      return response.ok({
        body: {
          total: Object.keys(esResp).length,
          data: esResp
        }
      });
    } catch (error) {
      console.log(JSON.stringify(error));
      return errorResponse(response, error);
    }
  });

  /**
   * Gets entity by id.
   *
   * the response format differs from different resource types. e.g.
   *
   * for internal user, response will look like:
   * {
   *   "hash": "",
   *   "reserved": false,
   *   "hidden": false,
   *   "backend_roles": [],
   *   "attributes": {},
   *   "static": false
   * }
   *
   * for role, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "description": "Allow full access to all indices and all cluster APIs",
   *   "cluster_permissions": ["*"],
   *   "index_permissions": [{
   *     "index_patterns": ["*"],
   *     "fls": [],
   *     "masked_fields": [],
   *     "allowed_actions": ["*"]
   *   }],
   *   "tenant_permissions": [{
   *     "tenant_patterns": ["*"],
   *     "allowed_actions": ["opensearch_dashboards_all_write"]
   *   }],
   *   "static": false
   * }
   *
   * for roles mapping, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "description": "Allow full access to all indices and all cluster APIs",
   *   "cluster_permissions": ["*"],
   *   "index_permissions": [{
   *     "index_patterns": ["*"],
   *     "fls": [],
   *     "masked_fields": [],
   *     "allowed_actions": ["*"]
   *   }],
   *   "tenant_permissions": [{
   *     "tenant_patterns": ["*"],
   *     "allowed_actions": ["opensearch_dashboards_all_write"]
   *   }],
   *   "static": false
   * }
   *
   * for action groups, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "allowed_actions": ["indices:data/read*", "indices:admin/mappings/fields/get*"],
   *   "type": "index",
   *   "description": "Allow all read operations",
   *   "static": false
   * }
   *
   * for tenant, response will look like:
   * {
   *   "reserved": true,
   *   "hidden": false,
   *   "description": "Global tenant",
   *   "static": false
   * },
   */
  router.get({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.getResource', {
        resourceName: request.params.resourceName,
        id: request.params.id
      });
      return response.ok({
        body: esResp[request.params.id]
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Deletes an entity by id.
   */
  router.delete({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string({
          minLength: 1
        })
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.deleteResource', {
        resourceName: request.params.resourceName,
        id: request.params.id
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Update object with out Id. Resource identification is expected to computed from headers. Eg: auth headers
   *
   * Request sample:
   * /configuration/account
   * {
   *   "password": "new-password",
   *   "current_password": "old-password"
   * }
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      validateRequestBody(request.params.resourceName, request.body);
    } catch (error) {
      return response.badRequest({
        body: error
      });
    }
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.saveResourceWithoutId', {
        resourceName: request.params.resourceName,
        body: request.body
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Update entity by Id.
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/{resourceName}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        resourceName: _configSchema.schema.string(),
        id: _configSchema.schema.string({
          validate: validateEntityId
        })
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      validateRequestBody(request.params.resourceName, request.body);
    } catch (error) {
      return response.badRequest({
        body: error
      });
    }
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.saveResource', {
        resourceName: request.params.resourceName,
        id: request.params.id,
        body: request.body
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets authentication info of the user.
   *
   * The response looks like:
   * {
   *   "user": "User [name=admin, roles=[], requestedTenant=__user__]",
   *   "user_name": "admin",
   *   "user_requested_tenant": "__user__",
   *   "remote_address": "127.0.0.1:35044",
   *   "backend_roles": [],
   *   "custom_attribute_names": [],
   *   "roles": ["all_access", "security_manager"],
   *   "tenants": {
   *     "another_tenant": true,
   *     "admin": true,
   *     "global_tenant": true,
   *     "aaaaa": true,
   *     "test tenant": true
   *   },
   *   "principal": null,
   *   "peer_certificates": "0",
   *   "sso_logout_url": null
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/auth/authinfo`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    let esResp;
    try {
      esResp = await client.callAsCurrentUser('opensearch_security.authinfo');
      return response.ok({
        body: esResp
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });
  router.get({
    path: `${_common.API_PREFIX}/auth/dashboardsinfo`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    let esResp;
    try {
      esResp = await client.callAsCurrentUser('opensearch_security.dashboardsinfo');
      return response.ok({
        body: esResp
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets audit log configuration。
   *
   * Sample payload:
   * {
   *   "enabled":true,
   *   "audit":{
   *     "enable_rest":false,
   *     "disabled_rest_categories":[
   *       "FAILED_LOGIN",
   *       "AUTHENTICATED"
   *     ],
   *     "enable_transport":true,
   *     "disabled_transport_categories":[
   *       "GRANTED_PRIVILEGES"
   *     ],
   *     "resolve_bulk_requests":true,
   *     "log_request_body":false,
   *     "resolve_indices":true,
   *     "exclude_sensitive_headers":true,
   *     "ignore_users":[
   *       "admin",
   *     ],
   *     "ignore_requests":[
   *       "SearchRequest",
   *       "indices:data/read/*"
   *     ]
   *   },
   *   "compliance":{
   *     "enabled":true,
   *     "internal_config":false,
   *     "external_config":false,
   *     "read_metadata_only":false,
   *     "read_watched_fields":{
   *       "indexName1":[
   *         "field1",
   *         "fields-*"
   *       ]
   *     },
   *     "read_ignore_users":[
   *       "opensearchdashboardsserver",
   *       "operator/*"
   *     ],
   *     "write_metadata_only":false,
   *     "write_log_diffs":false,
   *     "write_watched_indices":[
   *       "indexName2",
   *       "indexPatterns-*"
   *     ],
   *     "write_ignore_users":[
   *       "admin"
   *     ]
   *   }
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/configuration/audit`,
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.getAudit');
      return response.ok({
        body: esResp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode,
        body: parseEsErrorResponse(error)
      });
    }
  });

  /**
   * Update audit log configuration。
   *
   * Sample payload:
   * {
   *   "enabled":true,
   *   "audit":{
   *     "enable_rest":false,
   *     "disabled_rest_categories":[
   *       "FAILED_LOGIN",
   *       "AUTHENTICATED"
   *     ],
   *     "enable_transport":true,
   *     "disabled_transport_categories":[
   *       "GRANTED_PRIVILEGES"
   *     ],
   *     "resolve_bulk_requests":true,
   *     "log_request_body":false,
   *     "resolve_indices":true,
   *     "exclude_sensitive_headers":true,
   *     "ignore_users":[
   *       "admin",
   *     ],
   *     "ignore_requests":[
   *       "SearchRequest",
   *       "indices:data/read/*"
   *     ]
   *   },
   *   "compliance":{
   *     "enabled":true,
   *     "internal_config":false,
   *     "external_config":false,
   *     "read_metadata_only":false,
   *     "read_watched_fields":{
   *       "indexName1":[
   *         "field1",
   *         "fields-*"
   *       ]
   *     },
   *     "read_ignore_users":[
   *       "kibanaserver",
   *       "operator/*"
   *     ],
   *     "write_metadata_only":false,
   *     "write_log_diffs":false,
   *     "write_watched_indices":[
   *       "indexName2",
   *       "indexPatterns-*"
   *     ],
   *     "write_ignore_users":[
   *       "admin"
   *     ]
   *   }
   * }
   */
  router.post({
    path: `${_common.API_PREFIX}/configuration/audit/config`,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.saveAudit', {
        body: request.body
      });
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Deletes cache.
   *
   * Sample response: {"message":"Cache flushed successfully."}
   */
  router.delete({
    path: `${_common.API_PREFIX}/configuration/cache`,
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const esResp = await wrapRouteWithDataSource(dataSourceEnabled, context, request, 'opensearch_security.clearCache');
      return response.ok({
        body: {
          message: esResp.message
        }
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets permission info of current user.
   *
   * Sample response:
   * {
   *   "user": "User [name=admin, roles=[], requestedTenant=__user__]",
   *   "user_name": "admin",
   *   "has_api_access": true,
   *   "disabled_endpoints": {}
   * }
   */
  router.get({
    path: `${_common.API_PREFIX}/restapiinfo`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.restapiinfo');
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return response.badRequest({
        body: error
      });
    }
  });

  /**
   * Validates DLS (document level security) query.
   *
   * Request payload is an ES query.
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/validatedls/{indexName}`,
    validate: {
      params: _configSchema.schema.object({
        // in legacy plugin implmentation, indexName is not used when calling ES API.
        indexName: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.validateDls', {
        body: request.body
      });
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets index mapping.
   *
   * Calling ES _mapping API under the hood. see
   * https://www.elastic.co/guide/en/elasticsearch/reference/current/indices-get-mapping.html
   */
  router.post({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/index_mappings`,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.getIndexMappings', {
        index: request.body.index.join(','),
        ignore_unavailable: true,
        allow_no_indices: true
      });
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });

  /**
   * Gets all indices, and field mappings.
   *
   * Calls ES API '/_all/_mapping/field/*' under the hood. see
   * https://www.elastic.co/guide/en/elasticsearch/reference/current/indices-get-mapping.html
   */
  router.get({
    path: `${_common.API_PREFIX}/${_common.CONFIGURATION_API_PREFIX}/indices`,
    validate: false
  }, async (context, request, response) => {
    const client = context.security_plugin.esClient.asScoped(request);
    try {
      const esResponse = await client.callAsCurrentUser('opensearch_security.indices');
      return response.ok({
        body: esResponse
      });
    } catch (error) {
      return errorResponse(response, error);
    }
  });
}

/**
 * A helper function to wrap API calls with the appropriate client. If the multiple datasources feature is disabled, it will use
 * the existing client provided by the security plugin. Otherwise, it will use the one provided by the datasource plugin based on the id
 * that we extract via the UI.
 */
const wrapRouteWithDataSource = async (dataSourceEnabled, context, request, endpoint, body) => {
  var _request$query;
  if (!dataSourceEnabled || !((_request$query = request.query) !== null && _request$query !== void 0 && _request$query.dataSourceId)) {
    const client = context.security_plugin.esClient.asScoped(request);
    return await client.callAsCurrentUser(endpoint, body);
  } else {
    var _request$query2;
    const client = context.dataSource.opensearch.legacy.getClient((_request$query2 = request.query) === null || _request$query2 === void 0 ? void 0 : _request$query2.dataSourceId);
    return await client.callAPI(endpoint, body);
  }
};
function parseEsErrorResponse(error) {
  if (error.response) {
    try {
      const esErrorResponse = JSON.parse(error.response);
      return esErrorResponse.reason || error.response;
    } catch (parsingError) {
      return error.response;
    }
  }
  return error.message;
}
function errorResponse(response, error) {
  return response.custom({
    statusCode: error.statusCode,
    body: parseEsErrorResponse(error)
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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