﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sso-admin/SSOAdminRequest.h>
#include <aws/sso-admin/SSOAdmin_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSOAdmin {
namespace Model {

/**
 */
class ListAccountAssignmentsRequest : public SSOAdminRequest {
 public:
  AWS_SSOADMIN_API ListAccountAssignmentsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAccountAssignments"; }

  AWS_SSOADMIN_API Aws::String SerializePayload() const override;

  AWS_SSOADMIN_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the IAM Identity Center instance under which the operation will be
   * executed. For more information about ARNs, see <a
   * href="/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
   * (ARNs) and Amazon Web Services Service Namespaces</a> in the <i>Amazon Web
   * Services General Reference</i>.</p>
   */
  inline const Aws::String& GetInstanceArn() const { return m_instanceArn; }
  inline bool InstanceArnHasBeenSet() const { return m_instanceArnHasBeenSet; }
  template <typename InstanceArnT = Aws::String>
  void SetInstanceArn(InstanceArnT&& value) {
    m_instanceArnHasBeenSet = true;
    m_instanceArn = std::forward<InstanceArnT>(value);
  }
  template <typename InstanceArnT = Aws::String>
  ListAccountAssignmentsRequest& WithInstanceArn(InstanceArnT&& value) {
    SetInstanceArn(std::forward<InstanceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Web Services account from which to list the
   * assignments.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  ListAccountAssignmentsRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the permission set from which to list assignments.</p>
   */
  inline const Aws::String& GetPermissionSetArn() const { return m_permissionSetArn; }
  inline bool PermissionSetArnHasBeenSet() const { return m_permissionSetArnHasBeenSet; }
  template <typename PermissionSetArnT = Aws::String>
  void SetPermissionSetArn(PermissionSetArnT&& value) {
    m_permissionSetArnHasBeenSet = true;
    m_permissionSetArn = std::forward<PermissionSetArnT>(value);
  }
  template <typename PermissionSetArnT = Aws::String>
  ListAccountAssignmentsRequest& WithPermissionSetArn(PermissionSetArnT&& value) {
    SetPermissionSetArn(std::forward<PermissionSetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to display for the assignment.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAccountAssignmentsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token for the list API. Initially the value is null. Use the
   * output of previous API calls to make subsequent calls.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAccountAssignmentsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceArn;

  Aws::String m_accountId;

  Aws::String m_permissionSetArn;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_instanceArnHasBeenSet = false;
  bool m_accountIdHasBeenSet = false;
  bool m_permissionSetArnHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSOAdmin
}  // namespace Aws
