﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class DescribeAssociationRequest : public SSMRequest {
 public:
  AWS_SSM_API DescribeAssociationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeAssociation"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the SSM document.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  DescribeAssociationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The managed node ID.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  DescribeAssociationRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The association ID for which you want information.</p>
   */
  inline const Aws::String& GetAssociationId() const { return m_associationId; }
  inline bool AssociationIdHasBeenSet() const { return m_associationIdHasBeenSet; }
  template <typename AssociationIdT = Aws::String>
  void SetAssociationId(AssociationIdT&& value) {
    m_associationIdHasBeenSet = true;
    m_associationId = std::forward<AssociationIdT>(value);
  }
  template <typename AssociationIdT = Aws::String>
  DescribeAssociationRequest& WithAssociationId(AssociationIdT&& value) {
    SetAssociationId(std::forward<AssociationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the association version to retrieve. To view the latest version,
   * either specify <code>$LATEST</code> for this parameter, or omit this parameter.
   * To view a list of all associations for a managed node, use
   * <a>ListAssociations</a>. To get a list of versions for a specific association,
   * use <a>ListAssociationVersions</a>. </p>
   */
  inline const Aws::String& GetAssociationVersion() const { return m_associationVersion; }
  inline bool AssociationVersionHasBeenSet() const { return m_associationVersionHasBeenSet; }
  template <typename AssociationVersionT = Aws::String>
  void SetAssociationVersion(AssociationVersionT&& value) {
    m_associationVersionHasBeenSet = true;
    m_associationVersion = std::forward<AssociationVersionT>(value);
  }
  template <typename AssociationVersionT = Aws::String>
  DescribeAssociationRequest& WithAssociationVersion(AssociationVersionT&& value) {
    SetAssociationVersion(std::forward<AssociationVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_instanceId;

  Aws::String m_associationId;

  Aws::String m_associationVersion;
  bool m_nameHasBeenSet = false;
  bool m_instanceIdHasBeenSet = false;
  bool m_associationIdHasBeenSet = false;
  bool m_associationVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
