﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/route53/Route53_EXPORTS.h>
#include <aws/route53/model/ComparisonOperator.h>
#include <aws/route53/model/Dimension.h>
#include <aws/route53/model/Statistic.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Route53 {
namespace Model {

/**
 * <p>A complex type that contains information about the CloudWatch alarm that
 * Amazon Route 53 is monitoring for this health check.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/route53-2013-04-01/CloudWatchAlarmConfiguration">AWS
 * API Reference</a></p>
 */
class CloudWatchAlarmConfiguration {
 public:
  AWS_ROUTE53_API CloudWatchAlarmConfiguration() = default;
  AWS_ROUTE53_API CloudWatchAlarmConfiguration(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_ROUTE53_API CloudWatchAlarmConfiguration& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_ROUTE53_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>For the metric that the CloudWatch alarm is associated with, the number of
   * periods that the metric is compared to the threshold.</p>
   */
  inline int GetEvaluationPeriods() const { return m_evaluationPeriods; }
  inline bool EvaluationPeriodsHasBeenSet() const { return m_evaluationPeriodsHasBeenSet; }
  inline void SetEvaluationPeriods(int value) {
    m_evaluationPeriodsHasBeenSet = true;
    m_evaluationPeriods = value;
  }
  inline CloudWatchAlarmConfiguration& WithEvaluationPeriods(int value) {
    SetEvaluationPeriods(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For the metric that the CloudWatch alarm is associated with, the value the
   * metric is compared with.</p>
   */
  inline double GetThreshold() const { return m_threshold; }
  inline bool ThresholdHasBeenSet() const { return m_thresholdHasBeenSet; }
  inline void SetThreshold(double value) {
    m_thresholdHasBeenSet = true;
    m_threshold = value;
  }
  inline CloudWatchAlarmConfiguration& WithThreshold(double value) {
    SetThreshold(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For the metric that the CloudWatch alarm is associated with, the arithmetic
   * operation that is used for the comparison.</p>
   */
  inline ComparisonOperator GetComparisonOperator() const { return m_comparisonOperator; }
  inline bool ComparisonOperatorHasBeenSet() const { return m_comparisonOperatorHasBeenSet; }
  inline void SetComparisonOperator(ComparisonOperator value) {
    m_comparisonOperatorHasBeenSet = true;
    m_comparisonOperator = value;
  }
  inline CloudWatchAlarmConfiguration& WithComparisonOperator(ComparisonOperator value) {
    SetComparisonOperator(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For the metric that the CloudWatch alarm is associated with, the duration of
   * one evaluation period in seconds.</p>
   */
  inline int GetPeriod() const { return m_period; }
  inline bool PeriodHasBeenSet() const { return m_periodHasBeenSet; }
  inline void SetPeriod(int value) {
    m_periodHasBeenSet = true;
    m_period = value;
  }
  inline CloudWatchAlarmConfiguration& WithPeriod(int value) {
    SetPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the CloudWatch metric that the alarm is associated with.</p>
   */
  inline const Aws::String& GetMetricName() const { return m_metricName; }
  inline bool MetricNameHasBeenSet() const { return m_metricNameHasBeenSet; }
  template <typename MetricNameT = Aws::String>
  void SetMetricName(MetricNameT&& value) {
    m_metricNameHasBeenSet = true;
    m_metricName = std::forward<MetricNameT>(value);
  }
  template <typename MetricNameT = Aws::String>
  CloudWatchAlarmConfiguration& WithMetricName(MetricNameT&& value) {
    SetMetricName(std::forward<MetricNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace of the metric that the alarm is associated with. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
   * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon
   * CloudWatch User Guide</i>.</p>
   */
  inline const Aws::String& GetNamespace() const { return m_namespace; }
  inline bool NamespaceHasBeenSet() const { return m_namespaceHasBeenSet; }
  template <typename NamespaceT = Aws::String>
  void SetNamespace(NamespaceT&& value) {
    m_namespaceHasBeenSet = true;
    m_namespace = std::forward<NamespaceT>(value);
  }
  template <typename NamespaceT = Aws::String>
  CloudWatchAlarmConfiguration& WithNamespace(NamespaceT&& value) {
    SetNamespace(std::forward<NamespaceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For the metric that the CloudWatch alarm is associated with, the statistic
   * that is applied to the metric.</p>
   */
  inline Statistic GetStatistic() const { return m_statistic; }
  inline bool StatisticHasBeenSet() const { return m_statisticHasBeenSet; }
  inline void SetStatistic(Statistic value) {
    m_statisticHasBeenSet = true;
    m_statistic = value;
  }
  inline CloudWatchAlarmConfiguration& WithStatistic(Statistic value) {
    SetStatistic(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For the metric that the CloudWatch alarm is associated with, a complex type
   * that contains information about the dimensions for the metric. For information,
   * see <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/DeveloperGuide/CW_Support_For_AWS.html">Amazon
   * CloudWatch Namespaces, Dimensions, and Metrics Reference</a> in the <i>Amazon
   * CloudWatch User Guide</i>.</p>
   */
  inline const Aws::Vector<Dimension>& GetDimensions() const { return m_dimensions; }
  inline bool DimensionsHasBeenSet() const { return m_dimensionsHasBeenSet; }
  template <typename DimensionsT = Aws::Vector<Dimension>>
  void SetDimensions(DimensionsT&& value) {
    m_dimensionsHasBeenSet = true;
    m_dimensions = std::forward<DimensionsT>(value);
  }
  template <typename DimensionsT = Aws::Vector<Dimension>>
  CloudWatchAlarmConfiguration& WithDimensions(DimensionsT&& value) {
    SetDimensions(std::forward<DimensionsT>(value));
    return *this;
  }
  template <typename DimensionsT = Dimension>
  CloudWatchAlarmConfiguration& AddDimensions(DimensionsT&& value) {
    m_dimensionsHasBeenSet = true;
    m_dimensions.emplace_back(std::forward<DimensionsT>(value));
    return *this;
  }
  ///@}
 private:
  int m_evaluationPeriods{0};

  double m_threshold{0.0};

  ComparisonOperator m_comparisonOperator{ComparisonOperator::NOT_SET};

  int m_period{0};

  Aws::String m_metricName;

  Aws::String m_namespace;

  Statistic m_statistic{Statistic::NOT_SET};

  Aws::Vector<Dimension> m_dimensions;
  bool m_evaluationPeriodsHasBeenSet = false;
  bool m_thresholdHasBeenSet = false;
  bool m_comparisonOperatorHasBeenSet = false;
  bool m_periodHasBeenSet = false;
  bool m_metricNameHasBeenSet = false;
  bool m_namespaceHasBeenSet = false;
  bool m_statisticHasBeenSet = false;
  bool m_dimensionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Route53
}  // namespace Aws
