﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/migration-hub-refactor-spaces/MigrationHubRefactorSpacesRequest.h>
#include <aws/migration-hub-refactor-spaces/MigrationHubRefactorSpaces_EXPORTS.h>
#include <aws/migration-hub-refactor-spaces/model/ApiGatewayProxyInput.h>
#include <aws/migration-hub-refactor-spaces/model/ProxyType.h>

#include <utility>

namespace Aws {
namespace MigrationHubRefactorSpaces {
namespace Model {

/**
 */
class CreateApplicationRequest : public MigrationHubRefactorSpacesRequest {
 public:
  AWS_MIGRATIONHUBREFACTORSPACES_API CreateApplicationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateApplication"; }

  AWS_MIGRATIONHUBREFACTORSPACES_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A wrapper object holding the API Gateway endpoint type and stage name for the
   * proxy. </p>
   */
  inline const ApiGatewayProxyInput& GetApiGatewayProxy() const { return m_apiGatewayProxy; }
  inline bool ApiGatewayProxyHasBeenSet() const { return m_apiGatewayProxyHasBeenSet; }
  template <typename ApiGatewayProxyT = ApiGatewayProxyInput>
  void SetApiGatewayProxy(ApiGatewayProxyT&& value) {
    m_apiGatewayProxyHasBeenSet = true;
    m_apiGatewayProxy = std::forward<ApiGatewayProxyT>(value);
  }
  template <typename ApiGatewayProxyT = ApiGatewayProxyInput>
  CreateApplicationRequest& WithApiGatewayProxy(ApiGatewayProxyT&& value) {
    SetApiGatewayProxy(std::forward<ApiGatewayProxyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateApplicationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the environment.</p>
   */
  inline const Aws::String& GetEnvironmentIdentifier() const { return m_environmentIdentifier; }
  inline bool EnvironmentIdentifierHasBeenSet() const { return m_environmentIdentifierHasBeenSet; }
  template <typename EnvironmentIdentifierT = Aws::String>
  void SetEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    m_environmentIdentifierHasBeenSet = true;
    m_environmentIdentifier = std::forward<EnvironmentIdentifierT>(value);
  }
  template <typename EnvironmentIdentifierT = Aws::String>
  CreateApplicationRequest& WithEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    SetEnvironmentIdentifier(std::forward<EnvironmentIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name to use for the application. </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateApplicationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The proxy type of the proxy created within the application. </p>
   */
  inline ProxyType GetProxyType() const { return m_proxyType; }
  inline bool ProxyTypeHasBeenSet() const { return m_proxyTypeHasBeenSet; }
  inline void SetProxyType(ProxyType value) {
    m_proxyTypeHasBeenSet = true;
    m_proxyType = value;
  }
  inline CreateApplicationRequest& WithProxyType(ProxyType value) {
    SetProxyType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to assign to the application. A tag is a label that you assign to an
   * Amazon Web Services resource. Each tag consists of a key-value pair.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateApplicationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateApplicationRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the virtual private cloud (VPC).</p>
   */
  inline const Aws::String& GetVpcId() const { return m_vpcId; }
  inline bool VpcIdHasBeenSet() const { return m_vpcIdHasBeenSet; }
  template <typename VpcIdT = Aws::String>
  void SetVpcId(VpcIdT&& value) {
    m_vpcIdHasBeenSet = true;
    m_vpcId = std::forward<VpcIdT>(value);
  }
  template <typename VpcIdT = Aws::String>
  CreateApplicationRequest& WithVpcId(VpcIdT&& value) {
    SetVpcId(std::forward<VpcIdT>(value));
    return *this;
  }
  ///@}
 private:
  ApiGatewayProxyInput m_apiGatewayProxy;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_environmentIdentifier;

  Aws::String m_name;

  ProxyType m_proxyType{ProxyType::NOT_SET};

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_vpcId;
  bool m_apiGatewayProxyHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_environmentIdentifierHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_proxyTypeHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_vpcIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace MigrationHubRefactorSpaces
}  // namespace Aws
